require("util.nut");

class SnakeAI extends AIController 
{
	maxVehicles=2500;
    ut = Util();
    function SetCompanyName();
    function AttemptToConnectIndustry();
    function BuildRoadStop(location, type, station);
    function BuildRoadDepotNear(location);
	function TweakVehicleNumber();
	function FindPosNear();
	function ConnectIndToInd(in1, ind2, cargo);
    function Start();
}

function SnakeAI::SetCompanyName()
{
    if (!AICompany.SetName("Snake")) {
        local i = 2;
        while (!AICompany.SetName("Snake #" + i)) {
            i = i + 1;
        }
    }
}

function SnakeAI::ConnectToNearestRoad(pre, location, maxIterations){
	foreach(sign, value in AISignList()){
		if (AITile.GetDistanceManhattanToTile(AISign.GetLocation(sign), location)<30){
			AISign.RemoveSign(sign);
		}
	}


	local roads=[ ];
	local oldnewroads = [[pre, location]]
	if ((!ut.IsClearableTile(location) && !AIRoad.IsRoadTile(location)) || AITile.GetSlope(location)!=AITile.SLOPE_FLAT){
		return;
	}
	local done=false;
	for (local i=0; i<maxIterations && !done; i++){
		local newroads = [];
		foreach (road in oldnewroads){
			foreach (tile in ut.GetDirectlyAdjacentTiles(road[1])){
				local found=false;
				foreach(road2 in roads){
					if (road2[1]==tile){
						found=true;
						break;
					}
				}
				if (found){
				
				}
				else if (tile!=location && !AIRoad.IsRoadStationTile(tile) && AIRoad.IsRoadTile(tile) &&
					    AIRoad.CanBuildConnectedRoadPartsHere(road[1], road[0], tile)){
					//AISign.BuildSign(tile, "final");
					newroads.push([road[1],tile]);
					done=true;
					break;
				}
				else if (ut.IsClearableTile(tile) && AIRoad.CanBuildConnectedRoadPartsHere(road[1], road[0], tile)){
					//AISign.BuildSign(tile, "not final");
					newroads.push([road[1],tile]);
				}
				else {
				}
			}
			
			if (done) {
				break;
			}
		}
		foreach (road in oldnewroads){
			roads.push(road);
		}
		oldnewroads=[];
		foreach (road in newroads){
			oldnewroads.push(road);
		}
	}
	
	if (done) {
		local lastRoad = oldnewroads[oldnewroads.len() - 1];
		ut.TryBuildRoad(lastRoad[0], lastRoad[1]);
		while (lastRoad[0] != roads[0][1]){
			local firstTile = null;
			foreach (road in roads){
				if (road[1]==lastRoad[0]){
					ut.TryBuildRoad(road[0], road[1]);
					lastRoad = road;
					break;
				}
			}
		}
		return true;
	}
	else {
		return false;
	}
}

function SnakeAI::BuildRoadStop(location, type, station) {
	if (AIRoad.IsRoadStationTile(location)){
		local adjacentTiles = ut.GetDirectlyAdjacentTiles(location);
		foreach (l2 in adjacentTiles){
			if (ut.IsClearableTile(l2) && AITile.GetSlope(l2)==AITile.SLOPE_FLAT){
				if (AIRoad.BuildDriveThroughRoadStation(l2, l2 + AIRoad.GetRoadStationFrontTile(location) - location
					, type, station)){
					ConnectToNearestRoad(l2, AIRoad.GetDriveThroughBackTile(l2), 9);
					ConnectToNearestRoad(l2, AIRoad.GetRoadStationFrontTile(l2), 9);
					
					ut.TryBuildRoad(l2, AIRoad.GetDriveThroughBackTile(l2));
					ut.TryBuildRoad(l2, AIRoad.GetRoadStationFrontTile(l2));
					return location;
				}
				else {
					AILog.Warning("failed to build road station: "+AIError.GetLastErrorString());
				}
			}
		}
		foreach (location in adjacentTiles){
			if (Util.IsStraightRoadTile(location) && !AIRoad.IsRoadStationTile(location)) {
				local tries=0;
				while (tries<10){
					tries+=5;
					if (BuildRoadStop(location, type, station)){
						tries=200;
					}
					else if (AIError.GetLastError()==AIError.ERR_VEHICLE_IN_THE_WAY){
						tries-=4;
						AILog.Info("vehicle in the way, trying again...");
						this.Sleep(15);
					}
					else {
						AILog.Info("failed to build station, "+AIError.GetLastErrorString());
					}
				};
				break;
			}
			
		}
		return location;
	}

    local adjacentTiles = ut.GetDirectlyAdjacentTiles(location);
    foreach (tile in adjacentTiles){
        if (AIRoad.AreRoadTilesConnected(location, tile)){
            local station = AIRoad.BuildDriveThroughRoadStation(location,tile,type,station);
			local backTile = AIRoad.GetDriveThroughBackTile(location);
			if (!ut.IsClearableTile(backTile)){
				backTile = AIRoad.GetRoadStationFrontTile(location);
			}
			if (ut.IsClearableTile(backTile) && AITile.GetSlope(backTile) == AITile.SLOPE_FLAT){
				
				ut.TryBuildRoad(location, backTile);
				ConnectToNearestRoad(location, backTile, 8);
			}
			else {
			}
			return station;
        }
    }
}

function SnakeAI::BuildRoadDepotNear(location) {
    local finishedTiles = [];
    local deadEnds = [location];
    local newDeadEnds = []
	local numIterations = 0;
    while (deadEnds.len() != 0 && numIterations < 50) {
		numIterations ++;
		if (numIterations == 10){
			AILog.Info("hmm, finding a depo seems to be taking longer than expected");
		}
		if (numIterations % 10 == 0){
			this.Sleep(1);
		}
		
        foreach (deadEnd in deadEnds){
            local surround = ut.GetDirectlyAdjacentTiles(deadEnd)
            foreach(tile in surround){
				if (AIRoad.IsRoadDepotTile(tile)
					&& AITile.GetOwner(tile) == AICompany.ResolveCompanyID(AICompany.COMPANY_SELF)){
					return tile;
				}
				else if (AIRoad.IsRoadDepotTile(tile)) {
					AILog.Info("tile is a depo, but owned by "+AITile.GetOwner(tile)+
					" (AM "+AICompany.ResolveCompanyID(AICompany.COMPANY_SELF)+")");
				}
				else if (AIRoad.AreRoadTilesConnected(deadEnd, tile) && AIBridge.IsBridgeTile(tile)) {
					local edge = AIBridge.GetOtherBridgeEnd(tile);
					if (ut.ArrayFind(finishedTiles,edge) || ut.ArrayFind(deadEnds,edge)){
                        //AISign.BuildSign(edge, "unnew dead end (bridge)");
                    }
                    else {
                        newDeadEnds.push(edge);
                        //AISign.BuildSign(edge, "new dead end (bridge)");
                    
                    }
				}
				else if (AIRoad.AreRoadTilesConnected(deadEnd, tile) && AITunnel.IsTunnelTile(tile)){
					local edge = AITunnel.GetOtherTunnelEnd(tile);
					if (ut.ArrayFind(finishedTiles,edge) || ut.ArrayFind(deadEnds,edge)){
                        //AISign.BuildSign(edge, "unnew dead end (bridge)");
                    }
                    else {
                        newDeadEnds.push(edge);
                        //AISign.BuildSign(edge, "new dead end (bridge)");
                    
                    }
				}
                else if (AIRoad.AreRoadTilesConnected(deadEnd, tile)){
                    
                    if (ut.ArrayFind(finishedTiles,tile) || ut.ArrayFind(deadEnds,tile)){
                        //AISign.BuildSign(tile, "unnew dead end");
                    }
                    else {
                        newDeadEnds.push(tile);
                        //AISign.BuildSign(tile, "new dead end ("+numIterations+")");
                    
                    }
                }
                else if (
						numIterations >= 5 &&
                         ut.IsClearableTile(tile) &&
                        !AIRoad.IsRoadStationTile(deadEnd) &&
						!AIRoad.IsRoadDepotTile(tile) &&
						!AIBridge.IsBridgeTile(deadEnd) &&
                         AITile.GetSlope(deadEnd)==AITile.SLOPE_FLAT &&
                         AITile.GetSlope(tile)==AITile.SLOPE_FLAT)
                {
					
                    if (AIRoad.BuildRoadDepot(tile, deadEnd)){
						ut.TryBuildRoad(deadEnd, tile);
						return tile;
					}
					else {
						AILog.Info("failed building depo: "+AIError.GetLastErrorString());
					}
                }
                else {
                    if (numIterations >= 5){
						if (!ut.IsClearableTile(tile)){
							//AISign.BuildSign(tile, "not clearable ("+numIterations+")");
						}
						else if (AIRoad.IsRoadStationTile(deadEnd)){
							//AISign.BuildSign(tile, "no junction possible ("+numIterations+")");
						}
						else if (AITile.GetSlope(tile)!=AITile.SLOPE_FLAT ||
							AITile.GetSlope(deadEnd)!=AITile.SLOPE_FLAT){
							//AISign.BuildSign(tile, "not flat ("+numIterations+")");
						}
						else {
							AISign.BuildSign(tile, "invalid ("+numIterations+")");
						}
					}
                }
            }
        }
        foreach(deadEnd in deadEnds){
            //AISign.BuildSign(deadEnd, "dead end");
            finishedTiles.push(deadEnd);
        }
        deadEnds=newDeadEnds;
        newDeadEnds=[];
    }
	
	AILog.Info("Depo build failed, "+finishedTiles.len()+" tiles tried");
	return 0;
    
}

function SnakeAI::ConnectIndustryCargo(industry1, cargo) {
    //local sign = 0;
    local industryList = AIIndustryList();
    industryList.Valuate(AIIndustry.IsCargoAccepted, cargo);
    industryList.KeepValue(AIIndustry.CAS_ACCEPTED);
    local in1loc = AIIndustry.GetLocation(industry1);
	industryList.RemoveItem(industry1);
	industryList.Valuate(AIIndustry.GetDistanceManhattanToTile, in1loc);
    industryList.KeepBelowValue(100);
	industryList.KeepAboveValue(10);
    industryList.Sort(AIList.SORT_BY_VALUE, true);
    
    if (industryList.IsEmpty()){
        AILog.Info("Industry failed: "+AIIndustry.GetName(industry1) + "(" + AICargo.GetCargoLabel(cargo)+")");
        return false;
    }
	
	return ConnectIndToInd(industry1, industryList.Begin(), cargo);
}

function SnakeAI::FindPosNear(industry, near, stopType, ts) {
	local tileset = ts(industry, AIStation.GetCoverageRadius(stopType));
    tileset = ut.FilterBuildableTiles(tileset, true);
    tileset.Valuate(AITile.GetDistanceManhattanToTile, near);
    tileset.Sort(AIList.SORT_BY_VALUE, true);
	local idealLoc = tileset.Begin();
	
	local tileset = ts(industry, AIStation.GetCoverageRadius(stopType));
	tileset.Valuate(AITile.GetOwner);
	tileset.KeepValue(AICompany.ResolveCompanyID(AICompany.COMPANY_SELF));
	tileset.Valuate(AITile.GetDistanceManhattanToTile, idealLoc);
	tileset.KeepBelowValue(8);
	tileset.Valuate(AIRoad.IsRoadStationTile);
	tileset.KeepValue(1);
	if (tileset.IsEmpty()){
		return [idealLoc, AIStation.STATION_JOIN_ADJACENT];
	}
	tileset.Valuate(AIStation.GetStationID);
	foreach (tile, station in tileset){
		if (AIStation.HasStationType(station, stopType)){
			return [tile, station];
		}
	}
	return [idealLoc, AIStation.STATION_JOIN_ADJACENT];
}

function SnakeAI::ConnectIndToInd(industry1, industry2, cargo) {
	local stopType = AICargo.HasCargoClass(cargo, AICargo.CC_PASSENGERS) ? AIStation.STATION_BUS_STOP : AIStation.STATION_TRUCK_STOP;
    local in1loc = AIIndustry.GetLocation(industry1);
    local in2loc = AIIndustry.GetLocation(industry2);
    
    in1loc = FindPosNear(industry1, in2loc, stopType, AITileList_IndustryProducing);
	local in1station = in1loc[1];
	in1loc = in1loc[0];
    in2loc = FindPosNear(industry2, in1loc, stopType, AITileList_IndustryAccepting);
	local in2station = in2loc[1];
	in2loc = in2loc[0];
    if (in1loc != null && AIMap.IsValidTile(in1loc)){
        //sign = AISign.BuildSign(in1loc, "loc1");
    }
    else {
        AILog.Info("no build location");
        return false;
    }
    if (in2loc != null && AIMap.IsValidTile(in2loc)){
        //AISign.BuildSign(in1loc, "loc2");
    }
    else {
        //if (AISign.IsValidSign(sign)){
            //AILog.Info("no accepting location");
            //AISign.RemoveSign(sign);
        //}
        return false;
    }
    
    if (in1loc != null && AIMap.IsValidTile(in1loc) && in2loc != null && AIMap.IsValidTile(in2loc)){
    
		AILog.Info("from "+AIIndustry.GetName(industry1)+" to "+AIIndustry.GetName(industry2));
        if (ut.BuildRoute(in1loc,in2loc)){
			
            local station1 = BuildRoadStop(in1loc,
				(stopType == AIStation.STATION_BUS_STOP ? AIRoad.ROADVEHTYPE_BUS : AIRoad.ROADVEHTYPE_TRUCK)
			    , in1station);
			if (!AIRoad.IsRoadStationTile(in1loc)){
				return false;
			}
			local engine = ut.FindRoadVehicle(cargo);
			if (AIEngine.IsValidEngine(engine) && AIEngine.IsBuildable(engine)){
				BuildRoadStop(in2loc,
							 (stopType == AIStation.STATION_BUS_STOP ? AIRoad.ROADVEHTYPE_BUS : AIRoad.ROADVEHTYPE_TRUCK)
							, in2station);
				if (!AIRoad.IsRoadStationTile(in2loc)){
					return false;
				}
				local depotTile = BuildRoadDepotNear(in1loc);
				if (depotTile != null && AIMap.IsValidTile(depotTile) && ReserveMoney(10000)){
					local vehicle = AIVehicle.BuildVehicle(depotTile, engine);
					if (AIVehicle.IsValidVehicle(vehicle)){
						AIVehicle.RefitVehicle(vehicle, cargo);
						local name = ut.CapLength(AICargo.GetCargoLabel(cargo), 4)+" "
								     +ut.CapLength(AIIndustry.GetName(industry1),10)+" to "
									 +ut.CapLength(AIIndustry.GetName(industry2),10)
						if (!AIVehicle.SetName(vehicle, name)){
							AILog.Info("name change to \""+name+"\" failed: "+AIError.GetLastErrorString());
						}
						AIOrder.AppendOrder(vehicle, in1loc, AIOrder.OF_FULL_LOAD);
						AIVehicle.StartStopVehicle(vehicle);
						AIOrder.AppendOrder(vehicle, in2loc, AIOrder.OF_UNLOAD);
						
						for (local i=0; i<3; i++){
							vehicle = AIVehicle.CloneVehicle(depotTile, vehicle, true);
							if (AIError.GetLastError() == AIVehicle.ERR_VEHICLE_TOO_MANY){
								AILog.Info("too many vehicles");
								maxVehicles = AIVehicleList().Count();
								return true; //don't try again
							}
							if (AIVehicle.IsStoppedInDepot(vehicle)){
								AIVehicle.StartStopVehicle(vehicle);
							}
							this.Sleep(10);
						}
						return true;
					}
					else {
						return false;
					}
				}
				else {
					return false;
				}
			}
			else {
				return false;
			}
        }
        else {
            return false;
        }
    }
}

function SnakeAI::AttemptToConnectIndustry() {
	local cargoList = AICargoList();
    local industries = AIIndustryList();
    industries.Valuate(ut.IndustryTotalCargo);
    industries.Sort(AIList.SORT_BY_VALUE, false);
    local industry = industries.Begin();
    local done=false;
    while (!industries.IsEnd() && !done) {
        foreach (cargo, prod in cargoList){
			if (AIIndustry.GetLastMonthProduction(industry, cargo)>40 &&
			        ut.IsCargoPickedUp(industry, cargo) &&
			        AIIndustry.GetLastMonthTransportedPercentage(industry, cargo) == 0 &&
			        ReserveMoney(50000)) {
				if (ConnectIndustryCargo(industry, cargo)){
					done = true;
					break;
				}
			}
		}
        industry = industries.Next();
        this.Sleep(1);
    }
	if (done){
	}
    
}

function SnakeAI::DeleteNegativeIncomeVehicles(){
	local busses=AIVehicleList();
	busses.Valuate(AIVehicle.GetProfitThisYear);
	busses.KeepBelowValue(-150);
	busses.Valuate(AIVehicle.GetProfitLastYear);
	busses.KeepBelowValue(-150);
	foreach (bus, n in busses){
		if (AIOrder.IsCurrentOrderPartOfOrderList(bus)){
			AIVehicle.SendVehicleToDepot(bus);
			if (AIVehicle.GetState(bus) == AIVehicle.VS_STOPPED && !AIVehicle.IsStoppedInDepot(bus)){
				AIVehicle.StartStopVehicle(bus);
			}
			AILog.Info("Sending \""+AIVehicle.GetName(bus)+"\" to depo");
			local i=0;
			while (!AIVehicle.SetName(bus, "delete"+i)){
				i++;
			};
			AIVehicle.ReverseVehicle(bus);
		}
	}
	
	busses=AIVehicleList();
	busses.Valuate(AIVehicle.IsStoppedInDepot);
	busses.KeepValue(1);
	busses.Valuate(AIVehicle.GetProfitLastYear);
	busses.KeepBelowValue(-250);
	foreach (bus, n in busses){
		if (ut.CapLength(AIVehicle.GetName(bus),6) == "delete"){
			AILog.Info("Selling \""+AIVehicle.GetName(bus));
			AIVehicle.SellVehicle(bus);
		}
		else {
			AILog.Info("not selling "+ut.CapLength(AIVehicle.GetName(bus),6));
		}
	}
}

function SnakeAI::TweakVehicleNumber() {
	local stations = AIStationList(AIStation.STATION_TRUCK_STOP);
	stations.AddList(AIStationList(AIStation.STATION_BUS_STOP));
	local cargo = AICargoList();
	cargo.Valuate(ut.FindRoadVehicle);
	
	stations.Valuate(ut.StationWaitingCargo);
	stations.KeepAboveValue(40);
	stations.Sort(AIList.SORT_BY_VALUE, false);
	if (stations.IsEmpty()){
		return;
	}
	else {
		foreach (most, x in stations){ //x is unused
			foreach (gr, engine in cargo){
				local amount = AIStation.GetCargoWaiting(most, gr);
				if (amount <=40){
					continue;
				}
				local capacity = AIEngine.GetCapacity(engine);
				local date = AIDate.GetCurrentDate();
				local vehicleList = AIVehicleList_Station(most);
				vehicleList.Valuate(ut.CarriesCargo, gr);
				vehicleList.KeepValue(1);
				vehicleList.Valuate(AIVehicle.GetAge);
				vehicleList.Sort(AIList.SORT_BY_VALUE, true);
				local vehicleNumber=vehicleList.Count()
				if (amount > capacity * (vehicleNumber)){
					if (vehicleList.IsEmpty()){
						//AILog.Info("no vehicle available");
					}
					else {
						local vehicle = vehicleList.Begin();
						if (AIVehicle.GetProfitThisYear(vehicle)-500<=0 && AIVehicle.GetProfitLastYear(vehicle)-1000<=0){
							//AILog.Info("not enough profit");
						}
						else if (AIVehicle.GetAge(vehicle)>50+(10*vehicleList.Count())){
							if (ReserveMoney(10000)){
								AILog.Info("station "+AIBaseStation.GetName(most)+" has "+amount+" "+AICargo.GetCargoLabel(gr)+
										   " waiting (capacity "+capacity+", "+(amount/capacity)+"/"+vehicleNumber+")");
								AILog.Info("cloning "+AIVehicle.GetName(vehicle)+" (profit this year: "+AIVehicle.GetProfitThisYear(vehicle)+
								           ", last year: "+AIVehicle.GetProfitLastYear(vehicle)+", age "+AIVehicle.GetAge(vehicle)+")");
								
								AILog.Info(AIVehicle.GetProfitThisYear(vehicle)-500);
								local depot = BuildRoadDepotNear(AIVehicle.GetLocation(vehicle));
								if (depot != null && AIMap.IsValidTile(depot)){
									local vclone = AIVehicle.CloneVehicle(depot, vehicle, true);
									if (AIVehicle.IsValidVehicle(vclone)){
										if (AIVehicle.IsStoppedInDepot(vclone)){
											AIVehicle.StartStopVehicle(vclone);
										}
										AILog.Info("cloned vehicle "+AIVehicle.GetName(vclone));
									}
									else if (AIError.GetLastError()==AIVehicle.ERR_VEHICLE_TOO_MANY){
										maxVehicles = AIVehicleList().Count();
										AILog.Info("max vehicles reached: "+maxVehicles);
									}
									else {
										AILog.Info("failed cloning because "+AIError.GetLastErrorString());
									}
									
									if (vehicleNumber % 4 == 3){
										for (local i=0; i<AIOrder.GetOrderCount(vclone);i++){
											if (AIOrder.IsValidVehicleOrder(vclone, i) && AIOrder.IsGotoStationOrder(vclone, i)){
												BuildRoadStop(AIOrder.GetOrderDestination(vclone i),
													AICargo.HasCargoClass(gr, AICargo.CC_PASSENGERS)?AIRoad.ROADVEHTYPE_BUS : AIRoad.ROADVEHTYPE_TRUCK,
													AIStation.STATION_JOIN_ADJACENT);
											}
										}
										AILog.Info("expanding station");
									}
								}
								else {
									AILog.Info("failed cloning vehicle "+AIVehicle.GetName(vehicle));
								}
							}
							else {
								//AILog.Info("not enough money");
								return; //no use trying anything else
							}
						}
						else {
							//AILog.Info("too new ("+AIVehicle.GetAge(vehicle)+"/40 days)");
						}
					}
					//AILog.Info("fix finished");
				}
				else {
					//AILog.Info("not enough for intervention (" + (amount / capacity)+ "/4)");
				}
			}
		}
	}
}

function SnakeAI::ReserveMoney(amount){
	if (AICompany.GetBankBalance(AICompany.COMPANY_SELF)>amount){
		return true;
	}
	else if (amount + AICompany.GetLoanAmount() < AICompany.GetMaxLoanAmount()
			+ AICompany.GetBankBalance(AICompany.COMPANY_SELF)){
		if (amount + AICompany.GetLoanAmount()
				- AICompany.GetBankBalance(AICompany.COMPANY_SELF) > 0){
			AICompany.SetMinimumLoanAmount(amount + AICompany.GetLoanAmount()
			- AICompany.GetBankBalance(AICompany.COMPANY_SELF));
		}
		return true;
	}
	else {
		return false;
	}
}

function SnakeAI::Start()
{
    SetCompanyName();
    AICompany.SetLoanAmount(0);

    while (true) {
		//handle events
		while (AIEventController.IsEventWaiting()) {
			local e = AIEventController.GetNextEvent();
			switch (e.GetEventType()) {
				case AIEvent.ET_VEHICLE_CRASHED:
					local ec = AIEventVehicleCrashed.Convert(e);
					local v  = ec.GetVehicleID();
					AILog.Info("We have a crashed vehicle (" + v + ")");
					/* Handle the crashed vehicle */
					if (AIVehicle.GetVehicleType(v)==AIVehicle.VT_ROAD){
						local depo=BuildRoadDepotNear(ec.GetCrashSite());
						if (depo != null){
							local bus2=AIVehicle.CloneVehicle(depo,v,true);
							if (bus2 != null){
								AIVehicle.StartStopVehicle(bus2);
								AILog.Info("Replaced vehicle: "+AIVehicle.GetName(bus2));
							}
							else {
								AILog.Info("Error building Vehicle: "+AIError.GetLastErrorString());
							}
							
						}
						else {
							AILog.Info("Replacement failed, no depo found");
						}
					}
					else {
						AILog.Info("was not RV");
					}
					break;
			}
		}
		ut.BuildUnbuiltRoads();
		if (AIVehicleList().Count()<maxVehicles){
			AILog.Info("#v "+AIVehicleList().Count()+" max: "+maxVehicles);
			if (ReserveMoney(60000)){
				AttemptToConnectIndustry();
			}
			if (ReserveMoney(15000)){
				TweakVehicleNumber();
			}
		}
		
		DeleteNegativeIncomeVehicles();
		
		AICompany.SetMinimumLoanAmount(AICompany.GetLoanAmount() - AICompany.GetBankBalance(AICompany.COMPANY_SELF));
        this.Sleep(100);
    }
}

 function SnakeAI::Save()
 {
   local table = {};	
   //TODO: Add your save data to the table.
   return table;
 }
 
 function SnakeAI::Load(version, data)
 {
   AILog.Info(" Loaded");
   //TODO: Add your loading routines.
 }