/* -*- Mode: C++; tab-width: 4 -*- */
/**
 *    This file is part of Awards
 *    (c) krinn@chez.com
 *
 *    It's free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    any later version.
 *
 *    You should have received a copy of the GNU General Public License
 *    with it.  If not, see <http://www.gnu.org/licenses/>.
 *
**/

class Awards
{
static awards_db = {}; /**< base where we store awards */
static ElementID = []; /**< array with elementID for every award / company */
static CustomPoints = [10, 20, 50, 100, 150, 200];
	ID	= null;
	Category = null;
	Unique = null;
	Own_By_Company = null;
	Clock = null;
	Points = null;
	Datas = null;
	Show = null;

	constructor(id)
		{
		if (typeof(id) != "integer")	{ GSLog.Error("AwardID is not an integer : "+id); return; }
		this.ID = id;
		this.Category = null;
		this.Unique = true;
		this.Own_By_Company = [];
		this.Points = PointsType.EASY;
		this.Datas = array(14, -1);
		this.Clock = 0; // week per default
		this.Show = false;
		if (!(this.ID in Awards.awards_db))	{ Awards.awards_db[this.ID] <- this; }
		}
}

/** @brief Check if a company have that award
 *
 * @param aID AwardID
 * @param c_ID CompanyID
 * @return true if company have the award, false if not
 *
 */
function Awards::HaveAward(aID, c_ID)
{
	local awd = Awards.Get(aID);
	if (awd == false)	{ return false; }
	if (Utils.INArray(c_ID, awd.Own_By_Company) != -1)	{ return true; }
	return false;
}

/** @brief Remove an award for a company
 *
 * @param awd an AwardObject
 * @param c_ID array of companyID
 * @return False if nothing change, true if AwardObject was modify
 *
 */
function Awards::RemoveAward(awd, c_ID)
{
	local update = false;
	if (awd.Category == CategoryType.MEDAL)	{ return false; }
	foreach (company in c_ID)
		{
		local idx = Utils.INArray(company, awd.Own_By_Company);
		if (idx != -1)
			{
			CompanyLayer.RemovePoints(company, awd.Points);
			awd.Own_By_Company.remove(idx);
			update = true;
			}
		}
	return update;
}

/** @brief Give award to a company
 *
 * @param aID the AwardID
 * @param c_ID array of companyID
 *
 */
function Awards::GrantAward(aID, c_ID)
{
	local awd = Awards.Get(aID);
	if (awd == false)	{ return; }
	local grant = [];
	local lost = [];
	foreach (company in c_ID)
		{
		if (Utils.INArray(company, awd.Own_By_Company) == -1 && !GSToyLib.IsToyAI(company))	{ grant.push(company); }
		}
	foreach (company in awd.Own_By_Company)
		{
		if (Utils.INArray(company, c_ID) == -1)	{ lost.push(company); }
		}
	local less = Awards.RemoveAward(awd, lost);
	if (grant.len() == 0)
			{
			if (awd.Show)	{ StoryTeller.DisplayAwards(aID); }
			return;
			}
    GSLog.Info("Grant Award #"+aID+" to "+Utils.ArrayListToString(grant)+" - "+Utils.ArrayListToString(awd.Datas));
    awd.Show = true;
	foreach (company in grant)
		{
		if (awd.Unique)
				{
				awd.Datas[0] = company;
				awd.Datas[1] = GSDate.GetCurrentDate();
				if (awd.Points == PointsType.MEDAL)	{ Scheduler.RemoveTask(aID); }
				}
		awd.Own_By_Company.push(company);
		CompanyLayer.AddPoints(company, awd.Points);
		local z = "STR_NEWS_";
		switch (awd.Category)
			{
			case	CategoryType.MEDAL:
				z += "MEDAL";
				break;
			case	CategoryType.VEHICLE:
				z += "VEHICLE";
				break;
			case	CategoryType.ECONOMY:
				z += "ECONOMY";
				break;
			case	CategoryType.MISC:
				z += "MISC";
				break;
			}
		local ntext = GSText(GSText.STR_NEWS, GSText(GSText[z]), company, awd.Points, GSText(GSText["STR_KEY_"+aID]));
		if (GSController.GetSetting("scriptNews"))	{ GSNews.Create(GSNews.NT_GENERAL, ntext, GSCompany.COMPANY_INVALID); }
		}
	StoryTeller.DisplayAwards(aID);
	StoryTeller.UpdateRank();
}

/** @brief Get the AwardObject that match AwardID
 *
 * @param aID The AwardID
 * @return false if not found, AwardObject if found
 *
 */
function Awards::Get(aID)
{
	if (aID in Awards.awards_db)	{ return Awards.awards_db[aID]; }
	GSLog.Error("Invalid award ID "+aID);
	return false;
}

/** @brief Split a string at _ char
 *
 * @param st The String to split
 * @return Array with splitting strings
 *
 */
function Awards::Splitter(st)
{
	local retValue=[];
	local buff="";
	for (local i = 0; i < st.len(); i++)
		{
		local c = st.slice(i, i+1);
		if (c == "_")	{ retValue.push(buff); buff=""; }
				else	buff+=c;
		}
	if (buff != "")	retValue.push(buff); // add last one found because eol
	return retValue;
}

/** @brief Read awards handled in english.txt and fill our award base with values found
 *
 */
function Awards::DiscoverAwards()
{
/* OPT_XX_Z_D
XX : award number
Z  : options: 0-3 points, +100 if multi, +30 if misc, +20 if economy, +10 if vehicle
*/
	local counter = 0;
	local medcounter = 0;
	local options = [];
	foreach (key, _ in GSText)
		{
		if (key.find("STR_KEY_") != null)
				{
				local id = key.slice(8).tointeger();
				local n_award = Awards(id);
				counter++;
				}
		if (key.find("STR_OPT_") != null)
			{
			options.push(key);
			}
		}
	foreach (key in options)
		{
		local test = Awards.Splitter(key);
		if (test.len() != 5)	{ GSLog.Error("Error in award "+key); continue; }
		local id = test[2].tointeger();
		local epts = test[3].tointeger();
		local delay = test[4].tointeger() * 74;
		local unique = true;
		local cat = CategoryType.MEDAL;
		if (epts > 100)	{ unique = false; epts -= 100; }
		if (epts > 30)	{ cat = CategoryType.MISC; epts -= 30; }
		if (epts > 20)	{ cat = CategoryType.ECONOMY; epts -= 20; }
		if (epts > 10)	{ cat = CategoryType.VEHICLE; epts -= 10; }
		switch (epts)
			{
			case	0:
				epts = PointsType.MEDAL; medcounter++;
				break;
			case	1:
				epts = PointsType.EASY;
				break;
			case	2:
				epts = PointsType.MEDIUM;
				break;
			case	3:
				epts = PointsType.HARD;
				break;
			}
		Awards.SetAwardOptions(id, epts, delay, unique, cat);
		if (delay != 0)	{ local ntask = Scheduler(id, delay); } // add it to scheduler
		}
	GSLog.Info("Registered "+(counter - medcounter)+" awards and "+medcounter+" medals.");
	if (counter == 0)	{ throw("Found 0 awards, something is bad. A language file might be broken"); }
	for (local i = 0; i < counter * 15; i++)
		{
		Awards.ElementID.push(-1);
		}
}

/** @brief Set some options an award can have
 *
 * @param id The AwardID
 * @param epts The award points
 * @param delay The delay to trigger the award in the scheduler
 * @param unique True if award can only be earn by 1 company, false for multicompanies
 * @param cat The award category
 *
 */
function Awards::SetAwardOptions(id, epts, delay, unique, cat)
{
	local awd = Awards.Get(id);
	awd.Points = epts;
	awd.Clock = delay;
	awd.Unique = unique;
	awd.Category = cat;
}

/** @brief Update award datas with the given parameters
 *
 * @param id The AwardID
 * @param ... award.Data[0] = the company owning it, award.Data[1] = date when own, award.Data[>1] will be filled by ...
 *
 */
function Awards::Update(id, ...)
{
	local awd = Awards.Get(id);
	for (local i=0; i < vargc; i++)
		{
		awd.Datas[i+2] = vargv[i];
		}
}

/** @brief Just return the datas part of an award
 *
 * @param id The AwardID
 * @return the Data part of that award
 *
 */
function Awards::GetDatas(id)
{
	local awd = Awards.Get(id);
	return awd.Datas;
}

/** @brief Return a string formatted with the state of an award for that company
 *
 * @param id The AwardID
 * @param c_id The CompanyID
 * @return The formatted GSText ready to be display
 *
 */
function Awards::AwardFormat(id, c_id)
{
	local awd = Awards.Get(id);
	local own = 0;
	if (awd.Own_By_Company.len() != 0)	{ own = 1; }
	if (Utils.INArray(c_id, awd.Own_By_Company) != -1)	{ own = 2; }
    if (awd.Category == CategoryType.MEDAL)	{ own = 3; }
	local text = null;
	local bigUI = (GSController.GetSetting("BigUI") == 1);
	switch (own)
		{
		case	0:
				text = GSText(GSText.STR_AWARD_BAD);
				if (bigUI)	text = GSText(GSText.STR_AWARD_BAD_BIG);
				break;
		case	1:
				text = GSText(GSText.STR_AWARD_BAD);
				if (bigUI)	text = GSText(GSText.STR_AWARD_BAD_BIG);
				break;
		case	2:
				text = GSText(GSText.STR_AWARD_GOOD);
				if (bigUI)	text = GSText(GSText.STR_AWARD_GOOD_BIG);
				break;
		case	3:
				text = GSText(GSText.STR_MEDAL_GOOD);
				if (bigUI)	text = GSText(GSText.STR_MEDAL_GOOD_BIG);
				break;
		}
	text.AddParam(GSText(GSText["STR_KEY_"+id]));
	if (own == 0)	{
					text.AddParam(GSText(GSText.STR_AWARD_HIDDEN));
					text.AddParam(GSText(GSText.STR_AWARD_NOT_OWN));
					}
			else	{
					if (awd.Unique == false)
								{
								text.AddParam(GSText(GSText["STR_DESC_"+id]));
								text.AddParam(GSText(GSText["STR_AWARD_OWN_MULTI"], awd.Own_By_Company.len()));
								for (local z = 0; z < 15; z++)
									{
									if (z < awd.Own_By_Company.len())	{ text.AddParam(awd.Own_By_Company[z]); }
																else	{ text.AddParam(GSCompany.COMPANY_INVALID); }
									}
								}
						else	{
								text.AddParam(GSText(GSText["STR_DESC_"+id]));
								text.AddParam(GSText(GSText["STR_AWARD_OWN"]));
								for (local z = 0; z < 2; z++)	{ text.AddParam(awd.Datas[z]); }
								text.AddParam(GSText(GSText["STR_INFO_"+id]));
								for (local z = 2; z < awd.Datas.len(); z++)	{ text.AddParam(awd.Datas[z]); }
								}
					}
	return text;
}

/** @brief Add/Remove money from a company
 *
 * @param company The CompanyID
 * @param money The amount to add or remove (can be negative so)
 *
 */
function Awards::HandleMoney(company, money)
{
	GSCompany.ChangeBankBalance(company, money, GSCompany.EXPENSES_OTHER);
	local balance = Cache.GetData("reward_balance", company);
	balance += money;
	Cache.SetData("reward_balance", company, balance);
}

/** @brief Check the Periodic Reward and calc it for all companies
 *
 */
function Awards::Reward_Periodic()
{
	if (!GSController.GetSetting("PeriodicReward"))	{ return; }
	local points = GSList();
	for (local i = 0; i < 15; i++)
		{
		points.AddItem(i, CompanyLayer.companyPoints[i]);
		}
	points.Sort(GSList.SORT_BY_VALUE, GSList.SORT_DESCENDING);
    local first = points.GetValue(points.Begin());
    local multi = GSController.GetSetting("MoneyIndex");
    local money = multi * GSCompany.GetLoanInterval();
    foreach (company, pts in points)
		{
		if (GSToy.IsToyAI(company))	continue; // avoid poor toy AI
		local perc = (pts.tofloat() / first) * money;
		Awards.HandleMoney(company, perc.tointeger());
		}
}


/** @brief Gives Points Reward to a company and the money for it, and remove money from other companies
 *
 * @param company The CompanyID that get the reward
 *
 */
function Awards::Reward_Points(company)
{
	local pointlist = GSList();
	local multi = GSController.GetSetting("MoneyIndex");
	local companies = Cache.Company_Helper();
	local moneylost = GSCompany.GetLoanInterval() * multi;
	local moneygain = moneylost  * (companies.len() - 1);
	local page = CompanyLayer.GetPageID(18, 0);
	foreach (comp, _ in companies)
		{
		if (GSToyLib.IsToyAI(company))	continue; // avoid poor toy AI
		if (comp == company)	{ Awards.HandleMoney(company, moneygain); }
						else	{ Awards.HandleMoney(comp, -moneylost); }
		}
}

/** @brief Check if a company reach the Points Reward trigger value
 *
 * @param company the CompanyID
 *
 */function Awards::CheckPointsReward(company)
{
	if (!GSController.GetSetting("RewardPoints"))	{ return; }
	local trigger = Cache.GetData("reward", company);
	local points = CompanyLayer.companyPoints[company];
	local cPoints = Awards.CustomPoints[GSController.GetSetting("RewardPointsTrigger")];
	local c = 0;
	while (points >= cPoints)	{ c++; points -= cPoints; }
	if (c > trigger)	{ Cache.SetData("reward", company, trigger+1); Awards.Reward_Points(company); }
}
