
/*
 * This file is part of CityBuilder, which is a GameScript for OpenTTD
 *
 * CityBuilder is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License
 *
 * CityBuilder is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CityBuilder; If not, see <http://www.gnu.org/licenses/> or
 * write to the Free Software Foundation, Inc., 51 Franklin Street, 
 * Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */
 
  // Contributors:

///////////////////////////////////////////////
// SCRIPT CONSTANTS
// Some values:
// Cargo_promilles: how much promille of the population of each cargo a town wants, per month (30.6 days).
// Pop_reqs: How much pop a town needs to have to enable each cargo type.
// sigma: deviation factor in the requirements 
// minimum: minimum requirement factor (should be between 0.5 and 1 - 2*sigma, 0 for pop.)
// _min_size__: A town below this size grows linearly.

require("QSort.nut");
require("ERPF.nut");
require("send.nut");
require("DemandSize.nut");
require("HouseList.nut");
require("Place.nut");

function GetVersion(){
local V_BINARY = GSController.GetVersion();
local V_MAJOR = V_BINARY >> 28;
local V_MINOR = (V_BINARY >> 24) % 0x08;
local V_REV = (V_BINARY >> 20) % 0x08;
local V = ""+V_MAJOR+"."+V_MINOR + "." +V_REV;
return V;
}
function _pop_sigma_(){return 0.1;}
function _cargo_minfactor_(){return 0.5;}
function _pop_minfactor_(){return 0.0;}
function month_length (month){
switch(month){
case(2): return 28.0;
case(4): return 30.0;
case(6): return 30.0;
case(9): return 30.0;
case(11): return 30.0;
default: return 31.0;
	}
}
///////////////////////////////////////////////////
class Town extends Place {
///////////////////////////////////////////////////
////////// STATIC VARIABLES
//////////////////////////////////////////////////
static gametype = GSController.GetSetting("gametype");
static store_factor = GSController.GetSetting("storefactor");
static store_min = GSController.GetSetting("store_min");
static stagnation_pct = GSController.GetSetting("stagpct").tofloat() / 1000.0;
static slow_factor = GSController.GetSetting("slow_factor").tofloat();
static assim_factor = GSController.GetSetting("assim_factor");
static fullratio = GSController.GetSetting("mgrpct").tofloat() / 1000.0 + 1.0;
static mailcargo = GSController.GetSetting("metro_need");
static inject = GSController.GetSetting("injection");
static lowcargo = GSController.GetSetting("lowcargo");
static has_news = GSController.GetSetting("info_broadcast");
static min_size = GSController.GetSetting("min_size_max_growth");
static regrow = GSController.GetSetting("town_regrow");
static wtr = GSController.GetSetting("wtrcargo");
static fod = GSController.GetSetting("fodcargo");
static version = GetVersion();

static size_reqs = [
GSController.GetSetting("size0"),
GSController.GetSetting("size1"),
GSController.GetSetting("size2"),
GSController.GetSetting("size3"),
GSController.GetSetting("size4"),
GSController.GetSetting("size5"),
GSController.GetSetting("size6"),
GSController.GetSetting("size7"),

GSController.GetSetting("size8"),
GSController.GetSetting("size9"),
GSController.GetSetting("size10"),
GSController.GetSetting("size11"),
GSController.GetSetting("size12"),
GSController.GetSetting("size13"),
GSController.GetSetting("size14"),
GSController.GetSetting("size15"),

GSController.GetSetting("size16"),
GSController.GetSetting("size17"),
GSController.GetSetting("size18"),
GSController.GetSetting("size19"),
GSController.GetSetting("size20"),
GSController.GetSetting("size21"),
GSController.GetSetting("size22"),
GSController.GetSetting("size23"),

GSController.GetSetting("size24"),
GSController.GetSetting("size25"),
GSController.GetSetting("size26"),
GSController.GetSetting("size27"),
GSController.GetSetting("size28"),
GSController.GetSetting("size29"),
GSController.GetSetting("size30"),
GSController.GetSetting("size31")];

static decay_rates = [
GSController.GetSetting("decay0"),
GSController.GetSetting("decay1"),
GSController.GetSetting("decay2"),
GSController.GetSetting("decay3"),
GSController.GetSetting("decay4"),
GSController.GetSetting("decay5"),
GSController.GetSetting("decay6"),
GSController.GetSetting("decay7"),

GSController.GetSetting("decay8"),
GSController.GetSetting("decay9"),
GSController.GetSetting("decay10"),
GSController.GetSetting("decay11"),
GSController.GetSetting("decay12"),
GSController.GetSetting("decay13"),
GSController.GetSetting("decay14"),
GSController.GetSetting("decay15"),

GSController.GetSetting("decay16"),
GSController.GetSetting("decay17"),
GSController.GetSetting("decay18"),
GSController.GetSetting("decay19"),
GSController.GetSetting("decay20"),
GSController.GetSetting("decay21"),
GSController.GetSetting("decay22"),
GSController.GetSetting("decay23"),

GSController.GetSetting("decay24"),
GSController.GetSetting("decay25"),
GSController.GetSetting("decay26"),
GSController.GetSetting("decay27"),
GSController.GetSetting("decay28"),
GSController.GetSetting("decay29"),
GSController.GetSetting("decay30"),
GSController.GetSetting("decay31")];

static trans_reqs = [
GSController.GetSetting("trans0"),
GSController.GetSetting("trans1"),
GSController.GetSetting("trans2"),
GSController.GetSetting("trans3"),
GSController.GetSetting("trans4"),
GSController.GetSetting("trans5"),
GSController.GetSetting("trans6"),
GSController.GetSetting("trans7"),

GSController.GetSetting("trans8"),
GSController.GetSetting("trans9"),
GSController.GetSetting("trans10"),
GSController.GetSetting("trans11"),
GSController.GetSetting("trans12"),
GSController.GetSetting("trans13"),
GSController.GetSetting("trans14"),
GSController.GetSetting("trans15"),

GSController.GetSetting("trans16"),
GSController.GetSetting("trans17"),
GSController.GetSetting("trans18"),
GSController.GetSetting("trans19"),
GSController.GetSetting("trans20"),
GSController.GetSetting("trans21"),
GSController.GetSetting("trans22"),
GSController.GetSetting("trans23"),

GSController.GetSetting("trans24"),
GSController.GetSetting("trans25"),
GSController.GetSetting("trans26"),
GSController.GetSetting("trans27"),
GSController.GetSetting("trans28"),
GSController.GetSetting("trans29"),
GSController.GetSetting("trans30"),
GSController.GetSetting("trans31")];

static cargo_promilles = [
GSController.GetSetting("cargo0"),
GSController.GetSetting("cargo1"),
GSController.GetSetting("cargo2"),
GSController.GetSetting("cargo3"),
GSController.GetSetting("cargo4"),
GSController.GetSetting("cargo5"),
GSController.GetSetting("cargo6"),
GSController.GetSetting("cargo7"),
                                  
GSController.GetSetting("cargo8"),
GSController.GetSetting("cargo9"),
GSController.GetSetting("cargo10"),
GSController.GetSetting("cargo11"),
GSController.GetSetting("cargo12"),
GSController.GetSetting("cargo13"),
GSController.GetSetting("cargo14"),
GSController.GetSetting("cargo15"),
                                  
GSController.GetSetting("cargo16"),
GSController.GetSetting("cargo17"),
GSController.GetSetting("cargo18"),
GSController.GetSetting("cargo19"),
GSController.GetSetting("cargo20"),
GSController.GetSetting("cargo21"),
GSController.GetSetting("cargo22"),
GSController.GetSetting("cargo23"),
                                  
GSController.GetSetting("cargo24"),
GSController.GetSetting("cargo25"),
GSController.GetSetting("cargo26"),
GSController.GetSetting("cargo27"),
GSController.GetSetting("cargo28"),
GSController.GetSetting("cargo29"),
GSController.GetSetting("cargo30"),
GSController.GetSetting("cargo31")
];

/////////////////////////////////////////
// END OF STATIC VARS
//////////////////////////////////////////
mail_tr = 0;
pax_tr = 0;

used = false;
req_any = false;
metro = false;
metro_nepax = true;
company = -1;
num_signs = 0;
nwsmsg = "";
transport_pct = 0;
first_demand_size = 200000;
// Use an array for the cargo supply and such (32 elements)
supply_cargo =  [0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0];
goal_cargo =    [0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0];
stocked_cargo = [0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0];
cargo_factors = [1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0];
pop_factors = 	[1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0];
cargo_introduced = [];
// user information.
cargo_signs =  [-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1];
basicsign = -1;
// private variables for algorithm.
rh_x = 0;
rh_y = 0;
Is_Desert = true;
is_city = false;
house_list = null;

// town_assimilate (V13 feature)
neighbours = [];
muncipalities = [];
ismuncipality = false;

// Goal UI (V15)
GoalID = [];
// StoryBoard (V102)
Story = [];
myStoryPageID = 0;
// Demand sizes (V52)
DS = null;
// working %age pax requirements
lmp_pax = 0;
lmp_mail = 0;
pxr = 0;
mlr = 0;

constructor(town_id, DS_id, intr) {	
	::Place.constructor(town_id);
	this.initStoryBook();
	this.house_list = XHouseList(town_id);
	this.cargo_introduced = intr;
	this.DS = DS_id;
	this.stocked_cargo = [0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0];
	this.supply_cargo =  [0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0];
	this.goal_cargo =    [0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0];
	this.cargo_signs =  [-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1];
	this.is_city = GSTown.IsCity(this.id);
	this.basicsign = -1;
	GetMySettings();
	local d = GSController.GetSetting("cargo_sigma").tofloat() / 288.6751346; // 500 div sqrt 3
	local cargo_sigma = d;
	local d2 = GSController.GetSetting("pop_sigma").tofloat() / 288.6751346;
	local pop_sigma = d2;		
	if(cargo_sigma > 0) {			
		if(GSController.GetSetting("debug_level") > 2)
			GSLog.Info(GSDate.GetSystemTime() + " Calculating Variations for: " + GSTown.GetName(id) + "...");
		for(local i = 0; i < 32; i++) {
			// Do 25 passes at 1/5 of actual sigma.
			for(local j = 0; j < 25; ++j) {
				d = (GSBase.RandRange(10000).tofloat() / 10000.0 * cargo_sigma * 2.0) - cargo_sigma;
					cargo_factors[i] += d;
					if(cargo_factors[i] < _cargo_minfactor_()){cargo_factors[i] = _cargo_minfactor_();}
				}
			}
		}
	if(pop_sigma > 0) {
		for(local i = 0; i < 32; i++) {
			// Do 25 passes at 1/5 of actual sigma.
			for(local j = 0; j < 25; ++j) {
				d = (GSBase.RandRange(10000).tofloat() / 10000.0 * pop_sigma * 2.0) - pop_sigma;
				pop_factors[i] += d;
				if(pop_factors[i] < _pop_minfactor_()){pop_factors[i] = _pop_minfactor_();}
				}
			}
		}
	local x;
	for(local i = 0; i < 32; ++i) {
		x = size_reqs[i];
		if(x > 0) {
			if(first_demand_size > x){first_demand_size = x;}
			}
		}
		this.DisableDefaultGoals();
		this.SetGoals();
	}
	// default function overrides	
		function ToString();
		function ToInteger();
		function Set(new_id, new_start_population, new_supply_cargo, new_stocked_cargo, new_company, new_metro, new_num_signs, new_cargo_signs, new_basicsign, new_muncipalities, new_ismuncipality);
		function _typeof();
	// settings
		function GetMySettings();
	// neighbours	
		function AddNeighbour();
		function GetNeighbours();
		function Assimilate();
		function Assim(cid);
	// score keeping	
		function GetScore();
	// Game loop extension
		function Manage();	
	// User Interface	
		function News();
		function Metro_News();
	// News helper functions
		function GetNewsGrowLine();
		function GetTownWindowGrowLine();
		function GetNewsCargoLine(cargo_id);
		function GetTownWindowCargoLine(cargo_id);
		function GetNewsCargoLineMonochrome(cargo_id);
		function GetTownWindowCargoLineMonochrome(cargo_id);
		function SetGrowthRateAsCity();
		function ManageAsCity();
	// cargo handling
		function AddCargo();
		function TakeCargo(f);
		function PeekCargo(cargotype);
		function AddAdvancedCargo(cargo, cargotype);	
	// establishing goals
		function SetGoals();
		function SetGoals_Tropic();	
		function UpdateGoal();
		function clearGoal();
		function clearGoalStory();
	// growing and shrinking the town
		function SetGrowthRate(mf = 1.0, gf = 1.0);
		function SetGrowthRateMetro();
		function SetRate();
		function CalculateGrowthRate(mf = 1.0, g = 1.0);
		function ExpandMe(ig);
		function ContractMe(ig);
	// miscellaneous	
		function SetMetro();
}
      
function Town::ManageAsCity(){
	if(cities_setting == 1 || cities_setting == 2) {
		this.SetGrowthRateAsCity();
	}
}

function Town::initStoryBook() {
	this.myStoryPageID = GSStoryPage.New(GSCompany.COMPANY_INVALID, GSTown.GetName(this.id));
	GSStoryPage.NewElement(myStoryPageID, GSStoryPage.SPET_LOCATION, GSTown.GetLocation(this.id), GSText(GSText.STR_CITYBUILDER_SB_GOTO));
}

function Town::SetGrowthRateAsCity(){
	// if regrowth is on:
this.transport_pct = GSTown.GetLastMonthTransportedPercentage(this.id,this.paxcargo);
local num_houses = GSTown.GetHouseCount(this.id);
if(this.regrow > 100) {
	if(pop < start_population) {
			this.growthRate = this.CalculateGrowthRate();
			return;
		}
	}
	if( transport_pct < req && this.pop > this.tsize) {
  // don't grow at all.
		this.is_growing = 0;
	} else {
  // grow at full speed.
		this.is_growing = 3;
		this.growthRate = this.CalculateGrowthRate();
	}
}

function Town::_typeof(){return "Town";}

function Town::Set(new_id, new_start_population, new_supply_cargo, new_stocked_cargo, new_company, 
					new_metro, new_num_signs, new_cargo_signs, new_basicsign, new_muncipalities, new_ismuncipality) {
	id					=new_id;
	start_population	=new_start_population;
	supply_cargo		=new_supply_cargo;
	stocked_cargo		=new_stocked_cargo;
	company				=new_company;
	metro				=new_metro;
	num_signs			=new_num_signs;
	cargo_signs			=new_cargo_signs;
	basicsign			=new_basicsign;
	muncipalities		=new_muncipalities;
	ismuncipality		=new_ismuncipality;
}

///////////////////////////////////////////////////////////
// Get a string representation of the town (the town's name).
///////////////////////////////////////////////////////////
function Town::ToString(){
	return GSTown.GetName(this.id);
}
///////////////////////////////////////////////////////////
// Get an integer representation of the object (it's ID)
///////////////////////////////////////////////////////////
function Town::ToInteger(){
	return this.id;
}
///////////////////////////////////////////////////////////
// Prevent the town growing normally
///////////////////////////////////////////////////////////
function Town::SetRate(){
// we do growth by hand.
// so set the game's growth rate to very slow.
// do this only once to save cycles.
	GSTown.SetGrowthRate(this.id, GSTown.TOWN_GROWTH_NONE);
	this.GetPop();
	this.GetNeighbours(); //<< Get all towns within 64 tiles, and at least up to 8 towns.
							// moved out of the constructor for performance reasons.
}
///////////////////////////////////////////////////////////
// Do all the actions it needs to do at the start of the month.
///////////////////////////////////////////////////////////
function Town::Manage(){
this.pax_tr = GSTown.GetLastMonthTransportedPercentage(this.id,this.paxcargo);
this.mail_tr = GSTown.GetLastMonthTransportedPercentage(this.id,this.mailcargo);
this.lmp_pax = GSTown.GetLastMonthProduction(this.id, paxcargo) * this.pax_tr / 100;
this.lmp_mail = GSTown.GetLastMonthProduction(this.id, mailcargo) * this.mail_tr / 100;
if(this.lmp_pax <= lowcargo) this.pxr = 0 else this.pxr =  GSController.GetSetting("paxTR");
if(this.lmp_mail <= lowcargo) this.mlr = 0 else this.mlr =  GSController.GetSetting("mailTR");
//if(this.Is_Desert == false){this.SetGoals_Tropic();} //< Doesn't work properly?
if(GSController.GetSetting("growascity") && company == -1 && cities_setting < 3)
	this.ManageAsCity();
else{
	this.AddCargo();
	if(metro){this.SetGrowthRateMetro();}
	else{this.SetGrowthRate();}
	this.Assimilate();
	this.SetGoals();
	}
	ApplyGrowth();
}

///////////////////////////////////////////////////////////
// Calculate the maximum growth rate
///////////////////////////////////////////////////////////
function Town::CalculateGrowthRate(mf = 1.0, g = 1.0) {
	return  this.slow_factor * 1000.0 / (g  * mf * max(this.pop, _min_size__()).tofloat());
}

///////////////////////////////////////////////////////////
// Set the growth rate of the town.
///////////////////////////////////////////////////////////
function Town::SetGrowthRate(mf = 1.0, gf = 1.0) { 
	// get the least-delivered cargo ratio and store in variable f.
	local f = this.fullratio;
	local t = 0;
	local g = 0.0;
	local ig = 0.0;
	local w = 0.0
	local gc;
	this.used = true;
	// if regrowth is on:
	if(this.regrow > 100){
		if(this.pop < start_population){
			is_growing = 3;
			this.growthRate = this.CalculateGrowthRate(mf, 1.0); 
			this.TakeCargo(1.0);	
			return;	
		}	
	}
   if (this.pop >= tsize && GSTown.GetLastMonthReceived(this.id,GSCargo.TE_PASSENGERS) == 0){
		f = 1.0;
	}
for(local i = 0; i < 32; i++){
// skip a cargo if population is under minimum to demand it (or it ain't demanded).	
		gc = goal_cargo[i];
		if(gc == 0 || this.pop < size_reqs[i] * pop_factors[i]){continue;}
		t = (supply_cargo[i] + stocked_cargo[i]).tofloat() / gc.tofloat();
		w = 1.0 - gc.tofloat() / (lowcargo.tofloat());
		if(gc < lowcargo){t=t>w?t:w;}
		if(f > t){f = t;}
	}
	
// This ratio determines how fast the town grows
// shrink the town below 1 (remove houses) at inverse speed.
	if(f < 1.0) {
		this.is_growing = 0;
		g = (1.0-f) / (fullratio - 1.0);
		if(g > 1.0 || f == 0.0){g = 1.0;}
		if(g != 0.0 && this.pop * 100 > regrow * this.start_population) {
			this.growthRate = this.CalculateGrowthRate(mf, g); 
		} else {
			this.growthRate = 0;
		}
	}
// we have the grow case here.
	else if (f >= 1.0 + stagnation_pct && mail_tr >= this.mlr && pax_tr >= this.pxr) {
		this.is_growing = 1;
		g = (f-1.0-stagnation_pct) / (fullratio - 1.0 - stagnation_pct);
		if(g > 1.0 || f == 1.0){g = 1.0;}
		if(g > 0.0) {
			this.growthRate = this.CalculateGrowthRate(mf, g); 	
		} else {
			this.is_growing = 2;
			this.growthRate = 0;
		}
		if(f == this.fullratio){ 
			this.is_growing = 3;
		}
	}
// now remove and add cargo.
	this.TakeCargo(f>1.0?f:1.0);
}
///////////////////////////////////////////////////////////
// Set the growth rate of the town. Metro Only!
///////////////////////////////////////////////////////////
function Town::SetGrowthRateMetro() {
	// determine if the metro has sufficient service first.
	if(GetLastMonthTransportedPercentage(this.id, paxcargo) < GSController.GetSetting("metro_cgg")) {
	this.metro_nepax = true;
	return;};
	// get the least-delivered cargo ratio and store in variable f.
	this.metro_nepax = false;
	local mf = GSController.GetSetting("metro_mod");
	local gf = GSController.GetSetting("metro_gmod");
	this.SetGrowthRate(mf, gf);
}

///////////////////////////////////////////////////////////
// Set all the cargo goals.
// The commented out code is for debugging purposes.
///////////////////////////////////////////////////////////
function Town::SetGoals() {
	if(this.pop >= this.tsize)
	{
		local mult = 0.0;
		local t = 0;
		local date = GSDate.GetCurrentDate();
		local month = GSDate.GetMonth(date);
		if(GSController.GetSetting("debug_level") > 3)
			Log.Info("Creating cargo Requirement for " + GSTown.GetName(this.id), Log.LVL_INFO);
		for(local i = 0; i < 32; i++)
		{
		t = this.pop-this.size_reqs[i];
		if(t < 0 || cargo_introduced[i] > 0)
			{
				goal_cargo[i] = 0;
			}
			else
			{
				if	(t - this.trans_reqs[i] > 0)
				{
					goal_cargo[i] = (((this.cargo_promilles[i] * cargo_factors[i] *this.pop).tofloat()) * month_length(month) / 31000.0).tointeger();	
				}
				else{
					mult = ((this.pop - this.size_reqs[i]).tofloat()) / ((this.trans_reqs[i]).tofloat());
					goal_cargo[i] = (mult * ((this.cargo_promilles[i] * cargo_factors[i] *this.pop).tofloat())* month_length(month) / 31000.0).tointeger();	
				}
				if(goal_cargo[i] == 0 && this.cargo_promilles[i] > 0){goal_cargo[i] = 1;}
			}
		}
		// on tropic maps, set water requirement to 0 (zero) in non-desert towns
	// doesn't seem to be working, sets req to 0 always. ?!?
		if(GSGame.GetLandscape() == GSGame.LT_TROPIC && GSController.GetSetting("swapfw") == 1 )
		{
			if(wtr >= 0 && GSTile.GetTerrainType(GSTown.GetLocation(this.id)) == GSTile.TERRAIN_RAINFOREST)
			{
				goal_cargo[fod] += goal_cargo[wtr];
				goal_cargo[wtr] = 0; 
			}
		}
	}
if(req_any == false || this.used && this.pop < this.tsize){
this.req_any = false;
}
else{this.req_any = true;} 
	// add own production to pax and mailcargo!
local month = GSDate.GetMonth(GSDate.GetCurrentDate());
if(GSController.GetSetting("local_to_req")) {
		local multiplier = month_length(month) / month_length((month+11)%12).tofloat();
		if(goal_cargo[paxcargo] > 0)
			goal_cargo[paxcargo] +=  (this.lmp_pax  * multiplier).tointeger();
		if (goal_cargo[mailcargo] > 0)
			goal_cargo[mailcargo] += (this.lmp_mail * multiplier).tointeger();
	}
//Log.Info("Any Monthly transport service.", Log.LVL_INFO)
if(this.is_city)
	{
	switch(this.cities_setting)
		{
			case(3): break; // no change
			case(4): goal_cargo[this.paxcargo] = 0; break;
			case(5): goal_cargo[this.mailcargo] = 0;  break;
			case(6): goal_cargo[this.mailcargo] = 0; goal_cargo[this.paxcargo] = 0;  break;
		}
	}
}

///////////////////////////////////////////////////////////
// Tropic map modifications
//
///////////////////////////////////////////////////////////
function Town::SetGoals_Tropic(){
if(this.pop >= this.tsize)
	{
		local f = GSController.GetSetting("fodcargo");
		local w = GSController.GetSetting("wtrcargo");
		goal_cargo[f] += goal_cargo[w];
		goal_cargo[w] = 0;
	}
}
///////////////////////////////////////////////////////////
// Add the delivered simple cargoes to the town (stuff that houses accepts)
///////////////////////////////////////////////////////////
function Town::AddCargo(){
for(local i = 0; i < 32; i++){
supply_cargo[i] = 0;
	for (local cid = GSCompany.COMPANY_FIRST; cid < GSCompany.COMPANY_LAST; cid++)
		{
		if (GSCompany.ResolveCompanyID(cid) != GSCompany.COMPANY_INVALID)
			{
			supply_cargo[i] += GSCargoMonitor.GetTownDeliveryAmount(cid, i, this.id, true);
			}
		}
// note: stocked cargo above ~2 billion will not decay, 
// except 100% decay (which is easy to calculate)./
	if(stocked_cargo[i] < 2147483){
		stocked_cargo[i] *= (1000 - decay_rates[i]);
		stocked_cargo[i] /= 1000;
		} else {
		stocked_cargo[i] /= 1000;	
		stocked_cargo[i] *= (1000 - decay_rates[i]);	
		}
	}
}
///////////////////////////////////////////////////////////
// Peek at the cargo delivered (for the user)
///////////////////////////////////////////////////////////
function Town::PeekCargo(cargotype){

// This function currently has no use.

}

///////////////////////////////////////////////////////////
// Add all the 'advanced' cargo kinds (stuff that industry accepts)
// Turns out to be pretty simple!
///////////////////////////////////////////////////////////
function Town::AddAdvancedCargo(cargo, cargotype){
supply_cargo[cargotype] += cargo;
}
///////////////////////////////////////////////////////////
//Let the town take up cargo for one day. 
///////////////////////////////////////////////////////////
function Town::TakeCargo(f){
// take cargo from delivered first
for(local i = 0; i < 32; i++)
	{
	stocked_cargo[i] += supply_cargo[i];
	stocked_cargo[i] -= (f * goal_cargo[i]).tointeger();
	//if(this.company != -1){Log.Info("Y"+stocked_cargo[i], Log.LVL_INFO);}
// if we took too many that's just rounding errors.
// so just ignore this if it happens.	
	if(stocked_cargo[i] < 0){stocked_cargo[i] = 0;}
	}
// limit cargo storage.
if(this.store_factor > 0)
	{
	for(local i = 0; i < 32; i++)
		{

		if((this.goal_cargo[i] * this.store_factor + this.store_min) < stocked_cargo[i])
			{
			stocked_cargo[i] = this.goal_cargo[i] * this.store_factor + this.store_min;
			}
		}
	}
// Do the news
if(this.metro) {this.Metro_News();}
else {this.News();}
	
}
///////////////////////////////////////////////////////////
//Set the town as the METROPOLIS!
// the metropolis requires you to transport it's cargo. 
// it requires less cargo transported to it.
///////////////////////////////////////////////////////////
function Town::SetMetro(){
this.metro = true;
}

///////////////////////////////////////////////////////////
//Boring stuff.
///////////////////////////////////////////////////////////
function Town::GetMySettings(){
if(GSGame.GetLandscape() == GSGame.LT_TROPIC)
	{
	if(GSTile.IsDesertTile(GSTown.GetLocation(this.id)) == false)
		{
		this.Is_Desert = false;
		}
	}
}

function Town::News(){
// Modify Goal Window
	this.UpdateGoal();
// cargo numbers of goals
local tgoal_numbers = [];
local q = 0;
for(local i = 0; i < 32; i++)
	{
	if(goal_cargo[i] > 0 && this.pop > size_reqs[i])
		{
		tgoal_numbers.append(i);
		++q
		}
	}
local nwsmsg = "";
local next = DS.Get_Next_Demand(this.pop);
//GSLog.Info(q + " at " + this.pop);
// Now we have 4 equally sized arrays containing cargo numbers, goal, storage, and delivered.
// Depending on the length of the arrays, we present it to the user in different ways.
switch(q) {
	case(0):
		if(this.pop < tsize) {
			if(next != -1) {
					GSTown.SetText(id, GSText(GSText.STR_CITYBUILDER_CONCAT01, 
						GSText(GSText.STR_CITYBUILDER_TW_HAMLET), 
						GSText(GSText.STR_CITYBUILDER_TW_NEXT, 1 << next, size_reqs[next])));
				} else {
					GSTown.SetText(id, GSText(GSText.STR_CITYBUILDER_TW_HAMLET));
				}
			} else {
				if(next != -1) {
					GSTown.SetText(id, GSText(GSText.STR_CITYBUILDER_CONCAT002,  
							GetTownWindowGrowLine(), 
							GSText(GSText.STR_CITYBUILDER_TW_VILLAGE), 
							GSText(GSText.STR_CITYBUILDER_TW_NEXT, 1 << next, size_reqs[next])));
				}
			else{
					GSTown.SetText(id, GSText(GSText.STR_CITYBUILDER_CONCAT2,  
							GetTownWindowGrowLine(), 
							GSText(GSText.STR_CITYBUILDER_TW_VILLAGE)));
				}
			}		
		break;	
	case(1):
		if(next != -1) {
				GSTown.SetText(id, GSText(GSText.STR_CITYBUILDER_CONCAT042, 
						GetTownWindowGrowLine(), 
						GetTownWindowCargoLine(tgoal_numbers[0]),
						GSText(GSText.STR_CITYBUILDER_TW_NEXT, 1 << next, size_reqs[next])));
			} else {
				GSTown.SetText(id, GSText(GSText.STR_CITYBUILDER_CONCAT04, 
						GetTownWindowGrowLine(), 
						GetTownWindowCargoLine(tgoal_numbers[0])));
			}
		break;
	case(2):
		if(next != -1) {
				GSTown.SetText(id, GSText(GSText.STR_CITYBUILDER_CONCAT0442, 
						GetTownWindowGrowLine(), 
						GetTownWindowCargoLine(tgoal_numbers[0]), 
						GetTownWindowCargoLine(tgoal_numbers[1]),
						GSText(GSText.STR_CITYBUILDER_TW_NEXT, 1 << next, size_reqs[next])));
			} else {
				GSTown.SetText(id, GSText(GSText.STR_CITYBUILDER_CONCAT044, 
						GetTownWindowGrowLine(), 
						GetTownWindowCargoLine(tgoal_numbers[0]), 
						GetTownWindowCargoLine(tgoal_numbers[1])));
			}
		break;
	case(3):
		if(next != -1) {
				GSTown.SetText(id, GSText(GSText.STR_CITYBUILDER_CONCAT04442, 
						GetTownWindowGrowLine(), 
						GetTownWindowCargoLine(tgoal_numbers[0]), 
						GetTownWindowCargoLine(tgoal_numbers[1]), 
						GetTownWindowCargoLine(tgoal_numbers[2]),
						GSText(GSText.STR_CITYBUILDER_TW_NEXT, 1 << next, size_reqs[next])));
			} else {
				GSTown.SetText(id, GSText(GSText.STR_CITYBUILDER_CONCAT0444, 
						GetTownWindowGrowLine(), 
						GetTownWindowCargoLine(tgoal_numbers[0]), 
						GetTownWindowCargoLine(tgoal_numbers[1]), 
						GetTownWindowCargoLine(tgoal_numbers[2])));
			}		
		break;
	default:
// point towards the goal window.
		if(GSController.GetSetting("point_to_goal") == 1) {
			GSTown.SetText(this.id, GSText(GSText.STR_CITYBUILDER_SEEGOALWINDOW, this.company));
		} else {
			local q = 0;
			nwsmsg = "";
			switch(this.is_growing) {
				case(0): 
					nwsmsg += GSTown.GetName(this.id)+ " is not growing. D/S/R: ";
					break;
				case(1):
					nwsmsg += GSTown.GetName(this.id)+ " is growing slowly. D/S/R: ";
					break;		
				case(2):
					nwsmsg += GSTown.GetName(this.id)+ " is stagnant. D/S/R: ";
					break;	
				case(3):
					nwsmsg +=GSTown.GetName(this.id)+ " is growing! D/S/R: ";
					break;			
			}
		for(local i = 0; i < 32; i++) {
			if(goal_cargo[i] > 0 && this.pop > size_reqs[i]) {
			if(q) nwsmsg += ", "
			q = 1;
			if(decay_rates[i] != 1000) {
				nwsmsg += (supply_cargo[i]).tostring() + "/" +(stocked_cargo[i]).tostring() + "/" +
					((goal_cargo[i]).tointeger()).tostring() + " " + GSCargo.GetCargoLabel(i);
					} else {
				nwsmsg += (supply_cargo[i]).tostring() + "/" +
					((goal_cargo[i]).tointeger()).tostring() + " " + GSCargo.GetCargoLabel(i);
					}
				}
			}
		nwsmsg += ".";
		if(q == 0) {
			if(this.pop < tsize) {
				nwsmsg += " The town does not have any demands yet."
				} else {
				nwsmsg += " Supply a monthly transport service to start growth. ";
				}
			}
			if(this.pop >= tsize) {
			local m = this.mlr;
			local p = this.pxr;
			if(m > 0)
				if(mail_tr > m)
					nwsmsg += "Not enough mail has been transported to meet requirement. Transport at least " + m + "percent.";
				else
					nwsmsg += "Mail requirement fulfilled ";				
			if(p > 0)
				if(pax_tr > p)
					nwsmsg += "Not enough passengers have been transported to meet requirement. Transport at least " + m + "percent.";
				else
					nwsmsg += "Passengers requirement fulfilled ";	
			}			
//SendAdmin(this.company + "{}" + nwsmsg);}
			GSTown.SetText(this.id,nwsmsg);							
		}
	break;	
	}
}

function Town::Metro_News(){
local q = 0;
nwsmsg = "";
if(this.metro_nepax)
	{
	nwsmsg += "The metropolis demands additional transport services for its people before it will grow any further!";
	}
else
	{
	switch(this.is_growing)
		{
		case(0): 
			nwsmsg += GSTown.GetName(this.id)+ " is not growing. Delivered/Storage/Required: ";
			break;
		case(1):
			nwsmsg += GSTown.GetName(this.id)+ " is growing slowly. Delivered/Storage/Required: ";
			break;		
		case(2):
			nwsmsg += GSTown.GetName(this.id)+ " is stagnant. Delivered/Storage/Required: ";
			break;	
		case(3):
			nwsmsg +=GSTown.GetName(this.id)+ " is growing! Delivered/Storage/Required: ";
			break;			
		}
	for(local i = 0; i < 32; i++)
		{
		if(goal_cargo[i] > 0  && this.pop > size_reqs[i])
			{
			if(q) nwsmsg += ", "
			q = 1;
			nwsmsg +=(supply_cargo[i]).tostring() + "/" +(stocked_cargo[i]).tostring() + "/" +
				((goal_cargo[i]).tointeger()).tostring() + " " + GSCargo.GetCargoLabel(i);
			}
		}
	}
if(q == 0)
	{
	if(this.pop < tsize) 
		{
		nwsmsg += "The town does not have any demands yet."
		}
	else
		{
		nwsmsg += " Supply a monthly transport service to start growth.";
		}
	}
if(this.pop >= tsize)
	{
	local m = GSController.GetSetting("mailTR");
	local p = GSController.GetSetting("paxTR");
	if(m > 0)
		if(mail_tr > m)
			this.addGoalStory(GSText(GSText.STR_CITYBUILDER_NEWS_MAILTR_A, mail_tr));
		else
			this.addGoalStory(GSText(GSText.STR_CITYBUILDER_NEWS_MAILTR_B, mail_tr));						
	if(p > 0)
		if(pax_tr > p)
			this.addGoalStory(GSText(GSText.STR_CITYBUILDER_NEWS_PAXTR_A, pax_tr));
		else
			Gthis.addGoalStory(GSText(GSText.STR_CITYBUILDER_NEWS_PAXTR_B, pax_tr));		
	}		
	
	else nwsmsg += " Deliver additional cargo to grow the town!";
	this.UpdateGoal();
	GSTown.SetText(this.id,nwsmsg);
}

function Town::GetNeighbours() {
local towns = GSTownList(); // Get a list of all towns
local id2 = towns.Begin();
local j = 0;
local d = 0;
local distances = [];
local towns_a = [];
foreach(t, _ in towns)
	{
	d = GSTown.GetDistanceSquareToTile(this.id, GSTown.GetLocation(t));
	distances.append(d);
	towns_a.append(t);
	}
// Now we have an array towns_a with all the towns,
// and a second array with the distances. 
local towns_s = QSort(distances, towns_a);
Log.Info(GSDate.GetSystemTime() + " Town done:" + GSTown.GetName(this.id), Log.LVL_INFO);
// up to 3 are 
// The first town in the array is itself so we skip it (j starts at 1)
local _l = towns_a.len();
while(j < 3 && j < _l)
	{
	neighbours.append(towns_s[j]);
	//d =   GSTown.GetDistanceSquareToTile(this.id, GSTown.GetLocation(towns_s[j]));
	j++;
	//Log.Info(GSTown.GetName(this.id) + " nb of " + GSTown.GetName(towns_s[j]), Log.LVL_INFO);
	}
}

function Town::Assimilate() {
if( GSController.GetSetting("assim_factor") != 0)
	{
	local loc = GSTown.GetLocation(this.id);
	local tid;
	for(local i = 0; i<this.neighbours.len(); i++)
		{
		tid = this.neighbours[i];
		if(tid != this.id)
			{
			if(this.pop > assim_factor * GSTown.GetPopulation(tid) && (cities_setting >= 3 || GSTown.IsCity(tid) == false))
				{
	// The test is to see if townzone #3 overlaps the other town's center or zone #3.
				local f1 = GSTown.GetHouseCount(this.id) / 8 * 5 - 5;
				if(f1 < 0){f1 = 0;}
				local f2 = GSTown.GetHouseCount(tid) / 8 * 5 - 5;
				if(f2 < 0){f2 = 0;}
				local c = f1 + f2; // + sqrt(f1) * sqrt(f2) * 2 << Removed for optimization...
				Log.Info(GSTown.GetName(tid) + " and "+ GSTown.GetName(this.id) + "req " + c);
	// optimization: reuse f1
				f1 = GSTown.GetDistanceSquareToTile(tid,loc);
				if(c > f1)
					{
	// assimilate the second town
					this.neighbours.remove(i);
					this.Assim(tid);
					}	
				}
			else if(this.pop < GSTown.GetPopulation(tid) / assim_factor  && (cities_setting >= 3 || GSTown.IsCity(this.id) == false))
				{
				local f1 = GSTown.GetHouseCount(this.id) / 8 * 5 - 15;
				if(f1 < 0){f1 = 0;}
				local f2 = GSTown.GetHouseCount(tid) / 8 * 5 - 15;
				if(f2 < 0){f2 = 0;}
				local c = f1 + f2;  // + sqrt(f1) * sqrt(f2) * 2 << Removed for optimization...
	// optimization: reuse f1
				f1 = GSTown.GetDistanceSquareToTile(tid,loc);
				if(c > f1)
					{
					this.ismuncipality = true;
					}
				}
			}
		}
	}
}

function Town::Assim(tid){
	Log.Info("Merging Towns " + GSTown.GetName(tid) + " to "+ GSTown.GetName(this.id), Log.LVL_INFO);
		for (local cid = GSCompany.COMPANY_FIRST; cid <= GSCompany.COMPANY_LAST; cid++)
				{
				if (GSCompany.ResolveCompanyID(cid) != GSCompany.COMPANY_INVALID)
					{
					GSNews.Create(GSNews.NT_GENERAL, GSText(GSText.STR_CITYBUILDER_TAKEOVER, tid, this.id),cid);
					}
				}
	this.muncipalities.append(tid);

	// add another neighbour
	this.AddNeighbour();
}

function Town::AddNeighbour() {
local towns = GSTownList(); // Get a list of all towns
local id2 = towns.Begin();
local j = 2 + muncipalities.len();
local d = 0;
local distances = [];
local towns_a = [];
foreach(t, _ in towns)
	{
	d = GSTown.GetDistanceSquareToTile(this.id, GSTown.GetLocation(t));
	distances.append(d);
	towns_a.append(t);
	}
// Now we have an array towns_a with all the towns,
// and a second array with the distances. 
local towns_s = QSort(distances, towns_a);
// NOTE: towns within MD 64 are all checked.
// beyond that, up to 8 are. 
// The first town in the array is itself so we skip it (j starts at 1)
local _l = towns_s.len();
if(j < _l)
	{
	neighbours.append(towns_s[j]);
	d =  GSTile.GetDistanceManhattanToTile(GSTown.GetLocation(this.id), GSTown.GetLocation(towns_s[j]));
	j++;
	}
}

function Town::GetScore() {
local score = 0;
foreach(t in this.muncipalities)
	{
	score += GSTown.GetPopulation(t);
	}
score += this.pop;
return score;
}
// Updates the town's goals
function Town::UpdateGoal(){
this.clearGoalStory();
this.addStory(GSText(GSText.STR_CITYBUILDER_SB_POPULATION,this.pop));
local has_goal = false;
// Add new ones.
		this.addGoalStory(GSText(GSText.STR_CITYBUILDER_GW_GOALS, (fullratio * 100).tointeger() + 1));
switch(this.is_growing)
	{
	case(0): 		// decrease
		this.addGoalStory(GSText(GSText.STR_CITYBUILDER_GW_NOT_GROWING));
		break;
	case(1): 		// growth (slow)
		this.addGoalStory(GSText(GSText.STR_CITYBUILDER_GW_B_GROWING));
		break;
	case(2):		// stagnant
		this.addGoalStory(GSText(GSText.STR_CITYBUILDER_GW_STAGNANT));
		break;
	case(3):		// growth(fast)
		this.addGoalStory(GSText(GSText.STR_CITYBUILDER_GW_A_GROWING));
		break;
	}
	this.addGoalStory("");
for(local i = 0; i < 32; i++)
	{
	if(cargo_promilles[i] > 0 && this.goal_cargo[i] > 0)
		{
		has_goal = true;
		if(this.decay_rates[i] != 1000)
			{
			if(supply_cargo[i] >= goal_cargo[i] * fullratio)
				{
				this.addGoalStory(GSText(
								GSText.STR_CITYBUILDER_GW_G, 
								1 << i,  
								supply_cargo[i], 
								goal_cargo[i],
								(stocked_cargo[i]), 
								min(100 * stocked_cargo[i] / (goal_cargo[i] * store_factor + store_min), 100)
								));		
				}
			else if(supply_cargo[i] >= goal_cargo[i])
				{this.addGoalStory(
							GSText(
								GSText.STR_CITYBUILDER_GW_A, 
								1 << i,  
								supply_cargo[i], 
								goal_cargo[i],
								(stocked_cargo[i]), 
								min(100 * stocked_cargo[i] / (goal_cargo[i] * store_factor + store_min), 100)
								));		
				}
			else if(supply_cargo[i] + stocked_cargo[i] >= goal_cargo[i])
				{
				this.addGoalStory(
							GSText(
								GSText.STR_CITYBUILDER_GW_B, 
								1 << i, 
								supply_cargo[i], 
								goal_cargo[i],
								(stocked_cargo[i]), 
								min(100 * stocked_cargo[i] / (goal_cargo[i] * store_factor + store_min), 100)
								));
				}
			else{	
			this.addGoalStory(
							GSText(
								GSText.STR_CITYBUILDER_GW_C, 
								1 << i,  
								supply_cargo[i], 
								goal_cargo[i],
								(stocked_cargo[i]), 
								min(100 * stocked_cargo[i] / (goal_cargo[i] * store_factor + store_min), 100)
								));		
				}
			}
			else
			{
			if(supply_cargo[i] * 1000 >= goal_cargo[i] * (1000 + fullratio))
				{this.addGoalStory(
							GSText(
								GSText.STR_CITYBUILDER_GW_F, 
								1 << i,  
								supply_cargo[i], 
								goal_cargo[i]
								));
				}
			else if(supply_cargo[i] + stocked_cargo[i] >= goal_cargo[i])
				{this.addGoalStory(
							GSText(
								GSText.STR_CITYBUILDER_GW_D, 
								1 << i,  
								supply_cargo[i], 
								goal_cargo[i]
								));
				}
			else{	this.addGoalStory(
							GSText(
								GSText.STR_CITYBUILDER_GW_E, 
								1 << i, 
								supply_cargo[i], 
								goal_cargo[i]
								));		
				}
			}
		}
	}
	if(!has_goal)
	{
	if(this.pop < tsize)  this.addGoalStory(GSText(GSText.STR_CITYBUILDER_GW_HAMLET));
	else this.addGoalStory(GSText(GSText.STR_CITYBUILDER_GW_VILLAGE));
	}
			if(this.pop >= tsize)
			{
			local m = GSController.GetSetting("mailTR");
			local p = GSController.GetSetting("paxTR");
			if(m > 0)
				if(mail_tr > m)
					this.addGoalStory(GSText(GSText.STR_CITYBUILDER_GW_MAILTR_A, mail_tr));
				else
					this.addGoalStory(GSText(GSText.STR_CITYBUILDER_GW_MAILTR_B, mail_tr));						
			if(p > 0)
				if(pax_tr > p)
					this.addGoalStory(GSText(GSText.STR_CITYBUILDER_GW_PAXTR_A, pax_tr));
				else
					this.addGoalStory(GSText(GSText.STR_CITYBUILDER_GW_PAXTR_B, pax_tr));		
			}
	local next = DS.Get_Next_Demand(this.pop);
	if(next != -1){
		this.addGoalStory(GSText(GSText.STR_CITYBUILDER_GW_NEXT, 1 >> next, DS.Get_Amount(next)));
	}
}
function Town::clearGoal() {
	// Remove the old goals.
	foreach(ID in GoalID) {
		GSGoal.Remove(ID);
	}
	GoalID = [];	
}

// Clear the storyboard page and goal window for this town
function Town::clearGoalStory() {
	this.clearGoal();
	// Clear the story book.
	foreach(ID in Story){
		GSStoryPage.RemoveElement(ID);
	}
	Story = [];
	// Refresh the date in the story book
	GSStoryPage.SetDate(this.myStoryPageID,GSDate.GetCurrentDate()); 	
}
// Add one line of text to the story board and goal window. 
function Town::addGoalStory(goalText) {
	// if we are the metropolis...
	if(this.metro) {
		for (local cid = GSCompany.COMPANY_FIRST; cid <= GSCompany.COMPANY_LAST; cid++) {
			if (GSCompany.ResolveCompanyID(cid) != GSCompany.COMPANY_INVALID) {
				this.GoalID.push(GSGoal.New(cid, goalText, GSGoal.GT_NONE, 0));				
			}
		}
	}
	else if(this.company >= 0) {
		this.GoalID.push(GSGoal.New(company, goalText, GSGoal.GT_NONE, 0));
	}
	// in all cases: update the story board!
	this.addStory(goalText);
}

function Town::addStory(goalText) {
	Story.push(GSStoryPage.NewElement(this.myStoryPageID, GSStoryPage.SPET_TEXT, 0, goalText));
}
// Create the line of text saying whether the town is growing/not, TW Edition.
function Town::GetTownWindowGrowLine(){
switch(this.is_growing)
	{
	case(0): 
		return GSText(GSText.STR_CITYBUILDER_TW_NOT_GROWING);
	case(1):
		return GSText(GSText.STR_CITYBUILDER_TW_B_GROWING);
	case(2):
		return GSText(GSText.STR_CITYBUILDER_TW_STAGNANT);
	case(3):
		return GSText(GSText.STR_CITYBUILDER_TW_A_GROWING);
	}
}
function Town::GetNewsGrowLine(){
switch(this.is_growing)
	{
	case(0): 
		return GSText(GSText.STR_CITYBUILDER_NEWS_NOT_GROWING);
	case(1):
		return GSText(GSText.STR_CITYBUILDER_NEWS_B_GROWING);
	case(2):
		return GSText(GSText.STR_CITYBUILDER_NEWS_STAGNANT);
	case(3):
		return GSText(GSText.STR_CITYBUILDER_NEWS_A_GROWING);
	}
}
// Get the cargo line for the cargo_id. 
function Town::GetTownWindowCargoLine(cargo_id){
if(decay_rates[cargo_id] != 1000)
	{
	if(supply_cargo[cargo_id] >= goal_cargo[cargo_id] * fullratio)
		{
		return GSText(GSText.STR_CITYBUILDER_TW_G,1 << cargo_id,supply_cargo[cargo_id],goal_cargo[cargo_id],stocked_cargo[cargo_id]);
		}
	else if(supply_cargo[cargo_id] >= goal_cargo[cargo_id])
		{
		return GSText(GSText.STR_CITYBUILDER_TW_A,1 << cargo_id,supply_cargo[cargo_id],goal_cargo[cargo_id],stocked_cargo[cargo_id]);
		}
	else if(supply_cargo[cargo_id] + stocked_cargo[cargo_id] >= goal_cargo[cargo_id])
		{
		return GSText(GSText.STR_CITYBUILDER_TW_B,1 << cargo_id,supply_cargo[cargo_id],goal_cargo[cargo_id],stocked_cargo[cargo_id]);
		}
	else{
		return GSText(GSText.STR_CITYBUILDER_TW_C,1 << cargo_id,supply_cargo[cargo_id],goal_cargo[cargo_id],stocked_cargo[cargo_id]);
		}
	}
else{
	if(supply_cargo[cargo_id]  >= goal_cargo[cargo_id] * fullratio)
		{
		return GSText(GSText.STR_CITYBUILDER_TW_F,1 << cargo_id,supply_cargo[cargo_id],goal_cargo[cargo_id], 0); // Extra param to fool the GSText lib.
		}
	else if(supply_cargo[cargo_id] + stocked_cargo[cargo_id] >= goal_cargo[cargo_id])
		{
		return GSText(GSText.STR_CITYBUILDER_TW_D,1 << cargo_id,supply_cargo[cargo_id],goal_cargo[cargo_id], 0);
		}
	else{
		return GSText(GSText.STR_CITYBUILDER_TW_E,1 << cargo_id,supply_cargo[cargo_id],goal_cargo[cargo_id], 0);
		}	
	}
}

function Town::GetNewsCargoLine(cargo_id){
if(decay_rates[cargo_id] != 1000)
	{
	return GSText(GSText.STR_CITYBUILDER_NEWS_A,1 << cargo_id,supply_cargo[cargo_id],goal_cargo[cargo_id],stocked_cargo[cargo_id]);
	}
else{
	return GSText(GSText.STR_CITYBUILDER_NEWS_D,1 << cargo_id,supply_cargo[cargo_id],goal_cargo[cargo_id], 0); 
	}
}

// Get the cargo line for the cargo_id. 
function Town::GetTownWindowCargoLineMonochrome(cargo_id){
if(decay_rates[cargo_id] != 1000)
	{
	return GSText(GSText.STR_CITYBUILDER_TW_MA,1 << cargo_id, ""+supply_cargo[cargo_id] + " / " + goal_cargo[cargo_id] + " | Storage: " + stocked_cargo[cargo_id]);
	}
else
	{
	return GSText(GSText.STR_CITYBUILDER_TW_MD,1 << cargo_id, ""+supply_cargo[cargo_id] + " / " + goal_cargo[cargo_id]);
	}
}

function Town::GetNewsCargoLineMonochrome(cargo_id){
if(decay_rates[cargo_id] != 1000)
	{
	return GSText(GSText.STR_CITYBUILDER_NEWS_MA,1 << cargo_id, ""+supply_cargo[cargo_id] + " / " + goal_cargo[cargo_id] + " | Storage: " + stocked_cargo[cargo_id]);
	}
else{
	return GSText(GSText.STR_CITYBUILDER_NEWS_MD,1 << cargo_id, ""+supply_cargo[cargo_id] + " / " + goal_cargo[cargo_id]);
	}
}