/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

require ("config.nut");

class NWD
{
	p = null;
	towns = null;
	industries = null;
	wait = null;
	
	constructor (v)
	{
		p = v;
		industries = [];
		towns = [];
		wait = 0;
	}
	
	function Init ()
	{
		GSLog.Info ("--------------------------------------");
		GSLog.Info ("New World Disorder v0.2.1 - 12.10.2016");
		GSLog.Info ("Design by LaChupacabra");
		GSLog.Info ("Programming by Milek7");
		GSLog.Info ("--------------------------------------");
		GSLog.Info ("Initialization...");
		
		local list;
		
		list = GSIndustryList ();
		for (local v = list.Begin (); !list.IsEnd (); v = list.Next ())
			industries.push(Industry (v));
		
		list = GSTownList ();
		for (local v = list.Begin (); !list.IsEnd (); v = list.Next ())
			towns.push (Town (this, v));
		
		GSLog.Info ("OK!");
		GSLog.Info ("--------------------------------------");
	}
	
	function AssignStation (station)
	{
		foreach (v in towns)
			if (GSStation.GetDistanceManhattanToTile (station, GSTown.GetLocation (v.t)) < Config.StationDistance ||
			    GSStation.IsWithinTownInfluence (station, v.t))
				v.stations.push(station);
	}
	
	function AssignIndustry (industry)
	{
		local i = Industry(industry);
		industries.push(i);
		foreach (v in towns)
			v.PushIndustry(i);
	}
	
	function OnLoop ()
	{
		wait--;
		if (wait > 0)
			return;

		local bdate = GSDate.GetCurrentDate();
		
		GSLog.Info ("--------------------------------------");
		GSLog.Info ("Calculating...");

		foreach (k, v in industries)
			if (!GSIndustry.IsValidIndustry (v.i))
				industries.remove (k);

		foreach (v in towns)
			v.ComputeGrowth ();
		
		GSLog.Info ("OK!");
		GSLog.Info ("--------------------------------------");

		wait = Config.UpdateInterval - (GSDate.GetCurrentDate() - bdate);
		if (wait < 0)
			GSLog.Warning("WARNING: Can't keep up with UpdateInterval by " + (-wait) + " days!");
	}
	
	function OnEvent (e)
	{
		switch (e.GetEventType ())
		{
			case GSEvent.ET_STATION_FIRST_VEHICLE:
			AssignStation (GSEventStationFirstVehicle.Convert (e).GetStationID ());
			break;
			
			case GSEvent.ET_INDUSTRY_OPEN:
			AssignIndustry (GSEventIndustryOpen.Convert (e).GetIndustryID ());
			break;
			
			case GSEvent.ET_TOWN_FOUNDED:
			towns.push (Town (this, GSEventTownFounded.Convert (e).GetTownID ()));
			break;
		}
	}
}

class Industry
{
	i = null;

	nearest_town = null;
	nearest_town_distance = null;

	constructor (id)
	{
		i = id;
		nearest_town_distance = 10000;
	}
}

class Town
{
	t = null;
	p = null;
	industries = null;
	stations = null;
	
	lasthouses = null;
	lastvalues = null;
	
	constructor (v, b)
	{
		t = b;
		p = v;
		lasthouses = GSTown.GetHouseCount (t);
		RebuildStationsList ();
		RebuildIndustriesList ();
		GSTown.SetGrowthRate (t, 0);
	}
	
	function RebuildStationsList ()
	{
		stations = [];
		
		local list = GSStationList (GSStation.STATION_ANY);
		for (local v = list.Begin (); !list.IsEnd (); v = list.Next ())
		{
			local distance = GSStation.GetDistanceManhattanToTile (v, GSTown.GetLocation (t));
		
			if (distance < Config.StationDistance || GSStation.IsWithinTownInfluence (v, t))
				stations.push (v);
		}
	}
	
	function RebuildIndustriesList ()
	{
		industries = [];
		
		foreach (v in p.industries)
			PushIndustry(v);
	}

	function PushIndustry (v)
	{
		if (!GSIndustry.IsValidIndustry (v.i))
			return;
		local distance = GSIndustry.GetDistanceManhattanToTile (v.i, GSTown.GetLocation (t));
		if (distance < Config.IndustryDistance || GSTile.IsWithinTownInfluence(GSIndustry.GetLocation (v.i), t))
		{
			industries.push(v);
			if (distance < v.nearest_town_distance)
			{
				v.nearest_town_distance = distance;
				v.nearest_town = this;
			}
		}
	}

	function GetCategorizedCargo (station, provider, cargo)
	{
		local t = {};
		t.intercity <- 0;
		t.innercity <- 0;
		local c = provider (station, cargo);
		for (local x = c.Begin (); !c.IsEnd (); x = c.Next ())
		{
			if (station == x)
				continue;
			
			local inner = false;
			foreach (z in stations)
				if (z == x)
				{
					inner = true;
					break;
				}
			
			if (inner)
				t.innercity += c.GetValue (x);
			else
				t.intercity += c.GetValue (x);
		}
		return t;
	}
	
	function ComputeGrowth ()
	{
		if (GSTown.GetHouseCount (t) > lasthouses + 10)
		{
			RebuildStationsList();
			lasthouses = GSTown.GetHouseCount (t);
		}
		
		local values =
		{
			industry = 0
			trade = 0
			build = 0
			transport = 0
			transport_in = 0
			transport_out = 0
		};

		foreach (k, v in stations)
		{
			if (!GSStation.IsValidStation (v))
			{
				stations.remove (k);
				continue;
			}
			
			local c;
			
			foreach (h, j in Config.TransportCargo)
			{
				//incoming
				c = GetCategorizedCargo (v, GSStationList_CargoPlannedByFrom, h);
				values.transport_in += c.intercity * j;
				
				//outcoming
				c = GetCategorizedCargo (v, GSStationList_CargoPlannedByVia, h);
				values.transport += c.innercity * j;
				values.transport_out += c.intercity * j;
			}
		}
		
		foreach (k, v in industries)
		{
			if (!GSIndustry.IsValidIndustry (v.i))
			{
				industries.remove (k);
				continue;
			}
			
			local type = GSIndustry.GetIndustryType (v.i);
			local produced = GSIndustryType.GetProducedCargo (type);
			local accepted = GSIndustryType.GetAcceptedCargo (type);
			local multipier;
			if (v.nearest_town == this)
				multipier = 1.0;
			else
				multipier = (1.0 - (GSIndustry.GetDistanceManhattanToTile (v.i, GSTown.GetLocation (t)) / Config.IndustryDistance)) * Config.NonNearestMultipier;

			for (local c = produced.Begin (); !produced.IsEnd (); c = produced.Next ())
			{
				local m;
				m = 1;
				if (c in Config.IndustryMultipier)
					m = Config.IndustryMultipier[c];
				values.industry += GSIndustry.GetLastMonthProduction (v.i, c) * m * multipier;
			}

			for (local c = accepted.Begin (); !accepted.IsEnd (); c = accepted.Next ())
			{
				local m, a;

				a = MonitorIndustry (c, v.i);

				m = 0;
				if (c in Config.TradeCargoMultipier)
					m = Config.TradeCargoMultipier[c];
				values.trade += a * m * multipier;
				
				m = 0;
				if (c in Config.BuildCargoMultipier)
					m = Config.BuildCargoMultipier[c];
				values.build += a * m * multipier;
			}
		}

		local population = GSTown.GetPopulation (t);

		local total_pax = values.transport + values.transport_in + values.transport_out;
		local produced_pax = 0;
		foreach (a, b in Config.TransportCargo)
		{
			produced_pax += GSTown.GetLastMonthProduction (t, a) * b;
		}
		
		values.industry = safediv(values.industry, 200.0 / 1000.0 * population);
		values.trade = safediv(values.trade, 100.0 / 1000.0 * population);
		values.build = safediv(values.build,  50.0 / 1000.0 * population);

		values.transport = min(safediv(values.transport, total_pax) / 0.5, 1.0) * safediv(total_pax, produced_pax);
		
		values.transport_inter <- safediv(values.transport_in, produced_pax) * 0.5
								+ safediv(values.transport_out, produced_pax) * 0.5;

		foreach (a, b in values)
		{
			if (b > 1.0)
				values[a] = 1 + sqrt(b - 1.0) * 3.0;
			if (lastvalues != null)
				values[a] = lastvalues[a] + (b - lastvalues[a]) / 4.0;
		}
		lastvalues = values;
		
		local total = (values.industry + values.trade + values.build + values.transport + values.transport_inter) / 5.0;

		local unemployment = max((1 - total) * 0.4, 0);
		
		local growth = (355 * pow(1.0 - total, 2)).tointeger();
		
		if (growth > 200)
			growth = GSTown.TOWN_GROWTH_NONE;
		if (growth < 1)
			growth = 1;

		GSTown.SetGrowthRate (t, growth);
		
		GSTown.SetText (t, GSText (GSText.STR_INFO_WINDOW,
							GetColor (1 - unemployment / 0.4), (unemployment * 100).tointeger(),
							GetColor (values.transport), (values.transport * 100).tointeger(),
							GetColor (values.transport_inter), (values.transport_inter * 100).tointeger(),
							GetColor (100), (100).tointeger(),
							GetColor (values.industry), (values.industry * 100).tointeger(),
							GetColor (values.trade), (values.trade * 100).tointeger(),
							GetColor (values.build), (values.build * 100).tointeger()));
	}
}

function max (a, b)
{
	if (a < b)
		return b;
	return a;
}

function min (a, b)
{
	if (a > b)
		return b;
	return a;
}

function safediv (a, b)
{
	if (b == 0)
		return 1;
	return a / b;
}

function GetColor (v)
{
	if (v >= 0.75)
		return GSText (GSText.STR_COLOR_OK);
	if (v >= 0.25)
		return GSText (GSText.STR_COLOR_MIDDLE);

	return GSText (GSText.STR_COLOR_BAD);
}

function MonitorIndustry (cargo, industry) //terrible hack!
{
	local r = 0;
	for (local i = GSCompany.COMPANY_FIRST; i <= GSCompany.COMPANY_LAST; i++)
		if (GSCompany.ResolveCompanyID (i) != GSCompany.COMPANY_INVALID)
			r += GSCargoMonitor.GetIndustryDeliveryAmount (i, cargo, industry, true);
	return r;
}

function Find (v, c)
{
	foreach (x in v)
		if (x == c)
			return true;
	return false;
}
