#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <signal.h>
#include <gtk/gtk.h>
#if defined(_WIN32)
# include <gdk/gdkwin32.h>
# define HAVE_WIN32
#else
# include <gdk/gdkx.h>
# define HAVE_X
#endif
#ifdef HAVE_EDV2
#include <endeavour2.h>
#endif
#include <unistd.h>

#include "guiutils.h"
#include "cdialog.h"
#include "pdialog.h"
#include "fprompt.h"
#include "fb.h"

#include "avscanop.h"
#include "cfg.h"
#include "cfg_fio.h"
#include "obj.h"
#include "objfio.h"
#include "win.h"
#include "winlist.h"
#include "winopcb.h"
#include "core.h"
#include "cfglist.h"
#include "config.h"

#include "images/icon_executable_20x20.xpm"
#include "images/icon_file_infected_20x20.xpm"
#include "images/icon_folder_closed_20x20.xpm"
#include "images/icon_folder_opened_20x20.xpm"
#include "images/icon_virus_pattern_20x20.xpm"


static void MainSignalCB(int s);
static gint MainTOCB(gpointer data);

static cfg_item_struct *MainCfgListNew(void);

win_struct *MainNewAVScan(core_struct *core, gint argc, gchar **argv);


static gint	*avscan_scan_stop_count;


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	UNIX signal callback.
 */
static void MainSignalCB(int s)
{
	switch(s)
	{
	  case SIGINT:
	    /* Is this process running as a scan process? */
	    if(avscan_scan_stop_count != NULL)
	    {
		gint *stop_count = avscan_scan_stop_count;

		/* Increment the scan's stop count so that the scan
		 * process will stop nicely
		 *
		 * When its stop count is increment it will wait for
		 * the current (if any) ClamAV call to finish and
		 * continue to execute without calling any additional
		 * ClamAV calls except AVScanOPClamAVShutdown()
		 * in order to shut down ClamAV nicely (so that there
		 * are no ClamAV tempory files) and finish execution
		 */
		*stop_count = (*stop_count) + 1;
	    }
	    else
	    {
		exit(4);	/* User aborted */
	    }
	    break;

	  case SIGTERM:
	    exit(4);	/* User aborted */
	    break;

	  case SIGSEGV:
	    exit(8);	/* Segmentation fault */
	    break;

	  case SIGSTOP:
	    break;

	  case SIGCONT:
	    break;
	}
}

/*
 *	Main timeout callback.
 */
static gint MainTOCB(gpointer data)
{
	gint total_windows = 0;
	GList *glist;
	win_struct *win;
	core_struct *core = CORE(data);
	if(core == NULL)
	    return(FALSE);

	/* Manage all windows, count the number of windows that are
	 * mapped and delete any unmapped windows
	 */
	for(glist = core->win;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    win = WIN(glist->data);
	    if(win == NULL)
		continue;

	    if(WinIsMapped(win))
	    {
		total_windows++;
	    }
	    else
	    {
		WinDelete(win);
		glist->data = NULL;
		continue;
	    }
	}

	/* If no windows are mapped then break out of the main loop */
	if((total_windows == 0) || core->close_all_windows)
	{
	    gtk_main_quit();
	    return(FALSE);
	}

	return(TRUE);
}


/*
 *	Creates a new default Configuration List.
 */
static cfg_item_struct *MainCfgListNew(void)
{
	const cfg_item_struct src_cfg_list[] = CFG_LIST;
	cfg_item_struct *cfg_list = CFGItemListCopyList(src_cfg_list);
	gchar cwd[PATH_MAX];

	if(getcwd(cwd, sizeof(cwd)) != NULL)
	    cwd[sizeof(cwd) - 1] = '\0';
	else
	    *cwd = '\0';

	CFGItemListSetValueI(
	    cfg_list, CFG_PARM_VERSION_MAJOR,
	    PROG_VERSION_MAJOR, FALSE
	);
	CFGItemListSetValueI(
	    cfg_list, CFG_PARM_VERSION_MINOR,
	    PROG_VERSION_MINOR, FALSE
	);
	CFGItemListSetValueI(
	    cfg_list, CFG_PARM_VERSION_RELEASE,
	    PROG_VERSION_RELEASE, FALSE
	);

	CFGItemListSetValueS(
	    cfg_list, CFG_PARM_DB_LOCATION,
	    AVSCAN_DB_LOCATION, FALSE
	);
	CFGItemListSetValueL(
	    cfg_list, CFG_PARM_DB_LAST_UPDATE,
	    0l, FALSE
	);
 
	CFGItemListSetValueI(
	    cfg_list, CFG_PARM_WIN_X,
	    0, FALSE
	);
	CFGItemListSetValueI(
	    cfg_list, CFG_PARM_WIN_Y,
	    0, FALSE
	);
	CFGItemListSetValueI(
	    cfg_list, CFG_PARM_WIN_SHOW_TOOL_BAR,
	    TRUE, FALSE
	);
	CFGItemListSetValueI(
	    cfg_list, CFG_PARM_WIN_SHOW_STATUS_BAR,
	    TRUE, FALSE
	);


	return(cfg_list);
}


/*
 *	Creates a new Win.
 */
win_struct *MainNewAVScan(core_struct *core, gint argc, gchar **argv)
{
	gchar *path;
	GList *glist;
	win_struct *win = WinNew(core);
	if(win == NULL)
	    return(NULL);

	GUIWindowApplyArgs(GTK_WINDOW(win->toplevel), argc, argv);

	/* Clear the scan settings */
	WinScanSettingsUpdate(win, NULL);

	/* Reset status progress, message, and time */
	WinStatusProgress(win, 0.0f, FALSE);
	WinStatusMessage(win, "Ready", FALSE);
	WinStatusTime(win, 0l, FALSE);

	/* Map the window */
	WinMap(win);

	WinSetBusy(win, TRUE);

	/* Open the scan list from file */
	path = g_strconcat(
	    core->prog_data_dir,
	    G_DIR_SEPARATOR_S,
	    AVSCAN_SCAN_LIST_FILE,
	    NULL
	);
	WinScanListOpenFromFile(win, path);
	g_free(path);

	WinSetBusy(win, FALSE);

	/* Add pointer to the list */
	for(glist = core->win;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    if(glist->data == NULL)
	    {
		glist->data = win;
		return(win);
	    }
	}
	core->win = g_list_append(core->win, win);

	return(win);
}


int main(int argc, char *argv[])
{
	gboolean initialized_gtk = FALSE;
	gint i;
	const gchar *s, *arg, *cwd;
	gchar *path;
	GdkWindow *root;
	cfg_item_struct *cfg_list;
#ifdef HAVE_EDV2
	edv_context_struct *ctx;
#endif
	win_struct *win;
	core_struct *core;

	/* Initialize the globals */
	avscan_scan_stop_count = NULL;

	/* Set up the signal callbacks */
	signal(SIGINT, MainSignalCB);
	signal(SIGTERM, MainSignalCB);
	signal(SIGSEGV, MainSignalCB);
	signal(SIGSTOP, MainSignalCB);
	signal(SIGCONT, MainSignalCB);
	signal(SIGPIPE, MainSignalCB);

	/* Get the current working directory */
	cwd = g_get_current_dir();
	if(cwd == NULL)
	   cwd = "/";

	/* Parse arguments */
	for(i = 1; i < argc; i++)
	{
	    arg = argv[i];
	    if(STRISEMPTY(arg))
		continue;

	    /* Help */
	    if(!g_strcasecmp(arg, "--help") ||
	       !g_strcasecmp(arg, "-help") ||
	       !g_strcasecmp(arg, "--h") ||
	       !g_strcasecmp(arg, "-h") ||
	       !g_strcasecmp(arg, "-?")
	    )
	    {
		g_print("%s", PROG_USAGE_MESG);
		return(0);
	    }
	    /* Version */
	    else if(!g_strcasecmp(arg, "--version") ||
		    !g_strcasecmp(arg, "-version")
	    )
	    {
		gchar *av_engine_name_version =
		    AVScanGetAVEngineNameVersionString();
		g_print(
"%s Version %s\n\
%s\n\
AntiVirus Engine: %s\n",
		    PROG_NAME_FULL, PROG_VERSION, PROG_COPYRIGHT,
		    (av_engine_name_version != NULL) ?
			av_engine_name_version : "None"
		);
		g_free(av_engine_name_version);
		return(0);
	    }
	    /* Scan */
	    else if(!g_strcasecmp(arg, "--scan"))
	    {
		/* Do scan procedure and return */
		gint status, stop_count = 0;
		gchar *db_path = NULL;
		GList *path_list = NULL;
		gboolean	recursive = FALSE,
				executables_only = FALSE,
				ignore_links = FALSE;

		/* Set the stop count pointer for the signal callback */
		avscan_scan_stop_count = &stop_count;

		/* Collect subsequent arguments */
		for(i++; i < argc; i++)
		{
		    arg = argv[i];
		    if(arg == NULL)
			continue;

		    /* Database Path */
		    if(!g_strcasecmp(arg, "--database") ||
		       !g_strcasecmp(arg, "-db")
		    )
		    {
			i++;
			arg = (i < argc) ? argv[i] : NULL;
			if(arg != NULL)
			    db_path = STRDUP(arg);
			else
			    g_printerr(
				"%s: Requires argument.\n",
				argv[i - 1]
			    );
		    }
		    /* Recursive */
		    else if(!g_strcasecmp(arg, "--recursive") ||
			    !g_strcasecmp(arg, "-r")
		    )
		    {
			recursive = TRUE;
		    }
		    /* Executables Only */
		    else if(!g_strcasecmp(arg, "--executables-only") ||
		            !g_strcasecmp(arg, "--executables_only") ||
			    !g_strcasecmp(arg, "-x")
		    )
		    {
			executables_only = TRUE;
		    }
		    /* Ignore Links */
		    else if(!g_strcasecmp(arg, "--ignore-links") ||
		            !g_strcasecmp(arg, "--ignore_links") ||
			    !g_strcasecmp(arg, "-l")
		    )
		    {
			ignore_links = TRUE;
		    }
		    /* All else assume path */
		    else if((*arg != '-') && (*arg != '+'))
		    {
			gchar *path = g_path_is_absolute(arg) ?
			    STRDUP(arg) :
			    g_strconcat(cwd, G_DIR_SEPARATOR_S, arg, NULL);
			path_list = g_list_append(path_list, path);
		    }
		}

		/* Do scan procedure */
		status = AVScanOP(
		    db_path,		/* Database Path */
		    path_list,		/* Paths to objects to scan */
		    recursive,		/* Recursive */
		    executables_only,	/* Executables Only */
		    ignore_links,	/* Ignore Links */
		    &stop_count
		);

		/* Unset the stop count pointer */
		avscan_scan_stop_count = NULL;

		g_list_foreach(path_list, (GFunc)g_free, NULL);
		g_list_free(path_list);

		g_free(db_path);

		return(status);
	    }
	}


	/* Initialize GTK as needed */
	if(!initialized_gtk)
	{
	    if(!gtk_init_check(&argc, &argv))
	    {
		g_printerr(
"Unable to initialize GTK.\n"
		);
		return(1);
	    }

	    /* Initialize GDK RGB buffers */
	    gdk_rgb_init();

	    initialized_gtk = TRUE;
	}

	root = GDK_ROOT_PARENT();

	/* Initialize dialogs */
	CDialogInit();
	PDialogInit();
	FPromptInit();
	FileBrowserInit();


	/* Begin creating the core */
	core = CORE(
	    g_malloc0(sizeof(core_struct))
	);

	/* Configuration List */
	core->cfg_list = cfg_list = MainCfgListNew();

#ifdef HAVE_EDV2
	/* Endeavour Mark II context */
	core->edv_ctx = ctx = EDVContextNew();
#endif

	/* Pixmaps */
	core->scan_item_pixmap = GDK_PIXMAP_NEW_FROM_XPM_DATA(
	    &core->scan_item_mask,
	    (guint8 **)icon_executable_20x20_xpm
	);
	core->file_infected_pixmap = GDK_PIXMAP_NEW_FROM_XPM_DATA(
	    &core->file_infected_mask,
	    (guint8 **)icon_file_infected_20x20_xpm
	);
	core->folder_closed_pixmap = GDK_PIXMAP_NEW_FROM_XPM_DATA(
	    &core->folder_closed_mask,
	    (guint8 **)icon_folder_closed_20x20_xpm
	);
	core->folder_opened_pixmap = GDK_PIXMAP_NEW_FROM_XPM_DATA(
	    &core->folder_opened_mask,
	    (guint8 **)icon_folder_opened_20x20_xpm
	);
	core->db_item_pixmap = GDK_PIXMAP_NEW_FROM_XPM_DATA(
	    &core->db_item_mask,
	    (guint8 **)icon_virus_pattern_20x20_xpm
	);

	/* Cursors */
	core->busy_cur = gdk_cursor_new(GDK_WATCH);
	if(TRUE) 
	{
	    GdkColor	c_fg = { 0l, 0xffff, 0xffff, 0xffff },
			c_bg = { 0l, 0x0000, 0x0000, 0x0000 };
	    const gint width = 24, height = 24;
	    const gchar color_mask_data[] = {
0x03, 0x00, 0x00,
0x05, 0x00, 0x00,
0x09, 0x00, 0x00,
0x11, 0x00, 0x00,
0x21, 0x00, 0x00,
0x41, 0x00, 0x00,
0x81, 0x00, 0x00,
0x01, 0x01, 0x00,
0x01, 0xc2, 0x7f,
0xc1, 0x03, 0x00,
0x49, 0x80, 0x3f,
0x95, 0x80, 0x3f,
0x93, 0x80, 0x35,
0x20, 0x01, 0x1b,
0x20, 0x01, 0x0e,
0xc0, 0x00, 0x04,
0x00, 0x00, 0x0a,
0x00, 0x00, 0x1f,
0x00, 0x80, 0x3b,
0x00, 0x80, 0x35,
0x00, 0x80, 0x2a,
0x00, 0x00, 0x00,
0x00, 0xc0, 0x7f,
0x00, 0x00, 0x00
	    };
	    const gchar trans_mask_data[] = {
0x03, 0x00, 0x00,
0x07, 0x00, 0x00,
0x0f, 0x00, 0x00,
0x1f, 0x00, 0x00,
0x3f, 0x00, 0x00,
0x7f, 0x00, 0x00,
0xff, 0x00, 0x00,
0xff, 0xe1, 0xff,
0xff, 0xe3, 0xff,
0xff, 0xe3, 0xff,
0x7f, 0xc0, 0x7f,
0xf7, 0xc0, 0x7f,
0xf3, 0xc0, 0x7f,
0xe0, 0x81, 0x3f,
0xe0, 0x01, 0x1f,
0xc0, 0x00, 0x0e,
0x00, 0x00, 0x1f,
0x00, 0x80, 0x3f,
0x00, 0xc0, 0x7f,
0x00, 0xc0, 0x7f,
0x00, 0xc0, 0x7f,
0x00, 0xe0, 0xff,
0x00, 0xe0, 0xff,
0x00, 0xe0, 0xff
	    };
	    GdkBitmap *color_mask = gdk_bitmap_create_from_data(
		root, color_mask_data, width, height
	    );
	    GdkBitmap *trans_mask = gdk_bitmap_create_from_data(
		root, trans_mask_data, width, height
	    );
	    core->passive_busy_cur = gdk_cursor_new_from_pixmap(
		color_mask, trans_mask,
		&c_fg, &c_bg,
		0, 0
	    );
	    GDK_BITMAP_UNREF(color_mask);
	    GDK_BITMAP_UNREF(trans_mask);
	}
	else
	{
	    core->passive_busy_cur = gdk_cursor_new(GDK_WATCH);
	}

	core->close_all_windows = FALSE;

	core->win = NULL;

	core->prog_name = NULL;
	core->prog_path = NULL;
	core->prog_data_dir = NULL;

#ifdef HAVE_EDV2
	/* Load Endeavour configuration from file */
	EDVContextLoadConfigurationFile(ctx, NULL);
#endif

	/* Get the full path to the program */
	arg = (argc > 0) ? argv[0] : "*unknown*";
	core->prog_name = STRDUP(arg);
	core->prog_path = STRDUP(arg);

#ifdef HAVE_EDV2
	/* Get the data directory */
	s = EDVGetS(ctx, EDV_CFG_PARM_DIR_LOCAL);
	if(s == NULL)
	    s = g_getenv("HOME");
	if(s == NULL)
	    s = cwd;
	core->prog_data_dir = STRDUP(s);
#else
	s = g_getenv("HOME");
	if(s == NULL)
	    s = cwd;
	core->prog_data_dir = STRDUP(s);
#endif

	/* Open the configuration from file */
	path = g_strconcat(
	    core->prog_data_dir,
	    G_DIR_SEPARATOR_S,
	    AVSCAN_CFG_FILE,
	    NULL
	);
	CFGFileOpen(path, cfg_list);
	g_free(path);


	/* Create the AVScan window */
	win = MainNewAVScan(core, argc, argv);
	if(win != NULL)
	{
	    GList *paths_list = NULL;

	    /* Parse the arguments */
	    for(i = 1; i < argc; i++)
	    {
		arg = argv[i];
		if(STRISEMPTY(arg))
		    continue;

		/* Database Path */
		if(!g_strcasecmp(arg, "--database") ||
		   !g_strcasecmp(arg, "-db")
		)
		{
		    i++;
		    arg = (i < argc) ? argv[i] : NULL;
		    if(arg != NULL)
			CFGItemListSetValueS(
			    cfg_list, CFG_PARM_DB_LOCATION,
			    arg, FALSE
			);
		    else
			g_printerr(
			    "%s: Requires argument.\n",
			    argv[i - 1]
			);
		}
		/* Recursive */
		else if(!g_strcasecmp(arg, "--recursive") ||
			!g_strcasecmp(arg, "-r")
		)
		{
		    gtk_toggle_button_set_active(
			GTK_TOGGLE_BUTTON(win->recursive_check),
			TRUE 
		    );
		}
		/* Executables Only */
		else if(!g_strcasecmp(arg, "--executables-only") ||
		        !g_strcasecmp(arg, "--executables_only") ||
			!g_strcasecmp(arg, "-x")
		)
		{
		    gtk_toggle_button_set_active(
			GTK_TOGGLE_BUTTON(win->executables_only_check),
			TRUE
		    );
		}
		/* Ignore Links */
		else if(!g_strcasecmp(arg, "--ignore-links") ||
		        !g_strcasecmp(arg, "--ignore_links") ||
			!g_strcasecmp(arg, "-l")
		)
		{
		    gtk_toggle_button_set_active(
			GTK_TOGGLE_BUTTON(win->ignore_links_check),
			TRUE
		    );
		}
		/* Skip GUI arguments that have a an argument */
		else if(!g_strcasecmp(arg, "--geometry") ||
			!g_strcasecmp(arg, "-geometry") ||
			!g_strcasecmp(arg, "--fn") ||
			!g_strcasecmp(arg, "-fn") ||
			!g_strcasecmp(arg, "--fg") ||
			!g_strcasecmp(arg, "-fg") ||
			!g_strcasecmp(arg, "--bg") ||
			!g_strcasecmp(arg, "-bg") ||
			!g_strcasecmp(arg, "--sfg") ||
			!g_strcasecmp(arg, "-sfg") ||
			!g_strcasecmp(arg, "--sbg") ||
			!g_strcasecmp(arg, "-sbg") ||
			!g_strcasecmp(arg, "--cfg") ||
			!g_strcasecmp(arg, "-cfg") ||
			!g_strcasecmp(arg, "--cbg") ||
			!g_strcasecmp(arg, "-cbg") ||
			!g_strcasecmp(arg, "--bg-pixmap") ||
			!g_strcasecmp(arg, "-bg-pixmap") ||
			!g_strcasecmp(arg, "--sbg-pixmap") ||
			!g_strcasecmp(arg, "-sbg-pixmap") ||
			!g_strcasecmp(arg, "--name") ||
			!g_strcasecmp(arg, "-name") ||
			!g_strcasecmp(arg, "--class") ||
			!g_strcasecmp(arg, "-class") ||
			!g_strcasecmp(arg, "--title") ||
			!g_strcasecmp(arg, "-title")
		)
		{
		    i++;
		}
		/* All else assume path */
		else if((*arg != '-') && (*arg != '+'))
		{
		    gchar *path = g_path_is_absolute(arg) ?
			STRDUP(arg) :
			g_strconcat(cwd, G_DIR_SEPARATOR_S, arg, NULL);
		    paths_list = g_list_append(paths_list, path);
		}
	    }

	    /* Need to start scan on startup? */
	    if(paths_list != NULL)
	    {
		gchar *s, *location = STRDUP("");
		GList *glist;
		GtkEntry *entry = GTK_ENTRY(win->location_entry);

		for(glist = paths_list; glist != NULL; glist = g_list_next(glist))
		{
		    s = g_strconcat(
			location,
			(const gchar *)glist->data,
			(g_list_next(glist) != NULL) ? "," : "",
			NULL
		    );
		    g_free(location);
		    location = s;
		}
		gtk_entry_set_text(entry, location);
		gtk_entry_set_position(entry, -1);
		g_free(location);

		WinStartCB(win->start_btn, win);

		g_list_foreach(paths_list, (GFunc)g_free, NULL);
		g_list_free(paths_list);
	    }
	}

	/* Set main timeout callback */
	gtk_timeout_add(1000l, MainTOCB, core);
	gtk_main();


	/* Save the configuration to file */
	path = g_strconcat(
	    core->prog_data_dir,
	    G_DIR_SEPARATOR_S,
	    AVSCAN_CFG_FILE,
	    NULL
	);
	CFGFileSave(path, cfg_list);
	g_free(path);


	/* Begin deleting core */
	g_list_foreach(core->win, (GFunc)WinDelete, NULL);
	g_list_free(core->win);
	core->win = NULL;

	/* Cursors */
	GDK_CURSOR_DESTROY(core->busy_cur);
	GDK_CURSOR_DESTROY(core->passive_busy_cur);

	/* Pixmaps */
	GDK_PIXMAP_UNREF(core->scan_item_pixmap);
	GDK_BITMAP_UNREF(core->scan_item_mask);
	GDK_PIXMAP_UNREF(core->file_infected_pixmap);
	GDK_BITMAP_UNREF(core->file_infected_mask);
	GDK_PIXMAP_UNREF(core->folder_closed_pixmap);
	GDK_BITMAP_UNREF(core->folder_closed_mask);
	GDK_PIXMAP_UNREF(core->folder_opened_pixmap);
	GDK_BITMAP_UNREF(core->folder_opened_mask);
	GDK_PIXMAP_UNREF(core->db_item_pixmap);
	GDK_BITMAP_UNREF(core->db_item_mask);

	/* Configuration List */
	CFGItemListDeleteList(core->cfg_list);

#ifdef HAVE_EDV2
	/* Endeavour Mark II context */
	EDVContextDelete(core->edv_ctx);
#endif

	g_free(core->prog_name);
	g_free(core->prog_path);
	g_free(core->prog_data_dir);

	g_free(core->title);
	g_free(core->geometry);

	g_free(core);
	core = NULL;


	/* Shutdown dialogs */
	FileBrowserShutdown();
	FPromptShutdown();
	PDialogShutdown();
	CDialogShutdown();

	return(0);
}
