/*--------------------------------------------------------------------*/
/*--- Callgrind                                                    ---*/
/*---                                                       hash.c ---*/
/*--------------------------------------------------------------------*/

/*
   This file is part of Callgrind, a Valgrind tool for call tracing.

   Copyright (C) 2002-2005, Josef Weidendorfer (Josef.Weidendorfer@gmx.de)

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.

   The GNU General Public License is contained in the file COPYING.
*/


// Generic resizable hash table implementation
// See hash.h for documentation

#include "hash.h"

void CLG_(rhash_init)(rhash* h)
{
  Int i;

  h->entries = 0;
  h->resizes = 0;
  h->table   = (rhash_entry**) CLG_MALLOC(h->size * sizeof(rhash_entry*));

  for (i = 0; i < h->size; i++)
     h->table[i] = 0;

  CLG_DEBUG(0, "rhash_init: Initial size of %s: %d\n", h->name, h->size);

}

void CLG_(rhash_resize)(rhash* h, UInt new_size)
{
  UInt i, conflicts1 = 0, conflicts2 = 0;
  rhash_entry **new_table, *curr, *next;
  UInt new_idx;

  new_table = (rhash_entry**) CLG_MALLOC(new_size * sizeof(rhash_entry*));

  if (!new_table) return;

  for (i = 0; i < new_size; i++)
    new_table[i] = NULL;

  for (i = 0; i < h->size; i++) {
    if (h->table[i] == NULL) continue;

    curr = h->table[i];
    while (NULL != curr) {
      next = curr->next;

      new_idx = (UInt) curr->hash % new_size;

      curr->next = new_table[new_idx];
      new_table[new_idx] = curr;
      if (curr->next) {
	conflicts1++;
	if (curr->next->next)
	  conflicts2++;
      }
      
      curr = next;
    }
  }

  VG_(free)(h->table);


  CLG_DEBUG(0, "Resize %s hashtable: %d => %d (entries %d, conflicts %d/%d)\n",
	   h->name, h->size, new_size,
	   h->entries, conflicts1, conflicts2);

  h->size  = new_size;
  h->table = new_table;
  h->resizes++;
}


static rhash_entry* new_entry(rhash* h, void* data)
{
  UInt idx;
  rhash_entry* new = (*h->new_entry)(data);
  if (!new) return 0;

  new->hash = h->hash_value ? (*h->hash_value)(data) : (UWord)data;

  h->entries++;
  if (10* h->entries / h->size > 8)
    CLG_(rhash_resize)(h, 2* h->size +3);

  idx = (UInt) new->hash % h->size;
  new->next = h->table[idx];
  h->table[idx] = new;

  CLG_DEBUG(3, "rhash: New_entry in %s, now %d\n", h->name, h->entries);

  return new;
}

rhash_entry* CLG_(rhash_lookup)(rhash* h, void* data)
{
  UWord hash = h->hash_value ? (*h->hash_value)(data) : (UWord)data;
  UInt idx = (UInt) hash % h->size;
  rhash_entry* e = h->table[idx];
 
  while(e) {
      if (e->hash == hash) {
	  if (h->has_key == 0) break;
	  if ( (*h->has_key)(e, data) ) break;
      }
      e = e->next;
  }
  return e;
}

rhash_entry* CLG_(rhash_get)(rhash* h, void* data)
{
  rhash_entry* e = CLG_(rhash_lookup)(h, data); 

  if (!e)
    e = new_entry(h,data);

  return e;
}

rhash_entry* CLG_(rhash_remove)(rhash* h, void* data)
{
  UWord hash = h->hash_value ? (*h->hash_value)(data) : (UWord)data;
  UInt idx = (UInt) hash % h->size;
  rhash_entry* e = h->table[idx];
  rhash_entry** pe = &(h->table[idx]);
 
  while(e) {
      if (e->hash == hash) {
	  if (h->has_key == 0) break;
	  if ( (*h->has_key)(e, data) ) break;
      }
      pe = &(e->next);
      e = e->next;
  }

  if (e) {
      *pe = e->next;
      h->entries--;
  }

  CLG_DEBUG(3, "rhash_remove in %s, %sfound, now %d\n",
	    h->name, e ? "":"NOT ", h->entries);
      
  return e;
}

void CLG_(rhash_forall)(rhash* h, void (*f)(rhash_entry*) )
{
  UInt i;
  rhash_entry* e;

  for(i=0;i<h->size; i++) {
    e = h->table[i];
    while(e) {
      (*f)(e);
      e = e->next;
    }
  }
}
