/*
				Endeavour MIME Type
 */

#ifndef EDV_MIME_TYPE_H
#define EDV_MIME_TYPE_H

#include <glib.h>
#include "edv_types.h"


typedef struct _edv_mime_type_command_struct	edv_mime_type_command_struct;
#define EDV_MIME_TYPE_COMMAND(p)	((edv_mime_type_command_struct *)(p))
typedef struct _edv_mime_type_struct	edv_mime_type_struct;
#define EDV_MIME_TYPE(p)		((edv_mime_type_struct *)(p))


/*
 *	Version <= 2.6.x MIME TYpe compatability
 *
 *	These are equvical declarations for MIME Type C identifier names
 *	used in version 2.6.x and older.
 */
typedef struct _edv_mime_type_struct	edv_mimetype_struct;
#define EDV_MIMETYPE(p)			((edv_mime_type_struct *)(p))

#define EDV_MIMETYPE_CLASS_SYSTEM	EDV_MIME_TYPE_CLASS_SYSTEM
#define EDV_MIMETYPE_CLASS_FORMAT	EDV_MIME_TYPE_CLASS_FORMAT
#define EDV_MIMETYPE_CLASS_PROGRAM	EDV_MIME_TYPE_CLASS_PROGRAM
#define EDV_MIMETYPE_CLASS_UNIQUE	EDV_MIME_TYPE_CLASS_UNIQUE
#define edv_mimetype_class		edv_mime_type_class

#define EDV_MIMETYPE_HANDLER_COMMAND	EDV_MIME_TYPE_HANDLER_COMMAND
#define EDV_MIMETYPE_HANDLER_EDV_ARCHIVER	EDV_MIME_TYPE_HANDLER_EDV_ARCHIVER
#define EDV_MIMETYPE_HANDLER_EDV_IMAGE_BROWSER	EDV_MIME_TYPE_HANDLER_EDV_IMAGE_BROWSER
#define EDV_MIMETYPE_HANDLER_EDV_RECYCLE_BIN	EDV_MIME_TYPE_HANDLER_EDV_RECYCLE_BIN
#define edv_mimetype_handler		edv_mime_type_handler

#define EDV_MIMETYPE_ICON_STATE_STANDARD	EDV_MIME_TYPE_ICON_STATE_STANDARD
#define EDV_MIMETYPE_ICON_STATE_SELECTED	EDV_MIME_TYPE_ICON_STATE_SELECTED
#define EDV_MIMETYPE_ICON_STATE_EXTENDED	EDV_MIME_TYPE_ICON_STATE_EXTENDED
#define EDV_MIMETYPE_ICON_STATE_HIDDEN	EDV_MIME_TYPE_ICON_STATE_HIDDEN
#define EDV_MIMETYPE_TOTAL_ICON_STATES	4
#define edv_mimetype_icon_state		edv_mime_type_icon_state


/*
 *	MIME Type Icon States:
 */
typedef enum{
	EDV_MIME_TYPE_ICON_STATE_STANDARD,
	EDV_MIME_TYPE_ICON_STATE_SELECTED,	/* Directories = Expanded
						 * Other = Selected */
	EDV_MIME_TYPE_ICON_STATE_EXTENDED,	/* Links = Broken
						 * Other = Active/Opened/Reserved */
	EDV_MIME_TYPE_ICON_STATE_HIDDEN
} edv_mime_type_icon_state;
#define EDV_MIME_TYPE_TOTAL_ICON_STATES	4


/*
 *      Inode MIME Types:
 *
 *      Used for EDV_MIME_TYPE_CLASS_SYSTEM MIME Types.
 */
#define EDV_MIME_TYPE_TYPE_INODE_UNKNOWN	"inode/unknown"
#define EDV_MIME_TYPE_TYPE_INODE_ERROR		"inode/error"
#define EDV_MIME_TYPE_TYPE_INODE_FILE		"inode/file"
#define EDV_MIME_TYPE_TYPE_INODE_EXECUTABLE	"inode/executable"
#define EDV_MIME_TYPE_TYPE_INODE_DIRECTORY	"inode/directory"
#define EDV_MIME_TYPE_TYPE_INODE_LINK		"inode/link"
#define EDV_MIME_TYPE_TYPE_INODE_DEV_BLOCK	"inode/dev-block"
#define EDV_MIME_TYPE_TYPE_INODE_DEV_CHARACTER	"inode/dev-character"
#define EDV_MIME_TYPE_TYPE_INODE_FIFO		"inode/fifo"
#define EDV_MIME_TYPE_TYPE_INODE_SOCKET		"inode/socket"


/*
 *	MIME Type Command:
 */
struct _edv_mime_type_command_struct {
	gchar		*name,
			*command;
};


/*
 *	MIME Type:
 */
struct _edv_mime_type_struct {

	edv_mime_type_class	mt_class;

	/* Value, its contents is determined by mt_class:
	 *
	 * mt_class			value is a
	 * --------                     ----------
	 * EDV_MIME_TYPE_CLASS_SYSTEM	*Ignored*
	 * EDV_MIME_TYPE_CLASS_FORMAT	Space separated list of extensions
	 * EDV_MIME_TYPE_CLASS_PROGRAM	Path to the program object
	 * EDV_MIME_TYPE_CLASS_UNIQUE	Path to the unique object
	 */
	gchar		*value;

	/* The MIME Type's type (aka the MIME Type's name) */
	gchar		*type;

	/* A one line verbose description */
	gchar		*description;

	/* Small 20x20 icons */
	gchar		*small_icon_file[EDV_MIME_TYPE_TOTAL_ICON_STATES];

	/* Medium 32x32 icons */
	gchar		*medium_icon_file[EDV_MIME_TYPE_TOTAL_ICON_STATES];

	/* Large 48x48 icons */
	gchar		*large_icon_file[EDV_MIME_TYPE_TOTAL_ICON_STATES];

	/* Handler, determines if a command should be used to open
	 * objects assicated with this MIME Type (in the case of member
	 * handler = EDV_MIME_TYPE_HANDLER_COMMAND) or to use one of
	 * Endeavour's windows to open the object
	 */
	edv_mime_type_handler	handler;

	/* Commands List
	 *
	 * Each item in the list points to a
	 * edv_mime_type_command_struct
	 *
	 * Command are only used if the handler is set to
	 * EDV_MIME_TYPE_HANDLER_COMMAND
	 *
	 * The first command is considered the default
	 *
	 * The intention of each command is related by the mt_class as
	 * follows:
	 *
	 * mt_class			value is a
	 * --------			----------
	 * EDV_MIME_TYPE_CLASS_SYSTEM	*Ignored*
	 * EDV_MIME_TYPE_CLASS_FORMAT	Program commands or program aliases
	 * EDV_MIME_TYPE_CLASS_PROGRAM	Program commands (no aliases)
	 * EDV_MIME_TYPE_CLASS_UNIQUE	Program commands or program aliases
	 */
	GList           *commands_list;

	/* Time stamps */
	gulong		access_time,	/* In systime seconds */
			modify_time,
			change_time;

};


/*
 *	Gets the command value of the MIME Type command that matches
 *	the specified command name.
 */
extern const gchar *EDVMimeTypeGetCommandValueByName(
	edv_mime_type_struct *m, const gchar *name
);

/*
 *	Gets the MIME Type command that matches the specified command
 *	name.
 */
extern edv_mime_type_command_struct *EDVMimeTypeGetCommandByName(
	edv_mime_type_struct *m, const gchar *name
);


/*
 *	Creates a new MIME Type command.
 */
extern edv_mime_type_command_struct *EDVMimeTypeCommandNew(void);

/*
 *	Deletes the MIME Type command.
 */
extern void EDVMimeTypeCommandDelete(edv_mime_type_command_struct *cmd);


/*
 *      Creates a new MIME Type.
 */
extern edv_mime_type_struct *EDVMimeTypeNew(
	const edv_mime_type_class mt_class,
	const gchar *value,
	const gchar *type,		/* MIME Type name */
	const gchar *description	/* Short verbose description */
);

/*
 *      Coppies the MIME Type.
 */
extern edv_mime_type_struct *EDVMimeTypeCopy(const edv_mime_type_struct *m);

/*
 *      Deletes the MIME Type.
 */
extern void EDVMimeTypeDelete(edv_mime_type_struct *m);


#endif	/* EDV_MIME_TYPE_H */
