/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qfile.h>
#include <qregexp.h>
#include <qlistbox.h>
#include <qcheckbox.h>
#include <qspinbox.h>

#include "powerkadu.h"
#include "cmdline.h"
#include "cmdline_hint.h"
#include "cmdline_hist.h"

#include "usergroup.h"
#include "chat_manager.h"
#include "chat.h"
#include "config_dialog.h"
#include "misc.h"
#include "debug.h"
#include "custom_input.h"
#include "gadu_rich_text.h"
#include "emoticons.h"


CmdLine::CmdLine()
{
	kdebugf();
	history = new CmdLineHistory();
	connect(chat_manager, SIGNAL(chatCreated(const UserGroup*)), this, SLOT(chatCreated(const UserGroup*)));
	ChatList chatList = chat_manager->chats();
	for ( ChatList::iterator it = chatList.begin(); it != chatList.end(); ++it )
		chatCreated(*it);

	addCmd("help", this, SLOT( helpCmd(Chat*, const UserGroup*, QString&, QStringList&, QCString&) ));

	ConfigDialog::addVGroupBox("PowerKadu", "PowerKadu messages",
			QT_TRANSLATE_NOOP("@default", "Ignored messages"), NULL, Advanced);
	ConfigDialog::addCheckBox("PowerKadu", "Ignored messages",
			QT_TRANSLATE_NOOP("@default", "Let all unknown commands through"),
			"powerkadu_process_unknown_messages", false, NULL, NULL, Advanced);
	ConfigDialog::addListBox("PowerKadu", "Ignored messages",
			QT_TRANSLATE_NOOP("@default", "Ignored messages list"),
			QT_TRANSLATE_NOOP("@default", "Add the commands to be ignored by PowerKadu, that\n"
				"are used by another modules, e. g. \"shell\", or \"rshell\" in shellexec module."),
			NULL, Advanced);


	ConfigDialog::addHBox("PowerKadu", "Ignored messages", "Ignored Messages buttons and text",
			NULL, Advanced);
	ConfigDialog::addPushButton("PowerKadu", "Ignored Messages buttons and text",
			QT_TRANSLATE_NOOP("@default", "Add command"), NULL,
			QT_TRANSLATE_NOOP("@default", "Adds the command to the list of ignored messages."),
			NULL, Advanced);
	ConfigDialog::addPushButton("PowerKadu", "Ignored Messages buttons and text",
			QT_TRANSLATE_NOOP("@default", "Remove command"), NULL,
			QT_TRANSLATE_NOOP("@default", "Removes the command from the list of ignored messages."),
			NULL, Advanced);
	ConfigDialog::addLineEdit2("PowerKadu", "Ignored Messages buttons and text",
			QT_TRANSLATE_NOOP("@default", "Command: "),
			NULL, NULL, NULL, Advanced);

	ConfigDialog::registerSlotOnCreateTab("PowerKadu", this, SLOT(onCreateTab()));
	ConfigDialog::registerSlotOnApplyTab("PowerKadu", this, SLOT(onApplyTab()));
	ConfigDialog::registerSlotOnCloseTab("PowerKadu", this, SLOT(onCloseTab()));

	readCfg();

	kdebugf2();
}

void CmdLine::onCreateTab()
{
	kdebugf();
	QCheckBox *letUnknownThrough = ConfigDialog::getCheckBox("PowerKadu", "Let all unknown commands through");
	QPushButton *addButton = ConfigDialog::getPushButton("PowerKadu", "Add command");
	QPushButton *removeButton = ConfigDialog::getPushButton("PowerKadu", "Remove command");
	QLineEdit *command = ConfigDialog::getLineEdit("PowerKadu", "Command: ");
	QListBox *commandsList = ConfigDialog::getListBox("PowerKadu", "Ignored messages list");

	commandsList->setEnabled(!letUnknownThrough->isChecked());
	command->setEnabled(!letUnknownThrough->isChecked());
	addButton->setEnabled(false);
	removeButton->setEnabled(false);
	commandsList->insertStringList(ignoredCmdList);
	connect(addButton, SIGNAL(clicked()), this, SLOT(onAddCommand()));
	connect(removeButton, SIGNAL(clicked()), this, SLOT(onRemoveCommand()));
	connect(letUnknownThrough, SIGNAL(clicked()), this, SLOT(onProcessUnknownClicked()));

	connect(commandsList, SIGNAL(highlighted(int)), this, SLOT(onHighlighted(int)));
	connect(command, SIGNAL(textChanged(const QString &)), this, SLOT(onCommandChanged(const QString &)));
	kdebugf2();
}

void CmdLine::onApplyTab()
{
	kdebugf();
	QListBox *commandsList = ConfigDialog::getListBox("PowerKadu", "Ignored messages list");
	ignoredCmdList.clear();
	for(uint i = 0; i < commandsList->count(); i++)
		ignoredCmdList += commandsList->text(i);
	kdebugf2();
}

void CmdLine::onCloseTab()
{
	kdebugf();
	QCheckBox *letUnknownThrough = ConfigDialog::getCheckBox("PowerKadu", "Let all unknown commands through");
	QPushButton *addButton = ConfigDialog::getPushButton("PowerKadu", "Add command");
	QPushButton *removeButton = ConfigDialog::getPushButton("PowerKadu", "Remove command");
	QLineEdit *command = ConfigDialog::getLineEdit("PowerKadu", "Command: ");
	QListBox *commandsList = ConfigDialog::getListBox("PowerKadu", "Ignored messages list");
	disconnect(addButton, SIGNAL(clicked()), this, SLOT(onAddCommand()));
	disconnect(removeButton, SIGNAL(clicked()), this, SLOT(onRemoveCommand()));
	disconnect(letUnknownThrough, SIGNAL(clicked()), this, SLOT(onProcessUnknownClicked()));

	disconnect(commandsList, SIGNAL(highlighted(int)), this, SLOT(onHighlighted(int)));
	disconnect(command, SIGNAL(textChanged(const QString &)), this, SLOT(onCommandChanged(const QString &)));
	kdebugf2();
}

void CmdLine::onProcessUnknownClicked()
{
	kdebugf();
	QLineEdit *command = ConfigDialog::getLineEdit("PowerKadu", "Command: ");
	QListBox *commandsList = ConfigDialog::getListBox("PowerKadu", "Ignored messages list");
	QCheckBox *letUnknownThrough = ConfigDialog::getCheckBox("PowerKadu", "Let all unknown commands through");
	// TODO: Push buttons should be enables / disabled too!!
	setButtons();
	commandsList->setEnabled(!letUnknownThrough->isChecked());
	command->setEnabled(!letUnknownThrough->isChecked());
	kdebugf2();
}

void CmdLine::onAddCommand()
{
	kdebugf();
	QPushButton *addButton = ConfigDialog::getPushButton("PowerKadu", "Add command");
	QPushButton *removeButton = ConfigDialog::getPushButton("PowerKadu", "Remove command");
	QLineEdit *command = ConfigDialog::getLineEdit("PowerKadu", "Command: ");
	QListBox *commandsList = ConfigDialog::getListBox("PowerKadu", "Ignored messages list");
	kdebugm(KDEBUG_INFO, "adres: %d\t%d\n", command, commandsList);
	if(command->text() != "" && !commandsList->findItem(command->text(), Qt::ExactMatch))
	{
		commandsList->insertItem(command->text());
		addButton->setEnabled(false);
		removeButton->setEnabled(true);
	}
	commandsList->sort();
	kdebugf2();
}

void CmdLine::onRemoveCommand()
{
	kdebugf();
	QLineEdit *command = ConfigDialog::getLineEdit("PowerKadu", "Command: ");
	QListBox *commandsList = ConfigDialog::getListBox("PowerKadu", "Ignored messages list");
	QListBoxItem *item;
	if( (item = commandsList->findItem(command->text(), Qt::ExactMatch)) )
		commandsList->removeItem(commandsList->index(item));
	kdebugf2();
}

void CmdLine::onHighlighted(int index)
{
	kdebugf();
	QListBox *commandsList = ConfigDialog::getListBox("PowerKadu", "Ignored messages list");
	QLineEdit *command = ConfigDialog::getLineEdit("PowerKadu", "Command: ");
	command->setText(commandsList->text(index));
	setButtons();
	kdebugf2();
}

void CmdLine::onCommandChanged(const QString &newCommand)
{
	kdebugf();
	setButtons();
	kdebugf2();
}

void CmdLine::setButtons()
{
	kdebugf();
	QCheckBox *letUnknownThrough = ConfigDialog::getCheckBox("PowerKadu", "Let all unknown commands through");
	QLineEdit *command = ConfigDialog::getLineEdit("PowerKadu", "Command: ");
	QListBox *commandsList = ConfigDialog::getListBox("PowerKadu", "Ignored messages list");
	QPushButton *addButton = ConfigDialog::getPushButton("PowerKadu", "Add command");
	QPushButton *removeButton = ConfigDialog::getPushButton("PowerKadu", "Remove command");
	kdebugm(KDEBUG_INFO, "command: %s\n", command->text().data());
	if(letUnknownThrough->isChecked())
	{
		addButton->setEnabled(false);
		removeButton->setEnabled(false);
	}
	else
	{
		if((command->text() != "") && (!commandsList->findItem(command->text(), Qt::ExactMatch)))
		        addButton->setEnabled(true);
		else
		        addButton->setEnabled(false);
		if((command->text() != "") && (commandsList->findItem(command->text(), Qt::ExactMatch)))
		        removeButton->setEnabled(true);
		else
		        removeButton->setEnabled(false);
	}
	kdebugf2();
}

CmdLine::~CmdLine()
{
	kdebugf();
	disconnect(chat_manager, SIGNAL(chatCreated(const UserGroup*)), this, SLOT(chatCreated(const UserGroup*)));
	ConfigDialog::unregisterSlotOnCreateTab("PowerKadu", this, SLOT(onCreateTab()));
	ConfigDialog::unregisterSlotOnApplyTab("PowerKadu", this, SLOT(onApplyTab()));
	ConfigDialog::unregisterSlotOnCloseTab("PowerKadu", this, SLOT(onCloseTab()));
	ConfigDialog::removeControl("PowerKadu", "Command: ");
	ConfigDialog::removeControl("PowerKadu", "Remove command");
	ConfigDialog::removeControl("PowerKadu", "Add command");
	ConfigDialog::removeControl("PowerKadu", "Ignored Messages buttons and text");
	ConfigDialog::removeControl("PowerKadu", "Ignored messages list");
	ConfigDialog::removeControl("PowerKadu", "Let all unknown commands through");
	ConfigDialog::removeControl("PowerKadu", "Ignored messages");

	writeCfg();
	kdebugf2();
}

void CmdLine::readCfg()
{
	kdebugf();
	QString commands = powerKadu->cfg()->readEntry("PowerKadu", "powerkadu_ignored_cmds_list");
	if(commands != "")
		ignoredCmdList = QStringList::split(',', commands);
	else
	{
		ignoredCmdList += "shell";
		ignoredCmdList += "rshell";
	}
	kdebugf2();
}

void CmdLine::writeCfg()
{
	kdebugf();
	QString commands;
	for(QStringList::Iterator it = ignoredCmdList.begin(); it != ignoredCmdList.end(); ++it)
		commands += "," + (*it);
	commands = commands.right(commands.length() - 1);
	powerKadu->cfg()->writeEntry("PowerKadu", "powerkadu_ignored_cmds_list", commands);
	kdebugf2();
}

void CmdLine::chatCreated(const UserGroup *group)
{
	kdebugf();
	chatCreated( chat_manager->findChat(group) );
	kdebugf2();
}

void CmdLine::chatCreated(Chat *chat)
{
	kdebugf();
	connect(
		chat, SIGNAL( messageFiltering(const UserGroup*, QCString&, bool&) ),
		this, SLOT( handleChatMsg(const UserGroup*, QCString&, bool&) )
	);
	connect(
		chat->edit(), SIGNAL( keyPressed(QKeyEvent*, CustomInput*, bool&) ),
		this, SLOT( handleChatKey(QKeyEvent*, CustomInput*, bool&) )
	);
	connect(
		chat->edit(), SIGNAL( keyReleased(QKeyEvent*, CustomInput*, bool&) ),
		this, SLOT( handleChatKeyRelease(QKeyEvent*, CustomInput*, bool&) )
	);
	kdebugf2();
}

void CmdLine::handleChatMsg(const UserGroup* users, QCString& msg, bool &stop)
{
	kdebugf();

	Chat *chat = chat_manager->findChat(users);
	QString _msg = chat->edit()->textLine(0);
	QStringList args = QStringList::split(" ", _msg);
	if (args.count() < 1)
		return;

	// Input history
	history->messageSent( chatUniqKey(chat), chat->edit()->text() );

	// Command handling
	QString cmd = args[0];
	args.remove(args.begin());
	if (cmd[0] == '/')
	{
		cmd = cmd.right(cmd.length()-1);
		stop = true;

		if (cmdList.findIndex(cmd) > -1)
		{
			chat->edit()->setText("");
			emit cmdCall(chat, users, cmd, args, msg);
		}
		else if((ignoredCmdList.findIndex(cmd) == -1) &&
			(!config_file.readBoolEntry("PowerKadu", "powerkadu_process_unknown_messages", false)))
		{
			QString notFound = tr("No such command. Use: /help");
			powerKadu->showPkMsg(chat, notFound);
		}
		else
			stop = false;
	}

	kdebugf2();
}

void CmdLine::addCmd(QString cmd, QObject *receiver, const char* slot)
{
	kdebugf();
	if (cmdList.findIndex(cmd) == -1)
		cmdList.append(cmd);

	connect(
		this, SIGNAL(cmdCall(Chat*, const UserGroup*, QString&, QStringList&, QCString&)),
		receiver, slot
	);
	kdebugf2();
}

void CmdLine::helpCmd(Chat* chat, const UserGroup* users, QString &cmd, QStringList &args, QCString &message)
{
	kdebugf();
	CHECK_CMD("help");
	QString msg;
	if (args.count() == 0)
	{
		msg = tr("Commands list:")+" "+cmdList.join(", ")+"<br>"
			+tr("For more see: /help <code>&lt;command&gt;</code>")+"<br>";
	}
	else
	{
		QString helpWith = args[0],
				syntax, desc;

		QFile helpFile(dataPath("kadu/modules/data/powerkadu/cmdhelp.data"));
		if (helpFile.open(IO_ReadOnly))
		{
			QTextStream s(&helpFile);
			QString line;
			bool found = false;
			while (!s.atEnd())
			{
				line = s.readLine();
				if (found)
				{
					if (line.find(QRegExp("\\[\\w+\\]")) > -1)
					{
						break;
					}
					else
					{
						if (line.find("syntax:") == 0)
						{
							QStringList tmp = QStringList::split(" ", line);
							tmp.remove(tmp.begin());
							tmp.remove(tmp.begin());
							syntax = tmp.join(" ");
						}
						else
						{
							if (line.find("desc:") == -1)
							{
								desc += line+" ";
							}
						}
					}
				}
				else
				{
					if (line == "["+helpWith+"]")
					{
						found = true;
					}
				}
			}
			helpFile.close();
			if (found)
				msg = tr("Syntax: <code><b>%1</b> %2</code><br><br>%3").arg(helpWith).arg(syntax).arg(desc);
			else
				msg = tr("No help for <b>%1</b> command.").arg(helpWith);
		}
		else
		{
			msg = tr("Error while trying to open help file!<br>(%1)").arg(helpFile.name());
		}
	}
	powerKadu->showPkMsg(chat, msg);
	kdebugf2();
}

void CmdLine::sortCmds()
{
	kdebugf();
	cmdList.sort();
	kdebugf2();
}

void CmdLine::handleChatKey(QKeyEvent* e, CustomInput* input, bool &handled)
{
	kdebugf();
	if ( e->state() == Qt::ControlButton )
	{
		if ( e->key() == Qt::Key_Up )
		{
			historyPrev( input );
			handled = true;
		}
		else if ( e->key() == Qt::Key_Down )
		{
			historyNext(input);
			handled = true;
		}
	}
	else if (e->key() == Qt::Key_Tab)
	{
		complete(input);
		handled = true;
	}

	kdebugf2();
}

void CmdLine::handleChatKeyRelease(QKeyEvent* e, CustomInput* input, bool &handled)
{
	if (e->key() == Qt::Key_Tab)
		handled = true;
}

void CmdLine::complete(CustomInput *input)
{
	kdebugf();

/*	QWidget *p = input->parentWidget();
	while (p && !p->inherits("Chat"))
		p = p->parentWidget();
	if (!p)
	{
		kdebugm(KDEBUG_INFO, "this widget is not in chat\n");
		return;
	}*/
	//Chat* chat = (Chat*)p;
	Chat* chat = getChatByInput(input);
	if (!chat)
	{
		kdebugm(KDEBUG_INFO, "input widget is not in chat\n");
		return;
	}

	QString txt = input->text();
	delSide = LEFT;

	txt.replace("\n", "\r\n");
	void *format; // just a template needed for unformatGGMessage
	unsigned int length; // just a template needed for unformatGGMessage
	txt = unformatGGMessage(txt, length, format);

	QStringList lines = QStringList::split("\n", txt);

	int para, index, wordStart, wordEnd;
	input->getCursorPosition(&para, &index);
	QString line = lines[para];
	wordStart = line.findRev(QRegExp("[\\s\\n]"), index-1);
	wordEnd = line.find(QRegExp("[\\s\\n]"), index);
	if (wordStart == -1)
		wordStart = 0;
	else
		wordStart++;

	if (wordEnd == -1)
		wordEnd = line.length();

	if (wordStart == wordEnd)
		return;

	if (wordStart >= index)
		delSide = RIGHT;

	QString wordToComplete = line.mid(wordStart, wordEnd-wordStart);

	CompletionType cType = UNDEFINED;
	if (para == 0 && wordStart == 0)
	{
		if (wordToComplete[0] == '/')
			cType = COMMAND;
	}
	if (cType == UNDEFINED)
	{
		if (QRegExp("^\\<[\\S\\ ]*\\>?$").exactMatch(wordToComplete))
			cType = EMOTICON;
		else
			cType = NICK_OR_UID;
	}

	QStringList matches;
	QMap<QString,QString> desc;
	switch (cType)
	{
		case COMMAND:
		{
			matches = cmdList.grep(
					QRegExp( "^"+QRegExp::escape( wordToComplete.mid(1, wordToComplete.length()-1) )+".*")
				);

			for ( uint i = 0; i < matches.count(); i++ )
				matches[i].prepend("/");

			break;
		}
		case EMOTICON:
		{
			QString content;
			if ( QRegExp("^\\<[\\S\\ ]*\\>$").exactMatch(wordToComplete) )
				content = wordToComplete.mid( 1, wordToComplete.length()-2 );
			else
				content = wordToComplete.mid( 1, wordToComplete.length()-1 );

			for ( int i = 0; i < emoticons->selectorCount(); i++ )
			{
				QString emot = emoticons->selectorString( i );
				if ( QRegExp("^\\<"+QRegExp::escape( content )+"[\\S\\ ]*\\>$", false).exactMatch( emot ) )
					matches.append( emot );
			}
			break;
		}
		case NICK_OR_UID:
		{
			UserListElements ules = userlist->toUserListElements();
			for ( uint i = 0; i < ules.count(); i++ )
			{
				UserListElement ule = ules[i];
				QString altNick = ule.altNick();
				if ( QRegExp( "^"+QRegExp::escape( wordToComplete )+".*", false ).exactMatch( altNick ) )
					matches.append( altNick );

				QStringList protocols = ule.protocolList();
				for ( uint j = 0; j < protocols.count(); j++ )
				{
					QString id = ule.ID( protocols[j] );
					if ( QRegExp( "^"+QRegExp::escape( wordToComplete )+".*" ).exactMatch( id ) )
					{
						matches.append( id );
						desc[id] = altNick;
					}
				}
			}
			break;
		}
		default:
			return;
	}

	deleteLength = wordToComplete.length();
	if (matches.count() == 1)
		put( matches[0], chat );
	else if (matches.count())
		new CmdLineHint(chat, matches, desc);

	kdebugf2();
}

void CmdLine::put(QString word, Chat* chat)
{
	/*
	 * Function uses QTextEdit::ActionBackspace instead of QTextEdit::ActionWordBackspace
	 * becouse emoticon patters won't work with 'word' actions (it's all about non-alpha chars).
	 */
	kdebugf();
	int para, index;
	chat->edit()->getCursorPosition(&para, &index);
	QString line = chat->edit()->text( para );
	if (delSide == LEFT)
	{
		// If word is on the left side from cursor,
		// then make sure that curosor is at the end of the word.
		int lastIndex = -1;
		while ( line[index] != ' ' && lastIndex != index && line[index] != '\n' )
		{
			lastIndex = index;
			chat->edit()->moveCursor(QTextEdit::MoveForward, false);
			chat->edit()->getCursorPosition(&para, &index);
		}

		// And then delete the word
		for ( int i = 0; i < deleteLength; i++ )
			chat->edit()->doKeyboardAction(QTextEdit::ActionBackspace);
	}
	else
	{
		for ( int i = 0; i < deleteLength; i++ )
			chat->edit()->doKeyboardAction(QTextEdit::ActionDelete);
	}

	chat->edit()->getCursorPosition(&para, &index);
	chat->edit()->insertAt(word, para, index);

	for ( uint i = 0; i < word.length(); i++ )
		chat->edit()->moveCursor(QTextEdit::MoveForward, false);

	kdebugf2();
}

Chat* CmdLine::getChatByInput(CustomInput *input)
{
	kdebugf();
	QWidget *p = input->parentWidget();
	while (p && !p->inherits("Chat"))
		p = p->parentWidget();

	if (!p)
		return 0;

	kdebugf2();
	return (Chat*)p;
}

QString CmdLine::chatUniqKey(Chat* chat)
{
	kdebugf();
	const UserGroup* users = chat->users();
	QStringList usersList = users->altNicks();
	usersList.sort();
	kdebugf2();
	return usersList.join("_");
}

void CmdLine::historyPrev(CustomInput *input)
{
	kdebugf();
	Chat* chat = getChatByInput(input);
	if (!chat)
	{
		kdebugm(KDEBUG_INFO, "input widget is not in chat\n");
		return;
	}

	QString key = chatUniqKey(chat);
	QString val = history->getPrev( key, input->text() );
	if ( val == QString::null )
		return;

	input->setText( val );
	input->moveCursor( QTextEdit::MoveHome, false );
	input->moveCursor( QTextEdit::MoveEnd, true );
	kdebugf2();
}

void CmdLine::historyNext(CustomInput *input)
{
	kdebugf2();
	Chat* chat = getChatByInput(input);
	if (!chat)
	{
		kdebugm(KDEBUG_INFO, "input widget is not in chat\n");
		return;
	}

	QString key = chatUniqKey(chat);
	QString val = history->getNext( key );
	if ( val == QString::null )
		return;

	input->setText( val );
	input->moveCursor( QTextEdit::MoveHome, false );
	input->moveCursor( QTextEdit::MoveEnd, true );
	kdebugf2();
}

CmdLineHistory* CmdLine::getCmdLineHistory()
{
	return history;
}
