/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qregexp.h>
#include <qstring.h>

#include "word_fix.h"
#include "powerkadu.h"

#include "chat_manager.h"
#include "chat.h"
#include "debug.h"
#include "html_document.h"
#include "usergroup.h"
#include "config_dialog.h"
#include "misc.h"

#include <qlistbox.h>
#include <qfile.h>

WordFix::WordFix(PowerKadu *pk)
{
	kdebugf();
	connect(chat_manager, SIGNAL(chatCreated(const UserGroup*)), this, SLOT(chatCreated(const UserGroup*)));
	connect(chat_manager, SIGNAL(chatDestroying(const UserGroup*)), this, SLOT(chatDestroying(const UserGroup*)));

	for ( uint i = 0; i < chat_manager->chats().count(); i++ )
		connectToChat(chat_manager->chats()[i]);

	// Loading list
	QString data = pk->cfg()->readEntry("PowerKadu", "WordFix_list", "");
	QStringList list = QStringList::split("\t\t", data);
	if (!list.count())
	{
		QFile defList(dataPath("kadu/modules/data/powerkadu/wf_default_list.data"));
		if (defList.open(IO_ReadOnly))
		{
			QTextStream s(&defList);
			QStringList pair;
			while (!s.atEnd())
			{
				pair = QStringList::split('|', s.readLine());
				if (pair.count() <= 0)
					continue;

				wordsList[pair[0]] = pair[1];
			}
			defList.close();
		}
		else
		{
			kdebug(QString("Can't open file: %1").arg(defList.name()));
		}
	}
	else
	{
		for ( uint i = 0; i < list.count(); i++ )
		{
			QStringList sp = QStringList::split('\t', list[i]);
			wordsList[sp[0]] = sp[1];
		}
	}

	// Config dialog
	ConfigDialog::addVGroupBox("PowerKadu", "PowerKadu", QT_TRANSLATE_NOOP("@default", "Words fix"));
		ConfigDialog::addHBox("PowerKadu", "Words fix", "Words fix HBOX");
			ConfigDialog::addListBox("PowerKadu", "Words fix HBOX", QT_TRANSLATE_NOOP("@default", "Words to fix list:"));
			ConfigDialog::addVBox("PowerKadu", "Words fix HBOX", "Words fix VBOX");
				ConfigDialog::addLineEdit2("PowerKadu", "Words fix VBOX", QT_TRANSLATE_NOOP("@default", "Current fix value: "), QString::null,
					QT_TRANSLATE_NOOP("@default", "Selected word will be replaced with value in this field."));
				ConfigDialog::addLineEdit2("PowerKadu", "Words fix VBOX", QT_TRANSLATE_NOOP("@default", "Change current fix value to: "), QString::null,
					QT_TRANSLATE_NOOP("@default", "Enter value which you want to replace selected word with.\nIt will be applied after you click 'Change selected'."));
				ConfigDialog::addPushButton("PowerKadu", "Words fix VBOX", QT_TRANSLATE_NOOP("@default", "Change selected"), QString::null,
					QT_TRANSLATE_NOOP("@default", "Applies value entered in above field to selected word."));
				ConfigDialog::addPushButton("PowerKadu", "Words fix VBOX", QT_TRANSLATE_NOOP("@default", "Delete selected"), QString::null,
					QT_TRANSLATE_NOOP("@default", "Deletes selected word from the list."));
		ConfigDialog::addVBox("PowerKadu", "Words fix", "Add new word to fix");
			ConfigDialog::addLineEdit2("PowerKadu", "Add new word to fix", QT_TRANSLATE_NOOP("@default", "New key word: "), QString::null,
				QT_TRANSLATE_NOOP("@default", "Enter new word to be added to the list."));
			ConfigDialog::addLineEdit2("PowerKadu", "Add new word to fix", QT_TRANSLATE_NOOP("@default", "New value word: "), QString::null,
				QT_TRANSLATE_NOOP("@default", "Enter value that new word will be replaced with."));
			ConfigDialog::addPushButton("PowerKadu", "Add new word to fix", QT_TRANSLATE_NOOP("@default", "Add new"), QString::null,
				QT_TRANSLATE_NOOP("@default", "Adds above entered word and its value to the list."));

	kdebugf2();
}

WordFix::~WordFix()
{
	kdebugf();
	disconnect(chat_manager, SIGNAL(chatCreated(const UserGroup*)), this, SLOT(chatCreated(const UserGroup*)));
	disconnect(chat_manager, SIGNAL(chatDestroying(const UserGroup*)), this, SLOT(chatDestroying(const UserGroup*)));

	for ( uint i = 0; i < chat_manager->chats().count(); i++ )
		disconnectFromChat(chat_manager->chats()[i]);

	QStringList list;
	for ( uint i = 0; i < wordsList.keys().count(); i++ )
	{
		QString word = wordsList.keys()[i];
		list.append(word+"\t"+wordsList[word]);
	}
	powerKadu->cfg()->writeEntry("PowerKadu", "WordFix_list", list.join("\t\t"));

	// Config dialog
	ConfigDialog::removeControl("PowerKadu", "Add new");
	ConfigDialog::removeControl("PowerKadu", "New value word: ");
	ConfigDialog::removeControl("PowerKadu", "New key word: ");
	ConfigDialog::removeControl("PowerKadu", "Add new word to fix");
	ConfigDialog::removeControl("PowerKadu", "Delete selected");
	ConfigDialog::removeControl("PowerKadu", "Change selected");
	ConfigDialog::removeControl("PowerKadu", "Change current fix value to: ");
	ConfigDialog::removeControl("PowerKadu", "Current fix value: ");
	ConfigDialog::removeControl("PowerKadu", "Words fix VBOX");
	ConfigDialog::removeControl("PowerKadu", "Words to fix list:");
	ConfigDialog::removeControl("PowerKadu", "Words fix HBOX");
	ConfigDialog::removeControl("PowerKadu", "Words fix");


	kdebugf2();
}

void WordFix::chatCreated(const UserGroup *group)
{
	kdebugf();
	connectToChat(chat_manager->findChat(group));
	kdebugf2();
}

void WordFix::chatDestroying(const UserGroup *group)
{
	kdebugf();
	disconnectFromChat(chat_manager->findChat(group));
	kdebugf2();
}

void WordFix::connectToChat(Chat* chat)
{
	kdebugf();
	connect(chat, SIGNAL(messageSendRequested(Chat*)), this, SLOT(sendRequest(Chat*)));
	kdebugf2();
}

void WordFix::disconnectFromChat(Chat* chat)
{
	kdebugf();
	disconnect(chat, SIGNAL(messageSendRequested(Chat*)), this, SLOT(sendRequest(Chat*)));
	kdebugf2();
}

void WordFix::sendRequest(Chat *chat)
{
	kdebugf();

	// Reading chat input to html document.
	HtmlDocument doc;
	doc.parseHtml( chat->edit()->text() );

	// Parsing and replacing.
	for ( int i = 0; i < doc.countElements(); i++ )
	{
		if (!doc.isTagElement(i))
			doReplace( doc.elementText(i) );
	}

	// Putting back corrected text.
	chat->edit()->setText( doc.generateHtml() );

	kdebugf2();
}

void WordFix::doReplace(QString &text)
{
	kdebugf();

	// Make text as " "+text+" ", to match words at the start and end of text.
	QString txt = text;
	txt = " "+txt+" ";

	// Replacing
	QString key;
	for ( uint i = 0; i < wordsList.keys().count(); i++ )
	{
		key = wordsList.keys()[i];
		txt.replace(
				QRegExp(
						"[\\s\\!\\.\\,\\(\\)\\[\\]\\;\\:\\\"\\']{1}"
						+key
						+"[\\s\\!\\.\\,\\(\\)\\[\\]\\;\\:\\\"\\']{1}"
					),
				" "+wordsList[key]+" "
			);
	}

	// Cutting off " " from start and end.
	txt = txt.mid(1, txt.length()-2);
	text = txt;
	kdebugf2();
}

void WordFix::onCreateConfig()
{
	kdebugf();

	QLineEdit* currValue = ConfigDialog::getLineEdit("PowerKadu", "Current fix value: ");
	QListBox *listbox = ConfigDialog::getListBox("PowerKadu", "Words to fix list:");
	QPushButton *change = ConfigDialog::getPushButton("PowerKadu", "Change selected");
	QPushButton *del = ConfigDialog::getPushButton("PowerKadu", "Delete selected");
	QPushButton *add = ConfigDialog::getPushButton("PowerKadu", "Add new");
	QLineEdit* newWord = ConfigDialog::getLineEdit("PowerKadu", "New key word: ");
	QLineEdit* newValue = ConfigDialog::getLineEdit("PowerKadu", "New value word: ");
	QLineEdit* changeToValue = ConfigDialog::getLineEdit("PowerKadu", "Change current fix value to: ");

	listbox->clear();
	listbox->insertStringList(wordsList.keys());
	currValue->setReadOnly(true);

	connect(listbox, SIGNAL(highlighted(const QString&)), this, SLOT(wordSelected(const QString&)));
	connect(change, SIGNAL(clicked()), this, SLOT(changeSelected()));
	connect(del, SIGNAL(clicked()), this, SLOT(deleteSelected()));
	connect(add, SIGNAL(clicked()), this, SLOT(addNew()));
	connect(newWord, SIGNAL(returnPressed()), this, SLOT(moveToNewValue()));
	connect(newValue, SIGNAL(returnPressed()), this, SLOT(addNew()));
	connect(changeToValue, SIGNAL(returnPressed()), this, SLOT(changeSelected()));

	kdebugf2();
}

void WordFix::onDestroyConfig()
{
	kdebugf();

	QListBox *listbox = ConfigDialog::getListBox("PowerKadu", "Words to fix list:");
	QPushButton *change = ConfigDialog::getPushButton("PowerKadu", "Change selected");
	QPushButton *del = ConfigDialog::getPushButton("PowerKadu", "Delete selected");
	QPushButton *add = ConfigDialog::getPushButton("PowerKadu", "Add new");
	QLineEdit* newWord = ConfigDialog::getLineEdit("PowerKadu", "New key word: ");
	QLineEdit* newValue = ConfigDialog::getLineEdit("PowerKadu", "New value word: ");
	QLineEdit* changeToValue = ConfigDialog::getLineEdit("PowerKadu", "Change current fix value to: ");

	disconnect(listbox, SIGNAL(highlighted(const QString&)), this, SLOT(wordSelected(const QString&)));
	disconnect(change, SIGNAL(clicked()), this, SLOT(changeSelected()));
	disconnect(del, SIGNAL(clicked()), this, SLOT(deleteSelected()));
	disconnect(add, SIGNAL(clicked()), this, SLOT(addNew()));
	disconnect(newWord, SIGNAL(returnPressed()), this, SLOT(moveToNewValue()));
	disconnect(newValue, SIGNAL(returnPressed()), this, SLOT(addNew()));
	disconnect(changeToValue, SIGNAL(returnPressed()), this, SLOT(changeSelected()));

	kdebugf2();
}

void WordFix::wordSelected(const QString &word)
{
	kdebugf();

	QLineEdit* currValue = ConfigDialog::getLineEdit("PowerKadu", "Current fix value: ");
	QLineEdit* changeToValue = ConfigDialog::getLineEdit("PowerKadu", "Change current fix value to: ");

	currValue->setText(wordsList[word]);
	changeToValue->setText(wordsList[word]);

	kdebugf2();
}

void WordFix::changeSelected()
{
	kdebugf();
	QLineEdit* currValue = ConfigDialog::getLineEdit("PowerKadu", "Current fix value: ");
	QLineEdit* changeToValue = ConfigDialog::getLineEdit("PowerKadu", "Change current fix value to: ");
	QListBox *listbox = ConfigDialog::getListBox("PowerKadu", "Words to fix list:");

	if (listbox->currentItem() < 0)
		return;

	QString word = listbox->currentText(),
			value = changeToValue->text();

	wordsList[word] = value;
	currValue->setText(value);
	kdebugf2();
}

void WordFix::deleteSelected()
{
	kdebugf();
	QLineEdit* currValue = ConfigDialog::getLineEdit("PowerKadu", "Current fix value: ");
	QLineEdit* changeToValue = ConfigDialog::getLineEdit("PowerKadu", "Change current fix value to: ");
	QListBox *listbox = ConfigDialog::getListBox("PowerKadu", "Words to fix list:");

	QString word = listbox->currentText();
	int idx = listbox->currentItem();
	if (idx < 0)
		return;

	listbox->removeItem(idx);
	currValue->setText("");
	changeToValue->setText("");
	wordsList.remove(word);
	kdebugf2();
}

void WordFix::addNew()
{
	kdebugf();
	QListBox *listbox = ConfigDialog::getListBox("PowerKadu", "Words to fix list:");
	QLineEdit* newWord = ConfigDialog::getLineEdit("PowerKadu", "New key word: ");
	QLineEdit* newValue = ConfigDialog::getLineEdit("PowerKadu", "New value word: ");

	QString word = newWord->text(),
			value = newValue->text();

	if (word.isEmpty())
		return;

	listbox->insertItem(word);
	wordsList[word] = value;

	newWord->setText("");
	newValue->setText("");

	kdebugf2();
}

void WordFix::moveToNewValue()
{
	kdebugf();
	QLineEdit* newValue = ConfigDialog::getLineEdit("PowerKadu", "New value word: ");
	newValue->setFocus();
	kdebugf2();
}
