#ifndef KADU_CHAT_H
#define KADU_CHAT_H

#include <qdatetime.h>
#include <qstring.h>
#include <qmainwindow.h>
#include <qvaluelist.h>

#include "config_file.h"
#include "custom_input.h"
#include "emoticons.h"
#include "gadu.h"
#include "usergroup.h"

class QAccel;
class QHBox;
class QMimeSourceFactory;
class QObject;
class QPushButton;

class Chat;
class ChatManager;
class ChatMessage;
class ChatStyle;
class ColorSelector;
class EmoticonSelector;
class KaduSplitter;
class KaduTextBrowser;
class OwnChatColors;
class UserBox;
class UserChatColors;

/** \typedef QValueList<Chat*> ChatList */
typedef QValueList<Chat*> ChatList;

/**
	Klasa tworzca okno rozmowy, rejestrujca przyciski,
	formatujca wiadomo itp.
	\class Chat
	\brief Okno rozmowy
**/

class Chat : public QMainWindow
{
	Q_OBJECT

	private:
		friend class ChatManager;

		QValueList<ChatMessage *> ChatMessages; /*!< wiadomoci wpisane
							w oknie rozmowy */

		UserGroup *Users; /*!< lista uytkownikw w danym oknie */
		int index;	/*!< nr okna (z chat menadera) */
		QString title_buffer; /*!< tytu okna */
		QTimer* title_timer;  /*!< zmienna przechowujca czas od ostatniego
					odwieenia tytuu okna */
		QColor actcolor; /*!< zmienna przechowujca aktualny kolor */

		CustomInput* Edit; /*!< okno do wpisywania wiadomoci */
		QMimeSourceFactory *bodyformat; /*!< zmienna ustawiajca format */
		EmoticonSelector* emoticon_selector; /*!< okienko do
							wyboru emotikonek */
		ColorSelector* color_selector; /*!< okienko do wyboru koloru */
		bool AutoSend; /*!< okrela czy Return powoduje wysanie wiadomoci */
		bool ScrollLocked; /*!< blokuje przewijanie okna rozmowy */
		bool WaitingForACK;
		UserBox* userbox; /*!< lista kontaktw przydatna gdy jestemy w
						 konferencji */
		QString myLastMessage;/*!< zmienna przechowujca
					nasz ostatni wiadomo */
		unsigned int myLastFormatsLength; /* ?? */
		void* myLastFormats; /* ?? */
		int seq; /* ?? */
		KaduSplitter *vertSplit, *horizSplit; /*!< obiekty oddzielajce
							 kontrolki od siebie */
		int ParagraphSeparator; /*!< odstp midzy kolejnymi wypowiedziami */

		QDateTime lastMsgTime; /*!< czas ostatniej wiadomoci */

		//pomijanie nagwkw w wiadomociach
		QString PreviousMessage;           //pamitamy od kogo bya ostatnia wiadomo
		bool CfgNoHeaderRepeat; /*!< okrela czy uywamy usuwania nagwkw */
		int CfgHeaderSeparatorHeight; /*!< wysoko separatora nagwkw */
		int CfgNoHeaderInterval; /*!< interwa po jakim przywracany jest nagwek */

		ChatStyle* Style; /*!< styl okna rozmowy */

		time_t LastTime;

		/**
			\fn void pruneWindow()
			Funkcja czyszczca okno rozmowy
		**/
		void pruneWindow();

		KaduTextBrowser* body; /*!< historia rozmowy, prosz NIE dawa dostpu na zewntrz do tej zmiennej */

	private slots:

		/**
			\fn void changeColor()
			Slot zmieniajcy kolor czcionki
		**/
		void changeColor(const QWidget* activating_widget);

		/**
			\fn void addMyMessageToHistory()
			Slot dodaje wysan wiadomo do historii rozmowy
		**/
		void addMyMessageToHistory();

		/**
			\fn void pageUp()
			Slot przewijajcy histori rozmowy w gr
		**/
		void pageUp();

		/**
			\fn void pageDown()
			Slot przewijajcy histori rozmowy w d
		**/
		void pageDown();

		/**
			\fn void imageReceivedAndSaved(UinType sender,uint32_t size,uint32_t crc32,const QString& path)
			Slot obsugujcy odebranie i zapis obrazka
			\param sender osoba, ktra wysaa obrazek
			\param size rozmiar obrazka
			\param crc32 suma kontrolna obrazka
			\param path cieka do obrazka
		**/
		void imageReceivedAndSaved(UinType sender,uint32_t size,uint32_t crc32,const QString& path);

		/**
			\fn void connectAcknowledgeSlots();
			Slot podczajcy potwierdzenie dostarczenia wiadomoci
		**/
		void connectAcknowledgeSlots();

		/**
			\fn void disconnectAcknowledgeSlots();
			Slot odczajcy potwierdzenie dostarczenia wiadomoci
		**/
		void disconnectAcknowledgeSlots();

		void selectedUsersNeeded(const UserGroup*& user_group);

	protected:
		/**
			\fn virtual void closeEvent(QCloseEvent* e)
			Funkcja obsugujca zamknicie okna
		**/
		virtual void closeEvent(QCloseEvent* e);

		/**
			\fn virtual void windowActivationChange(bool oldActive)
			Funkcja sterujca mruganiem napisu okna
		**/
		virtual void windowActivationChange(bool oldActive);

		/**
			\fn virtual void keyPressEvent(QKeyEvent* e)
			Funkcja obsugujca nacinicie przycisku
			\param e wskanik do obiektu obsugujcego klawisze
		**/
		virtual void keyPressEvent(QKeyEvent* e);

		/**
			\fn virtual void keyReleaseEvent(QKeyEvent* e)
			Funkcja obsugujca zwolnienie przycisku
			\param e wskanik do obiektu obsugujcego klawisze
		**/
		virtual void mouseReleaseEvent(QMouseEvent *e);
		
		bool keyPressEventHandled(QKeyEvent *);

	public:

		/**
			Konstruktor okna rozmowy
			\fn Chat(UserListElements usrs, QWidget* parent = 0, const char* name = 0)
			\param usrs lista kontaktw, z ktrymi prowadzona jest rozmowa
			\param parent rodzic okna
			\param name nazwa obiektu
		**/
		Chat(UserListElements usrs, QWidget* parent = 0, const char* name = 0);

		/**
			\fn ~Chat()
			Destruktor okna rozmowy
		**/
		~Chat();

		/**
			\fn void formatMessages(QValueList<ChatMessage *> &msgs)
			Formatuje wszystkie wiadomoci na html'a,
			dodaje kolory ta i czcionek
			\param msgs lista wiadomoci do sformatowania
		**/
		void formatMessages(QValueList<ChatMessage *> &msgs);
		/**
			\fn void formatMessage(ChatMessage& msg,
				const OwnChatColors* own_colors = NULL,
				const UserChatColors* user_colors = NULL,
				EmoticonsStyle style = (EmoticonsStyle)
					config_file.readNumEntry("Chat","EmoticonsStyle"));

			Formats message to html
			\param msg message to be formatted
			\param own_colors our chat colors
			\param user_colors user chat colors
			\param style emoticons style
		**/
		void formatMessage(ChatMessage& msg,
			const OwnChatColors* own_colors = NULL,
			const UserChatColors* user_colors = NULL,
			EmoticonsStyle style = (EmoticonsStyle)
				config_file.readNumEntry("Chat","EmoticonsStyle"));

		/**
			\fn void newMessage(const QString &protocolName, UserListElements senders, const QString &msg, time_t time)
			Dodaje wiadomo do okna
			\param protocolName nazwa protokou
			\param senders lista uytkownikw
			\param msg wiadomoc
			\param time czas
		**/
		void newMessage(const QString &protocolName, UserListElements senders, const QString &msg, time_t time);

		/**
			\fn void writeMessagesFromHistory(UserListElements senders, time_t time)
			Wpisuje do okna histori rozmowy
			\param senders lista uytkownikw identyfikujcych okno
			\param time ..
		**/
		void writeMessagesFromHistory(UserListElements senders, time_t time);

		/**
			\fn const UserGroup *users() const
			Zwraca list numerw rozmowcw.
		**/
		const UserGroup *users() const;

		/**
			\fn QValueList<ChatMessage*>& chatMessages()
			Daje dostp do wiadomoci aktualnie przechowywanych
			w oknie chat. Metody tej mona uy do zmiany treci
			lub waciwoci ktrej z wiadomoci w odpowiedzi
			na jakie zdarzenie.
		**/
		QValueList<ChatMessage*>& chatMessages();

		/**
			\fn void repaintMessages()
			Odwiea zawarto okna uwzgldniajc ewentualne
			zmiany dokonane w ktrej wiadomoci z listy
			uzyskanej za pomoc metody chatMessages(),
			dodanie nowych wiadomoci lub usunicie istniejcych.
		**/
		void repaintMessages();

		/**
			\fn const QString& title() const
			Zwraca aktualny tytu okna
		**/
		const QString& title() const;

		/**
			\fn CustonInput* edit()
			Zwraca wskanik do okna wpisywania wiadomoci
		**/
		CustomInput* edit();

		/**
			\fn bool autoSend() const
			Zwraca warto okrelajc, czy Return powoduje
			wysanie wiadomoci.
		**/
		bool autoSend() const;

		bool waitingForACK() const;

		virtual void dragEnterEvent(QDragEnterEvent *e);
		virtual void dropEvent(QDropEvent *e);
		virtual void dragMoveEvent(QDragMoveEvent *e);

		void scrollHistoryToBottom();
		virtual bool eventFilter(QObject *watched, QEvent *e);

	public slots:

		/**
			\fn void changeAppearance()
			Slot zmieniajcy kolory i czcionki w oknie
		**/
		void changeAppearance();

		/**
			\fn void refreshTitle()
			Slot ustawiajcy tytu okna zgodnie z konfiguracj
		**/
		void refreshTitle();

		/**
			\fn void addEmoticon(QString)
			Slot dodaje ikonke do pola wpisywania wiadomoci
			\param emot ikonka np. <lol>
		**/
		void addEmoticon(QString);

		/**
			\fn void scrollMessages(const QValueList<ChatMessage *> &)
			Slot dodaj wiadomoci do okna
			\param messages lista wiadomoci
		**/
		void scrollMessages(const QValueList<ChatMessage *> &);

		/**
			\fn void alertNewMessage()
			Slot informujcy o nowej wiadomoci
			mruganie jeli okno jest nieaktywne
		**/
		void alertNewMessage();

		/**
			\fn void sendMessage()
			Slot wywoywany po naciniciu przycisku
			do wysyania wiadomoci
		**/
		void sendMessage();

		/**
			\fn void cancelMessage()
			Slot wywoywany po nacinieciu przycisku anulowania
			wysyaniu wiadomoci
		**/
		void cancelMessage();

		/**
			\fn void writeMyMessage()
			Slot wpisujcy wiadomo do okna
			\see sendMessage
		**/
		void writeMyMessage();

		/**
			\fn void changeTitle()
			Slot zmieniajcy tytu
			\see setTitle
		**/
		void changeTitle();

		/**
			\fn void curPosChanged(int para, int pos)
			Slot wywoywany kiedy pozycja kursosa zmienia si
			\param  para nie obslugiwane
			\param pos nie obslugiwane
		**/
		void curPosChanged(int para, int pos);

		/**
			\fn void specialKeyPressed(int key)
			Slot zostaje wywoany gdy nacinito specjalny klawisz (skrt)
			odpowiadajcy np KEY_BOLD
			\param key warto nacinitego specjalnego klawisza
		**/
		void specialKeyPressed(int key);

		/**
			\fn void colorChanged(const QColor& color)
			Slot zostaje wywowany gdy uytkownik zmieni kolor czcionki
			ktr bdzie pisa
			\param color warto zmienionego koloru
		**/
		void colorChanged(const QColor& color);

		/**
			\fn void colorSelectorAboutToClose()
			Slot zostaje wywoany przy zamykaniu okna wyboru ikonek
		**/
		void colorSelectorAboutToClose();

		/**
			\fn void messageBlockedSlot(int seq, UinType uin)
			Slot zostaje wywoany gdy nasza wiadomo zostaa zablokowana
			przez serwer
			\param seq
			\param uin
		**/
		void messageBlockedSlot(int seq, UinType uin);

		/**
			\fn void messageBoxFullSlot(int seq, UinType uin)
			Slot zostaje wywoany gdy skrzynka na wiadomoci
			zostaa zapeniona
			\param seq
			\param uin
		**/
		void messageBoxFullSlot(int seq, UinType uin);

		/**
			\fn void messageNotDeliveredSlot(int seq, UinType uin)
			Slot zostaje wywoany gdy wiadomo nie zostaa dostarczona
			\param seq
			\param uin
		**/
		void messageNotDeliveredSlot(int seq, UinType uin);

		/**
			\fn void messageAcceptedSlot(int seq, UinType uin)
			Slot zostaje wywoany gdy wiadomo zostaa poprawnie wysana
			\param seq
			\param uin
		**/
		void messageAcceptedSlot(int seq, UinType uin);

		/**
			\fn void clearChatWindow()
			Slot czyszczcy okno rozmowy
		**/
		void clearChatWindow();

		/**
			\fn void setAutoSend(bool auto_send)
			Ustaw autowysylanie
		**/
		void setAutoSend(bool auto_send);

		/**
			\fn void setScrollLocked(bool locked)
			Ustaw blokade przewijania
		**/
		void setScrollLocked(bool locked);

		/**
			\fn void emoticonSelectorClicked()
			Slot wywietlajcy okno z wyborem emotikonek
		**/
		void openEmoticonSelector(const QWidget* activating_widget);

		/**
			\fn void insertImage()
			Slot wklejajcy obrazek do okna rozmowy (wybr obrazka)
		**/
		void insertImage();

	signals:
		/**
			\fn void messageSendRequested(Chat* chat)
			Sygnal jest emitowany gdy uzytkownik wyda polecenie
			wyslania wiadomosci, np klikajac na guzik "wyslij".
			\param chat wskanik do okna ktre emituje sygna
		**/
		void messageSendRequested(Chat* chat);

		/**
			\fn void messageFiltering(const UserGroup *users, QCString& msg, bool& stop)
			Sygnal daje mozliwosc operowania na wiadomosci
			ktora ma byc wyslana do serwera juz w jej docelowej
			formie po konwersji z unicode i innymi zabiegami.
			Tresc wiadomosci mozna zmienic podmieniajac wskaznik
			msg na nowy bufor i zwalniajac stary (za pomoca free).
			Mozna tez przerwac dalsza jej obrobke ustawiajac
			wskaznik stop na true.
			\param users lista uytkownikw
			\param msg wiadomo
			\param stop zakoczenie dalszej obrbki sygnau
		**/
		void messageFiltering(const UserGroup *users, QCString &msg, bool &stop);

		/**
			\fn void messageSent(Chat* chat)
			Sygnal jest emitowany gdy zakonczy sie proces
			wysylania wiadomosci i zwiazanych z tym czynnosci.
			Oczywiscie nie wiemy czy wiadomosc dotarla.
			\param chat wskanik do okna rozmowy,
			 ktre emitowao sygna
		**/
		void messageSent(Chat* chat);

		/**
			\fn void messageSentAndConfirmed(UserListElements receivers, const QString& message)
			This signal is emited when message was sent
			and it was confirmed.
			When confirmations are turned off signal is
			emited immediately after message was send
			just like messageSent() signal.
			\param receivers list of receivers
			\param message the message
		**/
		void messageSentAndConfirmed(UserListElements receivers, const QString& message);

		/**
			\fn void fileDropped(const UserGroup *users, const QString& fileName)
			Sygna jest emitowany, gdy w oknie Chat
			upuszczono plik.
			\param users lista uytkownikw
			\param fileName nazwa pliku
		**/
		void fileDropped(const UserGroup *users, const QString &fileName);

		/**
			\fn void windowActivationChanged(bool b, const UserGroup* users)
			\param b ??
			\param users lista uytkownikw
		**/
		void windowActivationChanged(bool b, const UserGroup *users);
};

#endif
