/***************************************************************************
 *   Weather module for Kadu communicator                                  *
 *   2006 Micha Maek (bysk)                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "weather_global.h"

#include "misc.h"
#include "config_file.h"

const int WeatherGlobal::CONNECTION_TIMEOUT = 15000;
const int WeatherGlobal::CONNECTION_COUNT = 2;
const QString WeatherGlobal::WeatherConfigPath = dataPath("kadu/modules/data/weather/");
const QString WeatherGlobal::WeatherIconPath = dataPath("kadu/modules/data/weather/icons/");
const unsigned int WeatherGlobal::RECENT_LOCATIONS_COUNT = 10;
const int WeatherGlobal::KEEP_FORECAST = 1; // w godzinach

WeatherGlobal::WeatherGlobal()
{
	QStringList serverList = QStringList::split( ";", config_file.readEntry("Weather", "Servers") );
	QStringList serversUsing = QStringList::split( ";", config_file.readEntry("Weather", "ServersUsing") );

	QDir dir(WeatherConfigPath,"*.ini");
	dir.setFilter(QDir::Files);

	QStringList serverFilesList;
	for(unsigned int i=0; i<dir.count(); ++i)
	{
		serverFilesList.append( dir[i] );
	}
	
	for(unsigned int i=0; i<serverList.count(); ++i)
	{
		QStringList::iterator it = serverFilesList.find( serverList[i] );
		if( it == serverFilesList.end() )
		{
			serverList.remove( serverList.at(i) );
			serversUsing.remove( serversUsing.at(i) );
		}
		else
			serverFilesList.remove(it);
	}
	
	serverList += serverFilesList;
	serversUsing.insert( serversUsing.end(), serverFilesList.count(), "1" );
	for(unsigned int i=0; i<serverList.count(); ++i)
	{
		PlainConfigFile wConfig( WeatherConfigPath + serverList[i] );
		servers_.append( Server( wConfig.readEntry("Header","Name"), serverList[i], serversUsing[i] == "1" ) );
	}

	// load recent locations search list
	for( unsigned int i=0; i<RECENT_LOCATIONS_COUNT; i++)
	{
		QString city = config_file.readEntry("Weather", QString("Location%1").arg(i+1));
		if( !city.isEmpty() && !recentLocations_.contains( city ) )
			recentLocations_.push_back( city );
	}

}

WeatherGlobal::~WeatherGlobal()
{
	unsigned int i;
	for( i=0; i<recentLocations_.count(); i++ )
		config_file.writeEntry("Weather", QString("Location%1").arg(i+1), recentLocations_[i]);
	for( ; i<RECENT_LOCATIONS_COUNT; i++ )
		config_file.writeEntry("Weather", QString("Location%1").arg(i+1), "" );
		
	
	QStringList serverList;
	QStringList serversUsing;
	
	SERVERITERATOR server = servers_.begin();
	while( server != servers_.end() )
	{
		serverList.append( (*server).configFile_ );
		serversUsing.append( (*server).use_ ? "1" : "0" );
		++server;
	}
	
	config_file.writeEntry("Weather", "Servers", serverList.join(";") );
	config_file.writeEntry("Weather", "ServersUsing", serversUsing.join(";") );
}

WeatherGlobal::SERVERITERATOR WeatherGlobal::beginServer() const
{
	SERVERITERATOR it = servers_.begin();
	while( it != servers_.end() )
	{
		if( (*it).use_ )
			return it;
		else
			++it;
	}
	return servers_.end();
}

WeatherGlobal::SERVERITERATOR WeatherGlobal::nextServer( SERVERITERATOR it ) const
{
	++it;
	while( it != servers_.end() )
	{
		if( (*it).use_ )
			return it;
		else
			++it;
	}
	return servers_.end();
}

void WeatherGlobal::getConfigFile( const QString& serverName, QString* configFile ) const
{
	SERVERITERATOR it = servers_.begin();
	while( it != servers_.end() )
	{
		const Server& server = *it;
		if( server.name_ == serverName )
		{
			*configFile = server.configFile_;
			return;
		}
		++it;
	}

	*configFile = QString::null;
}

void WeatherGlobal::getServerName( const QString& configFile, QString* serverName ) const
{
	SERVERITERATOR it = servers_.begin();
	while( it != servers_.end() )
	{
		const Server& server = *it;
		if( server.configFile_ == configFile )
		{
			*serverName = server.name_;
			return;
		}
		++it;
	}

	*serverName = QString::null;
}

bool WeatherGlobal::configFileExists( const QString& configFile ) const
{
	SERVERITERATOR it = servers_.begin();
	while( it != servers_.end() )
	{
		if( (*it).configFile_ == configFile )
		{
			return true;
		}
		++it;
	}
	return false;
}

void WeatherGlobal::setServerUsing( const QString serverName, bool use )
{
	QValueListIterator<Server> it = servers_.begin();
	while( it != servers_.end() )
	{
		Server& server = *it;
		if( server.name_ == serverName )
		{
			server.use_ = use;
			return;
		}
		++it;
	}
}

void WeatherGlobal::setServerPos( const QString& serverName, uint pos )
{
	uint count = servers_.count();
	for( uint i=0; i<count; ++i )
	{
		Server& server = servers_[i];
		if( server.name_ == serverName )
		{
			if( i != pos )
			{
				QValueListIterator<Server> it = servers_.at(i);
				servers_.insert( servers_.at(pos), server );
				servers_.remove( it );
			}
			return;
		}
	}
}

bool WeatherGlobal::insertRecentLocation( const QString& location )
{
	if( recentLocations_.find(location) == recentLocations_.end() )
	{
		recentLocations_.push_front(location);
		if( recentLocations_.count() > RECENT_LOCATIONS_COUNT )
			recentLocations_.remove( recentLocations_.at( RECENT_LOCATIONS_COUNT ) );
		
		return true;
	}
	else
		return false;
}

