/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qpixmap.h>
#include <qevent.h>
#include <qtimer.h>
#include <qstyle.h>
#include <qfileinfo.h>
#include <qimage.h>
#include <qtooltip.h>
#include <qdir.h>
#include <qpopupmenu.h>

#include <sys/stat.h>
#include <sys/types.h>
#include <errno.h>

#include "message_box.h"
#include "chat.h"
#include "config_dialog.h"
#include "config_file.h"
#include "debug.h"
#include "modules.h"
#include "screenshot.h"
#include "userlist.h"
#include "action.h"
#include "chat_manager.h"
#include "misc.h"
#include "icons_manager.h"
#include "custom_input.h"

#define MODULE_SCREENSHOT_VERSION 0.4.4

ScreenShot* screenShot;

extern "C" int screenshot_init()
{
	kdebugf();
	screenShot = new ScreenShot();
	return 0;
}

extern "C" void screenshot_close()
{
	kdebugf();
	delete screenShot;
}

//-----------------------------------------------------------------------------------

ShotSizeHint::ShotSizeHint()
	: QWidget(0, 0, WStyle_Customize|Qt::WStyle_NoBorder|Qt::WStyle_StaysOnTop)
{
	QGridLayout *g = new QGridLayout(this, 1, 1);
	geom = new QLabel(this, "");
	fileSize = new QLabel(this, "0 KB");
	g->addWidget(geom, 0, 0);
	g->addWidget(fileSize, 1, 0);
}

//-----------------------------------------------------------------------------------

ScreenShot::ScreenShot(QWidget * parent, const char * name, WFlags f)
 : QWidget(parent, name, Qt::WStyle_Customize|Qt::WStyle_NoBorder)
{
	kdebugf();

	sizeHint = new ShotSizeHint();
	hintTimer = new QTimer();
	connect(hintTimer, SIGNAL(timeout()), this, SLOT(updateHint()));

	// Chat windows menu
	menu = new QPopupMenu();
	popups[0] = menu->insertItem(tr("Simple shot"), this, SLOT(takeShot(int)));
	popups[1] = menu->insertItem(tr("With chat window hidden"), this, SLOT(takeShot(int)));

	// Config Dialog
	config = new ConfigFile(ggPath(QString("/kadu-screenshot.conf")));

	ConfigDialog::registerSlotOnCreateTab("ScreenShot", this, SLOT(onCreateConfig()));
	ConfigDialog::registerSlotOnCloseTab("ScreenShot", this, SLOT(onDestroyConfig()));
	ConfigDialog::registerSlotOnApplyTab("ScreenShot", this, SLOT(onApplyConfig()));

	QStringList opts = QStringList::fromStrList(QImageIO::outputFormats());
	QStringList warns, warnsVals;
	warns += QT_TRANSLATE_NOOP("@default", "Warn with hint");
	warns += QT_TRANSLATE_NOOP("@default", "Warn with dialog window");
	warnsVals += "hint";
	warnsVals += "window";
	ConfigDialog::addTab("ScreenShot", dataPath("kadu/modules/data/screenshot/camera.png"));
		ConfigDialog::addVRadioGroup(config, "ScreenShot", "ScreenShot", QT_TRANSLATE_NOOP("@default", "File format"), "fileFormat", opts, opts, "PNG", QT_TRANSLATE_NOOP("@default", "Select image format, which screenshots will be stored and sent in."));
		ConfigDialog::addSpinBox(config, "ScreenShot", "ScreenShot", QT_TRANSLATE_NOOP("@default", "Screenshots quality"), "quality", -1, 100, 1, -1, QT_TRANSLATE_NOOP("@default", "-1 for autodetection, 0 for best file compression, 100 for best image quality."));
		ConfigDialog::addLineEdit(config, "ScreenShot", "ScreenShot", QT_TRANSLATE_NOOP("@default", "Screenshots directory path"), "path", ggPath("images/"));
		ConfigDialog::addLineEdit(config, "ScreenShot", "ScreenShot", QT_TRANSLATE_NOOP("@default", "Screenshots filename prefix"), "filenamePrefix", "shot");
		ConfigDialog::addSpinBox(config, "ScreenShot", "ScreenShot", QT_TRANSLATE_NOOP("@default", "Screenshots quality"), "quality", -1, 100, 1, -1, QT_TRANSLATE_NOOP("@default", "-1 for autodetection, 0 for best file compression, 100 for best image quality."));
		ConfigDialog::addCheckBox(config, "ScreenShot", "ScreenShot", QT_TRANSLATE_NOOP("@default", "Paste [IMAGE] clause automatically"), "paste_clause", true, QT_TRANSLATE_NOOP("@default", "When enabled, module will automatically paste ready to send [IMAGE] clause in chat window."));
		ConfigDialog::addVGroupBox("ScreenShot", "ScreenShot", QT_TRANSLATE_NOOP("@default", "Shots directory size control"));
			ConfigDialog::addCheckBox(config, "ScreenShot", "Shots directory size control", QT_TRANSLATE_NOOP("@default", "Enable size warnings"), "dir_size_warns", true, QT_TRANSLATE_NOOP("@default", "When enabled, module will check for shots directory size and warn you if it gets size specified below."));
			ConfigDialog::addSpinBox(config, "ScreenShot", "Shots directory size control", QT_TRANSLATE_NOOP("@default", "Directory size to warn (in KBytes)"), "dir_size_limit", 10, 500000, 50, 10000, QT_TRANSLATE_NOOP("@default", "Define, how many kbytes can shots take in the directory before module will warn you."));
			ConfigDialog::addVRadioGroup(config, "ScreenShot", "Shots directory size control", QT_TRANSLATE_NOOP("@default", "Warning style"), "dir_size_style", warns, warnsVals, "hint", QT_TRANSLATE_NOOP("@default", "Hint mode means a standard Kadu hint. Dialog window mode means standart information window."));

	// Chat toolbar button
	Action* screenshot_action = new Action(icons_manager->loadIcon(dataPath("kadu/modules/data/screenshot/camera_small.png")),
		tr("ScreenShot"), "ScreenShotAction", Action::TypeChat);
	connect(screenshot_action, SIGNAL(activated(const UserGroup*, const QWidget*, bool)),
		this, SLOT(screenshotActionActivated(const UserGroup*, const QWidget*)));
	KaduActions.insert("ScreenShotAction", screenshot_action);

	// Rest stuff
	buttonPressed = false;
	warnedAboutSize = false;
}


ScreenShot::~ScreenShot()
{
	kdebugf();

	hintTimer->stop();
	delete hintTimer;
	delete sizeHint;
	delete config;
	delete menu;

	KaduActions.remove("ScreenShotAction");

	ConfigDialog::removeControl("ScreenShot", "Enable size warnings");
	ConfigDialog::removeControl("ScreenShot", "Directory size to warn (in KBytes)");
	ConfigDialog::removeControl("ScreenShot", "Warning style");
	ConfigDialog::removeControl("ScreenShot", "Shots directory size control");
	ConfigDialog::removeControl("ScreenShot", "Paste [IMAGE] clause automatically");
	ConfigDialog::removeControl("ScreenShot", "Screenshots filename prefix");
	ConfigDialog::removeControl("ScreenShot", "Screenshots directory path");
	ConfigDialog::removeControl("ScreenShot", "Screenshots quality");
	ConfigDialog::removeControl("ScreenShot", "File format");
	ConfigDialog::removeTab("ScreenShot");

	ConfigDialog::unregisterSlotOnApplyTab("ScreenShot", this, SLOT(onApplyConfig()));
	ConfigDialog::unregisterSlotOnCloseTab("ScreenShot", this, SLOT(onDestroyConfig()));
	ConfigDialog::unregisterSlotOnCreateTab("ScreenShot", this, SLOT(onCreateConfig()));
}

void ScreenShot::screenshotActionActivated(const UserGroup* grp, const QWidget* source)
{
	kdebugf();

	tempChat = chat_manager->findChat(grp);

	menu->popup(source->mapToGlobal(QPoint(0,20)));
}

void ScreenShot::mousePressEvent(QMouseEvent* e)
{
	kdebugf();
	if (e->button() == Qt::LeftButton)
	{
		region = QRect(e->pos(), e->pos());
		buttonPressed = true;

		int x = e->pos().x()+50,
			y = e->pos().y()+50;

		QRect screen = QApplication::desktop()->screenGeometry();
		if (x+150 > screen.width())
			x -= 150;

		if (y+100 > screen.height())
			y -= 100;

		sizeHint->move(x, y);

		sizeHint->geom->setText("0x0");
		sizeHint->fileSize->setText("0 KB");
		sizeHint->show();
		hintTimer->start(1000);
	}
}

void ScreenShot::mouseReleaseEvent(QMouseEvent* e)
{
	kdebugf();

	if (!buttonPressed)
		return;

	hintTimer->stop();
	sizeHint->hide();

	// Uwalnianie myszki, klawiatury
	buttonPressed = false;
	releaseMouse();
	releaseKeyboard();
	drawRegionRect();

	// Normalizowanie prostokta do zrzutu
	region.setBottomRight(e->pos());
	region = region.normalize();

	// Plik do zapisu:
	QDir dir(config->readEntry("ScreenShot", "path", ggPath("images")));
	if (!dir.exists())
	{
		int ret = mkdir(dir.path().local8Bit().data(), 0755);
		if (ret)
		{
			printf("Error while creating directory %s:\n", dir.path().local8Bit().data());
			switch (ret)
			{
				case ENAMETOOLONG:
					printf("Pathname too long.\n");
					break;
				case EACCES:
					printf("The parent directory does not allow write permission to the process, or one of the directories in pathname did not allow search (execute) permission.\n");
					break;
				case EFAULT:
					printf("Pathname points outside your accessible address space.\n");
					break;
				case EROFS:
					printf("pathname refers to a file on a read-only filesystem.\n");
					break;
				case ENOSPC:
					printf("The new directory cannot be created because the user's disk quota is exhausted.\n");
					break;
				case EPERM:
					printf("The filesystem containing pathname does not support the creation of directories. \n");
					break;
				case EEXIST:
					printf("pathname already exists (not necessarily as a directory). This includes the case where pathname is a symbolic link, dangling or not.\n");
					break;
				default:
					printf("Unknown error.\n");
					break;
			}
			return;
		}
	}

	QString path = QDir::cleanDirPath(
			    dir.path() + "/" +
				config->readEntry("ScreenShot", "filenamePrefix", "shot") +
				QString::number(QDateTime::currentDateTime().toTime_t()) + "." +
				config->readEntry("ScreenShot", "fileFormat", "PNG").lower()
		);

	// Zrzut
	QPixmap shot = QPixmap::grabWindow(winId(), region.x(), region.y(), region.width(), region.height());
	handleShot(shot, path);
}

void ScreenShot::handleShot(QPixmap p, QString path)
{
	bool ret = p.save(
			path, config->readEntry("ScreenShot", "fileFormat", "PNG").ascii(),
			config->readNumEntry("ScreenShot", "quality", -1)
		);

	// Chowanie widgeta zrzutu i przywrcenie kursora.
	hide();
	QApplication::restoreOverrideCursor();

	if (!ret)
	{
		printf("Can't write file %s.\nAccess denied or other problem.", path.local8Bit().data());
		return;
	}

	if (!tempChat)
	{
		printf("ScreenShot module can't determinate Chat window. It should not happen! It's a bug!\n Please report it at Kadu Forum or to module author.\n");
		return;
	}

	QFileInfo f(path);
	int size = f.size();

	if (size == 0)
	{
		printf("File %s has 0 size!\nIt should be bigger.", path.local8Bit().data());
		return;
	}

	if (shotType == 1)
		if (wasMaximized)
			tempChat->showMaximized();
		else
			tempChat->showNormal();

	// Wklejanie [IMAGE] do okna Chat
	if (config->readBoolEntry("ScreenShot", "paste_clause", true))
	{
		// Sprawdzanie rozmiaru zrzutu wobec rozmwcw
		UserListElements users = tempChat->users()->toUserListElements();
		if (users.count() > 1)
		{
			QStringList list;
			for ( uint i = 0; i < users.count(); i++ )
			{
				if ( ( users[i].protocolData("Gadu", "MaxImageSize").toInt() * 1024) < size)
					list.append(users[i].altNick()) ;
			}
			if (list.count() > 0)
			{
				if (list.count() == users.count())
				{
					MessageBox::wrn(
							tr("Image size is bigger than maximal image size\nset by all of conference contacts."),
							TRUE
						);
				}
				else
				{
					MessageBox::wrn(
							tr("Image size is bigger than maximal image size\nset by some of conference contacts:\n%1.")
								.arg(list.join(", ")),
							TRUE
						);
				}
			}
		}
		else
		{
			if ( ( users[0].protocolData("Gadu", "MaxImageSize").toInt() * 1024) < size)
			{
				bool send = MessageBox::ask(
					tr("Image size is bigger than maximal image size set by %1. Send it anyway?")
						.arg( users[0].altNick() )
					);

				if (!send)
					return;
			}
		}

		int x, y;
		tempChat->edit()->getCursorPosition(&y, &x);
		tempChat->edit()->insertAt(QString("[IMAGE ") + path + "]", y, x);
		tempChat->edit()->moveCursor(QTextEdit::MoveEnd, FALSE);
	}
	tempChat = NULL;
	checkShotsSize();
}

void ScreenShot::keyPressEvent(QKeyEvent* e)
{
	kdebugf();
	if (e->key() == Qt::Key_Escape)
	{
		releaseMouse();
		releaseKeyboard();
		hide();
	}
}

void ScreenShot::takeShot(int ident)
{
	kdebugf();

	// This code tells us which item from menu button was selected
	int id;
	for ( int i = 0; i < 2; i++ )
	{
		if (popups[i] == ident)
		{
			id = i;
			break;
		}
	}

	shotType = id;
	if (id == 1)
	{
		wasMaximized = tempChat->isMaximized();
		tempChat->showMinimized();
		QTimer::singleShot(600, this, SLOT(takeShot_Step2()));
	}
	else
		takeShot_Step2();
}

void ScreenShot::takeShot_Step2()
{
	pixmap = QPixmap::grabWindow(QApplication::desktop()->winId());
	resize(pixmap.size());
	setPaletteBackgroundPixmap(pixmap);
	showFullScreen();
	setCursor(crossCursor);

	QTimer::singleShot(100, this, SLOT(grabMouseSlot()));
}

void ScreenShot::grabMouseSlot()
{
	kdebugf();
	grabMouse();
	grabKeyboard();
}

void ScreenShot::mouseMoveEvent(QMouseEvent* e)
{
	kdebugf();
	if (!buttonPressed)
		return;

	drawRegionRect();
	region.setBottomRight(e->pos());
	drawRegionRect();

	QRect reg = region;
	reg = reg.normalize();

	sizeHint->geom->setText(
		QString("%1x%2")
			.arg(QString::number(reg.width()))
			.arg(QString::number(reg.height()))
		);
}

void ScreenShot::updateHint()
{
	QRect reg;
	reg.setTopLeft(region.topLeft());
	reg.setBottomRight(region.bottomRight());
	reg = reg.normalize();
	QPixmap shot = QPixmap::grabWindow(winId(), reg.x(), reg.y(), reg.width(), reg.height());
	bool ret = shot.save("/tmp/kadu_screenshot_tmp.png", "PNG", -1);
	if (ret)
	{
		QFileInfo f("/tmp/kadu_screenshot_tmp.png");
		sizeHint->fileSize->setText(QString::number(f.size()/1024)+" KB");
	}
}

void ScreenShot::drawRegionRect()
{
	QPainter painter;
	painter.begin( this );
	painter.setRasterOp( NotROP );
	painter.setPen( QPen( color0, 1 ) );
	painter.setBrush( NoBrush );
	style().drawPrimitive(
			QStyle::PE_FocusRect,
			&painter,
			region,
			colorGroup(),
			QStyle::Style_Default,
			QStyleOption(colorGroup().base())
		);

	painter.end();
}

void ScreenShot::onCreateConfig()
{
	kdebugf();
	modules_manager->moduleIncUsageCount("screenshot");
}

void ScreenShot::onApplyConfig()
{
	kdebugf();
	config->sync();
}

void ScreenShot::onDestroyConfig()
{
	kdebugf();
	onApplyConfig();
	modules_manager->moduleDecUsageCount("screenshot");
}

void ScreenShot::checkShotsSize()
{
	kdebugf();
	if (!config->readBoolEntry("ScreenShot", "dir_size_warns", true))
		return;

	int size = 0;

	int limit = config->readNumEntry("ScreenShot", "dir_size_limit", 10000);
	QString style = config->readEntry("ScreenShot", "dir_size_style", "hint");
	bool dialogWindow = (style == "window");
	QString type = config->readEntry("ScreenShot", "fileFormat", "PNG").lower();
	QDir dir(config->readEntry("ScreenShot", "path", ggPath("images")));

	QFileInfoList* list = (QFileInfoList*)dir.entryInfoList("*."+type, QDir::Files);
	QFileInfo *f = list->first();
	while (f)
	{
		size += f->size();
		f = list->next();
	}

	if (size/1024 >= limit)
	{
		printf("warn!, %d\n", dialogWindow);
	}
}
