//                          -*- mode: C++ -*-
// Copyright(C) 2005,2007 Stefan Siegl <stesie@brokenpipe.de>
// kopete_silc - silc plugin for kopete messenger
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef SILCPROTOCOLSILCACCOUNT_H
#define SILCPROTOCOLSILCACCOUNT_H

#include <kopeteprotocol.h>
#include <kopetepasswordedaccount.h>
#include <kopeteonlinestatus.h>


// include libsilc stuff into a separate namespace
namespace SilcTK {
#include <silc.h>
#include <silcclient.h>
}
#include <kopetechatsession.h>

// forward declare classes:
class SilcProtocol;
class SilcAccount;
class SilcServerContact;
class SilcChatSession;

// kopete_silc includes
#include "silcprotocol.h"
#include "silccontactmanager.h"
#include "silcbuddycontact.h"

/**
 * @brief Plugin reimplementation of Kopete::Account, supporting SILC protocol
 *
 * @author Stefan Siegl <ssiegl@gmx.de>
 */
class SilcAccount : public Kopete::PasswordedAccount
{
  Q_OBJECT
  friend class SilcChannelContact;
  friend class SilcBuddyContact;
  friend class SilcProtocol;
  friend class SilcContactManager;
  friend class SilcFileTransfer;
  friend class SilcChatSession;

public:
  SilcAccount(SilcProtocol *proto, const QString &accountID,
	      const char *name = 0L);
  
  ~SilcAccount();

  /**
   * @brief send a command to the SILC network
   */
  void sendSilcCommand(const QString &command, 
                       SilcTK::SilcClientCommandReply reply = NULL, 
                       void *context = NULL);

  SilcChatSession *chatSession(Kopete::ContactPtrList others);

  inline SilcBuddyContact *myself(void) {
    return static_cast<SilcBuddyContact *>(Kopete::Account::myself());
  }

  inline const SilcBuddyContact *myself(void) const {
    return static_cast<SilcBuddyContact *>(Kopete::Account::myself());
  }

  inline SilcProtocol *protocol(void) {
    return static_cast<SilcProtocol *>(Kopete::Account::protocol());
  }

  inline const SilcProtocol *protocol(void) const {
    return static_cast<SilcProtocol *>(Kopete::Account::protocol());
  }


  /**
   * @brief return absolute path of the public key file
   */
  QString pubKeyPath(void);

  /**
   * @brief return absolute path of the private key file
   */
  QString privKeyPath(void);

  /**
   * @brief SilcHash
   */
  SilcTK::SilcHash sha1hash;

protected:
  inline SilcTK::SilcPublicKey pubkey(void) const { return _pubkey; }
  inline SilcTK::SilcPrivateKey privkey(void) const { return _privkey; }

  inline SilcTK::SilcClient client(void) { return _client; }
  inline SilcTK::SilcClientConnection conn(void) { return _conn; }
  void copyKey(QString keyPath,bool prvKey=false);
  

signals:
  /**
   * @brief now connecting to the SILC network 
   *
   * This signal is emitted if the user has choosen to connect and the 
   * connection has been set up correctly.
   */
  void connected();
  
  /**
   * @brief user want's to disconnect from the SILC network 
   */
  void disconnected();

public slots:
  /** 
   * @brief set away status, leaving an optional away message
   */
   virtual void setAway(bool away, const QString &reason = QString::null);

  /**
   * @brief connect to the silc network
   */
  virtual void connect(const Kopete::OnlineStatus& initialStatus =
		       Kopete::OnlineStatus());

  /** 
   * @brief disconnect from the silc network 
   */
  virtual void disconnect();

  /**
   * @brief set the online status, optionally leaving away message 
   */
  virtual void setOnlineStatus(const Kopete::OnlineStatus& status,
			       const QString &reason = QString::null);

  /**
   * @brief create a new SilcContact
   */
  virtual bool createContact(const QString &contactId,
			     Kopete::MetaContact *parentContact);

  // this is not yet used, but an abstract function
  virtual void connectWithPassword(const QString &password);

  /**
   * @brief prepare actionMenu (the one popin up when clicking the network
   * icon in the lower right of the kopete application window)
   */
  virtual KActionMenu *actionMenu(void);

  /**
   * @brief ask the user which channel to join and afterwards join it
   */
  void slotJoinChannel(void);

  /** 
   * @brief join the channel with the provided name an dopen a chat window
   */
  void slotJoinChannel(const QString &channel, bool founder = false,
		       bool auth = false, const QString &password = "");

  /**
   * @brief show the server messages window
   */
  void slotShowServer(void);
            
   /**
    * @brief show fingerprint of this account's user 
    */
  void slotShowFingerprint(QString name=QString::null);
                 
public:
  /**
   * @brief types of handling bind
   */
  enum BindSelection {FT_BIND_ALWAYS, FT_BIND_NEVER, FT_BIND_NAT_TEST};

  static const QString CONFIG_NICKNAME;
  static const QString CONFIG_USERNAME;
  static const QString CONFIG_REALNAME;
  static const QString CONFIG_HOSTNAME;
  static const QString CONFIG_SIGN_CHAT_MESSAGE;
  static const QString CONFIG_SIGN_CHAT_ACTION;
  static const QString CONFIG_SIGN_PRIV_MESSAGE;
  static const QString CONFIG_SIGN_PRIV_ACTION;
  static const QString CONFIG_DISPLAY_IMAGES_INLINE;
  static const QString CONFIG_FT_USE_SILC_MIME;
  static const QString CONFIG_FT_AUTO_ACCEPT;
  static const QString CONFIG_FT_BIND_SELECTION;
  static const QString CONFIG_QUIT_MESSAGE;
  static const QString CONFIG_ATTR_GEO_ALLOWED;
  static const QString CONFIG_ATTR_GEO_LONG;
  static const QString CONFIG_ATTR_GEO_LAT;
  static const QString CONFIG_ATTR_GEO_ALT;
  static const QString CONFIG_ATTR_MESSAGE;
  static const QString CONFIG_ATTR_LANGUAGE;
  static const QString CONFIG_ATTR_TIMEZONE;
  static const QString CONFIG_ATTR_ALLOWED;

  static const QString CONFIG_ATTR_MOOD;
  static const QString CONFIG_ATTR_CONTACT;

  /** 
   * @brief set the nickname
   */
  void setNickName(const QString &nickname);

  /** 
   * @brief return set nickname
   */
  const QString nickName(void) const;

  /** 
   * @brief set local username
   */
  void setUserName(const QString &);

  /**
   * @brief return local username
   */
  const QString userName(void) const;

  /** 
   * @brief set real username
   */
  void setRealName(const QString &);

  /**
   * @brief return real username
   */
  const QString realName(void) const;

  /**
   * @brief set the name of the host to connect to
   */
  void setHostName(const QString &);

  /**
   * @brief return the name of the host to connect to
   */
  const QString hostName(void) const;

  /**
   * @brief return whether to display small images inline
   */
  void setDisplayImagesInline(const bool);

  /**
   * @brief set whether to prefer sending files using SilcMime
   */
  void setUseSilcMime(const bool);

  /**
   * @brief set whether incoming file requests should be accepted automatically
   */
  void setFtAutoAccept(const bool);

  /** 
   * @brief binding habbit when sending files:
   * 0 - always
   * 1 - never
   * 2 - autodetect nat
   */
  void setFtBind(int);

  /**
   * @brief get the BindSelection, see setFtBind */
  int getFtBind(void) const;

  /**
   * @brief whether to display small images inline
   */
  bool displayImagesInline(void) const;

  /** 
   * @brief whether to prefer to send files using SilcMime
   */
  bool useSilcMime(void) const;

  /** 
   * @brief return whether incoming file request shall be accepted automatically
   */
  bool ftAutoAccept(void) const;

  /**
   * @brief return whether or not to bind a port when sending files
   */
  bool ftNoBind(void) const;

  /** 
   * @brief set the quit message
   */
  void setQuitMessage(const QString &nickname);

  /** 
   * @brief return set quit message
   */
  const QString quitMessage(void) const;
  
  /**
   * @brief set whether outgoing channel messages ought to be signed
   */
  void setSignChannelMessages(const bool);

  /**
   * @brief return whether outgoing channel messages are sent digitally signed
   */
  bool signChannelMessages(void) const;

  /** 
   * @brief set whether or not outgoing channel actions ought to be signed
   */
  void setSignChannelActions(const bool);
  
  /**
   * @brief return whether outgoing channel actions are to be sent 
   * digitally signed.
   */
  bool signChannelActions(void) const;

  /**
   * @brief set whether to sign outgoing private messages
   */
  void setSignPrivateMessages(const bool);

  /**
   * @brief return whether we sign outgoing private messages
   */
  bool signPrivateMessages(void) const;

  /** 
   * @brief set whether to digitally sign outgoing private actions
   */
  void setSignPrivateActions(const bool);
  
  /** 
   * @brief return whether we sign outgoing private actions 
   */
  bool signPrivateActions(void) const;

  /** 
   * @brief set mood attribute
   */
  void setAttributeMood(SilcTK::SilcAttributeMood mood);

  /**
   * @brief get mood attribute
   */
  SilcTK::SilcAttributeMood getAttributeMood(void) const;
  
  /** 
   * @brief set contact attribute
   */
  void setAttributeContact(SilcTK::SilcAttributeContact contact);

  /**
   * @brief get contact attribute
   */
  SilcTK::SilcAttributeContact getAttributeContact(void) const;

  /**
   * @brief set geo informations
   */
  void setGeoInformations(bool allowed, double longitude, double latitude,
                          int altitude);

  
  /**
   * @brief if sending of attribute informations is allowed at all
   */
  void setAttributesAllowed(bool allowed);
  bool getAttributesAllowed(void) const;
  /**
   * @brief if sending of timezone informations is allowed 
   */
  void setAttributeTimezone(bool allowed);
  bool getAttributeTimezone(void) const;
  
  /** 
   * @brief set attribute status message
   */
  void setAttributeMessage(const QString &message);

  /** 
   * @brief return set attribute status message
   */
  const QString getAttributeMessage(void) const;
  
  /** 
   * @brief set attribute status language
   */
  void setAttributeLanguage(const QString &language);

  /** 
   * @brief return set attribute status message
   */
  const QString getAttributeLanguage(void) const;
  
  /**
   * @brief if sending of geo informations is allowed
   */
  bool getGeoAllowed(void) const;
  
  /**
   * @brief get attribute longitude
  */
  double getGeoLongitude(void) const;
  
  /**
   * @brief get attribute latitude
  */
  double getGeoLatitude(void) const;
  
  /**
   * @brief get attribute altitude
  */
  int getGeoAltitude(void) const;
  
  /**
   * @brief updates attributes
   */
  void updateAttributes(void);

  /**
   * @brief return pointer to the associated SilcContactManager instance
   */
  inline SilcContactManager *contactManager(void) { 
    return &_contactManager;
  }

  /**
   * @brief check wheater account is behind NAT
   */
  bool isBehindNat(void) const;

  /*
   * @brief get the local ip, becaue silc_net_localhost return crap
   */
  char *localIp(void) const;

protected:
  /**
   * @brief regularly call libsilc's mailoop
   */
  void timerEvent(QTimerEvent *);


private:
  /**
   * @brief the "Join Channel" entry in the account's popup menu
   */
  KAction *menuJoinChannel;
  KAction *menuShowServer;
  KAction *menuShowFingerprint; 

  /**
   * @brief the onlineStatus which shall be entered after connecting
   */
  Kopete::OnlineStatus _wantedOnlineStatus;

  /**
   * @brief pointer to the allocated SilcContactManager instance
   */
  SilcContactManager _contactManager;

  SilcServerContact *_myServer;
  SilcServerContact *myServer(void) { return _myServer; }

  /**
   * @brief the loaded public key
   */
  SilcTK::SilcPublicKey _pubkey;
  
  /**
   * @brief the loaded private key
   */
  SilcTK::SilcPrivateKey _privkey;

  /**
   * @brief pointer to the SilcClient structure
   */
  SilcTK::SilcClient _client;

  /**
   * @brief libsilc server connection handle
   */
  SilcTK::SilcClientConnection _conn;

  /**
   * @brief path to local identity picture (cached)
   */
  QString _globalIdentityPicture;

  /**
   * @brief set _globalIdentityPicture to Attributes
   */
  void setAttributePicture();

  /**
   * @brief this function is called by the libsilc when something changed on
   * connection (online, offline)
   */

  static void silc_connection_cb(SilcTK::SilcClient client,
                                 SilcTK::SilcClientConnection conn,
                                 SilcTK::SilcClientConnectionStatus status,
                                 SilcTK::SilcStatus error,
                                 const char *message,
                                 void *context);


  static void silc_command(SilcTK::SilcClient client, 
			   SilcTK::SilcClientConnection conn,
			   SilcTK::SilcBool success,
			   SilcTK::SilcCommand command, 
			   SilcTK::SilcStatus status,
                           SilcTK::SilcUInt32 argc,
                           unsigned char **argv);

  static void silc_command_reply(SilcTK::SilcClient client, 
				 SilcTK::SilcClientConnection conn,
				 SilcTK::SilcCommand command, 
                                 SilcTK::SilcStatus status,
				 SilcTK::SilcStatus error, 
                                 va_list va);

  static void silc_notify(SilcTK::SilcClient client, 
			  SilcTK::SilcClientConnection conn,
			  SilcTK::SilcNotifyType type, ...);


  static void silc_get_auth_method(SilcTK::SilcClient client, 
				   SilcTK::SilcClientConnection conn,
				   char *hostname, 
				   SilcTK::SilcUInt16 port,
                                   SilcTK::SilcAuthMethod authmethod,
				   SilcTK::SilcGetAuthMeth completion,
				   void *context);

  static void silc_verify_public_key(SilcTK::SilcClient client, 
                                     SilcTK::SilcClientConnection conn,
                                     SilcTK::SilcConnectionType conn_type,
                                     SilcTK::SilcPublicKey public_key,
                                     SilcTK::SilcVerifyPublicKey completion, 
                                     void *context);

  static void silc_ask_passphrase(SilcTK::SilcClient client, 
				  SilcTK::SilcClientConnection conn,
				  SilcTK::SilcAskPassphrase completion, 
				  void *context);

  static void silc_say(SilcTK::SilcClient client, 
		       SilcTK::SilcClientConnection conn,
		       SilcTK::SilcClientMessageType type, 
		       char *msg, ...)
    __attribute__ ((__format__ (__printf__, 4, 5)));

  static void silc_key_agreement(SilcTK::SilcClient client,
				 SilcTK::SilcClientConnection conn,
				 SilcTK::SilcClientEntry entry, 
		          	 const char *hostname, SilcTK::SilcUInt16 protocol,
                                 SilcTK::SilcUInt16 port);

  static void silc_ftp(SilcTK::SilcClient client,
		       SilcTK::SilcClientConnection conn,
		       SilcTK::SilcClientEntry client_entry,
		       SilcTK::SilcUInt32 session_id,
		       const char *hostname, SilcTK::SilcUInt16 port);

  static void setBuddyOnlineStatus(SilcContactManager *cm,
                                   SilcBuddyContact *buddy,
                                   SilcTK::SilcUInt32 mode);
  
  
  static SilcTK::SilcClientOperations ops;


  int libsilcTimerId;
  bool libsilcTimerLocked;


private slots:
  /**
   * @brief the SilcEngine successfully joined a channel (thus open chat window)
   */
  void slotJoinedChannel(const QString &);
  /**
   * @brief is called when disconnected to stop the timer
   */
  void slotStopTimer(void);
  /**
   * @brief global properties of kopete were changed (needed for userphotos)
   */
  void slotGlobalIdentityChanged (const QString &key, const QVariant &value);
};
#endif
