/*=========================================================================
| Open Office QuickStarter
|--------------------------------------------------------------------------
| (c) 2002  Kumaran Santhanam  <kumaran@alumni.stanford.org>
|
| This project is released under the GNU General Public License.
| Please see the file COPYING for more details.
|--------------------------------------------------------------------------
| model.cxx
|
| This file contains all of the logic, separate from the various
| front-end interfaces.
 ========================================================================*/

/*=========================================================================
| CONSTANTS
 ========================================================================*/
static const int   MIN_TIME = 2; // seconds

static const char *BIN_SH   = "/bin/sh";
static const char *SOFFICE  = "program/soffice";
static const char *SWRITER  = "private:factory/swriter";
static const char *SCALC    = "private:factory/scalc";
static const char *SDRAW    = "private:factory/sdraw";
static const char *SIMPRESS = "private:factory/simpress";

static const char *PLUGIN   = "-plugin";
static const char *QSTART   = "-quickstart";


/*=========================================================================
| INCLUDES
 ========================================================================*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>

#include "model.h"


/*=========================================================================
| REAP CHILDREN
 ========================================================================*/
static void sigchld (int sig) {
    wait(0);
    signal(SIGCHLD, sigchld);
}


/*=========================================================================
| UTILITY FUNCTIONS
 ========================================================================*/
// Fork and execute a child process
static int startProcess (const char *path,
                         const char *arg1, const char *arg2, const char *arg3)
{
    int pid;

    // Install the reaper
    signal(SIGCHLD, sigchld);

    // Fork and return the child's process ID
    if ((pid = fork()) != 0)  return pid;

    // Child process execution point
    execl(path, path, arg1, arg2, arg3, 0);
    
    // Execution should never reach this point.
    // If it does, there was an error.
    exit(1);
}

// Send a termination signal to the given process
static void stopProcess (int pid) {
    if (pid != 0)  kill(pid, SIGTERM);
}

// Return true if the given process is running
static bool isRunning (int pid) {
    if (pid == 0)  return false;

    int result = waitpid(pid, 0, WNOHANG);
    if (result < 0 && errno == ECHILD)  return false;
    return true;
}


/*=========================================================================
| CONSTRUCTOR / DESTRUCTOR
 ========================================================================*/
Model::Model () {
    _enabled    = false;
    _daemonPID  = 0;
    _tick       = 0;
    _status     = MODEL_OFFICE_STOPPED;
    _sofficeDir = 0;
    _soffice    = 0;
}

Model::~Model () {
    stopProcess(_daemonPID);
    if (_sofficeDir)  free(_sofficeDir);
    if (_soffice)     free(_soffice);
}


/*=========================================================================
| OPEN OFFICE DIRECTORY
 ========================================================================*/
// Sets the open office directory if the path is valid
void Model::setOpenOfficeDir (const char *dir) {
    char path[1024];

    if (dir == 0)  dir = "";
    sprintf(path, "%s/%s", dir, SOFFICE);

    // Free the memory used by the previous path
    if (_soffice) free(_soffice);
    _soffice = 0;

    // Stat the path to see if it really exists
    struct stat buf;
    stat(path, &buf);
    if (S_ISREG(buf.st_mode)) {
        _soffice = strdup(path);
    }

    // Save away the directory
    if (_sofficeDir)  free(_sofficeDir);
    _sofficeDir = strdup(dir);
    
    // Make sure to stop the current running process
    stopProcess(_daemonPID);
    _daemonPID = 0;
    _tick      = time(0);
}


/*=========================================================================
| PROCESS
 ========================================================================*/
// This function should be called periodically by the application.
// A good time interval is about 250ms.  It is designed to do
// very little processing if nothing has changed in the system.
void Model::process () {
    // Check to see if the minimum time has elapsed.
    // This is used to prevent rapid re-spawning of the process.
    if (_tick != 0) {
        if ((int)(time(0)-_tick) >= MIN_TIME)
            _tick = 0;
        else
            return;
    }

    // Check the current running status of the daemon
    bool running = isRunning(_daemonPID);
    _status = running ? MODEL_OFFICE_RUNNING : MODEL_OFFICE_STOPPED;

    // If the quickstarter is enabled, attempt to keep
    // the background process running, even if it got
    // accidentally terminated.
    if (_enabled) {
        if (!running && _soffice != 0) {
            _daemonPID = startProcess(BIN_SH, _soffice, PLUGIN, QSTART);
            _status    = MODEL_OFFICE_STARTING;
            _tick      = time(0);
        }
    }

    // If the quickstarter is disabled, terminate the
    // child process.
    else {
        stopProcess(_daemonPID);
        _daemonPID = 0;
        _tick      = time(0);
    }
}


/*=========================================================================
| LAUNCHERS
 ========================================================================*/
#define START_APP(app) \
    if (isRunning(_daemonPID) && _soffice != 0) \
        { startProcess(BIN_SH, _soffice, app, 0); }

void Model::startWriter ()  const { START_APP(SWRITER);  }
void Model::startCalc ()    const { START_APP(SCALC);    }
void Model::startDraw ()    const { START_APP(SDRAW);    }
void Model::startImpress () const { START_APP(SIMPRESS); }
