#
# TinyUrl, version 2.0, for weechat version 0.1.6
#
#   Listens to all channels for long URLs, and submits them to tinyurl.com
#   for easier links.
#
# Usage:
#
#   By default, any URL longer than 30 characters in length is grabbed,
#   submitted to 'tinyurl', and printed in the channel for your eyes only.
#   For example, you may see something like this:
#
# [11:21] <@lack> http://www.cbc.ca/story/canada/national/2005/11/12/mcdona
#                 lds-051112.html?ref=rss 
# [11:21] -P- [AKA] http://tinyurl.com/9dthl
#
#   Now you can just cut&paste the easier, shorter URL into your favourite
#   browser.
# 
#   If you want to be extra-helpful (or annoying) to certain channels you
#   are in, you can actually have the script say the tinyurl.com equivalent
#   of all long URLs, by adding the channel to the 'activeChans' list.  In
#   that case, everyone in the channel would see the following:
#
# [11:25] <testuser> http://www.cbc.ca/story/canada/national/2005/11/12/mcdona
#                    lds-051112.html?ref=rss 
# [11:25] <@lack> [AKA] http://tinyurl.com/9dthl
#
#   This works on long URLs you may say in the channel as well.
#
# Configuration:
#
#   Run '/tinyurl help' for the actual usage for setting these options:
#
#   activeChans
#     A comma-delimited list of channels you will actually "say" the
#     tinyurl in.  By default the list is empty.  Be warned, some channels
#     won't appreciate extra help (or 'noise' as they like to call it), and
#     some channels already have bots that do this.  Please only enable
#     this in channels where the ops have given you permission.
#
#   urllength
#     An integer, default value 30.  Any URL this long or longer will
#     trigger a tinyurl event.
#
#   printall
#     Either "on" or "off", default "on".  When ON, you will see the
#     tinyurl printed in your window for any channels not in your
#     activeChans list.  When OFF, you will not see any tinyurls except in
#     your activeChans list.
#
# Requirements:
#
#  - Designed to run with weechat version 1.0.6 or better.
#      http://weechat.flashtux.org/
#
#  - Requires that 'curl' is in the path (tested with curl 7.15.0).
#      http://curl.haxx.se/
#
# Copyright (C) 2005 Jim Ramsay <i.am@jimramsay.com>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
# USA.
#
# Changelog:
#
# Version 1.1, Dec 2, 2005
#   Fixed undefined 'urlend' thanks to kolter@irc.freenode.org#weechat
#
# Version 2.0, Dec 13, 2005
#   Also catches https, ftp, and ftps URLs, thanks to kolter for the
#   suggestion!
#

import os, urllib
import weechat

weechat.register( "TinyUrl", "2.0", "", "Waits for URLs and sends them to 'tinyurl' for you" )
weechat.add_message_handler("privmsg", "checkTinyurl")
weechat.add_command_handler("tinyurl", "tinyurlMain", 
    "Sets/Gets 'tinyurl' settings.  Try '/tinyurl help' for usage.")

tinyurlParams = ("urllength","activechans","printall")

# Set default values:
if weechat.get_plugin_config('urllength') == "":
  weechat.set_plugin_config('urllength', "30")
if weechat.get_plugin_config('printall') != "on" and \
   weechat.get_plugin_config('printall') != "off":
  weechat.set_plugin_config('printall', "on")

def tinyurlGet( name = "" ):
  if name == "":
    weechat.prnt( "-TinyUrl- Get" )
    for name in tinyurlParams:
      weechat.prnt( "  %s = %s" % (name, weechat.get_plugin_config(name)) )
  else:
    weechat.prnt( "-TinyUrl- Get" )
    if name in tinyurlParams:
      weechat.prnt( "  %s = %s" % (name, weechat.get_plugin_config(name)) )
    else:
      weechat.prnt( "  Unknown parameter \'%s\"" % name )
  return

def tinyurlSet( name, value ):
  if value == "":
    tinyurlGet( name )
  else:
    weechat.prnt( "-TinyUrl- Set" )
    if name in tinyurlParams:
      if name == "printall":
        if value == "0" or value.lower() == "no" or value.lower() == "off":
          weechat.set_plugin_config(name, "off")
        elif value == "1" or value.lower() == "yes" or value.lower() == "on":
          weechat.set_plugin_config(name, "on")
        else:
          weechat.prnt( "  printall must be one of 'on' or 'off'" )
          weechat.prnt( "  value = '%s'" % value )
          return
      else:
        if name == "activechans" and value.find( " " ) > 0:
          value = ",".join(value.split(" "))
        if name == "activechans" and value[0] != "#":
          value = ""
        weechat.set_plugin_config(name, value)
      weechat.prnt( "  %s = %s" % (name, weechat.get_plugin_config(name)) )
    else:
      weechat.prnt( "  Unknown parameter \'%s\'" % name )
  return

def tinyurlUsage():
  weechat.prnt( "-TinyUrl- Usage:" )
  weechat.prnt( "" )
  weechat.prnt( "  /tinyurl" )
  weechat.prnt( "    -> Shows all current tinyurl settings" )
  weechat.prnt( "" )
  weechat.prnt( "  /tinyurl help" )
  weechat.prnt( "    -> Shows this usage information" )
  weechat.prnt( "" )
  weechat.prnt( "  /tinyurl urllength [[=] size]" )
  weechat.prnt( "    -> Gets/Sets the minimum URL length needed to" )
  weechat.prnt( "       trigger a tinyurl" )
  weechat.prnt( "       Default 30" )
  weechat.prnt( "" )
  weechat.prnt( "  /tinyurl activechans [[=] #chan1[,#chan2...]]" )
  weechat.prnt( "    -> Gets/Sets the list of active channels" )
  weechat.prnt( "       If a long URL appears in an active channel," )
  weechat.prnt( "       tinyurl.py will send a message to the channel" )
  weechat.prnt( "       (as you) with the tinyurl equivalent" )
  weechat.prnt( "" )
  weechat.prnt( "  /tinyurl printall [[=] on/off]" )
  weechat.prnt( "    -> Gets/Sets the printall status." )
  weechat.prnt( "       If \'on\', locally prints tinyurls in non-active channels" )

  return

def tinyurlMain( server, args ):
  args = args.split( " " )
  while '' in args:
    args.remove('')
  while ' ' in args:
    args.remove(' ')
  if len(args) == 0:
    tinyurlGet()
  elif args[0].lower() == "help":
    tinyurlUsage()
  else:
    name = args[0]
    value = ""
    if len(args) > 1:
      if args[1] == "=":
        value = " ".join(args[2:])
      else:
        value = " ".join(args[1:])
      tinyurlSet( args[0], value )
    else:
      tinyurlGet( name )
  return weechat.PLUGIN_RC_OK

def getTinyurl( url ):
  turl = None
  maxlen = int(weechat.get_plugin_config( "urllength" ))
  if len(url) < maxlen:
    return None
  cmd = "curl -d 'url=%s' http://tinyurl.com/create.php --stderr /dev/null" % \
           (urllib.quote(url))
  html = os.popen( cmd, "r" )
  if html is None:
    return None
  for line in html:
    if( line.startswith("<input type=hidden name=tinyurl value=\"") ):
      turlend = line[39:].find("\"")
      turl = line[39:][:turlend]
      break
  html.close()
  return turl

def printTinyurl( url, channel, server ):
  activeChans = weechat.get_plugin_config('activechans').split(',')
  if channel in activeChans:
    weechat.command( "/msg %s [AKA] %s" % ( channel, url) )
  else:
    weechat.prnt( "[AKA] %s" % (url), channel, server )

def findUrlstart( msg ):
  index = -1
  for prefix in ( "http://", "https://", "ftp://", "ftps://" ):
    index = msg.find( prefix )
    if index > -1:
      break
  return index 

def checkTinyurl( server, args ):
  activeChans = weechat.get_plugin_config('activechans').split(',')
  onlyActiveChans = weechat.get_plugin_config('printall') == "off"
  (source, type, channel, msg) = args.split(" ", 3)
  if onlyActiveChans and channel not in activeChans:
    return weechat.PLUGIN_RC_OK
  urlstart = findUrlstart( msg )
  if urlstart > -1:
    endurl = msg[urlstart:].find(" ")
    if endurl > -1:
      url = msg[urlstart:endurl]
    else:
      url = msg[urlstart:]
    tinyurl = getTinyurl(url)
    if tinyurl is not None:
      printTinyurl( tinyurl, channel, server )
  return weechat.PLUGIN_RC_OK

