/*
 * cpufreq plugin for Xfce4.
 *
 * Copyright (c) 2005 Joshua Kwan <joshk@triplehelix.org>
 *
 * Some code was used as a starter, but is mostly gone. Nevertheless, this code
 * was:
 * 
 * Copyright (c) 2003 Nicholas Penwarden <toth64@yahoo.com>
 * Copyright (c) 2003 Benedikt Meurer <benedikt.meurer@unix-ag.uni-siegen.de>
 * Copyright (c) 2003 edscott wilson garcia <edscott@users.sourceforge.net>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifdef ENABLE_NLS
#include <libintl.h>
#endif

#include <ctype.h>

#include <cpufreq.h>

#include <gtk/gtk.h>

#include <libxfce4util/i18n.h>
#include <libxfcegui4/dialogs.h>
#include <panel/global.h>
#include <panel/controls.h>
#include <panel/plugins.h>
#include <panel/xfce_support.h>

#define COLOR_LOW_DEFAULT "#ff0000"
#define COLOR_HI_DEFAULT "#00ff00"

struct cpufreq_monitor
{
	GtkWidget* pbar;
	GtkWidget* label;
	GtkWidget* vbox;

	unsigned long min, max;

	int orientation;
	int nr_cpus;
	int cpu;
	int timeoutid;

	GdkColor colorLow;
	GdkColor colorHi;

	gpointer extra; /* XXX */
};

static int
get_nr_cpus (void)
{
	FILE *f = fopen("/proc/stat", "r");
	char buf[512];
	int nr_cpus = 0;

	if (!f)
		return -1;

	while (fgets(buf, 512, f) != NULL)
	{
		if (strncmp(buf, "cpu", 3) == 0)
		{
			if (isdigit(buf[3]))
				nr_cpus++;
		}
	}

	fclose(f);

	return nr_cpus;
}

static char*
get_cpu_name (int cpu)
{
	FILE *f = fopen("/proc/cpuinfo", "r");
	int cur_cpu = -1;
	char buf[512];

	if (!f)
		return strdup(_("Processor model unknown"));
	
	while (fgets(buf, 512, f) != NULL)
	{
		if (strncmp(buf, "processor", 9) == 0)
		{
			cur_cpu++;
		}
		else if (cur_cpu == cpu && strncmp(buf, "model name", 10) == 0)
		{
			char* p = strchr(buf, ':');

			if (p)
			{
				fclose (f);
				return strdup(p + 2);
			}
		}
	}

	fclose(f);
	
	return strdup(_("Processor model unknown"));
}

static gboolean
update_cpufreq_status (struct cpufreq_monitor *c)
{
	unsigned int cur = cpufreq_get_freq_kernel(c->cpu);
	char buf[256];

	if (cur == 0) /* error */
	{
		GtkWindow* win = GTK_WINDOW(gtk_widget_get_toplevel(c->vbox));
		
		if (c->cpu == 0) /* MUST work, unless cpufreq support is not available */
		{
			GtkWidget* d = gtk_message_dialog_new(win, 0,
				GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
				_("CPUFreq support is not present. The CPUFreq Monitor will not function this session."));
			gtk_dialog_run(GTK_DIALOG(d));
			gtk_widget_destroy(d);

			/* hide the useless widgets */
			gtk_widget_hide(c->vbox);
			return TRUE; /* not an indicator of success */
		}
		else /* some other error, fall back to 0 */
		{
			GtkWidget* d = gtk_message_dialog_new(win, 0,
				GTK_MESSAGE_WARNING, GTK_BUTTONS_OK,
				_("CPU %d is not present. The CPUFreq Monitor will monitor CPU 0 for this session."), c->cpu);
			gtk_dialog_run(GTK_DIALOG(d));
			gtk_widget_destroy(d);

			c->cpu = 0;
			return TRUE; /* to get to the top of error-handling */
		}
	}
	
	if (cur < 1000000)
		snprintf(buf, 256, "<span size=\"small\">%d MHz</span>", cur / 1000);
	else
		snprintf(buf, 256, "<span size=\"small\">%.1f GHz</span>", (gdouble)cur / 1000000);

	if (cur < c->max)
		gtk_widget_modify_bg(c->pbar, GTK_STATE_PRELIGHT, &(c->colorLow));
	else
		gtk_widget_modify_bg(c->pbar, GTK_STATE_PRELIGHT, &(c->colorHi));

	gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(c->pbar), (gdouble)cur / c->max);
	gtk_label_set_markup(GTK_LABEL(c->label), buf);

	return TRUE;
}

static void
cpufreq_free (Control *ctrl)
{
	struct cpufreq_monitor *c = (struct cpufreq_monitor *)ctrl->data;

	if (c->timeoutid != 0)
	{
		g_source_remove(c->timeoutid);
		c->timeoutid = 0;
	}

	g_free(c);
}

static void
setup_cpufreq(Control *ctrl)
{
	struct cpufreq_monitor *c = (struct cpufreq_monitor *)ctrl->data;
	
	c->pbar = gtk_progress_bar_new();
	c->label = gtk_label_new("");

	gtk_label_set_use_markup (GTK_LABEL(c->label), TRUE);
	
	if (c->orientation == HORIZONTAL) {
		gtk_progress_bar_set_orientation(GTK_PROGRESS_BAR(c->pbar),
			GTK_PROGRESS_BOTTOM_TO_TOP);
		c->vbox = gtk_hbox_new(FALSE, 0);
	} else if (c->orientation == VERTICAL) {
		gtk_progress_bar_set_orientation(GTK_PROGRESS_BAR(c->pbar),
			GTK_PROGRESS_LEFT_TO_RIGHT);
		c->vbox = gtk_vbox_new(FALSE, 0);
	}

	gtk_box_pack_start(GTK_BOX(c->vbox), c->pbar, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(c->vbox), c->label, FALSE, FALSE, 0);
	
	gtk_container_add(GTK_CONTAINER(ctrl->base), c->vbox);
	gtk_container_set_border_width(GTK_CONTAINER(c->vbox), border_width);

	gtk_widget_set_size_request(ctrl->base, -1, -1);
	gtk_widget_show(c->vbox);
	gtk_widget_show(c->label);
	gtk_widget_show(c->pbar);

	update_cpufreq_status(c);
	
	c->timeoutid = g_timeout_add(1000, (GSourceFunc) update_cpufreq_status, c);
	
	return;
}

static gboolean
cpufreq_control_new(Control *ctrl)
{
	struct cpufreq_monitor* c = malloc(sizeof(struct cpufreq_monitor));

	c->cpu = 0;
	c->orientation = VERTICAL; /* a very good default */
	c->nr_cpus = get_nr_cpus();
	
	/* determine min/max */
	cpufreq_get_hardware_limits(c->cpu, &c->min, &c->max);
	
	ctrl->with_popup = FALSE;
	ctrl->data = c;
	
	gdk_color_parse(COLOR_LOW_DEFAULT, &(c->colorLow));
	gdk_color_parse(COLOR_HI_DEFAULT, &(c->colorHi));
	
	setup_cpufreq(ctrl);

	return (TRUE);
}

static void
cpufreq_attach_callback (Control *ctrl, const gchar *signal, GCallback cb, gpointer data)
{
	struct cpufreq_monitor *c = (struct cpufreq_monitor*)ctrl->data;

	g_signal_connect(c->vbox, signal, cb, data);
	g_signal_connect(c->pbar, signal, cb, data);
	g_signal_connect(c->label, signal, cb, data);
}

static void
cpufreq_set_orientation (Control *ctrl, int orientation)
{
	struct cpufreq_monitor *c = (struct cpufreq_monitor *)ctrl->data;

	if (c->orientation == orientation)
		return;
	
	c->orientation = orientation;

	if (c->timeoutid > 0)
	{
		g_source_remove(c->timeoutid);
		c->timeoutid = 0;
	}

	gtk_container_remove(GTK_CONTAINER(ctrl->base), c->vbox);

	setup_cpufreq(ctrl);
}

static void
cpufreq_read_config (Control *ctrl, xmlNodePtr parent)
{
	xmlChar *value;
	xmlNodePtr child;

	struct cpufreq_monitor *c = (struct cpufreq_monitor *)ctrl->data;

	if (!parent || !parent->children) return;

	child = parent->children;

	if (!xmlStrEqual(child->name, "CPUFreqMonitor")) return;

	if ((value = xmlGetProp(child, (const xmlChar *) "cpu_nr")) != NULL)
	{
		c->cpu = atoi(value);
		g_free(value);
	}
}

static void
cpufreq_write_config (Control *ctrl, xmlNodePtr parent)
{
	xmlNodePtr root;
	char buf[32];

	struct cpufreq_monitor *c = (struct cpufreq_monitor *)ctrl->data;

	root = xmlNewTextChild(parent, NULL, "CPUFreqMonitor", NULL);

	snprintf(buf, 32, "%d", c->cpu);
	xmlSetProp(root, "cpu_nr", buf);
}

static void
cpufreq_set_size (Control *ctrl, int size)
{
	struct cpufreq_monitor *c = (struct cpufreq_monitor *)ctrl->data;
	
	if (c->orientation == HORIZONTAL) {
		gtk_widget_set_size_request(c->pbar, 10, size);
	} else if (c->orientation == VERTICAL) {
		gtk_widget_set_size_request(c->pbar, size, 10);
	}
	
	gtk_widget_set_size_request(ctrl->base, -1, -1);
}

static void
spin_changed_cb (GtkSpinButton *sb, GtkScrollType st, gpointer data)
{
	struct cpufreq_monitor *c = (struct cpufreq_monitor *)data;
	int val = (int)gtk_spin_button_get_value(sb);
	char buf[256], *cpu_name;
	
	if (c->cpu == val) return;
	
	c->cpu = val;

	cpu_name = get_cpu_name(c->cpu); /* must free */
	snprintf(buf, 256, "<span size=\"small\">%s</span>", cpu_name);
	g_free(cpu_name);
	gtk_label_set_markup(GTK_LABEL(c->extra), buf);

	cpufreq_get_hardware_limits(c->cpu, &c->min, &c->max);
}

static void
cpufreq_create_options(Control *ctrl, GtkContainer *container, GtkWidget *done)
{
	struct cpufreq_monitor *c = (struct cpufreq_monitor *)ctrl->data;
	GtkWidget *hbox = gtk_hbox_new(FALSE, 0);
	GtkWidget *vbox = gtk_vbox_new(FALSE, 0);
	GtkWidget *sb = gtk_spin_button_new_with_range(0, c->nr_cpus - 1, 1);
	GtkWidget *label = gtk_label_new(_("Choose the CPU to monitor:"));
	GtkWidget *cpulabel = gtk_label_new("");
	char buf[256], *cpu_name;

	gtk_label_set_use_markup(GTK_LABEL(label), TRUE);

	gtk_spin_button_set_value(GTK_SPIN_BUTTON(sb), 0);
	
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
	gtk_box_pack_start(GTK_BOX(hbox), sb, FALSE, FALSE, 5);

	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 5);
	gtk_box_pack_start(GTK_BOX(vbox), cpulabel, FALSE, FALSE, 0);
	
	cpu_name = get_cpu_name(c->cpu); /* must free */
	snprintf(buf, 256, "<span size=\"small\">%s</span>", cpu_name);
	g_free(cpu_name);
	gtk_label_set_markup(GTK_LABEL(cpulabel), buf);

	c->extra = (gpointer)cpulabel;
	
	g_signal_connect(sb, "value_changed", G_CALLBACK(spin_changed_cb), c);

	gtk_container_add(container, vbox);

	gtk_widget_show(vbox);
	gtk_widget_show(hbox);
	gtk_widget_show(sb);
	gtk_widget_show(label);
	gtk_widget_show(cpulabel);
}

G_MODULE_EXPORT void
xfce_control_class_init(ControlClass *cc)
{
#ifdef ENABLE_NLS
    /* This is required for UTF-8 at least - Please don't remove it */
    bindtextdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
    bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
    textdomain (GETTEXT_PACKAGE);
#endif

	cc->name = "cpufreq";
	cc->caption = _("CPUFreq Monitor");

	cc->create_control = (CreateControlFunc)cpufreq_control_new;

	cc->free = cpufreq_free;
	cc->read_config = cpufreq_read_config;
	cc->write_config = cpufreq_write_config;
	cc->attach_callback = cpufreq_attach_callback;

	cc->set_size = cpufreq_set_size;
	cc->set_orientation = cpufreq_set_orientation;
	cc->create_options = cpufreq_create_options;
}

XFCE_PLUGIN_CHECK_INIT
