/*
 * Copyright (c) 2003 Benedikt Meurer (benedikt.meurer@unix-ag.uni-siegen.de)
 *               2004 Jean-Franois Wauthy (pollux@xfce.org)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#ifdef  HAVE_CONFIG_H
#include <config.h>
#endif /* !HAVE_CONFIG_H */

#include <string.h>
#include <stdlib.h>
#include <gtk/gtk.h>
#include <libxfce4util/libxfce4util.h>
#include <libxfcegui4/libxfcegui4.h>

#include <printing-system.h>

#include "queuedlg.h"

extern GList *printers_list;

/* Prototypes */
static gboolean delete_window_cb (GtkWidget * widget, GdkEvent * event, gpointer data);
static void close_window_cb (GtkWidget * widget, gpointer data);
static void icon_theme_changed_cb (XfceIconTheme * icon_theme, gpointer user_data);
static void tree_selection_changed_cb (GtkTreeSelection * selection, gpointer data);
static gboolean tree_button_pressed_cb (GtkWidget * widget, GdkEventButton * event, gpointer data);
static void remove_job_cb (GtkWidget * widget, gpointer data);
static gboolean timeout_fct (gpointer data);
static void refresh_tree_cb (GtkWidget * widget, gpointer data);

/**********************************************/
/* Show printer queue in a treeview and allow */
/* user to remove jobs                        */
/**********************************************/
void
show_printer_queue (const gchar * printer_name)
{
  QueueDlg *queuedlg;

  gchar *title;
  GtkWidget *vbox;

  GtkWidget *menubar;
  GtkWidget *menu_job;
  GtkWidget *separator;

  GtkListStore *store;
  GtkWidget *scrollwin;
  GtkTreeSelection *selection;

  GtkCellRenderer *cell_icon, *cell_name;
  GtkTreeViewColumn *column_name;

  GdkPixbuf *icon = NULL;

  /* Create window */
  queuedlg = g_new0 (QueueDlg, 1);

  queuedlg->accel_group = gtk_accel_group_new ();
  queuedlg->printer_name = g_strdup (printer_name);

  queuedlg->icontheme = xfce_icon_theme_get_for_screen (NULL);
  g_signal_connect (G_OBJECT (queuedlg->icontheme), "changed", G_CALLBACK (icon_theme_changed_cb), queuedlg);

  queuedlg->window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  title = g_strdup_printf ("%s - %s", _("Queue Manager"), (printer_lookup_byname (printers_list, printer_name))->alias);
  gtk_window_set_title (GTK_WINDOW (queuedlg->window), title);
  g_free (title);
  icon = xfce_icon_theme_load (queuedlg->icontheme, "xfprint", 48);
  if (icon) {
    gtk_window_set_icon (GTK_WINDOW (queuedlg->window), icon);
    g_object_unref (icon);
  }
  gtk_window_set_default_size (GTK_WINDOW (queuedlg->window), 450, 150);
  gtk_window_add_accel_group (GTK_WINDOW (queuedlg->window), queuedlg->accel_group);
  gtk_window_set_position (GTK_WINDOW (queuedlg->window), GTK_WIN_POS_NONE);
  g_signal_connect (G_OBJECT (queuedlg->window), "delete_event", G_CALLBACK (delete_window_cb), queuedlg);

  vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (queuedlg->window), vbox);

  /* Add menubar */
  menubar = gtk_menu_bar_new ();
  gtk_box_pack_start (GTK_BOX (vbox), menubar, FALSE, FALSE, 0);

  queuedlg->menuitem_job = gtk_menu_item_new_with_mnemonic (_("_Jobs"));
  gtk_container_add (GTK_CONTAINER (menubar), queuedlg->menuitem_job);

  menu_job = gtk_menu_new ();
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (queuedlg->menuitem_job), menu_job);

  queuedlg->menuitem_removejob = gtk_image_menu_item_new_with_mnemonic (_("_Remove job"));
  gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (queuedlg->menuitem_removejob),
                                 gtk_image_new_from_stock (GTK_STOCK_REMOVE, GTK_ICON_SIZE_MENU));
  g_signal_connect (G_OBJECT (queuedlg->menuitem_removejob), "activate", G_CALLBACK (remove_job_cb), queuedlg);
  gtk_container_add (GTK_CONTAINER (menu_job), queuedlg->menuitem_removejob);
  gtk_widget_set_sensitive (queuedlg->menuitem_removejob, FALSE);
  queuedlg->menuitem_refresh = gtk_image_menu_item_new_from_stock (GTK_STOCK_REFRESH, queuedlg->accel_group);
  g_signal_connect (G_OBJECT (queuedlg->menuitem_refresh), "activate", G_CALLBACK (refresh_tree_cb), queuedlg);
  gtk_container_add (GTK_CONTAINER (menu_job), queuedlg->menuitem_refresh);
  separator = gtk_menu_item_new ();
  gtk_widget_set_sensitive (separator, FALSE);
  gtk_container_add (GTK_CONTAINER (menu_job), separator);
  queuedlg->menuitem_close = gtk_image_menu_item_new_from_stock (GTK_STOCK_CLOSE, queuedlg->accel_group);
  g_signal_connect (G_OBJECT (queuedlg->menuitem_close), "activate", G_CALLBACK (close_window_cb), queuedlg);
  gtk_container_add (GTK_CONTAINER (menu_job), queuedlg->menuitem_close);

  /* Add treeview */
  store = gtk_list_store_new (JOBS_N_COLUMNS, GDK_TYPE_PIXBUF, G_TYPE_STRING,
                              G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING,
                              G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING);

  scrollwin = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrollwin), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_box_pack_start (GTK_BOX (vbox), scrollwin, TRUE, TRUE, 0);

  queuedlg->treeview = gtk_tree_view_new_with_model (GTK_TREE_MODEL (store));
  gtk_container_add (GTK_CONTAINER (scrollwin), queuedlg->treeview);
  g_signal_connect (G_OBJECT (queuedlg->treeview), "button-press-event", G_CALLBACK (tree_button_pressed_cb), queuedlg);


  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (queuedlg->treeview));
  g_signal_connect (G_OBJECT (selection), "changed", G_CALLBACK (tree_selection_changed_cb), queuedlg);


  /* Columns in treeview */
  cell_icon = gtk_cell_renderer_pixbuf_new ();
  cell_name = gtk_cell_renderer_text_new ();
  column_name = gtk_tree_view_column_new ();

  gtk_tree_view_column_pack_start (column_name, cell_icon, FALSE);
  gtk_tree_view_column_set_attributes (column_name, cell_icon, "pixbuf", JOBS_ICON_COLUMN, NULL);
  g_object_set (cell_icon, "xalign", 0.0, "ypad", 0, NULL);

  gtk_tree_view_column_pack_start (column_name, cell_name, TRUE);
  gtk_tree_view_column_set_attributes (column_name, cell_name, "markup", JOBS_NAME_COLUMN, NULL);
  g_object_set (cell_name, "ypad", 0, "yalign", 0.5, NULL);
  gtk_tree_view_column_set_title (column_name, _("Name"));

  gtk_tree_view_append_column (GTK_TREE_VIEW (queuedlg->treeview), column_name);
  gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (queuedlg->treeview),
                                               JOBS_ID_COLUMN, _("ID"), gtk_cell_renderer_text_new (),
                                               "text", JOBS_ID_COLUMN, NULL);
  gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (queuedlg->treeview),
                                               JOBS_USER_COLUMN, _("User"), gtk_cell_renderer_text_new (),
                                               "text", JOBS_USER_COLUMN, NULL);
  gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (queuedlg->treeview),
                                               JOBS_STATE_COLUMN, _("State"), gtk_cell_renderer_text_new (),
                                               "text", JOBS_STATE_COLUMN, NULL);
  gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (queuedlg->treeview),
                                               JOBS_SIZE_COLUMN, _("Size"), gtk_cell_renderer_text_new (),
                                               "text", JOBS_SIZE_COLUMN, NULL);
  gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (queuedlg->treeview),
                                               JOBS_PRIORITY_COLUMN, _("Priority"), gtk_cell_renderer_text_new (),
                                               "text", JOBS_PRIORITY_COLUMN, NULL);
  gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (queuedlg->treeview),
                                               JOBS_CREATION_TIME_COLUMN, _("Creation time"),
                                               gtk_cell_renderer_text_new (), "text", JOBS_CREATION_TIME_COLUMN, NULL);
  gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (queuedlg->treeview), JOBS_PROCESSING_TIME_COLUMN,
                                               _("Processing time"), gtk_cell_renderer_text_new (), "text",
                                               JOBS_PROCESSING_TIME_COLUMN, NULL);
  /* Load jobs in tree */
  refresh_tree_cb (NULL, queuedlg);

  /* Refresh job list every 20 sec */
  queuedlg->timeout_id = g_timeout_add (5000, timeout_fct, queuedlg);

  /* Customize dialog */
  printing_system_queue_dialog_customize (queuedlg);

  /* Show window */
  gtk_widget_show_all (queuedlg->window);
}

/* Callbacks */
static gboolean
delete_window_cb (GtkWidget * widget, GdkEvent * event, gpointer data)
{
  QueueDlg *dialog;

  dialog = (QueueDlg *) data;

  g_source_remove (dialog->timeout_id);
  g_free (dialog->printer_name);
  g_free (dialog);

  return FALSE;
}

static void
close_window_cb (GtkWidget * widget, gpointer data)
{
  QueueDlg *dialog;

  dialog = (QueueDlg *) data;

  g_source_remove (dialog->timeout_id);
  g_free (dialog->printer_name);
  g_free (dialog);
  gtk_widget_destroy (dialog->window);
}

/*****************************/
/* Manage icon theme changes */
/*****************************/
static gboolean
icon_theme_update_foreach_func (GtkTreeModel * model, GtkTreePath * path, GtkTreeIter * iter, gpointer data)
{
  QueueDlg *dlg;
  GdkPixbuf *icon;
  gint x,y;

  dlg = (QueueDlg *) data;

  gtk_tree_model_get (model, iter, JOBS_ICON_COLUMN, &icon, -1);

  if (icon)
    g_object_unref (icon);

  if (gtk_icon_size_lookup (GTK_ICON_SIZE_LARGE_TOOLBAR, &x, &y))
    icon = xfce_icon_theme_load (dlg->icontheme, "gnome-mime-application-postscript", x);
  else
    icon = xfce_icon_theme_load (dlg->icontheme, "gnome-mime-application-postscript", 24);  
  
  gtk_list_store_set (GTK_LIST_STORE (model), iter, JOBS_ICON_COLUMN, icon, -1);
  g_object_unref (icon);

  return FALSE;
}

static void icon_theme_changed_cb (XfceIconTheme * icon_theme, gpointer user_data)
{
  QueueDlg *dlg;
  GtkTreeModel *model;

  dlg = (QueueDlg *) user_data;

  model = gtk_tree_view_get_model (GTK_TREE_VIEW (dlg->treeview));

  if (model)
    gtk_tree_model_foreach (model, &icon_theme_update_foreach_func, dlg);
}

static gboolean
tree_button_pressed_cb (GtkWidget * widget, GdkEventButton * event, gpointer data)
{
  QueueDlg *dialog;

  dialog = (QueueDlg *) data;

  /* Right click draws the context menu */
  if ((event->button == 3) && (event->type == GDK_BUTTON_PRESS)) {
    GtkTreePath *path;

    if (gtk_tree_view_get_path_at_pos (GTK_TREE_VIEW (dialog->treeview), event->x, event->y, &path, NULL, NULL, NULL)) {
      GtkTreeSelection *selection;
      GtkWidget *menu_popup;
      GtkWidget *menuitem_removejob;
      GtkWidget *menuitem_separator;
      GtkWidget *menuitem_refresh;

      selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (dialog->treeview));
      gtk_tree_selection_unselect_all (selection);
      gtk_tree_selection_select_path (selection, path);

      /* Create the popup menu */
      menu_popup = gtk_menu_new ();

      menuitem_removejob = gtk_image_menu_item_new_with_mnemonic (_("_Remove job"));
      gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (menuitem_removejob),
                                     gtk_image_new_from_stock (GTK_STOCK_REMOVE, GTK_ICON_SIZE_MENU));
      gtk_container_add (GTK_CONTAINER (menu_popup), menuitem_removejob);
      menuitem_separator = gtk_separator_menu_item_new ();
      gtk_container_add (GTK_CONTAINER (menu_popup), menuitem_separator);
      menuitem_refresh = gtk_image_menu_item_new_from_stock (GTK_STOCK_REFRESH, dialog->accel_group);
      gtk_container_add (GTK_CONTAINER (menu_popup), menuitem_refresh);

      g_signal_connect (G_OBJECT (menuitem_removejob), "activate", G_CALLBACK (remove_job_cb), dialog);
      g_signal_connect (G_OBJECT (menuitem_refresh), "activate", G_CALLBACK (refresh_tree_cb), dialog);

      gtk_widget_show_all (menu_popup);
      gtk_menu_popup (GTK_MENU (menu_popup), NULL, NULL, NULL, NULL, event->button, gtk_get_current_event_time ());
      return TRUE;
    }
  }

  return FALSE;
}

static void
tree_selection_changed_cb (GtkTreeSelection * selection, gpointer data)
{
  GtkTreeModel *model;
  GtkTreeIter iter;
  QueueDlg *dialog;

  dialog = (QueueDlg *) data;

  if (gtk_tree_selection_get_selected (selection, &model, &iter))
    gtk_widget_set_sensitive (dialog->menuitem_removejob, TRUE);
  else
    gtk_widget_set_sensitive (dialog->menuitem_removejob, FALSE);
}

static void
refresh_tree_cb (GtkWidget * widget, gpointer data)
{
  QueueDlg *dialog;

  GtkTreeModel *model;
  GtkTreeSelection *selection = NULL;
  gchar *selected_id = NULL;
  GtkTreeIter iter;

  GList *jobs_list = NULL;
  GList *job = NULL;

  dialog = (QueueDlg *) data;

  /* Remember selected ID */
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (dialog->treeview));

  if (selection && gtk_tree_selection_get_selected (selection, &model, &iter))
    gtk_tree_model_get (model, &iter, JOBS_ID_COLUMN, &selected_id, -1);
  else
    model = gtk_tree_view_get_model (GTK_TREE_VIEW (dialog->treeview));

  /* Empty the list */
  gtk_list_store_clear (GTK_LIST_STORE (model));

  /* Obtain job list from cups server */
  jobs_list = printing_system_get_jobs_list (dialog->printer_name);

  /* Show job list */
  job = g_list_first (jobs_list);
  while (job) {
    gchar *string_id;
    GdkPixbuf *icon = NULL;
    gint x,y;

    Job *job_data = (Job *) job->data;

    if (gtk_icon_size_lookup (GTK_ICON_SIZE_LARGE_TOOLBAR, &x, &y))
      icon = xfce_icon_theme_load (dialog->icontheme, "gnome-mime-application-postscript", x);
    else
      icon = xfce_icon_theme_load (dialog->icontheme, "gnome-mime-application-postscript", 24);

    string_id = g_strdup_printf ("%u", job_data->id);

    gtk_list_store_append (GTK_LIST_STORE (model), &iter);
    gtk_list_store_set (GTK_LIST_STORE (model), &iter,
			JOBS_ICON_COLUMN, icon,
                        JOBS_NAME_COLUMN, job_data->name ? job_data->name : "",
                        JOBS_ID_COLUMN, string_id,
                        JOBS_USER_COLUMN, job_data->user ? job_data->user : "",
                        JOBS_STATE_COLUMN, job_data->state ? job_data->state : "",
                        JOBS_SIZE_COLUMN, job_data->size ? job_data->size : "",
                        JOBS_PRIORITY_COLUMN, job_data->priority ? job_data->priority : "",
                        JOBS_CREATION_TIME_COLUMN, job_data->creation_time ? job_data->creation_time : "",
                        JOBS_PROCESSING_TIME_COLUMN, job_data->processing_time ? job_data->processing_time : "", -1);

    g_free (string_id);
    g_object_unref (G_OBJECT (icon));
    job = g_list_next (job);
  }

  /* Free the jobs list */
  g_list_foreach (jobs_list, (GFunc) job_free, NULL);
  g_list_free (jobs_list);

  /* Reselect previously selected job if still available */
  if (selected_id && gtk_tree_model_get_iter_first (model, &iter)) {
    gchar *current_id;

    gtk_tree_model_get (model, &iter, JOBS_ID_COLUMN, &current_id, -1);
    if (strcmp (current_id, selected_id) == 0) {
      gtk_tree_selection_select_iter (selection, &iter);
      g_free (current_id);
    }
    else {
      g_free (current_id);
      while (gtk_tree_model_iter_next (model, &iter)) {
        gtk_tree_model_get (model, &iter, JOBS_ID_COLUMN, &current_id, -1);
        if (strcmp (current_id, selected_id) == 0) {
          gtk_tree_selection_select_iter (selection, &iter);
          g_free (current_id);
          break;
        }
        g_free (current_id);
      }
    }
  }

  g_free (selected_id);
}

static void
remove_job_cb (GtkWidget * widget, gpointer data)
{
  QueueDlg *dialog;

  GtkTreeModel *model;
  GtkTreeIter iter;
  GtkTreeSelection *selection;

  dialog = (QueueDlg *) data;

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (dialog->treeview));

  if (selection && gtk_tree_selection_get_selected (selection, &model, &iter)) {
    gint response = GTK_RESPONSE_NONE;
    gchar *string_id;
    gchar *string_title;
    gchar *dialog_message;

    gtk_tree_model_get (model, &iter, JOBS_NAME_COLUMN, &string_title, JOBS_ID_COLUMN, &string_id, -1);

    dialog_message = g_strdup_printf ("%s%s (%s) ?", _("Are you sure you want to remove the job "),
                                      string_id, string_title);

    response = xfce_message_dialog (GTK_WINDOW (dialog->window), _("Remove job"),
                                    GTK_STOCK_DIALOG_QUESTION,
                                    dialog_message, NULL,
                                    XFCE_CUSTOM_STOCK_BUTTON, _("Don't remove job"), GTK_STOCK_CANCEL,
                                    GTK_RESPONSE_CANCEL, XFCE_CUSTOM_STOCK_BUTTON, _("Remove job"), GTK_STOCK_REMOVE,
                                    GTK_RESPONSE_OK, NULL);
    if (response == GTK_RESPONSE_OK) {
      if (printing_system_remove_job (dialog->printer_name, atoi (string_id)))
        refresh_tree_cb (NULL, dialog);
      else
        xfce_err (_("An error occurred while removing job !"));
    }

    g_free (string_id);
    g_free (string_title);
    g_free (dialog_message);
  }
}

static gboolean
timeout_fct (gpointer data)
{
  QueueDlg *dlg;

  dlg = (QueueDlg *) data;

  refresh_tree_cb (dlg->treeview, dlg);

  return TRUE;
}
