/**
 * This is a test program for the Catalog class
 */

import java.io.*;
import com.arbortext.catalog.*;
import java.net.MalformedURLException;
import java.lang.Integer;

import java.util.Properties;
import java.util.Enumeration;
import java.util.Vector;

public class catalog {
    public static void main (String[] args)
	throws FileNotFoundException, MalformedURLException, IOException {

	if (args.length == 0) {
	    System.out.println("Usage: catalogtest [options] command");
	    System.out.println("Where options are:");
	    System.out.println("-c catalogfile");
	    System.out.println("-d debuglevel");
	    System.out.println("-p parser class");
	    System.out.println("-s");
	    System.out.println("");
	    System.out.println("The debug level should be an integer (0..3)");
	    System.out.println("The -c option can be repeated to load several catalogs");
	    System.out.println("The parser class is the name of a parser class to use for XML Catalogs");
	    System.out.println("If the -s option is specified, the system catalogs are loaded before any");
	    System.out.println("   catalogs specified by the -c options.");
	    System.out.println("");
	    System.out.println("Command is one of the following:");
	    System.out.println("  document");
	    System.out.println("  doctype  name publicid systemid");
	    System.out.println("  entity   name publicid systemid");
	    System.out.println("  notation name publicid systemid");
	    System.out.println("  public        publicid systemid");
	    System.out.println("  system                 systemid");
	    System.out.println("");
	    System.out.println("Arguments are positional, use the string \"null\" to indicate a null value.");
	    System.exit(1);
	}

	Vector catalogFiles = new Vector();
	String debugLevel = null;
	boolean debugSet = false;
	String parserClass = null;
	boolean loadSystemCatalogs = false;
	Vector commandArgs = new Vector();

	for (int i=0; i<args.length; i++) {
	    if (args[i].equals("-c")) {
		++i;
		catalogFiles.addElement(args[i]);
		continue;
	    }
	    if (args[i].equals("-d")) {
		++i;
		debugLevel = args[i];
		debugSet = true;
		continue;
	    }
	    if (args[i].equals("-p")) {
		++i;
		parserClass = args[i];
		continue;
	    }
	    if (args[i].equals("-s")) {
		loadSystemCatalogs = true;
		continue;
	    }

	    commandArgs.addElement(args[i]);
	}

	Catalog c = new Catalog();

	if (loadSystemCatalogs) {
	    System.out.println("Loading system catalogs.");
	    c.loadSystemCatalogs();
	} else {
	    System.out.println("Ignoring system catalogs.");
	}

	if (debugSet) {
	    try {
		int debug = Integer.parseInt(debugLevel);
		c.debug = debug;
		System.out.println("Set debug to: " + debug);
	    } catch (Exception e) {
		System.out.println("Cannot parse debug level: " + debugLevel);
		debugSet = false;
	    }
	}

	if (!debugSet) {
	    int defaultDebug = 3;
	    System.out.println("Setting debug to default: " + defaultDebug);
	    c.debug = defaultDebug;
	}

	if (parserClass != null) {
	    System.out.println("Set XML Parser class to: " + parserClass);
	    c.setParserClass(parserClass);
	}

	Enumeration cats = catalogFiles.elements();
	while (cats.hasMoreElements()) {
	    String catalog = (String) cats.nextElement();
	    System.out.println("Adding catalog: " + catalog);
	    c.parseCatalog(catalog);
	}

	int entityType = 0;
	String entityTypeName = null;
	String entityName = null;
	String publicId = null;
	String systemId = null;

	Enumeration argEnum = commandArgs.elements();
	while (argEnum.hasMoreElements()) {
	    String arg = (String) argEnum.nextElement();

	    if (entityType == 0) {
		entityTypeName = arg;
		if (arg.equalsIgnoreCase("DOCUMENT")) {
		    entityType = CatalogEntry.DOCUMENT;
		}
		if (arg.equalsIgnoreCase("DOCTYPE")) {
		    entityType = CatalogEntry.DOCTYPE;
		}
		if (arg.equalsIgnoreCase("ENTITY")) {
		    entityType = CatalogEntry.ENTITY;
		}
		if (arg.equalsIgnoreCase("NOTATION")) {
		    entityType = CatalogEntry.NOTATION;
		}
		if (arg.equalsIgnoreCase("PUBLIC")) {
		    entityType = CatalogEntry.PUBLIC;
		}
		if (arg.equalsIgnoreCase("SYSTEM")) {
		    entityType = CatalogEntry.SYSTEM;
		}
		if (entityType == 0) {
		    System.out.println("Invalid entity type: " + arg);
		    System.exit(1);
		}
		continue;
	    }

	    if ((entityType == CatalogEntry.DOCTYPE
		 || entityType == CatalogEntry.ENTITY
		 || entityType == CatalogEntry.NOTATION)
		&& entityName == null) {
		entityName = arg;
		continue;
	    }

	    if (publicId == null) {
		publicId = arg;
		continue;
	    }

	    systemId = arg;
	}

	if (entityName != null && entityName.equals("null")) {
	    entityName = null;
	}

	if (publicId != null && publicId.equals("null")) {
	    publicId = null;
	}

	if (systemId != null && systemId.equals("null")) {
	    systemId = null;
	}

	if (entityType == CatalogEntry.SYSTEM && systemId == null) {
	    systemId = publicId;
	    publicId = null;
	}

	System.out.println("Resolving " + entityTypeName + ":");
	if (entityType == CatalogEntry.DOCTYPE
	    || entityType == CatalogEntry.ENTITY
	    || entityType == CatalogEntry.NOTATION) {
	    System.out.println("\tName: " + entityName);
	}
	System.out.println("\tPublic: " + publicId);
	System.out.println("\tSystem: " + systemId);

	String resolved = null;

	switch (entityType) {
	case CatalogEntry.DOCUMENT: {
	    resolved = c.resolveDocument();
	    break;
	}
	case CatalogEntry.DOCTYPE: {
	    resolved = c.resolveDoctype(entityName, publicId, systemId);
	    break;
	}
	case CatalogEntry.ENTITY: {
	    resolved = c.resolveEntity(entityName, publicId, systemId);
	    break;
	}
	case CatalogEntry.NOTATION: {
	    resolved = c.resolveNotation(entityName, publicId, systemId);
	    break;
	}
	case CatalogEntry.PUBLIC: {
	    resolved = c.resolvePublic(publicId, systemId);
	    break;
	}
	case CatalogEntry.SYSTEM: {
	    resolved = c.resolveSystem(systemId);
	    break;
	}
	}

	System.out.println("\nResolved: " + resolved);

	if (systemId != null && resolved == null) {
	    System.out.println("So use " + systemId);
	}
    }
}
