/* ======================================
 * JFreeChart : a free Java chart library
 * ======================================
 *
 * Project Info:  http://www.jfree.org/jfreechart/index.html
 * Project Lead:  David Gilbert (david.gilbert@object-refinery.com);
 *
 * (C) Copyright 2000-2003, by Simba Management Limited and Contributors.
 *
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation;
 * either version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * ---------
 * Plot.java
 * ---------
 * (C) Copyright 2000-2003, by Simba Management Limited and Contributors.
 *
 * Original Author:  David Gilbert (for Simba Management Limited);
 * Contributor(s):   Sylvain Vieujot;
 *                   Jeremy Bowman;
 *                   Andreas Schneider;
 *                   Gideon Krause;
 *
 * $Id: Plot.java,v 1.1 2003/04/23 11:54:43 mungady Exp $
 *
 * Changes (from 21-Jun-2001)
 * --------------------------
 * 21-Jun-2001 : Removed redundant JFreeChart parameter from constructors (DG);
 * 18-Sep-2001 : Updated header info and fixed DOS encoding problem (DG);
 * 19-Oct-2001 : Moved series paint and stroke methods from JFreeChart class (DG);
 * 23-Oct-2001 : Created renderer for LinePlot class (DG);
 * 07-Nov-2001 : Changed type names for ChartChangeEvent (DG);
 *               Tidied up some Javadoc comments (DG);
 * 13-Nov-2001 : Changes to allow for null axes on plots such as PiePlot (DG);
 *               Added plot/axis compatibility checks (DG);
 * 12-Dec-2001 : Changed constructors to protected, and removed unnecessary 'throws' clauses (DG);
 * 13-Dec-2001 : Added tooltips (DG);
 * 22-Jan-2002 : Added handleClick(...) method, as part of implementation for crosshairs (DG);
 *               Moved tooltips reference into ChartInfo class (DG);
 * 23-Jan-2002 : Added test for null axes in chartChanged(...) method, thanks to Barry Evans for
 *               the bug report (number 506979 on SourceForge) (DG);
 *               Added a zoom(...) method (DG);
 * 05-Feb-2002 : Updated setBackgroundPaint(), setOutlineStroke() and setOutlinePaint() to better
 *               handle null values, as suggested by Sylvain Vieujot (DG);
 * 06-Feb-2002 : Added background image, plus alpha transparency for background and foreground (DG);
 * 06-Mar-2002 : Added AxisConstants interface (DG);
 * 26-Mar-2002 : Changed zoom method from empty to abstract (DG);
 * 23-Apr-2002 : Moved dataset from JFreeChart class (DG);
 * 11-May-2002 : Added ShapeFactory interface for getShape() methods, contributed by Jeremy
 *               Bowman (DG);
 * 28-May-2002 : Fixed bug in setSeriesPaint(int, Paint) for subplots (AS);
 * 25-Jun-2002 : Removed redundant imports (DG);
 * 30-Jul-2002 : Added 'no data' message for charts with null or empty datasets (DG);
 * 21-Aug-2002 : Added code to extend series array if necessary (refer to SourceForge bug
 *               id 594547 for details) (DG);
 * 17-Sep-2002 : Fixed bug in getSeriesOutlineStroke(...) method, reported by Andreas
 *               Schroeder (DG);
 * 23-Sep-2002 : Added getLegendItems() abstract method (DG);
 * 24-Sep-2002 : Removed firstSeriesIndex, subplots now use their own paint settings, there is a
 *               new mechanism for the legend to collect the legend items (DG);
 * 27-Sep-2002 : Added dataset group (DG);
 * 14-Oct-2002 : Moved listener storage into EventListenerList.  Changed some abstract methods
 *               to empty implementations (DG);
 * 28-Oct-2002 : Added a getBackgroundImage() method (DG);
 * 21-Nov-2002 : Added a plot index for identifying subplots in combined and overlaid charts (DG);
 * 22-Nov-2002 : Changed all attributes from 'protected' to 'private'.  Added dataAreaRatio
 *               attribute from David M O'Donnell's code (DG);
 * 09-Jan-2003 : Integrated fix for plot border contributed by Gideon Krause (DG);
 * 17-Jan-2003 : Moved to com.jrefinery.chart.plot (DG);
 * 23-Jan-2003 : Removed one constructor (DG);
 * 26-Mar-2003 : Implemented Serializable (DG);
 *
 */

package org.jfree.chart.plot;

import java.awt.AlphaComposite;
import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Composite;
import java.awt.Font;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Insets;
import java.awt.Paint;
import java.awt.Shape;
import java.awt.Stroke;
import java.awt.font.FontRenderContext;
import java.awt.geom.Rectangle2D;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;

import javax.swing.event.EventListenerList;

import org.jfree.chart.ChartRenderingInfo;
import org.jfree.chart.LegendItemCollection;
import org.jfree.chart.axis.AxisConstants;
import org.jfree.chart.event.AxisChangeEvent;
import org.jfree.chart.event.AxisChangeListener;
import org.jfree.chart.event.PlotChangeEvent;
import org.jfree.chart.event.PlotChangeListener;
import org.jfree.data.Dataset;
import org.jfree.data.DatasetChangeEvent;
import org.jfree.data.DatasetChangeListener;
import org.jfree.data.DatasetGroup;
import org.jfree.io.SerialUtilities;
import org.jfree.ui.Align;
import org.jfree.util.ObjectUtils;

/**
 * The base class for all plots in JFreeChart.  The {@link org.jfree.chart.JFreeChart} class 
 * delegates the drawing of axes and data to the plot.  This base class provides facilities common 
 * to most plot types.
 *
 * @author David Gilbert
 */
public abstract class Plot implements AxisChangeListener, 
                                      DatasetChangeListener,
                                      AxisConstants,
                                      Serializable {

    /** Useful constant representing zero. */
    public static final Number ZERO = new Integer(0);

    /** The default insets. */
    public static final Insets DEFAULT_INSETS = new Insets(4, 8, 4, 8);

    /** The default outline stroke. */
    public static final Stroke DEFAULT_OUTLINE_STROKE = new BasicStroke(1);

    /** The default outline color. */
    public static final Paint DEFAULT_OUTLINE_PAINT = Color.gray;

    /** The default foreground alpha transparency. */
    public static final float DEFAULT_FOREGROUND_ALPHA = 1.0f;

    /** The default background alpha transparency. */
    public static final float DEFAULT_BACKGROUND_ALPHA = 1.0f;

    /** The default background color. */
    public static final Paint DEFAULT_BACKGROUND_PAINT = Color.white;

    /** The minimum width at which the plot should be drawn. */
    public static final int MINIMUM_WIDTH_TO_DRAW = 10;

    /** The minimum height at which the plot should be drawn. */
    public static final int MINIMUM_HEIGHT_TO_DRAW = 10;

//    /** The plot index. */
//    private int index;

    /** The parent plot (null if this is the root plot). */
    private Plot parent;
   
    /** The dataset group (to be used for thread synchronisation). */
    private DatasetGroup datasetGroup;

    /** The data. */
    private Dataset dataset;

    /** A secondary dataset. */
    private Dataset secondaryDataset;

    /** The message to display if no data is available. */
    private String noDataMessage;

    /** The font used to display the 'no data' message. */
    private Font noDataMessageFont;

    /** The paint used to draw the 'no data' message. */
    private transient Paint noDataMessagePaint;

    /** Amount of blank space around the plot area. */
    private Insets insets;

    /** The Stroke used to draw an outline around the plot. */
    private transient Stroke outlineStroke;

    /** The Paint used to draw an outline around the plot. */
    private transient Paint outlinePaint;

    /** An optional color used to fill the plot background. */
    private transient Paint backgroundPaint;

    /** An optional image for the plot background. */
    private transient Image backgroundImage;  // not currently serialized

    /** The alignment for the background image. */
    private int backgroundImageAlignment = Align.FIT;
    
    /** The alpha-transparency for the plot. */
    private float foregroundAlpha;

    /** The alpha transparency for the background paint. */
    private float backgroundAlpha;

    /** Storage for registered change listeners. */
    private transient EventListenerList listenerList;

    /** Defines dataArea rectangle as the ratio formed from dividing height by width
     * (of the dataArea).  Modifies plot area calculations.
     * ratio>0 will attempt to layout the plot so that the
     * dataArea.height/dataArea.width = ratio.
     * ratio<0 will attempt to layout the plot so that the
     * dataArea.height/dataArea.width in plot units (not java2D units as when ratio>0)
     * = -1.*ratio.
     */         //dmo
    private double dataAreaRatio = 0.0;  //zero when the parameter is not set

    /**
     * Creates a new plot.
     *
     * @param data  the dataset.
     */
    protected Plot(Dataset data) {

        // set the data and register to receive change notifications...
        this.dataset = data;
        if (data != null) {
            this.datasetGroup = data.getGroup();
            data.addChangeListener(this);
        }
        
//        this.index = 0;
        this.parent = null;
        this.secondaryDataset = null;

        this.insets = DEFAULT_INSETS;
        this.backgroundPaint = DEFAULT_BACKGROUND_PAINT;
        this.backgroundAlpha = DEFAULT_BACKGROUND_ALPHA;
        this.backgroundImage = null;
        this.outlineStroke = DEFAULT_OUTLINE_STROKE;
        this.outlinePaint = DEFAULT_OUTLINE_PAINT;
        this.foregroundAlpha = DEFAULT_FOREGROUND_ALPHA;

        this.noDataMessage = null;
        this.noDataMessageFont = new Font("SansSerif", Font.PLAIN, 12);
        this.noDataMessagePaint = Color.black;

        this.listenerList = new EventListenerList();

    }

//    /**
//     * Returns the plot index.  This defaults to zero, but will be assigned a value if the plot
//    * is added to a combined or overlaid chart structure.
//     *
//     * @return the index.
//     */
//    protected int getIndex() {
//        return this.index;
//    }

//    /**
//     * Sets the plot index.
//     *
//     * @param index  the new index value.
//     */
//    protected void setIndex(int index) {
//        this.index = index;
//    }
    
    /**
     * Returns the dataset group for the plot.
     *
     * @return the dataset group.
     */
    public DatasetGroup getDatasetGroup() {
        return this.datasetGroup;
    }

    /**
     * Sets the dataset group.
     *
     * @param group  the dataset group.
     */
    protected void setDatasetGroup(DatasetGroup group) {
        this.datasetGroup = group;
    }

    /**
     * Returns the primary dataset for the plot.
     *
     * @return the dataset.
     */
    public Dataset getDataset() {
        return dataset;
    }

    /**
     * Sets the primary dataset for the chart, replacing any existing dataset.  Registered
     * listeners are notified that the dataset has been replaced.
     * <P>
     * The plot is automatically registered with the new dataset, to listen for any changes.
     *
     * @param data  the new dataset (<code>null</code> permitted).
     */
    public void setDataset(Dataset data) {

        // if there is an existing dataset, remove the plot from the list of change listeners...
        Dataset existing = this.dataset;
        if (existing != null) {
            existing.removeChangeListener(this);
        }

        // set the new dataset, and register the chart as a change listener...
        this.dataset = data;
        if (data != null) {
            this.datasetGroup = data.getGroup();
            data.addChangeListener(this);
        }

        // send a dataset change event to self...
        DatasetChangeEvent event = new DatasetChangeEvent(this, data);
        datasetChanged(event);

    }

    /**
     * Returns the secondary dataset for the plot.
     *
     * @return the secondary dataset.
     */
    public Dataset getSecondaryDataset() {
        return this.secondaryDataset;
    }

    /** 
     * Sets the secondary dataset.
     * 
     * @param dataset  the dataset.
     */
    public void setSecondaryDataset(Dataset dataset) {

        // if there is an existing dataset, remove the plot from the list of change listeners...
        Dataset existing = this.secondaryDataset;
        if (existing != null) {
            existing.removeChangeListener(this);
        }

        // set the new dataset, and register the chart as a change listener...
        this.secondaryDataset = dataset;
        if (dataset != null) {
            this.datasetGroup = dataset.getGroup();
            dataset.addChangeListener(this);
        }

        // send a dataset change event to self...
        DatasetChangeEvent event = new DatasetChangeEvent(this, dataset);
        datasetChanged(event);

    }

    /**
     * Returns the string that is displayed when the dataset is empty or <code>null</code>.
     *
     * @return The 'no data' message (<code>null</code> possible).
     */
    public String getNoDataMessage() {
        return this.noDataMessage;
    }

    /**
     * Sets the message that is displayed when the dataset is empty or null.
     *
     * @param message  the message (null permitted).
     */
    public void setNoDataMessage(String message) {
        this.noDataMessage = message;
    }

    /**
     * Returns the font used to display the 'no data' message.
     *
     * @return the font.
     */
    public Font getNoDataMessageFont() {
        return this.noDataMessageFont;
    }

    /**
     * Sets the font used to display the 'no data' message.
     *
     * @param font  the font.
     */
    public void setNoDataMessageFont(Font font) {
        this.noDataMessageFont = font;
    }

    /**
     * Returns the paint used to display the 'no data' message.
     *
     * @return the paint.
     */
    public Paint getNoDataMessagePaint() {
        return this.noDataMessagePaint;
    }

    /**
     * Sets the paint used to display the 'no data' message.
     *
     * @param paint  the paint.
     */
    public void setNoDataMessagePaint(Paint paint) {
        this.noDataMessagePaint = paint;
    }

    /**
     * Returns a short string describing the plot type.
     * <P>
     * Note: this gets used in the chart property editing user interface,
     * but there needs to be a better mechanism for identifying the plot type.
     *
     * @return a short string describing the plot type.
     */
    public abstract String getPlotType();

    /**
     * Returns the parent plot (or null if this plot is not part of a combined plot).
     *
     * @return the parent plot.
     */
    public Plot getParent() {
        return this.parent;
    }

    /**
     * Sets the parent plot.
     *
     * @param parent  the parent plot.
     */
    public void setParent(Plot parent) {
        this.parent = parent;
    }

    /**
     * Returns the root plot.
     * 
     * @return the root plot.
     */
    public Plot getRootPlot() {

        Plot parent = getParent();
        if (parent == null) {
            return this;
        }
        else {
            return parent.getRootPlot();
        }

    }

    /**
     * Returns true if this plot is part of a combined plot structure.
     *
     * @return <code>true</code> if this plot is part of a combined plot structure.
     */
    public boolean isSubplot() {
        return (getParent() != null);
    }


    /**
     * Returns the insets for the plot area.
     *
     * @return The insets.
     */
    public Insets getInsets() {
        return this.insets;
    }

    /**
     * Sets the insets for the plot and notifies registered listeners that the
     * plot has been modified.
     *
     * @param insets  the new insets.
     */
    public void setInsets(Insets insets) {
        this.setInsets(insets, true);
    }

    /**
     * Sets the insets for the plot and, if requested, notifies registered listeners that the
     * plot has been modified.
     *
     * @param insets  the new insets.
     * @param notify  a flag that controls whether the registered listeners are notified.
     */
    public void setInsets(Insets insets, boolean notify) {

        if (!this.insets.equals(insets)) {
            this.insets = insets;
            if (notify) {
                notifyListeners(new PlotChangeEvent(this));
            }
        }

    }

    /**
     * Returns the background color of the plot area.
     *
     * @return The paint (possibly <code>null</code>).
     */
    public Paint getBackgroundPaint() {
        return this.backgroundPaint;
    }

    /**
     * Sets the background color of the plot area.  A {@link PlotChangeEvent} is forwarded to
     * all registered listeners.
     *
     * @param paint  the paint (<code>null</code> permitted).
     */
    public void setBackgroundPaint(Paint paint) {

        if (paint == null) {
            if (this.backgroundPaint != null) {
                this.backgroundPaint = null;
                notifyListeners(new PlotChangeEvent(this));
            }
        }
        else {
            if (this.backgroundPaint != null) {
                if (this.backgroundPaint.equals(paint)) {
                    return;  // nothing to do
                }
            }
            this.backgroundPaint = paint;
            notifyListeners(new PlotChangeEvent(this));
        }

    }

    /**
     * Returns the alpha transparency of the plot area background.
     *
     * @return the alpha transparency.
     */
    public float getBackgroundAlpha() {
        return this.backgroundAlpha;
    }

    /**
     * Sets the alpha transparency of the plot area background, and notifies
     * registered listeners that the plot has been modified.
     *
     * @param alpha the new alpha value.
     */
    public void setBackgroundAlpha(float alpha) {

        if (this.backgroundAlpha != alpha) {
            this.backgroundAlpha = alpha;
            notifyListeners(new PlotChangeEvent(this));
        }

    }

    /**
     * Returns the background image that is used to fill the plot's background area.
     *
     * @return The image (possibly <code>null</code>).
     */
    public Image getBackgroundImage() {
        return this.backgroundImage;
    }

    /**
     * Sets the background image for the plot.
     *
     * @param image  the image (<code>null</code> permitted).
     */
    public void setBackgroundImage(Image image) {
        this.backgroundImage = image;
        notifyListeners(new PlotChangeEvent(this));
    }

    /**
     * Returns the background image alignment. Alignment constants are defined in the 
     * <code>com.jrefinery.ui.Align</code> class in the JCommon class library.
     * 
     * @return The alignment.
     */
    public int getBackgroundImageAlignment() {
        return this.backgroundImageAlignment;
    }
    
    /**
     * Sets the background alignment.
     * <p>
     * Alignment options are defined by the {@link org.jfree.ui.Align} class.
     * 
     * @param alignment  the alignment.
     */
    public void setBackgroundImageAlignment(int alignment) {
        if (this.backgroundImageAlignment != alignment) {
            this.backgroundImageAlignment = alignment;
            notifyListeners(new PlotChangeEvent(this));
        }
    }

    /**
     * Returns the pen/brush used to outline the plot area.
     *
     * @return the outline stroke (possibly null).
     */
    public Stroke getOutlineStroke() {
        return this.outlineStroke;
    }

    /**
     * Sets the pen/brush used to outline the plot area, and notifies
     * registered listeners that the plot has been modified.
     *
     * @param stroke  the new outline pen/brush (null permitted).
     */
    public void setOutlineStroke(Stroke stroke) {

        if (stroke == null) {
            if (this.outlineStroke != null) {
                this.outlineStroke = null;
                notifyListeners(new PlotChangeEvent(this));
            }
        }
        else {
            if (this.outlineStroke != null) {
                if (this.outlineStroke.equals(stroke)) {
                    return;  // nothing to do
                }
            }
            this.outlineStroke = stroke;
            notifyListeners(new PlotChangeEvent(this));
        }

    }

    /**
     * Returns the color used to draw the outline of the plot area.
     *
     * @return The color (possibly <code>null<code>).
     */
    public Paint getOutlinePaint() {
        return this.outlinePaint;
    }

    /**
     * Sets the color of the outline of the plot area, and notifies registered
     * listeners that the Plot has been modified.
     *
     * @param paint  the new outline paint (null permitted).
     */
    public void setOutlinePaint(Paint paint) {

        if (paint == null) {
            if (this.outlinePaint != null) {
                this.outlinePaint = null;
                notifyListeners(new PlotChangeEvent(this));
            }
        }
        else {
            if (this.outlinePaint != null) {
                if (this.outlinePaint.equals(paint)) {
                    return;  // nothing to do
                }
            }
            this.outlinePaint = paint;
            notifyListeners(new PlotChangeEvent(this));
        }

    }

    /**
     * Returns the alpha-transparency for the plot foreground.
     *
     * @return the alpha-transparency.
     */
    public float getForegroundAlpha() {
        return this.foregroundAlpha;
    }

    /**
     * Sets the alpha-transparency for the plot.
     *
     * @param alpha  the new alpha transparency.
     */
    public void setForegroundAlpha(float alpha) {

        if (this.foregroundAlpha != alpha) {
            this.foregroundAlpha = alpha;
            notifyListeners(new PlotChangeEvent(this));
        }

    }

    /**
     * Returns the legend items for the plot.
     * <P>
     * By default, this method returns <code>null</code>.  Subclasses should override to return a 
     * {@link LegendItemCollection}.
     *
     * @return the legend items for the plot.
     */
    public LegendItemCollection getLegendItems() {
        return null;
    }
    
    /**
     * Registers an object for notification of changes to the plot.
     *
     * @param listener  the object to be registered.
     */
    public void addChangeListener(PlotChangeListener listener) {
        listenerList.add(PlotChangeListener.class, listener);
    }

    /**
     * Unregisters an object for notification of changes to the plot.
     *
     * @param listener  the object to be unregistered.
     */
    public void removeChangeListener(PlotChangeListener listener) {
        listenerList.remove(PlotChangeListener.class, listener);
    }

    /**
     * Notifies all registered listeners that the plot has been modified.
     *
     * @param event  information about the change event.
     */
    public void notifyListeners(PlotChangeEvent event) {

        Object[] listeners = this.listenerList.getListenerList();
        for (int i = listeners.length - 2; i >= 0; i -= 2) {
            if (listeners[i] == PlotChangeListener.class) {
                ((PlotChangeListener) listeners[i + 1]).plotChanged(event);
            }
        }

    }

    /**
     * Draws the plot on a Java 2D graphics device (such as the screen or a printer).
     * <P>
     * This class does not store any information about where the individual
     * items that make up the plot are actually drawn.  If you want to collect
     * this information, pass in a ChartRenderingInfo object.  After the
     * drawing is complete, the info object will contain lots of information
     * about the chart.  If you don't want the information, pass in null.
     * *
     * @param g2  the graphics device.
     * @param plotArea  the area within which the plot should be drawn.
     * @param info  an object for collecting information about the drawing of the chart.
     */
    public abstract void draw(Graphics2D g2, Rectangle2D plotArea, ChartRenderingInfo info);

    /**
     * Draw the plot background.
     *
     * @param g2  the graphics device.
     * @param area  the area within which the plot should be drawn.
     */
    public void drawBackground(Graphics2D g2, Rectangle2D area) {

        if (backgroundPaint != null) {
            Composite originalComposite = g2.getComposite();
            g2.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_OVER,
                                                       this.backgroundAlpha));
            g2.setPaint(backgroundPaint);
            g2.fill(area);
            g2.setComposite(originalComposite);
        }

        if (backgroundImage != null) {
            Composite originalComposite = g2.getComposite();
            g2.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC, this.backgroundAlpha));
            Rectangle2D dest = new Rectangle2D.Double(0.0, 0.0,
                                                      this.backgroundImage.getWidth(null),
                                                      this.backgroundImage.getHeight(null));
            Align.align(dest, area, this.backgroundImageAlignment);
            g2.drawImage(this.backgroundImage,
                         (int) dest.getX(), (int) dest.getY(),
                         (int) dest.getWidth() + 1, (int) dest.getHeight() + 1, null);
            g2.setComposite(originalComposite);
        }

    }

    /**
     * Draw the plot outline
     *
     * @param g2  the graphics device.
     * @param area  the area within which the plot should be drawn.
     */
    public void drawOutline(Graphics2D g2, Rectangle2D area) {

        if ((outlineStroke != null) && (outlinePaint != null)) {
            g2.setStroke(outlineStroke);
            g2.setPaint(outlinePaint);
            g2.draw(area);
        }

    }

    /**
     * Draws a message to state that there is no data to plot.
     *
     * @param g2  the graphics device.
     * @param area  the area within which the plot should be drawn.
     */
    protected void drawNoDataMessage(Graphics2D g2, Rectangle2D area) {

        Shape savedClip = g2.getClip();
        g2.clip(area);
        String message = this.noDataMessage;
        if (message != null) {
            g2.setFont(this.noDataMessageFont);
            g2.setPaint(this.noDataMessagePaint);
            FontRenderContext frc = g2.getFontRenderContext();
            Rectangle2D bounds = noDataMessageFont.getStringBounds(message, frc);
            float x = (float) (area.getX() + area.getWidth() / 2 - bounds.getWidth() / 2);
            float y = (float) (area.getMinY() + (area.getHeight() / 2) - (bounds.getHeight() / 2));
            g2.drawString(message, x, y);
        }
        g2.clip(savedClip);

    }

    /**
     * Handles a 'click' on the plot.  Since the plot does not maintain any
     * information about where it has been drawn, the plot area is supplied as
     * an argument.
     *
     * @param x  the x coordinate.
     * @param y  the y coordinate.
     * @param info  an object for collecting information about the drawing of the chart.
     */
    public void handleClick(int x, int y, ChartRenderingInfo info) {

    }

    /**
     * Performs a zoom on the plot.  Subclasses should override if zooming is appropriate for
     * the type of plot.
     *
     * @param percent  the zoom percentage.
     */
    public void zoom(double percent) {
        // do nothing by default.
    }

    /**
     * Receives notification of a change to one of the plot's axes.
     *
     * @param event  information about the event (not used here).
     */
    public void axisChanged(AxisChangeEvent event) {
        notifyListeners(new PlotChangeEvent(this));
    }

    /**
     * Receives notification of a change to the plot's dataset.
     * <P>
     * The plot reacts by passing on a plot change event to all registered listeners.
     *
     * @param event  information about the event (not used here).
     */
    public void datasetChanged(DatasetChangeEvent event) {

        PlotChangeEvent newEvent = new PlotChangeEvent(this);
        notifyListeners(newEvent);

    }

    /**
     * Adjusts the supplied x-value.
     * 
     * @param x  the x-value.
     * @param w1  width 1.
     * @param w2  width 2.
     * @param location  the location (left or right).
     * 
     * @return the adjusted x-value.
     */
    protected double getRectX(double x, double w1, double w2, int location) {

        double result = x;
        if (location == LEFT) {
            result = result + w1;
        }
        else if (location == RIGHT) {
            result = result + w2;
        }
        return result;

    }

    /**
     * Adjusts the supplied y-value.
     * 
     * @param y  the x-value.
     * @param h1  height 1.
     * @param h2  height 2.
     * @param location  the location (top or bottom).
     * 
     * @return the adjusted y-value.
     */
    protected double getRectY(double y, double h1, double h2, int location) {

        double result = y;
        if (location == TOP) {
            result = result + h1;
        }
        else if (location == BOTTOM) {
            result = result + h2;
        }
        return result;

    }

    /**
     * Returns the location code for the opposite location to the one specified.
     * 
     * @param location  the location.
     * 
     * @return The opposite location.
     */
    protected int getOppositeAxisLocation(int location) {

        int result;
        switch (location) {
            case TOP:     result = BOTTOM;
                          break;
            case BOTTOM:  result = TOP;
                          break;
            case LEFT:    result = RIGHT;
                          break;
            case RIGHT:
            default:      result = LEFT;
        }
        return result;

    }

//    /**
//     * Assigns a unique index to this plot and each of its subplots.  The index is mainly used
//     * for default color selection, to make it possible for each subplot to use a unique selection
//     * of colors.
//     *
//     * @param index1  the index for regular plots.
//     * @param index2  the index for container plots.
//     * 
//     * @return The indices.
//     */
//    protected int[] indexSubplots(int index1, int index2) {
//
//        this.index = index1;
//        index1++;
//        return new int[] { index1, index2 };
//
//    }

    /**
     * Returns the data area ratio.
     * 
     * @return The ratio.
     */
    public double getDataAreaRatio() {
        return dataAreaRatio;
    }

    /**
     * Sets the data area ratio.
     * 
     * @param ratio  the ratio.
     */
    public void setDataAreaRatio(double ratio) {
        this.dataAreaRatio = ratio;
    }

    /**
     * Tests this plot for equality with another object.
     * 
     * @param obj  the object.
     * 
     * @return <code>true</code> or <code>false</code>.
     */
    public boolean equals(Object obj) {
    
        if (obj == null) {
            return false;
        }
        
        if (obj == this) {
            return true;
        }
    
        if (obj instanceof Plot) {
        
            Plot p = (Plot) obj;
            
            //boolean b0 = (this.index == p.index);
            boolean b1 = true;
            // ObjectUtils.equalOrBothNull(this.parent, p.parent);
            boolean b2 = true;  // todo: need to work on equality test for this 
            //ObjectUtils.equalOrBothNull(this.datasetGroup, p.datasetGroup);
            boolean b3 = ObjectUtils.equalOrBothNull(this.dataset, p.dataset);
            boolean b4 = ObjectUtils.equalOrBothNull(this.secondaryDataset, p.secondaryDataset);

            boolean b5 = ObjectUtils.equalOrBothNull(this.noDataMessage, p.noDataMessage);
            boolean b6 = ObjectUtils.equalOrBothNull(this.noDataMessageFont, p.noDataMessageFont);
            boolean b7 = ObjectUtils.equalOrBothNull(this.noDataMessagePaint, p.noDataMessagePaint);

            boolean b8 = ObjectUtils.equalOrBothNull(this.insets, p.insets);
            boolean b9 = ObjectUtils.equalOrBothNull(this.outlineStroke, p.outlineStroke);
            boolean b10 = ObjectUtils.equalOrBothNull(this.outlinePaint, p.outlinePaint);

            boolean b11 = ObjectUtils.equalOrBothNull(this.backgroundPaint, p.backgroundPaint);
            boolean b12 = ObjectUtils.equalOrBothNull(this.backgroundImage, p.backgroundImage);
            boolean b13 = (this.backgroundImageAlignment == p.backgroundImageAlignment);

            boolean b14 = (this.foregroundAlpha == p.foregroundAlpha);
            boolean b15 = (this.backgroundAlpha == p.backgroundAlpha);
            
            return b1 && b2 && b3 && b4 && b5 && b6 && b7 && b8 && b9 
                   && b10 && b11 && b12 && b13 && b14 && b15;
                   
        }
        
        return false;
            
    }
    
    /**
     * Provides serialization support.
     * 
     * @param stream  the output stream.
     * 
     * @throws IOException  if there is an I/O error.
     */
    private void writeObject(ObjectOutputStream stream) throws IOException {
        stream.defaultWriteObject();
        SerialUtilities.writePaint(this.noDataMessagePaint, stream);
        SerialUtilities.writeStroke(this.outlineStroke, stream);
        SerialUtilities.writePaint(this.outlinePaint, stream);
        // backgroundImage
        SerialUtilities.writePaint(this.backgroundPaint, stream);    
    }
    
    /**
     * Provides serialization support.
     * 
     * @param stream  the input stream.
     * 
     * @throws IOException  if there is an I/O error.
     * @throws ClassNotFoundException  if there is a classpath problem. 
     */
    private void readObject(ObjectInputStream stream) throws IOException, ClassNotFoundException {
        stream.defaultReadObject();
        this.noDataMessagePaint = SerialUtilities.readPaint(stream);
        this.outlineStroke = SerialUtilities.readStroke(stream);
        this.outlinePaint = SerialUtilities.readPaint(stream);
        // backgroundImage
        this.backgroundPaint = SerialUtilities.readPaint(stream);
        
        this.listenerList = new EventListenerList();
        
        if (this.dataset != null) {
            this.dataset.addChangeListener(this);
        }
        
        if (this.secondaryDataset != null) {
            this.secondaryDataset.addChangeListener(this);
        }
        
    }
    
}

