/* ================================================================
 * JCommon : a general purpose, open source, class library for Java
 * ================================================================
 *
 * Project Info:  http://www.object-refinery.com/jcommon/index.html
 * Project Lead:  David Gilbert (david.gilbert@object-refinery.com);
 *
 * (C) Copyright 2000-2002, by Simba Management Limited and Contributors.
 *
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation;
 * either version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * --------------------------
 * RelativeDayOfWeekRule.java
 * --------------------------
 * (C) Copyright 2000-2002, by Simba Management Limited and Contributors.
 *
 * Original Author:  David Gilbert (for Simba Management Limited);
 * Contributor(s):   -;
 *
 * $Id: RelativeDayOfWeekRule.java,v 1.1 2002/04/02 17:03:44 mungady Exp $
 *
 * Changes (from 26-Oct-2001)
 * --------------------------
 * 26-Oct-2001 : Changed package to com.jrefinery.date.*;
 *
 */

package com.jrefinery.date;

/**
 * An annual date rule that returns a date for each year based on (a) a reference rule; (b) a
 * day of the week; and (c) a selection parameter (SerialDate.PRECEDING, SerialDate.NEAREST,
 * SerialDate.FOLLOWING).
 * <P>
 * For example, Good Friday can be specified as 'the Friday PRECEDING Easter Sunday'.
 */
public class RelativeDayOfWeekRule extends AnnualDateRule implements Cloneable {

    /** A reference to the annual date rule on which this rule is based. */
    protected AnnualDateRule subrule;

    /** The day of the week (SerialDate.MONDAY, SerialDate.TUESDAY, and so on). */
    protected int dayOfWeek;

    /** Specifies which day of the week (PRECEDING, NEAREST or FOLLOWING). */
    protected int relative;

    /**
     * Default constructor - builds a rule for the Monday following 1 January.
     */
    public RelativeDayOfWeekRule() {
        this(new DayAndMonthRule(), SerialDate.MONDAY, SerialDate.FOLLOWING);
    }

    /**
     * Standard constructor - builds rule based on the supplied sub-rule.
     * @param subrule The rule that determines the reference date;
     * @param dayOfWeek The day-of-the-week relative to the reference date;
     * @param relative Indicates *which* day-of-the-week (preceding, nearest or following);
     */
    public RelativeDayOfWeekRule(AnnualDateRule subrule, int dayOfWeek, int relative) {
        this.subrule = subrule;
        this.dayOfWeek = dayOfWeek;
        this.relative = relative;
    }

    /**
     * Returns the sub-rule (also called the reference rule).
     * @return The annual date rule that determines the reference date for this rule;
     */
    public AnnualDateRule getSubrule() {
        return subrule;
    }

    /**
     * Sets the sub-rule;
     * @param subrule The annual date rule that determines the reference date for this rule;
     */
    public void setSubrule(AnnualDateRule subrule) {
        this.subrule = subrule;
    }

    /**
     * Returns the day-of-the-week for this rule.
     * @return The day-of-the-week for this rule;
     */
    public int getDayOfWeek() {
        return dayOfWeek;
    }

    /**
     * Sets the day-of-the-week for this rule.
     * @param dayOfWeek The day-of-the-week (SerialDate.MONDAY, SerialDate.TUESDAY, and so on);
     */
    public void setDayOfWeek(int dayOfWeek) {
        this.dayOfWeek = dayOfWeek;
    }

    /**
     * Returns the 'relative' attribute, that determines *which* day-of-the-week we are interested
     * in.
     * @return The 'relative' attribute (SerialDate.PRECEDING, SerialDate.NEAREST or
     *         SerialDate.FOLLOWING).
     */
    public int getRelative() {
        return relative;
    }

    /**
     * Sets the 'relative' attribute.
     * @param relative Determines *which* day-of-the-week is selected by this rule
     *                 (SerialDate.PRECEDING, SerialDate.NEAREST, SerialDate.FOLLOWING);
     */
    public void setRelative(int relative) {
        this.relative = relative;
    }

    /**
     * Creates a clone of this rule.
     */
    public Object clone() throws CloneNotSupportedException {
        RelativeDayOfWeekRule duplicate = null;
        duplicate = (RelativeDayOfWeekRule)super.clone();
        duplicate.subrule = (AnnualDateRule)duplicate.getSubrule().clone();
        return duplicate;
    }

    /**
     * Returns the date generated by this rule, for the specified year.
     * @param The year (1900 <= year <= 9999).
     * @return The date generated by the rule for the given year (null possible).
     */
    public SerialDate getDate(int year) {

        // check argument...
        if ((year<SerialDate.MINIMUM_YEAR_SUPPORTED) || (year>SerialDate.MAXIMUM_YEAR_SUPPORTED)) {
            throw new IllegalArgumentException("RelativeDayOfWeekRule.getDate(...): "
                                               +"year outside valid range.");
        }

        // calculate the date...
        SerialDate result = null;
        SerialDate base = subrule.getDate(year);

        if (base!=null) {

            switch (relative) {

                case(SerialDate.PRECEDING):
                    result = SerialDate.getPreviousDayOfWeek(dayOfWeek, base);
                    break;

                case(SerialDate.NEAREST):
                    result = SerialDate.getNearestDayOfWeek(dayOfWeek, base);
                    break;

                case(SerialDate.FOLLOWING):
                    result = SerialDate.getFollowingDayOfWeek(dayOfWeek, base);
                    break;

                default: ;

            }

        }

        return result;

    }

}