/*
 * Copyright 2000 Murray Cumming
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef BAKERY_DOCUMENT_H
#define BAKERY_DOCUMENT_H

#include "bakery/View/View.h"
#include <glibmm.h>

#include <iostream>

namespace Bakery
{

/** The Document is like the 'Model' in the Model-View-Controller framework.
 * Each App should have a Document.
 * Each View gets and sets data in its document.
 */
class Document
{
public: 
  Document();
  virtual ~Document();

  /* Saves the data to disk.
   * Asks the View to update this document before saving to disk,
   * but you should probably ensure that the document is updated more regularly than this,
   * so that different parts of the GUI are synchronized.
   * Only saves if the document has been modified.
   * bool indicates success.
   */
  virtual bool save();

  /* Loads data from disk, then asks the View to update itself.
   * bool indicates success.
   */
  virtual bool load();

  virtual bool get_modified() const;
  virtual void set_modified(bool bVal = true);

  ///Whether this just a default document.
  virtual bool get_is_new() const;
  ///Called by App_WithDoc::init_create_document().
  void set_is_new(bool bVal);

  virtual Glib::ustring get_contents() const;
  virtual void set_contents(const Glib::ustring& strVal);

  virtual Glib::ustring get_filepath() const;
  virtual void set_filepath(const Glib::ustring& strVal, bool bEnforceFileExtension = false);

  ///Gets filename part of filepath, or 'untitled'.
  virtual Glib::ustring get_name() const;
  static Glib::ustring util_filepath_get_name(const Glib::ustring& filePath);

  virtual bool get_read_only() const;
  virtual void set_read_only(bool bVal);
  
  ///If you don't want to use a View, then don't use set_view().
  virtual void set_view(ViewBase* pView);
  virtual ViewBase* get_view();

  virtual void set_file_extension(const Glib::ustring& strVal);
  virtual Glib::ustring get_file_extension() const;

  //Signals
  typedef SigC::Signal0<void> type_signal_modified;
  type_signal_modified& signal_modified();

  ///Allow app to update icons/title bar.

protected:
  /** overrideable.
   * Does anything which should be done after the data has been loaded from disk, but before updating the View.
   */
  virtual bool load_after();

  /** overrideable.
   * Does anything which should be done before the view has saved its data, before writing to disk..
   */
  virtual bool save_before();

  virtual bool read_from_disk();
  virtual bool write_to_disk();

  Glib::ustring m_strContents;
  Glib::ustring m_strFilePath;
  Glib::ustring m_strFileExtension;

  ViewBase* m_pView;

  type_signal_modified signal_modified_;

  bool m_bModified;
  bool m_bIsNew; //see get_is_new().
  bool m_bReadOnly;
};

} //namespace

#endif //GNOME_APPWITHDOCS_DOCUMENT_H
