
/*
 *  Diverse Bristol audio routines.
 *  Copyright (c) by Nick Copeland <nick.copeland@ntlworld.com> 1996,2002
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


#include <fcntl.h>

#include "brighton.h"
#include "brightonMini.h"

static int mixInit();
static int mixConfigure();
static int mixCallback(void *, int, int, float);

extern guimain global;

#include "brightonKeys.h"

#define CHAN_COUNT 16
#define BUS_COUNT 60
#define MASTER_COUNT 17
#define FUNCTION_COUNT 37
#define FIRST_DEV 0
#define PARAM_COUNT 21
#define ACTIVE_DEVS (PARAM_COUNT * CHAN_COUNT + BUS_COUNT + FUNCTION_COUNT + FIRST_DEV)

#define DISPLAY_DEV 0
#define DISPLAY_PANEL 7

#define CHAN_PANEL 0
#define BUS_PANEL (CHAN_COUNT)
#define MASTER_PANEL (BUS_PANEL + 1)
#define FUNCTION_PANEL (MASTER_PANEL + 1)

#define DEVICE_COUNT (ACTIVE_DEVS)

/*
 * This structure is for device definition. The structure is defined in 
 * include/brighton.h, further definitions in brighton/brightonDevtable.h and
 * include/brightoninternals.h
 *
 *	typedef int (*brightonCallback)(int, float);
 *	typedef struct BrightonLocations {
 *		int device; 0=rotary, 1=scale, etc.
 *		float relx, rely; relative position with regards to 1000 by 1000 window
 *		float relw, relh; relative height.
 *		int from, to;
 *		brightonCallback callback; specific to this dev
 *		char *image; bitmap. If zero take a device default.
 *		int flags;
 *	} brightonLocations;
 *
 * This example is for a mixBristol type synth interface.
 */
#define CD1 50
#define R1 50
#define R2 (R1 + CD1)
#define R3 (R2 + CD1)
#define R4 (R3 + CD1)
#define R5 (R4 + CD1)
#define R6 (R5 + CD1)
#define R7 (R6 + CD1)
#define R8 (R7 + CD1)
#define R9 (R8 + CD1)
#define R10 (R9 + CD1)
#define R11 (R10 + CD1)
#define R12 (R11 + CD1)
#define R13 (R12 + CD1)
#define R14 (R13 + CD1)
#define R15 (R14 + CD1)

#define D1 180

#define C1 200
#define C2 100
#define C3 600
#define C4 350

#define C5 75
#define C6 400
#define C7 725

#define C8 30
#define C9 275
#define C10 525
#define C11 760

#define S1 550
#define S2 80
#define SB3 225
#define SB4 22

static brightonLocations locations[PARAM_COUNT] = {
	{"", 2, C8, 10, SB3, SB4, 0, 1.01, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 2, C9, 10, SB3, SB4, 0, 1.01, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 2, C10, 10, SB3, SB4, 0, 1.01, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 2, C11, 10, SB3, SB4, 0, 1.01, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 0, C1, R1, S1, S2, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 0, C1, R2, S1, S2, 0, 1, 0, 0, 0, 0},
	{"", 0, C1, R3, S1, S2, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 0, C1, R4, S1, S2, 0, 1, 0, 0, 0, 0},
	{"", 0, C1, R5, S1, S2, 0, 1, 0, 0, 0, 0},
	{"", 0, C1, R6, S1, S2, 0, 1, 0, 0, 0, 0},
	{"", 0, C1, R7, S1, S2, 0, 1, 0, 0, 0, 0},
	{"", 0, C1, R8, S1, S2, 0, 1, 0, 0, 0, 0},
	{"", 0, C1, R9, S1, S2, 0, 1, 0, 0, 0, 0},
	{"", 0, C1, R10, S1, S2, 0, 1, 0, 0, 0, 0},
	{"", 0, C1, R11, S1, S2, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 0, C1, R12, S1, S2, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 2, C5, R13, SB3, SB4, 0, 1.01, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 2, C6, R13, SB3, SB4, 0, 1.01, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 2, C7, R13, SB3, SB4, 0, 1.01, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 1, C4, R14, 250, 250, 0, 1, 0,
		"bitmaps/knobs/sliderblack.xpm", 0, 0},
	{"", 3, 0, 970, 1000, 13, 0, 1, 0, "bitmaps/digits/display2.xpm", 0, 0}
};

#define BS1 90
#define BS2 80

#define BRD1 100
#define BRD2 66

#define BR1 11
#define BR2 (BR1 + BRD2)
#define BR3 (BR2 + BRD2)
#define BR4 (BR3 + BRD2)
#define BR5 (BR4 + BRD2)
#define BR6 (BR5 + BRD2)
#define BR7 (BR6 + BRD2)
#define BR8 (BR7 + BRD2)
#define BR9 (BR8 + BRD2)
#define BR10 (BR9 + BRD2 - 1)

#define BCD1 100
#define BCD2 125

#define BC1 35
#define BC2 200
#define BC3 350
#define BC4 550
#define BC5 700
#define BC6 850

#define BC7 (BC1 + BCD2)
#define BC8 (BC7 + BCD2)
#define BC9 (BC8 + BCD2)
#define BC10 (BC9 + BCD2)
#define BC11 (BC10 + BCD2)
#define BC12 (BC11 + BCD2)
#define BC13 (BC12 + BCD2)

static brightonLocations busses[BUS_COUNT] = {
	// Bus groups
	{"", 0, BC1, BR1, BS1, BS2, 0, 1, 0, 0, 0, 0},
	{"", 0, BC2, BR1, BS1, BS2, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 0, BC3, BR1, BS1, BS2, 0, 1, 0, 0, 0, 0},
	{"", 0, BC1, BR2, BS1, BS2, 0, 1, 0, 0, 0, 0},
	{"", 0, BC2, BR2, BS1, BS2, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 0, BC3, BR2, BS1, BS2, 0, 1, 0, 0, 0, 0},
	{"", 0, BC1, BR3, BS1, BS2, 0, 1, 0, 0, 0, 0},
	{"", 0, BC2, BR3, BS1, BS2, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 0, BC3, BR3, BS1, BS2, 0, 1, 0, 0, 0, 0},
	{"", 0, BC1, BR4, BS1, BS2, 0, 1, 0, 0, 0, 0},
	{"", 0, BC2, BR4, BS1, BS2, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 0, BC3, BR4, BS1, BS2, 0, 1, 0, 0, 0, 0},
	{"", 0, BC1, BR5, BS1, BS2, 0, 1, 0, 0, 0, 0},
	{"", 0, BC2, BR5, BS1, BS2, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 0, BC3, BR5, BS1, BS2, 0, 1, 0, 0, 0, 0},
	{"", 0, BC1, BR6, BS1, BS2, 0, 1, 0, 0, 0, 0},
	{"", 0, BC2, BR6, BS1, BS2, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 0, BC3, BR6, BS1, BS2, 0, 1, 0, 0, 0, 0},
	{"", 0, BC1, BR7, BS1, BS2, 0, 1, 0, 0, 0, 0},
	{"", 0, BC2, BR7, BS1, BS2, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 0, BC3, BR7, BS1, BS2, 0, 1, 0, 0, 0, 0},
	{"", 0, BC1, BR8, BS1, BS2, 0, 1, 0, 0, 0, 0},
	{"", 0, BC2, BR8, BS1, BS2, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 0, BC3, BR8, BS1, BS2, 0, 1, 0, 0, 0, 0},
	{"", 0, BC4, BR1, BS1, BS2, 0, 1, 0, 0, 0, 0},
	{"", 0, BC5, BR1, BS1, BS2, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 0, BC6, BR1, BS1, BS2, 0, 1, 0, 0, 0, 0},
	{"", 0, BC4, BR2, BS1, BS2, 0, 1, 0, 0, 0, 0},
	{"", 0, BC5, BR2, BS1, BS2, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 0, BC6, BR2, BS1, BS2, 0, 1, 0, 0, 0, 0},
	{"", 0, BC4, BR3, BS1, BS2, 0, 1, 0, 0, 0, 0},
	{"", 0, BC5, BR3, BS1, BS2, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 0, BC6, BR3, BS1, BS2, 0, 1, 0, 0, 0, 0},
	{"", 0, BC4, BR4, BS1, BS2, 0, 1, 0, 0, 0, 0},
	{"", 0, BC5, BR4, BS1, BS2, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 0, BC6, BR4, BS1, BS2, 0, 1, 0, 0, 0, 0},
	{"", 0, BC4, BR5, BS1, BS2, 0, 1, 0, 0, 0, 0},
	{"", 0, BC5, BR5, BS1, BS2, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 0, BC6, BR5, BS1, BS2, 0, 1, 0, 0, 0, 0},
	{"", 0, BC4, BR6, BS1, BS2, 0, 1, 0, 0, 0, 0},
	{"", 0, BC5, BR6, BS1, BS2, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 0, BC6, BR6, BS1, BS2, 0, 1, 0, 0, 0, 0},
	{"", 0, BC4, BR7, BS1, BS2, 0, 1, 0, 0, 0, 0},
	{"", 0, BC5, BR7, BS1, BS2, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 0, BC6, BR7, BS1, BS2, 0, 1, 0, 0, 0, 0},
	{"", 0, BC4, BR8, BS1, BS2, 0, 1, 0, 0, 0, 0},
	{"", 0, BC5, BR8, BS1, BS2, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 0, BC6, BR8, BS1, BS2, 0, 1, 0, 0, 0, 0},
	{"", 0, BC1 + 40, BR9, BS1, BS2, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 0, BC8 + 40, BR9, BS1, BS2, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 0, BC10 + 40, BR9, BS1, BS2, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
	{"", 0, BC12 + 40, BR9, BS1, BS2, 0, 1, 0, "bitmaps/knobs/knob2.xpm", 0, 0},
#define BSL 330
	{"", 1, BC1, BR10, 40, BSL, 0, 1, 0,
		"bitmaps/knobs/sliderblack.xpm", 0, 0},
	{"", 1, BC7, BR10, 40, BSL, 0, 1, 0,
		"bitmaps/knobs/sliderblack.xpm", 0, 0},
	{"", 1, BC8, BR10, 40, BSL, 0, 1, 0,
		"bitmaps/knobs/sliderblack.xpm", 0, 0},
	{"", 1, BC9, BR10, 40, BSL, 0, 1, 0,
		"bitmaps/knobs/sliderblack.xpm", 0, 0},
	{"", 1, BC10, BR10, 40, BSL, 0, 1, 0,
		"bitmaps/knobs/sliderblack.xpm", 0, 0},
	{"", 1, BC11, BR10, 40, BSL, 0, 1, 0,
		"bitmaps/knobs/sliderblack.xpm", 0, 0},
	{"", 1, BC12, BR10, 40, BSL, 0, 1, 0,
		"bitmaps/knobs/sliderblack.xpm", 0, 0},
	{"", 1, BC13, BR10, 40, BSL, 0, 1, 0,
		"bitmaps/knobs/sliderblack.xpm", 0, 0},
};

#define MC1 150
#define MC2 200
#define MC3 650

#define MS1 240

static brightonLocations master[MASTER_COUNT] = {
	{"", 0, MC2, BR8, 600, 600, 0, 1, 0, 0, 0, 0},
	{"", 1, MC1 + 50, BR10, 130, BSL, 0, 1, 0,
		"bitmaps/knobs/sliderblack.xpm", 0, 0},
	{"", 1, MC3, BR10, 130, BSL, 0, 1, 0,
		"bitmaps/knobs/sliderblack.xpm", 0, 0},
	{"", 0, MC1, BR1, MS1, MS1, 0, 1, 0, 0, 0, 0},
	{"", 0, MC1, BR2, MS1, MS1, 0, 1, 0, 0, 0, 0},
	{"", 0, MC1, BR3, MS1, MS1, 0, 1, 0, 0, 0, 0},
	{"", 0, MC1, BR4, MS1, MS1, 0, 1, 0, 0, 0, 0},
	{"", 0, MC1, BR5, MS1, MS1, 0, 1, 0, 0, 0, 0},
	{"", 0, MC1, BR6, MS1, MS1, 0, 1, 0, 0, 0, 0},
	{"", 0, MC1, BR7, MS1, MS1, 0, 1, 0, 0, 0, 0},
	{"", 0, MC3, BR1, MS1, MS1, 0, 1, 0, 0, 0, 0},
	{"", 0, MC3, BR2, MS1, MS1, 0, 1, 0, 0, 0, 0},
	{"", 0, MC3, BR3, MS1, MS1, 0, 1, 0, 0, 0, 0},
	{"", 0, MC3, BR4, MS1, MS1, 0, 1, 0, 0, 0, 0},
	{"", 0, MC3, BR5, MS1, MS1, 0, 1, 0, 0, 0, 0},
	{"", 0, MC3, BR6, MS1, MS1, 0, 1, 0, 0, 0, 0},
	{"", 0, MC3, BR7, MS1, MS1, 0, 1, 0, 0, 0, 0},
};

#define FD1 8
#define FD2 50
#define FD3 120

#define FPS1 83
#define FPS2 485

#define FR1 25

#define FR2 (FR1 + FPS1 * 6)
#define FR3 (FR2 + FPS1 - FD1)
#define FR4 (FR3 + FPS1 - FD1)
#define FR5 (FR4 + FPS1 - FD1)
#define FR6 (FR5 + FPS1 - FD1)
#define FR7 (FR6 + FPS1 - FD1)

#define FR8 (FR1 + 120)
#define FR9 (FR8 + 120)
#define FR10 (FR10 + 120)

#define FC1 15
#define FC2 500
#define FC3 (FC2 + FD2)
#define FC4 (FC3 + FD2)
#define FC5 (FC4 + FD2)
#define FC6 (FC5 + FD2)
#define FC7 (FC6 + FD2)
#define FC8 (FC7 + FD2)
#define FC9 (FC8 + FD2)
#define FC10 (FC9 + FD2)
#define FC11 (FC10 + FD2)

#define DISPLAY1 (FUNCTION_COUNT - 6)
#define DISPLAY2 (FUNCTION_COUNT - 5)
#define DISPLAY3 (FUNCTION_COUNT - 4)
#define DISPLAY4 (FUNCTION_COUNT - 3)
#define DISPLAY5 (FUNCTION_COUNT - 2)
#define DISPLAY6 (FUNCTION_COUNT - 1)

static brightonLocations functions[FUNCTION_COUNT] = {
	{"", 2, FC2, FR1, 40, 100, 0, 1.01, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 2, FC3, FR1, 40, 100, 0, 1.01, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 2, FC4, FR1, 40, 100, 0, 1.01, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 2, FC5, FR1, 40, 100, 0, 1.01, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 2, FC6, FR1, 40, 100, 0, 1.01, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 2, FC7, FR1, 40, 100, 0, 1.01, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 2, FC8, FR1, 40, 100, 0, 1.01, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 2, FC9, FR1, 40, 100, 0, 1.01, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 2, FC10, FR1, 40, 100, 0, 1.01, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 2, FC11, FR1, 40, 100, 0, 1.01, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 2, FC2, FR8, 40, 100, 0, 1.01, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 2, FC3, FR8, 40, 100, 0, 1.01, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 2, FC4, FR8, 40, 100, 0, 1.01, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 2, FC5, FR8, 40, 100, 0, 1.01, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 2, FC6, FR8, 40, 100, 0, 1.01, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 2, FC7, FR8, 40, 100, 0, 1.01, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 2, FC8, FR8, 40, 100, 0, 1.01, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 2, FC9, FR8, 40, 100, 0, 1.01, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 2, FC10, FR8, 40, 100, 0, 1.01, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 2, FC11, FR8, 40, 100, 0, 1.01, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 2, FC2, FR9, 40, 100, 0, 1.01, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 2, FC3, FR9, 40, 100, 0, 1.01, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 2, FC4, FR9, 40, 100, 0, 1.01, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 2, FC5, FR9, 40, 100, 0, 1.01, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 2, FC6, FR9, 40, 100, 0, 1.01, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 2, FC7, FR9, 40, 100, 0, 1.01, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 2, FC8, FR9, 40, 100, 0, 1.01, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 2, FC9, FR9, 40, 100, 0, 1.01, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 2, FC10, FR9, 40, 100, 0, 1.01, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 2, FC11, FR9, 40, 100, 0, 1.01, 0,
		"bitmaps/buttons/pressoff.xpm",
		"bitmaps/buttons/presson.xpm", 0},
	{"", 3, FC1, FR1, FPS2, FPS1 * 6 - 25, 0, 1, 0, "bitmaps/digits/display3.xpm", 0, 0},
	{"", 3, FC1, FR2, FPS2, FPS1, 0, 1, 0, 0, 0, 0},
	{"", 3, FC1, FR3, FPS2, FPS1, 0, 1, 0, 0, 0, 0},
	{"", 3, FC1, FR4, FPS2, FPS1, 0, 1, 0, 0, 0, 0},
	{"", 3, FC1, FR5, FPS2, FPS1, 0, 1, 0, 0, 0, 0},
	{"", 3, FC1, FR6, FPS2, FPS1, 0, 1, 0, 0, 0, 0},
	{"", 3, FC1, FR7, FPS2, FPS1, 0, 1, 0, 0, 0, 0},
};

#define PWB 20
#define PS1 4
#define PW1 38
#define PW2 (PW1 + PS1)

/*
 * This is a set of globals for the main window rendering. Again taken from
 * include/brighton.h
 */
brightonApp mixApp = {
	"mixer",
	0, // no blueprint on wood background.
	"bitmaps/textures/metal6.xpm",
	BRIGHTON_STRETCH,
	mixInit,
	mixConfigure, // 3 callbacks, unused?
	0,
	destroySynth,
	1000, 650,
	21, // panel count
	{
		{
			"Mix",
			"bitmaps/blueprints/m1chan.xpm",
			"bitmaps/textures/metal5.xpm",
			0, // flags
			0,
			0,
			mixCallback,
			PWB + PS1, PS1, PW1, 1000 - PS1 * 2,
			PARAM_COUNT,
			locations
		},
		{
			"Mix",
			"bitmaps/blueprints/m1chan.xpm",
			"bitmaps/textures/metal5.xpm",
			0, // flags
			0,
			0,
			mixCallback,
			PWB + PS1 + PW2 * 1, PS1, PW1, 1000 - PS1 * 2,
			PARAM_COUNT,
			locations
		},
		{
			"Mix",
			"bitmaps/blueprints/m1chan.xpm",
			"bitmaps/textures/metal5.xpm",
			0, // flags
			0,
			0,
			mixCallback,
			PWB + PS1 + PW2 * 2, PS1, PW1, 1000 - PS1 * 2,
			PARAM_COUNT,
			locations
		},
		{
			"Mix",
			"bitmaps/blueprints/m1chan.xpm",
			"bitmaps/textures/metal5.xpm",
			0, // flags
			0,
			0,
			mixCallback,
			PWB + PS1 + PW2 * 3, PS1, PW1, 1000 - PS1 * 2,
			PARAM_COUNT,
			locations
		},
		{
			"Mix",
			"bitmaps/blueprints/m1chan.xpm",
			"bitmaps/textures/metal5.xpm",
			0, // flags
			0,
			0,
			mixCallback,
			PWB + PS1 + PW2 * 4, PS1, PW1, 1000 - PS1 * 2,
			PARAM_COUNT,
			locations
		},
		{
			"Mix",
			"bitmaps/blueprints/m1chan.xpm",
			"bitmaps/textures/metal5.xpm",
			0, // flags
			0,
			0,
			mixCallback,
			PWB + PS1 + PW2 * 5, PS1, PW1, 1000 - PS1 * 2,
			PARAM_COUNT,
			locations
		},
		{
			"Mix",
			"bitmaps/blueprints/m1chan.xpm",
			"bitmaps/textures/metal5.xpm",
			0, // flags
			0,
			0,
			mixCallback,
			PWB + PS1 + PW2 * 6, PS1, PW1, 1000 - PS1 * 2,
			PARAM_COUNT,
			locations
		},
		{
			"Mix",
			"bitmaps/blueprints/m1chan.xpm",
			"bitmaps/textures/metal5.xpm",
			0, // flags
			0,
			0,
			mixCallback,
			PWB + PS1 + PW2 * 7, PS1, PW1, 1000 - PS1 * 2,
			PARAM_COUNT,
			locations
		},
		{
			"Mix",
			"bitmaps/blueprints/m1chan.xpm",
			"bitmaps/textures/metal5.xpm",
			0, // flags
			0,
			0,
			mixCallback,
			PWB + PS1 + PW2 * 8, PS1, PW1, 1000 - PS1 * 2,
			PARAM_COUNT,
			locations
		},
		{
			"Mix",
			"bitmaps/blueprints/m1chan.xpm",
			"bitmaps/textures/metal5.xpm",
			0, // flags
			0,
			0,
			mixCallback,
			PWB + PS1 + PW2 * 9, PS1, PW1, 1000 - PS1 * 2,
			PARAM_COUNT,
			locations
		},
		{
			"Mix",
			"bitmaps/blueprints/m1chan.xpm",
			"bitmaps/textures/metal5.xpm",
			0, // flags
			0,
			0,
			mixCallback,
			PWB + PS1 + PW2 * 10, PS1, PW1, 1000 - PS1 * 2,
			PARAM_COUNT,
			locations
		},
		{
			"Mix",
			"bitmaps/blueprints/m1chan.xpm",
			"bitmaps/textures/metal5.xpm",
			0, // flags
			0,
			0,
			mixCallback,
			PWB + PS1 + PW2 * 11, PS1, PW1, 1000 - PS1 * 2,
			PARAM_COUNT,
			locations
		},
		{
			"Mix",
			"bitmaps/blueprints/m1chan.xpm",
			"bitmaps/textures/metal5.xpm",
			0, // flags
			0,
			0,
			mixCallback,
			PWB + PS1 + PW2 * 12, PS1, PW1, 1000 - PS1 * 2,
			PARAM_COUNT,
			locations
		},
		{
			"Mix",
			"bitmaps/blueprints/m1chan.xpm",
			"bitmaps/textures/metal5.xpm",
			0, // flags
			0,
			0,
			mixCallback,
			PWB + PS1 + PW2 * 13, PS1, PW1, 1000 - PS1 * 2,
			PARAM_COUNT,
			locations
		},
		{
			"Mix",
			"bitmaps/blueprints/m1chan.xpm",
			"bitmaps/textures/metal5.xpm",
			0, // flags
			0,
			0,
			mixCallback,
			PWB + PS1 + PW2 * 14, PS1, PW1, 1000 - PS1 * 2,
			PARAM_COUNT,
			locations
		},
		{
			"Mix",
			"bitmaps/blueprints/m1chan.xpm",
			"bitmaps/textures/metal5.xpm",
			0, // flags
			0,
			0,
			mixCallback,
			PWB + PS1 + PW2 * 15, PS1, PW1, 1000 - PS1 * 2,
			PARAM_COUNT,
			locations
		},
#define PBW 200
#define PBH 760
		{
			"Bussing",
			"bitmaps/blueprints/m1bus.xpm",
			"bitmaps/textures/metal5.xpm",
			0, // flags
			0,
			0,
			mixCallback,
				PWB + PS1 + PW2 * 16,
				1000 - PBH + PS1, //PS1,
				PBW,
				PBH - PS1 * 2,//1000 - PS1 * 2,
			BUS_COUNT,
			busses
		},
		{
			"Mastering",
			"bitmaps/blueprints/m1master.xpm",
			"bitmaps/textures/metal5.xpm",
			0, // flags
			0,
			0,
			mixCallback,
				PWB + PS1 * 2 + PW2 * 16 + PBW,
				1000 - PBH + PS1, //PS1,
				1000 - PS1 * 3 - PW2 * 16 - PBW - PWB * 2,
				PBH - PS1 * 2,//1000 - PS1 * 2,
			MASTER_COUNT,
			master
		},
		{
			"Function Panel",
			"bitmaps/blueprints/mix.xpm",
			"bitmaps/textures/metal5.xpm",
			0, // flags
			0,
			0,
			mixCallback,
				PWB + PS1 + PW2 * 16,
				PS1, //PS1,
				1000 - PWB - PS1 * 2 - PWB - PW2 * 16,
				1000 - PBH - PS1,
			FUNCTION_COUNT,
			functions
		},
		{
			"Border Left",
			0,
			"bitmaps/textures/wood2.xpm",
			BRIGHTON_STRETCH|BRIGHTON_VERTICAL, // flags
			0,
			0,
			0,
			0, 0, PWB, 1000,
			0,
			0
		},
		{
			"Border Right",
			0,
			"bitmaps/textures/wood2.xpm",
			BRIGHTON_STRETCH|BRIGHTON_VERTICAL, // flags
			0,
			0,
			0,
			1000 - PWB, 0, PWB, 1000,
			0,
			0
		}
	}
};

//static dispatcher dispatch[DEVICE_COUNT];

static int
mixMidiSendMsg(void *synth, int fd, int chan, int c, int o, int v)
{
//printf("%i, %i, %i\n", c, o, v);
	bristolMidiSendMsg(fd, chan, c, o, v);
}

static int
mixMemory(guiSynth *synth, int fd, int chan, int c, int o, int v)
{
	brightonEvent event;

printf("mixMemory()\n");

	switch (c) {
		default:
		case 0:
			synth->location = synth->location * 10 + o;

			if (synth->location >= 1000)
				synth->location = o;
			displayPanelText(synth, "PRG", synth->location,
				DISPLAY_PANEL, DISPLAY_DEV);
			break;
		case 1:
			loadMemory(synth, "mix", 0, synth->location,
				synth->mem.active, FIRST_DEV, 0);
			displayPanelText(synth, "PRG", synth->bank + synth->location,
				DISPLAY_PANEL, DISPLAY_DEV);
			break;
		case 2:
			saveMemory(synth, "mix", 0, synth->location, FIRST_DEV);
			displayPanelText(synth, "PRG", synth->bank + synth->location,
				DISPLAY_PANEL, DISPLAY_DEV);
			break;
	}
}

static int
mixMidi(guiSynth *synth, int fd, int chan, int c, int o, int v)
{
	bristolMidiMsg msg;
	//printf("mixMidi(%i %i %i %i %i)\n", fd, chan, c, o, v);

	if (c == 1) {
		if (--synth->midichannel < 0)
			synth->midichannel = 0;
	} else {
		if (++synth->midichannel >= 16)
			synth->midichannel = 15;
	}

	bristolMidiSendMsg(global.controlfd, synth->sid,
		127, 0, BRISTOL_MIDICHANNEL|synth->midichannel);
	if (bristolMidiRead(global.controlfd, &msg) != BRISTOL_OK)
		cleanupBristol();

	displayPanelText(synth, "MIDI", synth->midichannel + 1,
		DISPLAY_PANEL, DISPLAY_DEV);
}

/*
 * For the sake of ease of use, links have been placed here to be called
 * by any of the devices created. They would be better in some other file,
 * perhaps with this as a dispatch.
 *
 * Param refers to the device index in the locations table given below.
 */
static int
mixCallback(void *cid, int panel, int index, float value)
{
	guiSynth *synth = findSynth(global.synths, (int) cid);
	int sendvalue;

	if (synth == 0)
		return;

	if (mixApp.resources[panel].devlocn[index].to == 1)
		sendvalue = value * C_RANGE_MIN_1;
	else
		sendvalue = value;

	switch (panel) {
		case CHAN_PANEL:
		case CHAN_PANEL + 1:
		case CHAN_PANEL + 2:
		case CHAN_PANEL + 3:
		case CHAN_PANEL + 4:
		case CHAN_PANEL + 5:
		case CHAN_PANEL + 6:
		case CHAN_PANEL + 7:
		case CHAN_PANEL + 8:
		case CHAN_PANEL + 9:
		case CHAN_PANEL + 10:
		case CHAN_PANEL + 11:
		case CHAN_PANEL + 12:
		case CHAN_PANEL + 13:
		case CHAN_PANEL + 14:
		case CHAN_PANEL + 15:
			index += panel * PARAM_COUNT;
			break;
		case BUS_PANEL:
			index += panel * PARAM_COUNT;
			break;
		case MASTER_PANEL:
			index += panel * PARAM_COUNT;
			break;
		case FUNCTION_PANEL:
			if (index == 0)
			{
				displayPanelText(synth,
					"ABCDEFGHIJKLMNOPQRSTUVWXYZ",
					synth->midichannel + 1, FUNCTION_PANEL, DISPLAY1);
				displayPanelText(synth,
					"BCDEFGHIJKLMNOPQRSTUVWXYZ",
					synth->midichannel + 1, FUNCTION_PANEL, DISPLAY2);
				displayPanelText(synth,
					"CDEFGHIJKLMNOPQRSTUVWXYZ",
					synth->midichannel + 1, FUNCTION_PANEL, DISPLAY3);
				displayPanelText(synth,
					"DEFGHIJKLMNOPQRSTUVWXYZ",
					synth->midichannel + 1, FUNCTION_PANEL, DISPLAY4);
				displayPanelText(synth,
					"EFGHIJKLMNOPQRSTUVWXYZ",
					synth->midichannel + 1, FUNCTION_PANEL, DISPLAY5);
				displayPanelText(synth,
					"FGHIJKLMNOPQRSTUVWXYZ",
					synth->midichannel + 1, FUNCTION_PANEL, DISPLAY6);

/*
				displayPanelText(synth,
					"Bass",
					synth->midichannel + 1, 0, PARAM_COUNT - 1);
				displayPanelText(synth,
					"GUITAR",
					synth->midichannel + 1, 1, PARAM_COUNT - 1);
				displayPanelText(synth,
					"BDRUM",
					synth->midichannel + 1, 2, PARAM_COUNT - 1);
				displayPanelText(synth,
					"SNARE",
					synth->midichannel + 1, 3, PARAM_COUNT - 1);
				displayPanelText(synth,
					"HH",
					synth->midichannel + 1, 4, PARAM_COUNT - 1);
				displayPanelText(synth,
					"CYM1",
					synth->midichannel + 1, 5, PARAM_COUNT - 1);
				displayPanelText(synth,
					"CYM2",
					synth->midichannel + 1, 6, PARAM_COUNT - 1);
				displayPanelText(synth,
					"TOMTOM",
					synth->midichannel + 1, 7, PARAM_COUNT - 1);
				displayPanelText(synth,
					"MINI",
					synth->midichannel + 1, 8, PARAM_COUNT - 1);
				displayPanelText(synth,
					"HAMMOND",
					synth->midichannel + 1, 9, PARAM_COUNT - 1);
				displayPanelText(synth,
					"ABCDEFGHIJKLM",
					synth->midichannel + 1, 10, PARAM_COUNT - 1);
				displayPanelText(synth,
					"ABCDEFGHIJKLM",
					synth->midichannel + 1, 11, PARAM_COUNT - 1);
				displayPanelText(synth,
					"ABCDEFGHIJKLM",
					synth->midichannel + 1, 12, PARAM_COUNT - 1);
				displayPanelText(synth,
					"ABCDEFGHIJKLM",
					synth->midichannel + 1, 13, PARAM_COUNT - 1);
				displayPanelText(synth,
					"ABCDEFGHIJKLM",
					synth->midichannel + 1, 14, PARAM_COUNT - 1);
				displayPanelText(synth,
					"ABCDEFGHIJKLM",
					synth->midichannel + 1, 15, PARAM_COUNT - 1);
*/
			}
			index += panel * PARAM_COUNT;
			break;
		default:
			printf("unknown panel %i\n", panel);
			break;
	}

	if ((synth->flags & OPERATIONAL) == 0)
		return;

//	printf("mixCallback(%i, %f): %x\n", index, value, synth);

	synth->mem.param[index] = value;

	synth->dispatch[index].routine(synth,
		global.controlfd, synth->sid,
		synth->dispatch[index].controller,
		synth->dispatch[index].operator,
		sendvalue);
#ifdef DEBUG
	else
		printf("dispatch[%x,%i](%i, %i, %i, %i, %i)\n", synth, index,
			global.controlfd, synth->sid,
			synth->dispatch[index].controller,
			synth->dispatch[index].operator,
			sendvalue);
#endif

	return(0);
}

static int
mixPassthrough(float value)
{
	//printf("mixPassthrough %f\n", value);
}

/*
 * Any location initialisation required to run the callbacks. For bristol, this
 * will connect to the engine, and give it some base parameters.
 * May need to generate some application specific menus.
 * Will also then make specific requests to some of the devices to alter their
 * rendering.
 *
 * For the mixer it is also reasonable to build the application structures, 
 * rather than have them defined in advance?
 */
static int
mixInit(void *reference)
{
	guiSynth *synth = findSynth(global.synths, (int) reference);
	dispatcher *dispatch;
	int i;
/*
	brightonEvent event;
	char bitmap[128];

	event.type = BRIGHTON_MEM;
	event.m = OP2XPM;
	brightonParamChange((void *) synth->connid,
		ALGOS_PANEL, ALGOS_COUNT - 1, &event);
*/

	if (synth == 0)
	{
		synth = findSynth(global.synths, (int) 0);
		if (synth == 0)
		{
			printf("cannot init\n");
			return(0);
		}
	}

	synth->connid = (int) reference;

	printf("Initialise the mix link to bristol: %x\n", synth->connid);

	synth->mem.param = (float *) brightonmalloc(DEVICE_COUNT * sizeof(float));
	synth->mem.count = DEVICE_COUNT;
	synth->mem.active = ACTIVE_DEVS;
	synth->dispatch = (dispatcher *)
		brightonmalloc(DEVICE_COUNT * sizeof(dispatcher));
	dispatch = synth->dispatch;

	/*
	 * We really want to have three connection mechanisms. These should be
	 *	1. Unix named sockets.
	 *	2. UDP sockets.
	 *	3. MIDI pipe.
	 */
	if (!global.libtest)
		if ((synth->sid = initConnection(&global, synth)) < 0)
			exit(0);
printf("CID: %i\n", synth->sid);

	for (i = 0; i < DEVICE_COUNT; i++)
		synth->dispatch[i].routine = (synthRoutine) mixPassthrough;
}

/*
 * This will be called to make any routine specific parameters available.
 */
static int
mixConfigure(int cid)
{
	guiSynth *synth = findSynth(global.synths, cid);

	if (synth == 0)
	{
		printf("problems going operational\n");
		return(-1);
	}

	if (synth->flags & OPERATIONAL)
		return;

printf("going operational %x %x\n", synth, synth->connid);

	synth->flags |= OPERATIONAL;
	synth->transpose = 36;

	displayPanelText(synth, "TRK", 1, 0, PARAM_COUNT - 1);
	displayPanelText(synth, "TRK", 2, 1, PARAM_COUNT - 1);
	displayPanelText(synth, "TRK", 3, 2, PARAM_COUNT - 1);
	displayPanelText(synth, "TRK", 4, 3, PARAM_COUNT - 1);
	displayPanelText(synth, "TRK", 5, 4, PARAM_COUNT - 1);
	displayPanelText(synth, "TRK", 6, 5, PARAM_COUNT - 1);
	displayPanelText(synth, "TRK", 7, 6, PARAM_COUNT - 1);
	displayPanelText(synth, "TRK", 8, 7, PARAM_COUNT - 1);
	displayPanelText(synth, "TRK", 9, 8, PARAM_COUNT - 1);
	displayPanelText(synth, "TRK", 10, 9, PARAM_COUNT - 1);
	displayPanelText(synth, "TRK", 11, 10, PARAM_COUNT - 1);
	displayPanelText(synth, "TRK", 12, 11, PARAM_COUNT - 1);
	displayPanelText(synth, "TRK", 13, 12, PARAM_COUNT - 1);
	displayPanelText(synth, "TRK", 14, 13, PARAM_COUNT - 1);
	displayPanelText(synth, "TRK", 15, 14, PARAM_COUNT - 1);
	displayPanelText(synth, "TRK", 16, 15, PARAM_COUNT - 1);
}

