/*
 * Caudium - An extensible World Wide Web server
 * Copyright  2000-2004 The Caudium Group
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * $Id: camas_addressbook_preferences.pike,v 1.16.2.4 2004/04/07 23:08:28 vida Exp $
 */

#include <module.h>
#include <camas/globals.h>
#include <camas/addressbook.h>
inherit "module";

constant cvs_version="$Id: camas_addressbook_preferences.pike,v 1.16.2.4 2004/04/07 23:08:28 vida Exp $";
constant module_type = MODULE_PROVIDER;
constant module_name = "CAMAS: Addressbook user's preferences";
constant module_doc  = "<p>Addressbook for CAMAS based on preferences module. This module provides specific methods for "
                       "managing addressbook using the preferences module. This is the most basic type of "
                       "addressbook and it is designed only for a small number of personnal entries "
                       "but it's also the most easy to setup.<br/>"
                       "It's better for you if use LDAP address book module since it's more complete, "
                       "faster and you can plug-in to your existing LDAP architecture with it. "
                       "The drawback is that it's more difficult to setup (you need LDAP knowledge and a "
                       "LDAP server).<br/>"
                       "This addressbook is compatible with the "
                       "default addressbook available in previous versions of Camas and IMHO.</p>"
                       "<i>Note</i>: You have to enable &quot;Features -&gt; User can change setup&quot; in the "
                       "Features module to get this module working.";
constant module_unique = 1;
constant thread_safe = 1;

// the order in which we store attributes 
constant attributes = ({ 
  AD_MAIL,
  AD_NAME,
  AD_SURNAME,
  AD_NICKNAME,
  AD_HOMEPHONE,
  AD_WORKPHONE,
  AD_TITLE,
  AD_COMPANY,
  AD_DEPARTMENT,
  AD_HOMEPOSTALADDRESS,
  AD_WORKPOSTALADDRESS,
  AD_WORKCITY,
  AD_WORKCOUNTRY,
  AD_WORKSTATE,
  AD_PAGER,
  AD_MOBILE,
  AD_URI,
  AD_DESCRIPTION,
  AD_POSTALCODE
});

void create()
{
#ifdef CAMAS_DEBUG
  defvar("debug",0,"Debug",TYPE_FLAG,"Debug the call / errors into Caudium "
         "error log ?");
#endif
  defvar("myname", "Personnal addressbook", "Name of this addressbook",
         TYPE_STRING, "The name of this addressbook that will appear to the user");

  defvar("isdefault", 1, "Is default ?", TYPE_FLAG,
         "If set to yes, users will have this addressbook selected when they "
         "enter the address book screen the first time during their session.");
}

void start(int cnt, object conf)
{
}

string query_provides()
{
  return("camas_addressbook");
}

void stop()
{
}

/*
 * What we provide here
 */

//
//! method: int version(void)
//!  Give the CAMAS_PREFERENCES api version
//!  supported by the module
//! returns:
//!  the version of the API
//
int version()
{
  return 1;
}

//! method: string get_name(void)
//!  get the name of this addressbook
string get_name()
{
  return QUERY(myname);
}

//! method: int writeable(void)
//!  Is this addressbook writeable ?
int writeable()
{
  return 1;
}

//! method: int isdefault(void)
//!  Is this addressbook the default one for users ?
int isdefault()
{
  return QUERY(isdefault);
}

//! method: array(string) get_all_attributes()
//!  Returns every attributes supported by this addressbook
array(string) get_all_attributes()
{
  return attributes;
}

//! method: int|string get_lastprotocol_error()
//!  Returns the last protocol error if any or 0 otherwise
//!  In this module there is no protocol so this function always returns 0
int|string get_lastprotocol_error()
{
  return 0;
}

//
//! method: array(mapping(int:string)) get_all(object id)
//!  Returns every entries on this addressbook, maybe disable by administrator/protocol
//!  Returns an empty string if there are no entries
array(mapping(int:array(string))) get_all(object id)
{
  array contacts = ({ });
  array(string) lines = CSESSION->addressbook / "\n" - ({ "" });
  foreach (lines, string line)
  {
    int i = 0;
    array fields = line / ":";
    mapping contact = ([ ]);
    foreach(attributes, int attribute)
    {
      string field;
      if(i >= sizeof(fields))
        field = "";
      else
        field = fields[i++];
      contact += ([ attribute: ({ field }) ]);
    }
    // This is a hack because previous version of Camas put the
    // email address at the place of the name
    if(has_value(contact[AD_NAME][0], "@") && !has_value(contact[AD_MAIL][0], "@"))
    {
      array mail = contact[AD_NAME];
      contact[AD_NAME] = contact[AD_MAIL];
      contact[AD_MAIL] = mail;
    }
    contacts += ({ contact });
  }
  return contacts;
}

//
//! method: array(mapping(int:array(string))) get(mapping(int:string) variables2search, object id, void|int or_search, void|int wildcardsearch)
//!   get every entries matching the given values in the variables2search mapping
//! arg: mapping(int:string) variables2search
//!  A mapping indexed by internal attributes value (internals means name will be replaced by cn later) and 
//!  containing for each key the value it must have. If several attributes are provided we simply make an AND
//! arg: object id
//!  Caudium id object
//! arg: void|int or_search
//!  Is this search and'ed or or'ed ?. By default attributes are and'ed
//! arg: void|int wildcardsearch 
//!  For a given attribute, do we search words matching a part of the search filter (wildcard type search) 
//!  or only the words matching ?
//! returns:
//!  Every entries present on this addressbook that match the given criterias
//!  An empty string if none match
array(mapping(int:array(string))) get(mapping(int:string) variables2search, object id, void|int or_search, 
    void|int wildcardsearch)
{
  return CAMAS.AddressBook2.get(variables2search, id, get_all(id), or_search, wildcardsearch);
}

//
//! method: int|void add(mapping(int:string) variables2insert, object id)
//!  Insert variables into this addressbook starting at the DN configured in the configuration interface
//! arg: mapping(int:string) variables2insert
//!  The variables to insert
//! arg: object id
//!  The Caudium id object 
//! returns:
//!  Missing field in case of errors
int|void add(mapping(int:string) variables2insert, object id)
{
  string line = "";
  CDEBUG(sprintf("variables2insert=%O\n", variables2insert));
  // if the contact email is already in the "database"
  if(sizeof(get(([ AD_MAIL: variables2insert[AD_MAIL] ]), id)) > 0)
  {
    report_warning("Email %s not added: already in this addressbook\n", variables2insert[AD_MAIL]);
    return;
  }
  foreach(attributes, int attribute)
  {
    if(variables2insert[attribute])
      line += replace(variables2insert[attribute], ":", "") + ":";
    else
      line += ":";
  }
  if (sizeof(CSESSION->addressbook) > 0)
   CSESSION->addressbook += "\n"+line;
  else
   CSESSION->addressbook = line;
}

//
//! method int|void modify(mapping(int:string) variables2insert, string oldemail, object id)
//!  Modify entry into this addressbook starting at the DN configured in the configuration interface
//! arg: mapping(int:string) variables2modify
//!  The variables to modify
//! arg: string oldemail
//!  The old email of the user
//! arg: object id
//!  The Caudium id object 
//! returns:
//!  Missing field in case of errors
int|void modify(mapping(int:string) variables2modify, string oldemail, object id)
{
  array all = get_all(id);
  array res = ({ });
  CDEBUG(sprintf("variables2modify=%O, oldemail=%O\n", variables2modify, oldemail));
  foreach(all, mapping contact)
  {
    int modifyme = 0;
    foreach(contact[AD_MAIL], string val)
      if(val == oldemail)
        // modify the entry
        modifyme = 1;
    if(modifyme)
      contact = contact + variables2modify;
    res += ({ contact });
  }
  // modify = erase and re-create everything
  // ok it's dirty but after all I told this method was not for lot of entries
  CSESSION->addressbook = "";
  foreach(res, mapping(int:array(string)) variables2insert)
  {
    // convert mapping(int:array(string) to mapping(int:string)
    mapping(int:string) _variables2insert = ([ ]);
    foreach(indices(variables2insert), int index)
      if(arrayp(variables2insert[index]))
        _variables2insert += ([ index: variables2insert[index] * "," ]);
      else
        _variables2insert += ([ index: variables2insert[index] ]);
    add(_variables2insert, id);
  }
}

//
//! method: int|void delete(string email, object id)
//!  Delete variables into this addressbook starting at the DN configured in the configuration interface
//! arg: mapping(string:string) variables2insert
//!  The variables to insert
//! arg: object id
//!  The Caudium id object 
//! returns:
//!  Missing field in case of errors
int|void delete(string email, object id)
{
  array all = get_all(id);
  array res = ({ });
  CDEBUG(sprintf("deleting email=%O\n", email));
  foreach(all, mapping contact)
  {
    foreach(contact[AD_MAIL], string val)
    {
      if(val != email)
      {
        res += ({ contact });
        break;
      }
    }
  }
  // modify = erase and re-create everything
  // ok it's dirty but after all I told this method was not for lot of entries
  CSESSION->addressbook = "";
  foreach(res, mapping(int:array(string)) _variables2insert)
  {
    mapping(int:string) variables2insert = ([ ]);
    foreach(indices(_variables2insert), int indice)
      variables2insert += ([ indice: _variables2insert[indice] * "," ]);
    add(variables2insert, id);
  }
}

/*
 * If you visit a file that doesn't contain these lines at its end, please     
 * cut and paste everything from here to that file.                            
 */                                                                            

/*
 * Local Variables:                                                            
 * c-basic-offset: 2                                                           
 * End:                                                                        
 *                                                                             
 * vim: softtabstop=2 tabstop=2 expandtab autoindent formatoptions=croqlt smartindent cindent shiftwidth=2
 */

