/*-----------------------------------------------------------------------------------------

 File        : pde_TschumperleDeriche2D.cpp

 Description : Implementation of the Tschumperle-Deriche's Regularization
               PDE, for 2D multivalued images, as described in the following articles :
 
 (1) PDE-Based Regularization of Multivalued Images and Applications.
     (D. Tschumperl). PhD Thesis. University of Nice-Sophia Antipolis, December 2002.
 (2) Diffusion PDE's on Vector-valued Images : Local Approach and Geometric Viewpoint.
     (D. Tschumperl and R. Deriche). IEEE Signal Processing Magazine, October 2002.
 (3) Vector-Valued Image Regularization with PDE's : A Common Framework for Different Applications.
     (D. Tschumperl and R. Deriche). CVPR'2003, Computer Vision and Pattern Recognition, Madison, United States, June 2003.  
 
 This code can be used to perform image restoration, inpainting, magnification or flow visualization.
 
 Author      : David Tschumperl

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

------------------------------------------------------------------------------------*/

#include "../CImg.h"
// The lines below are not necessary in your own code, it simply allows 
// the source compilation with compilers that do not respect the C++ standart.
#if ( defined(_MSC_VER) && _MSC_VER<=1200 ) || defined(__DMC__)
#define std
#endif
using namespace cimg_library;

int main(int argc,char **argv) {

  // Read command line arguments
  //-----------------------------
  cimg_usage("Tschumperle-Deriche's flow for 2D Image Restoration, Inpainting, Magnification or Flow visualization");
  const char *file_i  = cimg_option("-i",(char*)NULL,"Input image");
  const char *file_m  = cimg_option("-m",(char*)NULL,"Mask image (if Inpainting)");
  const char *file_f  = cimg_option("-f",(char*)NULL,"Flow image (if Flow visualization)");
  const char *file_o  = cimg_option("-o",(char*)NULL,"Output file");
  const double zoom   = cimg_option("-zoom",1.0,"Image magnification");

  const unsigned int nb_iter  = cimg_option("-iter",100000,"Number of iterations");
  const double dt     = cimg_option("-dt",20,"Adapting time step");
  const double alpha  = cimg_option("-alpha",0,"Gradient smoothing");
  const double sigma  = cimg_option("-sigma",0.5,"Structure tensor smoothing");
  const double a1     = cimg_option("-a1",1,"Diffusion coef along minimal variations");
  const double a2     = cimg_option("-a2",0.5,"Diffusion coef along maximal variations");
  const double noiseg = cimg_option("-ng",0.0,"Add gauss noise before aplying the algorithm");
  const double noiseu = cimg_option("-nu",0.0,"Add uniform noise before applying the algorithm");
  const double noises = cimg_option("-ns",0.0,"Add salt&pepper noise before applying the algorithm");
  const bool stflag   = cimg_option("-stats",false,"Display image statistics at each iteration");
  const unsigned int save     = cimg_option("-save",0,"Iteration saving step");
  const unsigned int visu     = cimg_option("-visu",10,"Visualization step (0=no visualization)");
  const unsigned int init     = cimg_option("-init",3,"Inpainting initialization (0=black, 1=white, 2=noise, 3=unchanged)");
  const unsigned int skip     = cimg_option("-skip",1,"Step of image geometry computation");
  bool view_t         = cimg_option("-d",false,"View tensor directions (useful for debug)");
  const double na1 = -1.0/a1, na2 = -1.0/a2;
  double xdt = 0;

  // Variable initialization
  //-------------------------
  CImg<> img, flow;
  CImg<int> mask;

  if (file_i) {
    img = CImg<>(file_i).resize(-100,-100,1,-100);
    if (file_m) mask = CImg<unsigned char>(file_m).resize(img.width,img.height,1,1);
    else if (zoom>1) {
      mask = CImg<int>(img.width,img.height,1,1,-1).resize((int)(img.width*zoom),(int)(img.height*zoom),1,1,4)+1;
      img.resize((int)(img.width*zoom),(int)(img.height*zoom),1,-100,3);
    }
  } else {
    if (file_f) {
      flow = CImg<>(file_f);
      img = CImg<>((int)(flow.width*zoom),(int)(flow.height*zoom),1,1,0).noise(100,2);
      flow.resize(img.width,img.height,1,2,3);
    } else throw CImgException("You need to specify at least one input image (option -i), or one flow image (option -f)");
  }
  img.noise(noiseg,0).noise(noiseu,1).noise(noises,2);
  CImgStats initial_stats(img,false);
  if (mask.data && init!=3) 
    cimg_mapXYV(img,x,y,k) if (mask(x,y)) 
      img(x,y,k)=(float)((init?
                  (init==1?initial_stats.max:((initial_stats.max-initial_stats.min)*cimg::rand())):
                  initial_stats.min));

  CImgDisplay *disp = visu?img.new_display("Iterated Image",1,0x12):NULL;
  CImg<> G(img.width,img.height,1,3,0), T(G,false), veloc(img,false), val(2), vec(2,2);

  // PDE main iteration loop
  //-------------------------
  for (unsigned int iter=0; iter<nb_iter && (!disp || !disp->closed && disp->key!=cimg::keyQ && disp->key!=cimg::keyESC); iter++) {
    std::printf("\riter %d , xdt = %g               ",iter,xdt); fflush(stdout);
    if (stflag) img.print();
    if (disp && disp->key==cimg::keySPACE) { view_t = !view_t; disp->key=0; }

    if (!(iter%skip)) {
      // Compute the tensor field T, used to drive the diffusion
      //---------------------------------------------------------
      
      // When using PDE for flow visualization
      if (flow.data) cimg_mapXY(flow,x,y) {
        const float
          u = flow(x,y,0,0),
          v = flow(x,y,0,1),
          n = (float)std::sqrt((double)(u*u+v*v)),
          nn = (n!=0)?n:1;
        T(x,y,0) = u*u/nn;
        T(x,y,1) = u*v/nn;
        T(x,y,2) = v*v/nn;
      } else {
        
        // Compute structure tensor field G
        const CImgl<> grad = img.get_gradientXY(3);
        if (alpha!=0) cimgl_map(grad,l) grad[l].blur((float)alpha);
        cimg_mapXYV(img,x,y,k) {
          const float ix = grad[0](x,y,k), iy = grad[1](x,y,k);
          G(x,y,0) += ix*ix;
          G(x,y,1) += ix*iy;
          G(x,y,2) += iy*iy;
        }
        if (sigma!=0) G.blur((float)sigma);
        
        // When using PDE for image restoration, inpainting or zooming
        T.fill(0);
        if (!mask.data) cimg_mapXY(G,x,y) {
          G.get_tensor(x,y).symeigen(val,vec);
          const double ng = std::sqrt((double)(1e-7+val[0]+val[1]));
          const float
            l1 = (float)std::pow(ng,na1),
            l2 = (float)std::pow(ng,na2),
            ux = vec[0],
            uy = vec[1];
          T(x,y,0) = l1*ux*ux + l2*uy*uy;
          T(x,y,1) = l1*ux*uy - l2*ux*uy;
          T(x,y,2) = l1*uy*uy + l2*ux*ux;
        }
        else cimg_mapXY(G,x,y) if (mask(x,y)) {
          G.get_tensor(x,y).symeigen(val,vec);
          const float
            ux = vec[0],
            uy = vec[1];
          T(x,y,0) = ux*ux;
          T(x,y,1) = ux*uy;
          T(x,y,2) = uy*uy;
        }
      }
    }

    // Compute the PDE velocity and update the iterated image
    //--------------------------------------------------------
    CImg_3x3(I,float);
    veloc.fill(0);
    cimg_mapV(img,k) cimg_map3x3(img,x,y,0,k,I) {
      const float 
        a = T(x,y,0), 
        b = T(x,y,1),
        c = T(x,y,2),
        ixx = Inc+Ipc-2*Icc, 
        iyy = Icn+Icp-2*Icc,
        ixy = 0.25f*(Ipp+Inn-Ipn-Inp);
      veloc(x,y,k) = a*ixx + 2*b*ixy + c*iyy;
    }
    const CImgStats stats(veloc,false);
    xdt = dt/cimg::max(std::fabs(stats.min),std::fabs(stats.max));
    img+=veloc*xdt;
    img.cut((float)initial_stats.min,(float)initial_stats.max);

    // Display and save iterations
    if (disp && !(iter%visu)) {
      if (!view_t) img.display(*disp);
      else {
        const unsigned char white[3] = {255,255,255};
        CImg<unsigned char> visu = img.get_resize(disp->width,disp->height).normalize(0,255);
        CImg<> isophotes(img.width,img.height,1,2,0);
        cimg_mapXY(img,x,y) if (!mask.data || mask(x,y)) {
          T.get_tensor(x,y).symeigen(val,vec);
          isophotes(x,y,0) = vec[2];
          isophotes(x,y,1) = vec[3];
        }
        visu.draw_quiver(isophotes,white,10,9,0,0.5f).display(*disp);
      }
    }
    if (save && file_o && !(iter%save)) img.save(file_o,iter);
    if (disp) disp->resize().display(img);
  }

  // Save result and exit.
  if (file_o) img.save(file_o);
  return 0;
}
