/*****************************************************************************
** FILE IDENTIFICATION
**
**   Name:          docs.cpp
**   Purpose:       Document routines for CTSim program
**   Programmer:    Kevin Rosenberg
**   Date Started:  July 2000
**
**  This is part of the CTSim program
**  Copyright (c) 1983-2001 Kevin Rosenberg
**
**  $Id: docs.cpp 7061 2003-09-07 06:34:45Z kevin $
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License (version 2) as
**  published by the Free Software Foundation.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
******************************************************************************/

#include "wx/wxprec.h"

#ifndef WX_PRECOMP
#include "wx/wx.h"
#endif
#include "wx/txtstrm.h"
#include "wx/file.h"
#include "wx/thread.h"

#if !wxUSE_DOC_VIEW_ARCHITECTURE
#error You must set wxUSE_DOC_VIEW_ARCHITECTURE to 1 in setup.h!
#endif

#include "ct.h"
#include "ctsim.h"
#include "docs.h"
#include "views.h"
#include "threadrecon.h"


// ImageFileDocument

IMPLEMENT_DYNAMIC_CLASS(ImageFileDocument, wxDocument)

bool ImageFileDocument::OnSaveDocument(const wxString& filename)
{
  if (! m_pImageFile->fileWrite (filename)) {
    *theApp->getLog() << "Unable to write image file " << filename << "\n";
    return false;
  }
  if (theApp->getVerboseLogging())
    *theApp->getLog() << "Wrote image file " << filename << "\n";
  Modify(false);
  return true;
}

bool ImageFileDocument::OnOpenDocument(const wxString& filename)
{
  if (! OnSaveModified())
    return false;

  if (! m_pImageFile->fileRead (filename.c_str())) {
    *theApp->getLog() << "Unable to read image file " << filename << "\n";
    m_bBadFileOpen = true;
    return false;
  }
  
  if (theApp->getVerboseLogging() && filename != "")
    *theApp->getLog() << "Read image file " << filename << "\n";
  
  SetFilename(filename, true);  
  Modify(false);
  getView()->setInitialClientSize();
  UpdateAllViews();
  m_bBadFileOpen = false;

  return true;
}

bool 
ImageFileDocument::IsModified(void) const
{
  return wxDocument::IsModified();
}

void 
ImageFileDocument::Modify(bool mod)
{
  wxDocument::Modify(mod);
}

ImageFileView* 
ImageFileDocument::getView() const
{ 
  return dynamic_cast<ImageFileView*>(GetFirstView()); 
}

bool
ImageFileDocument::Revert ()
{
  if (IsModified()) {
    wxString msg ("Revert to saved ");
    msg += GetFilename();
    msg += "?";
    wxMessageDialog dialog (getView()->getFrame(), msg, "Are you sure?", wxYES_NO | wxNO_DEFAULT);
    if (dialog.ShowModal() == wxID_YES) {
      if (theApp->getVerboseLogging())
        *theApp->getLog() << "Reverting to saved " << GetFilename() << "\n";
      Modify (false);
      OnOpenDocument (GetFilename());
    }
  }
  UpdateAllViews();

  return true;
}

void
ImageFileDocument::Activate()
{
#if CTSIM_MDI
  getView()->getFrame()->Activate();
#endif
};

// BackgroundProcessingDocument - Base Class

IMPLEMENT_DYNAMIC_CLASS(BackgroundProcessingDocument, wxDocument)
BEGIN_EVENT_TABLE(BackgroundProcessingDocument, wxDocument)
END_EVENT_TABLE()

#ifdef HAVE_WXTHREADS
void
BackgroundProcessingDocument::addBackgroundSupervisor (BackgroundSupervisor* pSupervisor)
{
  wxCriticalSectionLocker locker (m_criticalSection);
  if (pSupervisor)
    m_vecpBackgroundSupervisors.push_back (pSupervisor);
}

void
BackgroundProcessingDocument::removeBackgroundSupervisor (BackgroundSupervisor* pSupervisor)
{
  m_criticalSection.Enter();
  bool bFound = false;
  for (BackgroundContainer::iterator i = m_vecpBackgroundSupervisors.begin(); 
        i != m_vecpBackgroundSupervisors.end(); 
        i++) 
          if (*i == pSupervisor) {
            m_vecpBackgroundSupervisors.erase(i);
            bFound = true;
            break;
        }
  m_criticalSection.Leave();

  if (! bFound) 
     sys_error (ERR_SEVERE, "Could not find background task [OnRemoveBackground]");
}
#endif

void
BackgroundProcessingDocument::cancelRunningTasks()
{
#ifdef HAVE_WXTHREADS
  m_criticalSection.Enter();
  for (BackgroundContainer::iterator i = m_vecpBackgroundSupervisors.begin(); 
        i != m_vecpBackgroundSupervisors.end(); i++)
          (*i)->onCancel();
  m_criticalSection.Leave();

  while (m_vecpBackgroundSupervisors.size() > 0) {
     ::wxYield();
     ::wxUsleep(50);
  }
#endif
}


// ProjectionFileDocument

IMPLEMENT_DYNAMIC_CLASS(ProjectionFileDocument, BackgroundProcessingDocument)

bool 
ProjectionFileDocument::OnSaveDocument(const wxString& filename)
{
  if (! m_pProjectionFile->write (filename.c_str())) {
    *theApp->getLog() << "Unable to write projection file " << filename << "\n";
    return false;
  }
  if (theApp->getVerboseLogging())
    *theApp->getLog() << "Wrote projection file " << filename << "\n";
  Modify(false);
  return true;
}

ProjectionFileDocument::~ProjectionFileDocument()
{
  cancelRunningTasks();

  delete m_pProjectionFile;
}

bool 
ProjectionFileDocument::OnOpenDocument(const wxString& filename)
{
  if (! OnSaveModified())
    return false;

  if (! m_pProjectionFile->read (filename.c_str())) {
    *theApp->getLog() << "Unable to read projection file " << filename << "\n";
    m_bBadFileOpen = true;
    return false;
  }
  m_bBadFileOpen = false;

  if (theApp->getVerboseLogging() && filename != "")
    *theApp->getLog() << "Read projection file " << filename << "\n";

  SetFilename(filename, true);
  Modify(false);
  getView()->setInitialClientSize();
  UpdateAllViews();
  
  return true;
}

bool 
ProjectionFileDocument::IsModified(void) const
{
  return wxDocument::IsModified();
}

void 
ProjectionFileDocument::Modify(bool mod)
{
  wxDocument::Modify(mod);
}


ProjectionFileView* 
ProjectionFileDocument::getView() const
{ 
  return dynamic_cast<ProjectionFileView*>(GetFirstView()); 
}

void
ProjectionFileDocument::Activate()
{
#if CTSIM_MDI
  getView()->getFrame()->Activate();
#endif
};

// PhantomFileDocument

IMPLEMENT_DYNAMIC_CLASS(PhantomFileDocument, BackgroundProcessingDocument)

PhantomFileDocument::~PhantomFileDocument()
{
  cancelRunningTasks();
}

bool 
PhantomFileDocument::OnOpenDocument(const wxString& constFilename)
{
  if (! OnSaveModified())
    return false;

  wxString filename (constFilename);

  if (wxFile::Exists (filename)) {
    m_phantom.createFromFile (filename);
    if (theApp->getVerboseLogging())
      *theApp->getLog() << "Read phantom file " << filename << "\n";
  } else {
    filename.Replace (".phm", "");
    m_phantom.createFromPhantom (filename);
  }
  m_namePhantom = filename;
  SetFilename (filename, true);
  if (m_phantom.fail()) {
    *theApp->getLog() << "Failure creating phantom " << filename << "\n";
    m_bBadFileOpen = true;
    return false;
  }
  m_idPhantom = m_phantom.id();
  Modify(false);
  UpdateAllViews();
  m_bBadFileOpen = false;
  
  return true;
}

bool 
PhantomFileDocument::OnSaveDocument(const wxString& filename)
{
  if (! m_phantom.fileWrite (filename.c_str())) {
    *theApp->getLog() << "Unable to write phantom file " << filename << "\n";
    return false;
  }
  if (theApp->getVerboseLogging())
    *theApp->getLog() << "Wrote phantom file " << filename << "\n";
  Modify(false);
  return true;
}

bool 
PhantomFileDocument::IsModified(void) const
{
  return false;
}

void 
PhantomFileDocument::Modify(bool mod)
{
  wxDocument::Modify(mod);
}


PhantomFileView* 
PhantomFileDocument::getView() const
{ 
  return dynamic_cast<PhantomFileView*>(GetFirstView()); 
}

void
PhantomFileDocument::Activate()
{
#if CTSIM_MDI
  getView()->getFrame()->Activate();
#endif
};

// PlotFileDocument

IMPLEMENT_DYNAMIC_CLASS(PlotFileDocument, wxDocument)

bool 
PlotFileDocument::OnSaveDocument(const wxString& filename)
{
  m_namePlot = filename.c_str();
  if (! m_plot.fileWrite (filename)) {
    *theApp->getLog() << "Unable to write plot file " << filename << "\n";
    return false;
  }
  if (theApp->getVerboseLogging())
    *theApp->getLog() << "Wrote plot file " << filename << "\n";
  Modify(false);
  return true;
}

bool 
PlotFileDocument::OnOpenDocument(const wxString& filename)
{
  if (! OnSaveModified())
    return false;

  if (! m_plot.fileRead (filename.c_str())) {
    *theApp->getLog() << "Unable to read plot file " << filename << "\n";
    m_bBadFileOpen = true;
    return false;
  }
  m_bBadFileOpen = false;

  if (theApp->getVerboseLogging() && filename != "")
    *theApp->getLog() << "Read plot file " << filename << "\n";

  SetFilename (filename, true);
  m_namePlot = filename.c_str();
  Modify (false);
  getView()->setInitialClientSize();
  UpdateAllViews();
  
  return true;
}


bool 
PlotFileDocument::IsModified(void) const
{
  return wxDocument::IsModified();
}

void 
PlotFileDocument::Modify (bool mod)
{
  wxDocument::Modify(mod);
}

PlotFileView* 
PlotFileDocument::getView() const
{ 
  return dynamic_cast<PlotFileView*>(GetFirstView()); 
}

void
PlotFileDocument::Activate()
{
#if CTSIM_MDI
  getView()->getFrame()->Activate();
#endif
};

//////////////////////////////////////////////////////////////////////////
//
// TextFileDocument
//
//////////////////////////////////////////////////////////////////////////

IMPLEMENT_DYNAMIC_CLASS(TextFileDocument, wxDocument)

bool 
TextFileDocument::OnSaveDocument(const wxString& filename)
{
  TextFileView *view = getView();
  if (! view->getTextCtrl()->SaveFile(filename))
    return false;
  Modify(false);
  return true;
}

bool 
TextFileDocument::OnOpenDocument(const wxString& filename)
{
  TextFileView *view = getView();
  
  if (! view->getTextCtrl()->LoadFile(filename)) {
    m_bBadFileOpen = true;
    return false;
  }
  
  SetFilename (filename, true);
  Modify (false);
  UpdateAllViews();
  m_bBadFileOpen = false;
  return true;
}

bool 
TextFileDocument::IsModified(void) const
{
  return false;
  
  TextFileView *view = getView();
  
  if (view)
    return (wxDocument::IsModified() || view->getTextCtrl()->IsModified());
  else
    return wxDocument::IsModified();
}


TextFileView* 
TextFileDocument::getView() const
{ 
  return dynamic_cast<TextFileView*>(GetFirstView()); 
}

wxTextCtrl* 
TextFileDocument::getTextCtrl()
{ 
  return dynamic_cast<TextFileView*>(GetFirstView())->getTextCtrl(); 
}

//////////////////////////////////////////////////////////////////////////
//
// Graph3dFileDocument
//
//////////////////////////////////////////////////////////////////////////

#if wxUSE_GLCANVAS

IMPLEMENT_DYNAMIC_CLASS(Graph3dFileDocument, wxDocument)

Graph3dFileDocument::Graph3dFileDocument(void) 
: m_bBadFileOpen(false), m_nVertices(0), m_pVertices(0), m_pNormals(0),m_nx(0),m_ny(0),m_array(0)
{
}

Graph3dFileDocument::~Graph3dFileDocument() 
{
}

bool 
Graph3dFileDocument::OnSaveDocument(const wxString& filename)
{
  Modify(false);
  return true;
}

bool 
Graph3dFileDocument::OnOpenDocument(const wxString& filename)
{
  SetFilename (filename, true);
  Modify (false);
  getView()->setInitialClientSize();
  UpdateAllViews();
  m_bBadFileOpen = false;
  return true;
}

bool 
Graph3dFileDocument::IsModified(void) const
{
    return wxDocument::IsModified();
}


Graph3dFileView* 
Graph3dFileDocument::getView() const
{ 
  return dynamic_cast<Graph3dFileView*>(GetFirstView()); 
}

bool
Graph3dFileDocument::createFromImageFile (const ImageFile& rImageFile)
{
  m_nx = rImageFile.nx();
  m_ny = rImageFile.ny();
  m_array = rImageFile.getArray();

  return true;
}

void
Graph3dFileDocument::Activate()
{
#if CTSIM_MDI
  getView()->getFrame()->Activate();
#endif
};


#endif // wxUSE_GLCANVAS
