//Please refer to http://dansguardian.org/?page=copyright2
//for the license for this code.
//Written by Daniel Barron (daniel@//jadeb/.com).
//For support go to http://groups.yahoo.com/group/dansguardian

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "autoconf/platform.h"
#include <syslog.h>
#include "OptionContainer.hpp"
#include "RegExp.hpp"
#include <string>
#include <iostream>
#include <fstream>
#include <netdb.h>  // for gethostby
#include <netinet/in.h>  // for address structures
#include <arpa/inet.h>  // for inet_aton()
#include <sys/socket.h>

#include <unistd.h>  // remove

extern bool isDaemonised;

OptionContainer::OptionContainer()
:numfg(0)
{}

OptionContainer::~OptionContainer() {
    deleteFilterGroups();
}

void OptionContainer::reset() {
    deleteFilterGroups();
    exception_user_list.reset();
    exception_ip_list.reset();
    banned_ip_list.reset();
    banned_user_list.reset();
    html_template.reset();
    language_list.reset();
    conffile.clear();
    filter_groups_list.reset();
}

void OptionContainer::deleteFilterGroups() {
    for(int i = 0; i < numfg; i++) {
        if (fg[i] != NULL) {
            delete fg[i];  // delete extra FOptionContainer objects
            fg[i] = NULL;
        }
    }
    if (numfg > 0) {
        delete[] fg;
        numfg = 0;
    }
}

bool OptionContainer::read(const char* filename, int type) {
    conffilename = filename;
    try { // all sorts of exceptions could occur reading conf files
        std::string linebuffer;
        String temp;  // for tempory conversion and storage
        int j;  // counter
        ifstream conffiles(filename, ios::in);  // dansguardian.conf
        if (!conffiles.good()) {
            if (!isDaemonised) {
                std::cerr << "error reading: " << filename << std::endl;
            }
            syslog(LOG_ERR, "%s","error reading dansguardian.conf");
            return false;
        }
        while (!conffiles.eof()) {
            getline(conffiles, linebuffer);
            if (!conffiles.eof() && linebuffer.length() != 0) {
                if (linebuffer[0] != '#') {  // i.e. not commented out
                    for(j = 0; j < (signed)linebuffer.length(); j++) {
                        linebuffer[j] = tolower(linebuffer[j]);
                    }
                    temp = (char*)linebuffer.c_str();
                    if (temp.contains("#")) {
                        temp = temp.before("#");
                    }
                    temp.removeWhiteSpace();  // get rid of spaces at end of line
                    linebuffer = temp.toCharArray();
                    conffile.push_back(linebuffer);  // stick option in deque
                }
            }
        }
        conffiles.close();

        if (type == 0 || type == 2) {

            if ((ipc_filename = findoptionS("ipcfilename")) == "")
	        ipc_filename = "/tmp/.dguardianipc";

            if ((urlipc_filename = findoptionS("urlipcfilename")) == "")
	        urlipc_filename = "/tmp/.dguardianurlipc";

            if ((pid_filename = findoptionS("pidfilename")) == "") {
                pid_filename = __PIDDIR;
                pid_filename += "dansguardian.pid";
            }

            if ((log_location = findoptionS("loglocation")) == "") {
	        log_location = __LOGLOCATION;
                log_location += "access.log";
            }

            if (type == 0) {
                return true;
	    }
	}

	if ((daemon_user_name = findoptionS("daemonuser")) == "") {
	    daemon_user_name = __PROXYUSER;
        }

        if ((daemon_group_name = findoptionS("daemongroup")) == "") {
            daemon_group_name = __PROXYGROUP;
        }

        if (findoptionS("nodaemon") == "on") {
            no_daemon = 1;
        }
        else {
            no_daemon = 0;
        }

        if (findoptionS("nologger") == "on") {
            no_logger = 1;
        }
        else {
            no_logger = 0;
        }

        if (findoptionS("softrestart") == "on") {
            soft_restart = 1;
        }
        else {
            soft_restart = 0;
        }


        // the dansguardian.conf and pics files get amalgamated into one
        // deque.  They are only seperate files for clarity.



        max_children = findoptionI("maxchildren");
        if (!realitycheck(String(max_children), 1, 3, "maxchildren"))
            { return false; }  // check its a reasonable value
        min_children = findoptionI("minchildren");
        if (!realitycheck(String(min_children), 1, 2, "children"))
            { return false; }  // check its a reasonable value
        maxspare_children = findoptionI("maxsparechildren");
        if (!realitycheck(String(maxspare_children), 1, 2, "children"))
            { return false; }  // check its a reasonable value
        if (maxspare_children < min_children) {
            maxspare_children = min_children;
        }
        prefork_children = findoptionI("preforkchildren");
        if (!realitycheck(String(prefork_children), 1, 2, "children"))
            { return false; }  // check its a reasonable value
        minspare_children = findoptionI("minsparechildren");
        if (!realitycheck(String(minspare_children), 1, 2, "children"))
            { return false; }  // check its a reasonable value
        maxage_children = findoptionI("maxagechildren");
        if (!realitycheck(String(maxage_children), 1, 6, "children"))
            { return false; }  // check its a reasonable value

        max_upload_size = findoptionI("maxuploadsize") * 1024;
        if (!realitycheck(String(max_upload_size), 1, 8, "maxuploadsize"))
            { return false; }  // check its a reasonable value
        max_content_filter_size = findoptionI("maxcontentfiltersize") * 1024;
        if (!realitycheck(String(max_content_filter_size), 1, 8, "maxcontentfiltersize"))
            { return false; }  // check its a reasonable value

        url_cache_number = findoptionI("urlcachenumber");
        if (!realitycheck(String(url_cache_number), 1, 5, "urlcachenumber"))
            { return false; }  // check its a reasonable value

        url_cache_age = findoptionI("urlcacheage");
        if (!realitycheck(String(url_cache_age), 1, 5, "urlcacheage"))
            { return false; }  // check its a reasonable value
        phrase_filter_mode = findoptionI("phrasefiltermode");
        if (!realitycheck(String(phrase_filter_mode), 1, 1, "phrasefiltermode"))
            { return false; }
        preserve_case = findoptionI("preservecase");
        if (!realitycheck(String(preserve_case), 1, 1, "preservecase"))
            { return false; }
        hex_decode_content = findoptionI("hexdecodecontent");
        if (!realitycheck(String(hex_decode_content), 1, 1, "hex_decode_content"))
            { return false; }
        force_quick_search = findoptionI("forcequicksearch");
        if (!realitycheck(String(force_quick_search), 1, 1, "forcequicksearch"))
            { return false; }  // check its a reasonable value

        use_custom_banned_image = findoptionI("usecustombannedimage");
        if (!realitycheck(String(use_custom_banned_image), 1, 1, "usecustombannedimage"))
            { return false; }
        custom_banned_image_file = findoptionS("custombannedimagefile");
        banned_image.read(custom_banned_image_file.c_str());

        filter_port = findoptionI("filterport");
        if (!realitycheck(String(filter_port), 2, 6, "filterport"))
            { return false; }  // check its a reasonable value
        proxy_port = findoptionI("proxyport");
        if (!realitycheck(String(proxy_port), 2, 6, "proxyport"))
            { return false; }  // etc
        proxy_ip = findoptionS("proxyip");
        if (!realitycheck(String(proxy_ip.c_str()), 7, 15, "proxyip"))
            { return false; }
        filter_ip = findoptionS("filterip");
        if (!realitycheck(String(filter_ip.c_str()), 0, 15, "filterip"))
            { return false; }
        ll = findoptionI("loglevel");
        if (!realitycheck(ll, 1, 1, "loglevel"))
            { return false; }  // etc
        log_file_format = findoptionI("logfileformat");
        if (!realitycheck(log_file_format, 1, 1, "logfileformat"))
            { return false; }  // etc
        if (log_file_format < 1 || log_file_format > 4) {
            log_file_format = 1;
        }

        if (findoptionS("showweightedfound") == "on") {
            show_weighted_found = 1;
        }
        else {
            show_weighted_found = 0;
        }
        weighted_phrase_mode = findoptionI("weightedphrasemode");
        if (!realitycheck(String(weighted_phrase_mode), 1, 1, "weightedphrasemode"))
            { return false; }
        reporting_level = findoptionI("reportinglevel");
        if (!realitycheck(String(reporting_level), 1, 2, "reportinglevel"))
            { return false; }
        html_template_location = findoptionS("languagedir") + "/" + findoptionS("language") + "/template.html";

        if (findoptionS("forwardedfor") == "on") {
           forwarded_for = 1;
        }
        else {
           forwarded_for = 0;
        }
        if (findoptionS("logexceptionhits") == "on") {
           log_exception_hits = 1;
        }
        else {
           log_exception_hits = 0;
        }
        if (findoptionS("nonstandarddelimiter") == "off") {
           non_standard_delimiter = 0;
        }
        else {
           non_standard_delimiter = 1;
        }
        if (findoptionS("createlistcachefiles") == "off") {
           createlistcachefiles = 0;
        }
        else {
           createlistcachefiles = 1;
        }
        if (findoptionS("logconnectionhandlingerrors") == "on") {
           logconerror = 1;
        }
        else {
           logconerror = 0;
        }
        if (findoptionS("usernameidmethodproxyauth") == "on") {
           uim_proxyauth = 1;
        }
        else {
           uim_proxyauth = 0;
        }
        if (findoptionS("usernameidmethodntlm") == "on") {
           uim_ntlm = 1;
        }
        else {
           uim_ntlm = 0;
        }
        if (findoptionS("usernameidmethodident") == "on") {
           uim_ident = 1;
        }
        else {
           uim_ident = 0;
        }

        if (findoptionS("preemptivebanning") == "off") {
            preemptive_banning = 0;
        }
        else {
            preemptive_banning = 1;
        }

        if (findoptionS("reverseaddresslookups") == "on") {
           reverse_lookups = 1;
        }
        else {
           reverse_lookups = 0;
        }
        if (findoptionS("reverseclientiplookups") == "on") {
           reverse_client_ip_lookups = 1;
        }
        else {
           reverse_client_ip_lookups = 0;
        }


        if (findoptionS("usexforwardedfor") == "on") {
	   use_xforwardedfor = 1;
	}
	else {
	   use_xforwardedfor = 0;
	}

	filter_groups = findoptionI("filtergroups");

        if (!realitycheck(String(filter_groups), 1, 1, "filtergroups"))
            { return false; }
        if (filter_groups < 1) {
            if (!isDaemonised) {
                std::cerr << "filtergroups too small" << std::endl;
            }
            syslog(LOG_ERR, "%s", "filtergroups too small");
            return false;
        }

        filter_groups_list_location = findoptionS("filtergroupslist");
        banned_ip_list_location = findoptionS("bannediplist");
        banned_user_list_location = findoptionS("banneduserlist");
        exceptions_user_list_location = findoptionS("exceptionuserlist");
        exceptions_ip_list_location = findoptionS("exceptioniplist");
        language_list_location = findoptionS("languagedir") + "/" + findoptionS("language") + "/messages";
        access_denied_address = findoptionS("accessdeniedaddress");
        ada = access_denied_address.c_str();
        ada = ada.after("://");
        ada.removeWhiteSpace();
        if (ada.contains("/")) {
            ada = ada.before("/"); // ada now contains the FQ host nom of the
                               // server that serves the accessdenied.html
                               // file
        }
        if (ada.contains(":")) {
            ada = ada.before(":");  // chop off the port number if any
        }
        if (reporting_level == 1 || reporting_level == 2) {
            if (ada.length() < 4) {
                if (!isDaemonised) {
                    cerr << "accessdeniedaddress setting appears to be wrong." << endl;
                }
                syslog(LOG_ERR, "%s", "accessdeniedaddress setting appears to be wrong.");
                return false;
            }
        }

        if (!readfgfile(filter_groups_list_location.c_str())) {
            return false;
        }

        if (!readbilfile(banned_ip_list_location.c_str())) {
            return false;
        }  // read banned ip list
        if (!readbuslfile(banned_user_list_location.c_str())) {
            return false;
        }  // read banned user list
        if (!readeilfile(exceptions_ip_list_location.c_str())) {
            return false;
        }  // ip exceptions
        if (!readeulfile(exceptions_user_list_location.c_str())) {
            return false;
        }  // site exceptions

        if (!language_list.readLanguageList(language_list_location.c_str())) {
            return false;
        }  // messages language file

        if (reporting_level == 3) {  // only if reporting set to HTML templ
            if (!html_template.readTemplateFile(html_template_location.c_str())) {
                if (!isDaemonised) {
                    std::cerr << "Error reading HTML Template file:" << html_template_location << std::endl;
                }
                syslog(LOG_ERR, "%s", "Error reading HTML Template file.");
                return false;
                // HTML template file
            }
        }

        if (!readFilterGroupConf()) {
            if (!isDaemonised) {
                std::cerr << "Error reading filter group conf file(s)." << std::endl;
            }
            syslog(LOG_ERR, "%s", "Error reading filter group conf file(s).");           return false;

        }

    } catch (exception& e) {
        if (!isDaemonised) {
            std::cerr << e.what() << std::endl;  // when called the daemon has not
                                   // detached so we can do this
        }
	return false;
    }
    return true;
}

bool OptionContainer::readfgfile(const char* filename) {
    bool result = filter_groups_list.readItemList(filename, false, 0);
    if (!result) {
        if (!isDaemonised) {
            std::cerr << "Error opening filtergroupslist" << std::endl;
        }
        syslog(LOG_ERR, "%s","Error opening filtergroupslist");
        return false;
    }
    filter_groups_list.startsWithSort();
    return true;
}

bool OptionContainer::readbilfile(const char* filename) {

    bool result = banned_ip_list.readItemList(filename, false, 0);
    if (!result) {
        if (!isDaemonised) {
            std::cerr << "Error opening bannediplist" << std::endl;
        }
        syslog(LOG_ERR, "%s","Error opening bannediplist");
        return false;
    }
    banned_ip_list.startsWithSort();
    return true;
}

bool OptionContainer::readbuslfile(const char* filename) {

    bool result = banned_user_list.readItemList(filename, false, 0);
    if (!result) {
        if (!isDaemonised) {
            std::cerr << "Error opening banneduserlist" << std::endl;
        }
        syslog(LOG_ERR, "%s","Error opening banneduserlist");
        return false;
    }
    banned_user_list.startsWithSort();
    return true;
}



bool OptionContainer::readeulfile(const char* filename) {

    bool result = exception_user_list.readItemList(filename, false, 0);
    if (!result) {
        if (!isDaemonised) {
            std::cerr << "Error opening exceptionuserlist" << std::endl;
        }
        syslog(LOG_ERR, "%s","Error opening exceptionuserlist");
        return false;
    }
    exception_user_list.endsWithSort();
    return true;
}

bool OptionContainer::readeilfile(const char* filename) {

    bool result = exception_ip_list.readItemList(filename, false, 0);
    if (!result) {
        if (!isDaemonised) {
            std::cerr << "Error opening exceptioniplist" << std::endl;
        }
        syslog(LOG_ERR, "%s","Error opening exceptioniplist");
        return false;
    }
    exception_ip_list.endsWithSort();
    return true;
}



bool OptionContainer::inuserexceptions(const std::string *user) {
    if ((*user).length() < 1) {
        return false;
    }
    return exception_user_list.inList((char*)(*user).c_str());
}


bool OptionContainer::inipexceptions(const std::string *ip) {
    if ((*ip).length() < 1) {
        return false;
    }
    if (reverse_client_ip_lookups != 1) {
        return exception_ip_list.inList((char*)(*ip).c_str());
    }
    if (exception_ip_list.inList((char*)(*ip).c_str())) {
        return true;
    }
    std::deque<String> hostnames = (*fg[0]).ipToHostname((*ip).c_str());
    bool result;
    for (unsigned int i = 0; i < hostnames.size(); i++) {
        result = exception_ip_list.inList(hostnames[i].toCharArray());
        if (result) {
            return true;
        }
    }
    return false;
}

bool OptionContainer::inBannedUserList(const std::string *user) {
    if ((*user).length() < 1) {
        return false;
    }
    return banned_user_list.inList((char*)(*user).c_str());
}

bool OptionContainer::inBannedIPList(const std::string *ip) {
    if ((*ip).length() < 1) {
        return false;
    }
    if (reverse_client_ip_lookups != 1) {
        return banned_ip_list.inList((char*)(*ip).c_str());
    }
    if (banned_ip_list.inList((char*)(*ip).c_str())) {
        return true;
    }
    std::deque<String> hostnames = (*fg[0]).ipToHostname((*ip).c_str());
    bool result;
    for (unsigned int i = 0; i < hostnames.size(); i++) {
        result = banned_ip_list.inList(hostnames[i].toCharArray());
        if (result) {
            return true;
        }
    }
    return false;
}


int OptionContainer::findoptionI(const char* option) {
    int res = String(findoptionS(option).c_str()).toInteger();
    return res;
}


std::string OptionContainer::findoptionS(const char* option) {
      // findoptionS returns a found option stored in the deque
    String temp;
    String temp2;
    String o = option;
    for (int i = 0; i < (signed)conffile.size(); i++) {
        temp = conffile[i].c_str();
        temp2 = temp.before("=");
        while(temp2.endsWith(" ")) { // get rid of tailing spaces before =
            temp2.chop();
        }
        if (o == temp2) {
            temp = temp.after("=");
            while(temp.startsWith(" ")) { // get rid of heading spaces
                temp.lop();
            }
            if(temp.startsWith("'")) { // inverted commas
                temp.lop();
            }
            while(temp.endsWith(" ")) { // get rid of tailing spaces
                temp.chop();
            }
            if(temp.endsWith("'")) { // inverted commas
                temp.chop();
            }
            return temp.toCharArray();
        }
    }
    return "";
}

bool OptionContainer::realitycheck(String s, int minl, int maxl, char* emessage) {
      // realitycheck checks a String for certain expected criteria
      // so we can spot problems in the conf files easier
    if ((signed)s.length() < minl) {
        if (!isDaemonised) {
            std::cerr << emessage << std::endl;
                                   // when called we have not detached from
                                   // the console so we can write back an
                                   // error

            std::cerr << "Too short or missing." << std::endl;
        }
        syslog(LOG_ERR, "%s", emessage);
        syslog(LOG_ERR, "%s", "Too short or missing.");

        return false;
    }
    if ((signed)s.length() > maxl && maxl > 0) {
        if (!isDaemonised) {
            std::cerr << emessage << std::endl;
            std::cerr << "Too long or broken." << std::endl;
        }
        syslog(LOG_ERR, "%s", emessage);
        syslog(LOG_ERR, "%s", "Too long or broken.");
        return false;
    }
    return true;
}



bool OptionContainer::readFilterGroupConf() {
    String prefix = conffilename;
    prefix = prefix.before(".conf");
    prefix += "f";
    String file;
    for(int i = 1; i <= filter_groups; i++){
        file = prefix + String(i);
        file += ".conf";
        if (!readAnotherFilterGroupConf(file.toCharArray())) {
            if (!isDaemonised) {
                std::cerr << "Error opening filter list:" << file << std::endl;
            }
            syslog(LOG_ERR, "%s","Error opening filter list:");
            syslog(LOG_ERR, "%s",file.toCharArray());
            return false;
        }
    }
    return true;
}

bool OptionContainer::readAnotherFilterGroupConf(const char *filename) {
    #ifdef DGDEBUG
        std::cout << "adding filter group" << numfg << " " << filename << std::endl;
    #endif

    // array of pointers to FOptionContainer
    typedef FOptionContainer* PFOptionContainer;
    FOptionContainer** temp = new PFOptionContainer[numfg + 1];
    for(int i = 0; i < numfg; i++) {
        temp[i] = fg[i];
    }
    if (numfg > 0) {
        delete[] fg;
    }
    fg = temp;
    fg[numfg] = new FOptionContainer;

    #ifdef DGDEBUG
        std::cout << "added filter group" << numfg << " " << filename << std::endl;
    #endif

    // pass all the vars from OptionContainer needed
    (*fg[numfg]).weighted_phrase_mode = weighted_phrase_mode;
    (*fg[numfg]).force_quick_search = force_quick_search;
    (*fg[numfg]).createlistcachefiles = createlistcachefiles;
    (*fg[numfg]).reverse_lookups = reverse_lookups;
    (*fg[numfg]).ada = ada;

    #ifdef DGDEBUG
        std::cout << "passed variables to filter group" << numfg << " " << filename << std::endl;
    #endif

    bool rc = (*fg[numfg]).read(filename);
    #ifdef DGDEBUG
        std::cout << "read filter group" << numfg << " " << filename << std::endl;
    #endif

    numfg++;

    if (!rc) {
        return false;
    }
    return true;
}


