/**
 * Regular expression compiler and Discrete Finite Automaton
 * simulator.
 * @author Shaun Jackman <sdj@sfu.ca>
 * @copyright Copyright 2004 Shaun Jackman
 */


#include <stdio.h>
#include "grammar.h"
#include "scan.h"
#include "util.h"


/** Returns the next token in the specified string, and stores its
 * length in the specified pointer. */
static int
next_token( int* plength, const DFA* dfa, FILE* file)
{
	int state = INITIAL_STATE, token = 0, length = 0;
	char c;
	*plength = 0;
	for( c = getc( file); !feof( file); c = getc( file)) {
		assumex( within( c, '\t', CHAR_MAX), "Non-ASCII character");
		state = dfa->transition[state][(int)c];
		if( state == NO_TRANSITION)
			break;
		length++;
		if( contains_element( &dfa->final, state)) {
			token = dfa->tokens[state];
			*plength = length;
		}
	}
	return token;
}


/** Scans the specified file. */
void
scan( const Grammar* grammar, FILE* file)
{
	int line = 1, column = 1;
	for(;;) {
		int i, length, token;
		fpos_t pos;

		fgetpos( file, &pos);
		token = next_token( &length, &grammar->dfa, file);
		fsetpos( file, &pos);
		if( length == 0)
			break;

		printf( "%-15s \"", grammar->tokens[token]);
		for( i = 0; i < length; i++) {
			char c = getc( file);
			print_escaped_char( c, "\"");
			count_lines_columns( &line, &column, c);
		}
		puts( "\"");
	}
	assumex( getc( file) == EOF,
			"illegal token at line %d column %d", line, column);
}
