/**
 * Utility functions.
 * @author Shaun Jackman <sdj@sfu.ca>
 * @copyright Copyright 2004 Shaun Jackman
 */


#include <errno.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "util.h"


/** Counts the number of lines and columns seen. */
void
count_lines_columns( int* line, int* column, char c)
{
	switch( c) {
		case '\n':
			*line += 1;
			*column = 1;
			break;
		case '\t':
			*column += 4;
			break;
		default:
			*column += 1;
	}
}


/** Returns an escaped version of the given character. */
const char*
escape( char c)
{
	switch( c) {
		case '\a': return( "\\a");
		case '\b': return( "\\b");
		case '\t': return( "\\t");
		case '\n': return( "\\n");
		case '\v': return( "\\v");
		case '\f': return( "\\f");
		case '\r': return( "\\r");
		case '\\': return( "\\\\");
		default: return NULL;
	}
}


/** Returns an unescaped version of the given character. */
char
unescape( char c)
{
	switch( c) {
		case 'a': return '\a';
		case 'b': return '\b';
		case 't': return '\t';
		case 'n': return '\n';
		case 'v': return '\v';
		case 'f': return '\f';
		case 'r': return '\f';
		case '\\': return '\\';
		default: die( "unknown escape sequence '\\%c'", c);
	}
}


/** Unescapes the specified string. */
char*
unescape_string( char* str)
{
	char* dst = str;
	const char* src = str;
	char c;
	while( (c = *src++) != '\0')
		*dst++ = c != '\\' ? c : unescape( *src++);
	*dst++ = '\0';
	return dst;
}


/** Prints a character, escaping control characters. */
void
print_char( char c)
{
	const char* str = escape( c);
	if( str)
		printf( "%s", str);
	else
		putchar( c);
}


/** Prints an escaped character. */
void
print_escaped_char( char c, const char* escapes)
{
	if( strchr( escapes, c) != NULL)
		putchar( '\\');
	print_char( c);
}


/** Prints an string, escaping control characters. */
void
print_string( const char* string)
{
	while( *string)
		print_char( *string++);
}


/** Prints an escaped string. */
void
print_escaped_string( const char* string, const char* escapes)
{
	while( *string)
		print_escaped_char( *string++, escapes);
}


/** Returns a potentially escaped character. */
char
read_char( FILE* file)
{
	char c = getc( file);
	return c != '\\' ? c : unescape( getc( file));
}


/** Reads an escaped string until one of the specified delimiters and
 * allocates memory for it on the heap. The delimiter is not extracted
 * from the input. */
char*
read_string( FILE* file, const char* delimiters)
{
	int max = 1;
	char* str = allocate_memory( max);
	int count = 0;
	for(;;) {
		char c = getc( file);
		if( c == EOF) {
			free( str);
			return NULL;
		}
		if( strchr( delimiters, c) != NULL) {
			ungetc( c, file);
			break;
		}
		if( c == '\\') {
			c = getc( file);
			if( strchr( delimiters, c) == NULL)
				c = unescape( c);
		}
		if( count+1 > max) {
			max *= 2;
			str = reallocate_memory( str, max);
		}
		str[count++] = c;
	}
	str = reallocate_memory( str, count+1);
	str[count] = '\0';
	return str;
}


/** Returns true if the specified value is within the specified range.
 */
int
within( int x, int low, int hi)
{
	return low <= x && x <= hi;
}


/** Prints the specified error message. */
void
vprint_error( const char* format, va_list args)
{
	fflush( stdout);
	fprintf( stderr, "%s: ", program);
	vfprintf( stderr, format, args);
}


/** Prints the specified error message. */
void
print_error( const char* format, ...)
{
	va_list args;
	va_start( args, format);
	vprint_error( format, args);
	va_end( args);
}


/** Prints the specified error message and exits. */
void
die( const char* format, ...)
{
	va_list args;
	va_start( args, format);
	vprint_error( format, args);
	va_end( args);
	fputc( '\n', stderr);
	exit( EXIT_FAILURE);
}


/** If the specified assumption isn't true, print the specified error
 * message, system error message, and exit. */
void
assume( bool assumption, const char* format, ...)
{
	va_list args;
	va_start( args, format);
	if( !assumption) {
		vprint_error( format, args);
		fprintf( stderr, ": %s\n", strerror( errno));
		exit( EXIT_FAILURE);
	}
	va_end( args);
}


/** If the specified assumption isn't true, print the specified error
 * message, and exit. */
void
assumex( bool assumption, const char* format, ...)
{
	va_list args;
	va_start( args, format);
	if( !assumption) {
		fflush( stdout);
		vprint_error( format, args);
		putc( '\n', stderr);
		exit( EXIT_FAILURE);
	}
	va_end( args);
}


/** If the specified assumption isn't true, print the specified error
 * message, the next string in the specified file, and exit. */
void
assume_near( bool assumption, FILE* file, const char* format, ...)
{
	va_list args;
	va_start( args, format);
	if( !assumption) {
		char* string;
		fflush( stdout);
		vprint_error( format, args);
		fscanf( file, "%as", &string);
		fprintf( stderr, " near %s\n", string);
		exit( EXIT_FAILURE);
	}
	va_end( args);
}


/** Opens the specified file for reading. */
FILE*
open_file( const char* file_name)
{
	FILE* file = fopen( file_name, "r");
	assume( file != NULL, "%s", file_name);
	return file;
}


/** Allocates the specified number of bytes and returns a pointer to
 * the allocated memory. */
void*
allocate_memory( size_t size)
{
	void* ptr = malloc( size);
	assume( ptr != NULL, "unable to allocate memory");
	return ptr;
}


/** Reallocates the specified number of bytes and returns a pointer to
 * the reallocated memory. */
void*
reallocate_memory( void* ptr, size_t size)
{
	ptr = realloc( ptr, size);
	assume( ptr != NULL, "unable to reallocate memory");
	return ptr;
}
