/*   mp3db.c -- routines for storing mp3 info in a mysql database
 *
 * Copyright (c) 1998-2002  Mike Oliphant <oliphant@gtk.org>
 *
 *   http://www.nostatic.org/grip
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <glib.h>
#include <mysql.h>
#include "ddj.h"
#include "mp3db.h"

static gboolean GotData(MYSQL *mysql);

char sql_host[61];
char sql_user[17];
char sql_pswd[17];
char sql_dbname[33];

MYSQL *ConnectSQL(void)
{
  MYSQL *mysql;

  mysql=mysql_init(NULL);

  if(!(mysql_real_connect(mysql,sql_host,sql_user,sql_pswd,sql_dbname,
			  3306,NULL,0))) {
    printf("SQL error %d: %s\n",mysql_errno(mysql),mysql_error(mysql));

    return NULL;
  }

  return mysql;
}

void DisconnectSQL(MYSQL *mysql)
{
  mysql_close(mysql);
}

gboolean SQLQuery(MYSQL *mysql,char *query)
{
  mysql_query(mysql,query);

  return TRUE;
}

gboolean SingleIntQuery(MYSQL *mysql,char *query,int *result)
{
  MYSQL_RES *res=NULL;
  MYSQL_ROW row;

  if(mysql_query(mysql,query)||
     !(res = mysql_store_result(mysql))) {
    printf("Query error\n");
    
    return FALSE;
  }

  if(!mysql_num_rows(res)) {
    mysql_free_result(res);

    return FALSE;
  }

  row=mysql_fetch_row(res);

  *result=atoi(row[0]);

  mysql_free_result(res);

  return TRUE;
}

/* Get version info from the database */
gboolean GetDBInfo(MYSQL *mysql,int *db_version)
{
  MYSQL_RES *res=NULL;
  MYSQL_ROW row;

  if(mysql_query(mysql,"SELECT db_version FROM db_info")||
     !(res = mysql_store_result(mysql))) {
    printf("Query error\n");
    
    return FALSE;
  }

  if(!mysql_num_rows(res)) {
    mysql_free_result(res);

    return FALSE;
  }

  row=mysql_fetch_row(res);

  *db_version=atoi(row[0]);

  mysql_free_result(res);

  return TRUE;
}

/* Execute the contents of a database schema file */
gboolean ExecuteDBSchema(char *schemafile,MYSQL *mysql,char *rootpswd)
{
  FILE *sfile;
  char buf[1024];
  GString *sql_cmd;
  int len;

  if(!(mysql_real_connect(mysql,sql_host,"root",rootpswd,NULL,3306,NULL,0))) {
    printf("InitSQL(): Cannot connect to server: error %d: %s\n",
           mysql_errno(mysql),
           mysql_error(mysql));
 
    return FALSE;
  }

  sprintf(buf,"CREATE DATABASE `%s`",sql_dbname);

  if(mysql_query(mysql,buf))
    printf("InitSQL(): Cannot create database: error %d: %s\n",
           mysql_errno(mysql),
           mysql_error(mysql));

  if(mysql_select_db(mysql,sql_dbname)) {
    printf("SQL error: Unable to access %s database\n",sql_dbname);

    mysql_close(mysql);

    return FALSE;
  }

  sfile=fopen(schemafile,"r");

  if(!sfile) {
    printf("Unable to open schema file [%s]\n",schemafile);

    mysql_close(mysql);

    return FALSE;
  }

  sql_cmd=g_string_new(NULL);

  while(fgets(buf,1024,sfile)) {
    if(*buf!='#') {
      /* Check for the end of a command */
      len=strlen(buf);

      if(buf[len-1]=='\n') {
	len--;

	buf[len]=' ';
      }

      if(buf[len-1]==';') {
	buf[len-1]=' ';

	g_string_append(sql_cmd,buf);

	if(mysql_query(mysql,sql_cmd->str))
	  printf("ERROR: DB command failed\n");

	g_string_truncate(sql_cmd,0);
      }
      else g_string_append(sql_cmd,buf);
    }
  }

  g_string_free(sql_cmd,TRUE);

  mysql_close(mysql);

  return TRUE;
}

gboolean ReloadSQL(MYSQL *mysql,char *rootpswd)
{
  if(!(mysql_real_connect(mysql,sql_host,"root",rootpswd,NULL,3306,NULL,0))) {
    printf("ReloadSQL(): Could not connect: error %d: %s\n",
           mysql_errno(mysql),
           mysql_error(mysql));

    return FALSE;
  }

  if(mysql_query(mysql,"FLUSH PRIVILEGES")) {
    printf("ReloadSQL(): Could not flush privileges: error %d: %s\n",
           mysql_errno(mysql),
           mysql_error(mysql));
    return FALSE;
  }

  mysql_close(mysql);

  return TRUE;
}

gboolean MakeSQLUser(MYSQL *mysql,char *user,char *host,char *pswd,
		     char *rootpswd)
{
  char query[256];

  if(!(mysql_real_connect(mysql,sql_host,"root",rootpswd,"mysql",
			  3306,NULL,0))) {
    printf("MakeSQLUser(): Could not connect to database: error %d: %s\n",
           mysql_errno(mysql),
	   mysql_error(mysql));

    return FALSE;
  }

  snprintf(query,256,"INSERT INTO user (host,user,password) values('%s'"
	   ",'%s',password('%s'))",host,user,pswd);

  mysql_query(mysql,query);

  snprintf(query,256,"INSERT INTO db (host,db,user,Select_priv,Insert_priv,"
	   "Update_priv,Delete_priv) values ('%s','%s','%s','Y','Y','Y','Y')",
	   host,sql_dbname,user);

  mysql_query(mysql,query);

  mysql_close(mysql);

  return TRUE;
}

gboolean DeleteSQLUser(MYSQL *mysql,char *user,char *host,char *rootpswd)
{
  char query[256];

  if(!(mysql_real_connect(mysql,sql_host,"root",rootpswd,"mysql",
			  3306,NULL,0))) {
    printf("DeleteSQLUser(): Could not connect to database: error %d: %s\n",
           mysql_errno(mysql),
           mysql_error(mysql));
    return FALSE;
  }

  snprintf(query,256,"DELETE FROM user WHERE Host='%s' AND User='%s'",
	   host,user);

  mysql_query(mysql,query);

  snprintf(query,256,"DELETE FROM db WHERE Db='%s' AND HOST='%s' AND "
	   "USER='%s'",
	   sql_dbname,host,user);

  mysql_query(mysql,query);

  mysql_close(mysql);

  return TRUE;
}

void EscapeQuote(char *in,char *out,int maxlen)
{
  int inpos,outpos;

  for(inpos=outpos=0;outpos<maxlen;inpos++,outpos++) {
    if(!in[inpos]) {
      out[outpos]='\0';

      return;
    }

    if(in[inpos]=='\'') {
      out[outpos++]='\\';

      if(outpos==maxlen) return;
    }

    out[outpos]=in[inpos];
  }

  return;
}

int AddSQLArtist(MYSQL *mysql,char *artist)
{
  char query[1024];
  MYSQL_RES *res;
  MYSQL_ROW row;
  int id;

  snprintf(query,1024,"SELECT * FROM artist WHERE artist='%.80s'",artist);

  if(mysql_query(mysql,query)) {
    printf("SQL error: Artist query failed\n");

    return -1;
  }
  
  if((res=mysql_store_result(mysql))&&
     (row=mysql_fetch_row(res)))
    id=atoi(row[0]);
  else {
    snprintf(query,1024,"INSERT INTO artist (id,artist,num_plays,last_play,"
	     "date_added) VALUES (NULL,'%.80s',0,NULL,NOW())",artist);

    if(mysql_query(mysql,query)) {
      printf("SQL error: Artist insert failed\n");

      id=-1;
    }
    else id=mysql_insert_id(mysql);
  }

  if(res) mysql_free_result(res);

  return id;
}

int AddSQLDisc(MYSQL *mysql,char *disc,int artistid,char *genre,int year)
{
  char query[1024];
  int left=1024;
  MYSQL_RES *res;
  MYSQL_ROW row;
  int id;

  left-=snprintf(query,1024,
		 "SELECT * FROM disc WHERE title='%.80s' AND artistid=%d",
		 disc,artistid);

  if(mysql_query(mysql,query)) {
    printf("SQL error: Disc query failed\n");

    return -1;
  }
  
  if((res=mysql_store_result(mysql))&&
     (row=mysql_fetch_row(res)))
    id=atoi(row[0]);
  else {
    snprintf(query,1024,
	     "INSERT INTO disc (id,title,artistid,num_plays,last_play,"
	     "date_added,genre,year) VALUES (NULL,'%.80s',%d,0,NULL,NOW(),",
	     disc,artistid);

    if(*genre)
      left-=snprintf(query+strlen(query),left,"'%.20s',",genre);
    else left-=snprintf(query+strlen(query),left,"NULL,");

    if(year)
      left-=snprintf(query+strlen(query),left,"%d)",year);
    else 
      left-=snprintf(query+strlen(query),left,"NULL)");

    printf("query is %s\n",query);

    if(mysql_query(mysql,query)) {
      printf("SQL error: Disc insert failed\n");

      id=-1;
    }
    else id=mysql_insert_id(mysql);
  }

  if(res) mysql_free_result(res);

  return id;
}

static gboolean GotData(MYSQL *mysql)
{
  MYSQL_RES *res;
  MYSQL_ROW row;

  if((res=mysql_store_result(mysql))&&
     (row=mysql_fetch_row(res))) {
    if(res) mysql_free_result(res);

    return TRUE;
  }

  if(res) mysql_free_result(res);

  return FALSE;
}

int AddSQLEntry(MYSQL *mysql,
		char *filename,char *artist,char *sartist,char *title,
		char *disc_name,char *genre,int year,
		int track_num,int start_frame,int num_frames,int mins,
		int secs,int bpm,int gain)
{
  char query[1024];
  int left=1024;
  char tfilename[257],tartist[81],tsartist[81],ttitle[81],tdisc_name[81],
    tgenre[21],
    play_time[8];
  int artistid,discid;
  int sartistid;
  int id;

  EscapeQuote(filename,tfilename,256);
  EscapeQuote(artist,tartist,80);
  EscapeQuote(sartist,tsartist,80);
  EscapeQuote(title,ttitle,80);
  EscapeQuote(disc_name,tdisc_name,80);
  EscapeQuote(genre,tgenre,20);

  artistid=AddSQLArtist(mysql,tartist);

  if(*tsartist)
    sartistid=AddSQLArtist(mysql,tsartist);

  if(*tdisc_name)
    discid=AddSQLDisc(mysql,tdisc_name,artistid,tgenre,year);
  else discid=-1;

  if(discid!=-1)
    snprintf(query,1024,"SELECT * from song WHERE title='%.80s' AND "
	     "artistid=%d AND discid=%d",ttitle,
	     (*tsartist)?sartistid:artistid,discid);
  else
    snprintf(query,1024,"SELECT * from song WHERE title='%.80s' AND "
	     "artistid=%d AND discid is NULL",ttitle,
	     (*tsartist)?sartistid:artistid);

  if(mysql_query(mysql,query)) {
    printf("SQL error: Song query failed\n");

    return -1;
  }

  if(GotData(mysql)) {
    printf("MP3 already in database\n");

    return -1;
  }

  if((mins!=-1)&&(secs!=-1))
    snprintf(play_time,8,"'%02d:%02d'",mins,secs);
  else strcpy(play_time,"NULL");

  if(*tsartist)
    left-=snprintf(query,left,"INSERT INTO song "
		   "(id,date_added,filename,title,artistid,discid,genre,"
		   "track_num,start_frame,num_frames,play_time,bpm,num_plays,"
		   "last_play,year,gain_adjustment) VALUES "
		   "(NULL,NOW(),'%.256s','%.80s',%d,",
		   tfilename,ttitle,sartistid);
  else
    left-=snprintf(query,left,"INSERT INTO song "
		   "(id,date_added,filename,title,artistid,discid,genre,"
		   "track_num,start_frame,num_frames,play_time,bpm,num_plays,"
		   "last_play,year,gain_adjustment) VALUES "
		   "(NULL,NOW(),'%.256s','%.80s',%d,",
		   tfilename,ttitle,artistid);

  if(discid!=-1)
    left-=snprintf(query+strlen(query),left,"%d,",discid);
  else left-=snprintf(query+strlen(query),left,"NULL,");

  if(*tgenre)
    left-=snprintf(query+strlen(query),left,"'%.20s',",tgenre);
  else left-=snprintf(query+strlen(query),left,"NULL,");
  
  if(track_num!=-1)
    left-=snprintf(query+strlen(query),left,"%d,",track_num);
  else left-=snprintf(query+strlen(query),left,"NULL,");

  if(start_frame!=-1)
    left-=snprintf(query+strlen(query),left,"%d,",start_frame);
  else left-=snprintf(query+strlen(query),left,"NULL,");

  if(num_frames!=-1)
    left-=snprintf(query+strlen(query),left,"%d,",num_frames);
  else left-=snprintf(query+strlen(query),left,"NULL,");

  left-=snprintf(query+strlen(query),left,"%s,",play_time);

  if(bpm!=-1)
    left-=snprintf(query+strlen(query),left,"%d",bpm);
  else left-=snprintf(query+strlen(query),left,"NULL");

  left-=snprintf(query+strlen(query),left,",0,NULL");

  if(year)
    left-=snprintf(query+strlen(query),left,",%d",year);
  else
    left-=snprintf(query+strlen(query),left,",NULL");

  
  if(gain)
    left-=snprintf(query+strlen(query),left,",%d)",gain);
  else
    left-=snprintf(query+strlen(query),left,",NULL)");

  printf("Query is [%s]\n",query);

  if(mysql_query(mysql,query)) {
    printf("SQL error: Insert failed\n");
    id=-1;
  }
  else id=mysql_insert_id(mysql);

  return id;
}

gboolean DeleteSongByID(MYSQL *mysql,int id)
{
  char query[80];
  gboolean success=FALSE;

  snprintf(query,80,"DELETE from song where id=%d",id);

  if(mysql_query(mysql,query)) {
    printf("SQL error: Delete failed\n");
  }
  else success=TRUE;

  return success;
}

int DBNum(char *str)
{
  if(!str) return -1;
  else return atoi(str);
}

void DBString(char *dest,char *src,int len)
{
  if(!src||!*src) *dest='\0';
  else strncpy(dest,src,len);
}

