/**
 * Abstract base class for FileList builders
 *
 * @see DirScanner
 * @see LinkScanner
 *
 * @author Mark Howard, Ben Secrest &lt;blsecres@users.sourceforge.net&gt;
 * @version $Id: FileListBuilder.java,v 1.16 2002/09/29 05:42:20 blsecres Exp $
 */
abstract public class FileListBuilder {
    /** Filters for files to be included.  If empty, no filters will be used */
    private String[] includeFilters;

    /** Filters for files to be excluded */
    private String[] excludeFilters;

    /** List to which file should be appended */
    private FileList list;

    /** Object for logging */
    protected IGLog log;

    /** Object for parsing files */
    protected ParserChooser parser;

    /** Default log level filter */
    protected static final int LOGLEVEL = 9;


    /**
     * Constructor
     * @param logObj The object ot use for logging
     * @param parserChooser The parser to use for extracting info from files
     * @param incFilters Include filters
     * @param excFilters Exclude filters
     * @param targetList Object to build FileList in
     */
    public FileListBuilder(IGLog logObj, ParserChooser parserChooser,
	    String[] incFilters, String[] excFilters, FileList targetList) {
        log = logObj;
        if (LOGLEVEL >= IGLog.PROCEDURE)
            log.add(IGLog.PROCEDURE, "FileListBuilder.FileListBuilder(IGLog, "
                    + "String[], String[], FileList)");
        includeFilters = incFilters;
        excludeFilters = excFilters;
        list = targetList;
        parser = parserChooser;
    }


    /**
     * Scan a list of files or directories
     * @param scans List to be scanned
     */
    public void scan(String[] scans) throws java.io.IOException,
	    IllegalVariableException {
        if (LOGLEVEL >= IGLog.PROCEDURE)
            log.add(IGLog.PROCEDURE, "FileListBuilder.scan(String[])");

        if (scans.length < 1)
	    log.addWarning(110, "FLB_NO_ITEMS", null);

        for (int i = 0; i < scans.length; i++)
            scan(scans[i]);
    }


    /**
     * Abstract method which must be overridden.  This method should scan the
     * given file/directory adding any files found which pass the filters to
     * the file list.
     * @param file The file or directory to scan
     */
    abstract public void scan(String file) throws java.io.IOException,
	    IllegalVariableException;

    /**
     * Attempt to add an IGFile to the file list
     * @param file The IGFile object to add
     * @return True if file was added false otherwise
     */
    protected void addFile(IGFile file) throws IllegalVariableException {
        if (LOGLEVEL >= IGLog.DEBUG)
            log.addResource(IGLog.DEBUG, "FLB_CONSIDER",
                            new String[]{file.getLocation()});
        if (filePassesFilters(file.getLocation())) {
	    list.addFile(file);
	}
    }


    /**
     * Attempt to add a file to the file list
     * @param fileName The name of the file to be added to the list
     */
    protected void addFile(String fileName) throws IllegalVariableException{
        if (LOGLEVEL >= IGLog.DEBUG)
            log.addResource(IGLog.DEBUG, "FLB_CONSIDER",
                            new String[]{fileName});
        if (filePassesFilters(fileName))
            list.addFile(new IGFile(fileName));
    }


    /**
     * Check a filename against the index include and exclude filters
     * @param filename The name of the file to run through the filters
     * @return True if the file passes the filters, false otherwise
     */
    protected boolean filePassesFilters(String fileName) {
        if (LOGLEVEL >= IGLog.RESULT)
            log.addResource(IGLog.RESULT, "FLB_SCAN_EXFILTER",
		    new Object[]{new Integer(excludeFilters.length)});

        // scan exclude filters
        for (int i = 0; i < excludeFilters.length; i++) {
            if (excludeFilters[i].equals("")) {
                continue;
            }
            if (LOGLEVEL >= IGLog.PROCEDURE)
                log.addResource(IGLog.PROCEDURE, "FLB_CHECK_EXFILTER",
                                new String[]{excludeFilters[i]});
            if (fileName.indexOf(excludeFilters[i]) > -1) {
                if (LOGLEVEL >= IGLog.PROGRESS)
                    log.addResource(IGLog.PROGRESS, "FLB_FILE_EXCLUDED",
                                    new String[]{fileName});
                return false;
            }
        }

        // No include filters == allow all
        if (includeFilters.length == 0) {
            return true;
        }

        // scan include filters
        for (int i = 0; i < includeFilters.length; i++) {
            if (fileName.indexOf(includeFilters[i]) > -1) {
                return true;
            }
        }

        if (LOGLEVEL >= IGLog.PROGRESS)
            log.addResource(IGLog.PROGRESS, "FLB_FILE_MATCH",
                            new Object[]{ fileName } );
		

        return false;
    }
}
