/*
 *  ---------
 * |.**> <**.|  CardContact
 * |*       *|  Software & System Consulting
 * |*       *|  Minden, Germany
 * |**> <**|  Copyright (c) 1999. All rights reserved
 *  --------- 
 *
 * See file LICENSE for details on licensing
 *
 * Abstract :       Defines tools/procedures ECO5000 reader
 *
 * Author :         Frank Thater (FTH)
 *
 * Last modified:   12/10/1999
 *
 *****************************************************************************/

#include <stdio.h>
#include <string.h>

#ifdef LIMITED
#include <time.h>
#endif

#include "sercom.h"
#include "eco5000.h"
#include "ctapi.h"
#include "defines.h"



/*
 * Send a command to the ECO 5000 / Carddrive 330 reader and wait for the
 * reponse. If <outbyteslen> and <outbytes> are defined, then data is
 * send to the reader after the command byte. If <inbytes> is defined, then
 * either the number of bytes specified in <expinbytes> is read or the number
 * indicated in the first byte of the reponse returned from the reader. To
 * activate the later case, <expinbytes> shall be set to 0
 *
 */

int ecoCommand (struct eco5000_t *ctx,
                unsigned char com,
                int outbyteslen,
                unsigned char *outbytes,
                int expinbytes,
                unsigned char *inbytes)

{
    int rc, retry;
    int i = 0;
    unsigned char temp;
   
    /* If the transparent mode is still switched on, then                    */
    /* we need to switch to command mode first                               */

    if(ctx->RXStatus == ON)  {
        if (com == RX_ON)
            return 0;

        rs232LineControl(ctx->fh, 0, 0);      /* Set DTR and RTS off     */
        
	if (ctx->Indirect)
	    rs232Mode(ctx->fh, -1, 'E', -1, -1, 200);
        
        ctx->RXStatus = OFF;

        if (com != RX_OFF) {
            rc = ecoCommand(ctx, RX_OFF, 0, NULL, 0, NULL);
            if (rc < 0)
                return rc;
        }
    }

    if (!((com == RX_ON) && (ctx->disableRX_ON))) {
	    
	retry = 3;                          /* Sometimes we get parity errors    */
	while (1) {
	    if (rs232Write(ctx->fh, &com, 1) <= 0) {  /* Write command byte      */
#ifdef DEBUG
		printf("[ecoCommand] Error writing command byte\n");
#endif
		return ERR_TRANS;
	    }
	    
	    temp = 0;
	    if (rs232Read(ctx->fh, &temp, 1) <= 0) {  /* Read acknowledge        */
#ifdef DEBUG
		printf("[ecoCommand] Error reading command acknowledge\n");
#endif
		return ERR_TRANS;
	    }
	    
	    if ((temp == NACK_PAR) && retry--) {
		rs232Drain(ctx->fh);
		rs232Flush(ctx->fh);
		continue;                   /* Retry if parity error             */
	    } else if (temp == NACK_CMD)
		return ERR_TRANS;
	    
	    if ((outbyteslen != 0) && (outbytes != NULL)) {
		for (i = 0; i < outbyteslen; i++)
		    if (rs232Write(ctx->fh, &outbytes[i], 1) <=0)
			return ERR_TRANS;  
		
		temp = 0;
		if (rs232Read(ctx->fh, &temp, 1) <= 0) {
#ifdef DEBUG
		    printf("[ecoCommand] Error reading command acknowledge\n");
#endif
		    return ERR_TRANS;
		}
		
            if ((temp == NACK_PAR) && retry--) {
                rs232Drain(ctx->fh);
                rs232Flush(ctx->fh);
                continue;
            } else if (temp == NACK_CMD)
                return ERR_TRANS;
	    }
	    break;
	} /* while (1) */
    

	/* If we expect data from the reader, then inbytes points to a buffer    */
	/* If the reader return a variable number of bytes, we set expinbytes    */
	/* to zero. In the later case, the first byte transmitted is the number  */
	/* of byte following in the response                                     */
	
	if (inbytes != NULL) {
	    if (!expinbytes) {
		if (rs232Read(ctx->fh, inbytes, 1) < 1)
		    return ERR_CT;
		expinbytes = *inbytes++;
	    }
	    
	    if (rs232Read(ctx->fh, inbytes, expinbytes) < expinbytes)
		return ERR_CT;
	}
    } /* if */
    
    /* After sending an RX_ON or SET_RESET_RX_ON we are back in transparent  */
    /* mode. Set RTS accordingly                                             */

    if ((com == RX_ON) || (com == SET_RESET_RX_ON)) {
        ctx->RXStatus = ON;
        if (ctx->Indirect) 
            rs232Mode(ctx->fh, -1, 'O', -1, -1, -1);	    
	
	
	rs232LineControl(ctx->fh, 0, 1);           /* Set RTS on */
 
    }
  
    return temp;
}



int ecoChangeBaudrate(struct eco5000_t *ctx, int baudrate)

{
    int rc;
    char buff;
    
    buff = 0xFF - 14318000 / (32 * baudrate); /* 115200 = 0xFC */

    if ((rc = ecoCommand(ctx, SET_BAUDRATE, 1, &buff, 0, NULL)) < 0)
        return rc;

    rs232Mode(ctx->fh, baudrate, 0, -1, -1, -1);

    ctx->Baud = baudrate;

    return 0;
}



int getFirmware (struct eco5000_t *ctx)

{
    int response;
    unsigned char buffer[FWSIZE + 2];
   
    if((response = ecoCommand(ctx, RESET, 0, NULL, 0, NULL)) <= 0)
        return response;

    if((response = ecoCommand(ctx, GET_VERSION, 0, NULL, 0, buffer)) <= 0) 
        return response;

    buffer[buffer[0] + 1] = '\0';
    strcpy(ctx->Firmware, buffer + 1);

#ifdef LIMITED
    {
        time_t t;
        struct tm *tmp;

        t = time(NULL);
        tmp = localtime(&t);

        if (tmp->tm_mon > LIMITED)
            return -1;
    }
#endif
         
    return response;
}




/*
 * Decode the APDU to determine the case of the command,
 * determine the length of outgoing data (lc), the outgoing
 * data and the length of expected data.
 *
 * If any of the parameter lc, data or le is set the NULL,
 * then the parameter is ignored.
 *
 * Return < 0 if APDU has error or
 *          1 for Case 1
 *          2 for Case 2S
 *          3 for Case 3S
 *          4 for Case 4S
 *          5 for Case 2E
 *          6 for Case 3E
 *          7 for Case 4E
 */

int DecodeAPDU(unsigned int len, unsigned char *cmd,
               unsigned int *lc, unsigned char **data, unsigned int *le)

{
    int apducase,ext;
    unsigned int l;

    ext = 0;                            /* Short is default                  */

    if (len < 4)                        /* Need at least 4 bytes             */
        return -1;

    cmd += 4;
    len -= 4;

    if (len == 0) {                     /* Case 1                            */
        apducase = 1;
    } else {
        l = 0;
        if (!*cmd && (len > 1)) {       /* Extended format                   */
            if (len < 3)                /* requires 3 byte lc field minimum  */
                return -1;

            ext = 3;
            cmd++;
            l = (unsigned int)*cmd << 8;
            cmd++;
            len -= 2;
        }
            
        l += *cmd;
        len--;
        cmd++;
                                        /* cmd and len is now at body part   */
        if (!len) {                     /* Case 2 short or extended          */
            apducase = 2 + ext;
            if (!l) {                   /* l is le, for which 0 means ...    */
                if (ext)
                    l = 65536;          /* ... either 65536 or ...           */
                else
                    l = 256;            /* ... 256 bytes expected            */
            }

            if (le)                     /* Parameter requested by caller ?   */
                *le = l;
        } else {                        /* Somethings left in the body       */
            if (len < l)                /* l is lc now                       */
                return -1;

            if (lc)                     /* Parameter requested by caller ?   */
                *lc = l;

            if (data)                   /* Parameter requested by caller ?   */
                *data = cmd;
            
            len -= l;                   /* Move past body                    */
            cmd += l;

            if (!len)                   /* Case 3 short or extended          */
                apducase = 3 + ext;
            else {                      /* Case 4, le is following           */
                apducase = 4 + ext;
                l = 0;
                if (ext) {              /* le is 3 byte in extended mode     */
                    if (len != 3)
                        return -1;

                    cmd++;
                    l = (unsigned int)*cmd << 8;
                    cmd++;
                    len -= 2;
                }

                if (len != 1)
                    return -1;
                
                l += (unsigned int)*cmd;

                if (!l) {               /* 0 on le means maximum, which ...  */
                    if (ext)
                        l = 65536;      /* is 65536 in extended and ...      */
                    else
                        l = 256;        /* 256 in short mode                 */
                }

                if (le)                 /* le requested by caller ?          */
                    *le = l;
            }
        }
    }
    return apducase;
}



/*
 * Invert a region of memory by swapping the bit order and sign
 *
 */

void invert(unsigned char *buff, int len)

{
    unsigned char o,n;
    int i;

    while (len--) {
        o = *buff;

        n = 0;
        for (i = 8; i; i--) {
            n <<= 1;
            n |= (o & 1) ^ 1;
            o >>= 1;
        }

        *buff = n;
        buff++;
    }
}



/*
 * Read a block of data from the ICC and invert if in indirect mode
 *
 */

int iccRead(HANDLE fh, int inverse, unsigned char *buff, int len)

{
    int rc;
    
    rc = rs232Read(fh, buff, len);

    if (rc <= 0)
        return rc;

    if (inverse)
        invert(buff, rc);

    return rc;
}



/*
 * Write a block of data to the ICC and invert before, if in indirect mode
 *
 */

int iccWrite(HANDLE fh, int inverse, unsigned char *buff, int len)

{
    unsigned char tmp[MAX_APDULEN];

    if (inverse) {
        memcpy(tmp, buff, len);
        invert(tmp, len);
        return rs232Write(fh, tmp, len);
    } else 
        return rs232Write(fh, buff, len);
}



