/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-mime-tools.c,v 1.8 2003/12/15 16:08:53 hoa Exp $
 */

#include "etpan-mime-tools.h"
#include <libetpan/libetpan.h>
#if 0
#include "etpan-message-data-driver.h"
#endif
#include "etpan-errors.h"
#if 0
#include "etpan-security.h"
#endif
#include <stdlib.h>
#include <string.h>
#include <libgen.h>
#include "etpan-cfg-mime.h"
#include "etpan-tools.h"
#include "etpan-imf-helper.h"
#include "etpan-msg-new.h"
#include <unistd.h>
#include <unistd.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

/*
  etpan_mime_data_replace()
  
  write a mime part to a file and change the reference of mailmime_data
  to the file.
*/

static int etpan_mime_data_replace(int encoding_type,
    struct mailmime_data * data)
{
  char filename[PATH_MAX];
  FILE * f;
  size_t written;
  char * dup_filename;
  int res;
  int r;
  int decoded;

  if (data->dt_type != MAILMIME_DATA_TEXT) {
    res = NO_ERROR;
    goto err;
  }
  
  f = etpan_get_mime_tmp_file(filename, sizeof(filename));
  if (f == NULL) {
    res = ERROR_FILE;
    goto err;
  }
  
  decoded = 0;
  if (encoding_type != -1) {
    char * content;
    size_t content_len;
    size_t cur_token;
    
    cur_token = 0;
    r = mailmime_part_parse(data->dt_data.dt_text.dt_data,
        data->dt_data.dt_text.dt_length,
        &cur_token, encoding_type, &content, &content_len);
    
    if (r == MAILIMF_NO_ERROR) {
      /* write decoded */
      written = fwrite(content, 1, content_len, f);
      if (written != content_len) {
        fclose(f);
        unlink(filename);
        res = ERROR_FILE;
        goto err;
      }
      mmap_string_unref(content);
      
      decoded = 1;
      data->dt_encoded = 0;
    }
  }
  
  if (!decoded) {
    written = fwrite(data->dt_data.dt_text.dt_data, 1,
        data->dt_data.dt_text.dt_length, f);
    if (written != data->dt_data.dt_text.dt_length) {
      fclose(f);
      unlink(filename);
      res = ERROR_FILE;
      goto err;
    }
  }
  
  fclose(f);
  
  dup_filename = strdup(filename);
  if (dup_filename == NULL) {
    unlink(filename);
    res = ERROR_MEMORY;
    goto err;
  }
  
  data->dt_type = MAILMIME_DATA_FILE;
  data->dt_data.dt_filename = dup_filename;
  
  return NO_ERROR;
  
 err:
  return res;
}


/*
  recursive_replace_single_parts()
  
  write all parts of the given mime part to file.
*/

static int recursive_replace_single_parts(struct etpan_app * app,
    struct mailmime * mime)
{
  int r;
  int res;
  clistiter * cur;
  
  mime->mm_mime_start = NULL;
  
  switch(mime->mm_type) {
  case MAILMIME_SINGLE:
    if (mime->mm_data.mm_single != NULL) {
      int encoding_type;
      struct mailmime_single_fields single_fields;
      
      mailmime_single_fields_init(&single_fields, mime->mm_mime_fields,
          mime->mm_content_type);
      
      if (single_fields.fld_encoding != NULL)
        encoding_type = single_fields.fld_encoding->enc_type;
      else
        encoding_type = -1;
      
      r = etpan_mime_data_replace(encoding_type, mime->mm_data.mm_single);
      if (r != NO_ERROR) {
        res = r;
        goto err;
      }
    }
    break;
    
  case MAILMIME_MULTIPLE:
    if (mime->mm_data.mm_multipart.mm_preamble != NULL) {
      r = etpan_mime_data_replace(-1, mime->mm_data.mm_multipart.mm_preamble);
      if (r != NO_ERROR) {
        res = r;
        goto err;
      }
    }
    
    if (mime->mm_data.mm_multipart.mm_epilogue != NULL) {
      r = etpan_mime_data_replace(-1, mime->mm_data.mm_multipart.mm_epilogue);
      if (r != NO_ERROR) {
        res = r;
        goto err;
      }
    }
    
    for(cur = clist_begin(mime->mm_data.mm_multipart.mm_mp_list) ;
        cur != NULL ; cur = clist_next(cur)) {
      struct mailmime * child;
      
      child = cur->data;
      
      r = recursive_replace_single_parts(app, child);
      if (r != NO_ERROR) {
        res = r;
        goto err;
      }
    }
    
    break;
    
  case MAILMIME_MESSAGE:
    if (mime->mm_data.mm_message.mm_msg_mime != NULL) {
      r = recursive_replace_single_parts(app,
          mime->mm_data.mm_message.mm_msg_mime);
      if (r != NO_ERROR) {
        res = r;
        goto err;
      }
    }
    
    break;
  }
  
  return NO_ERROR;
  
 err:
  return res;
}

#if 0
/*
  etpan_get_mime()
  
  parse the message in MIME structure,
  all single MIME parts are stored in files.
  
  folder is the folder of the account.
  This is NOT the folder of any message.
*/

int etpan_get_mime(struct etpan_app * app, struct mailfolder * folder,
    char * content, size_t content_len, int check_security,
    struct mailmime ** result_mime)
{
  struct mailmime * mime;
  mailmessage * msg;
  int r;
  int res;
  
  /*
    use message data driver, get bodystructure and
    convert all the data part in MAILMIME_SINGLE to files.
  */
  
  msg = etpan_message_data_init(content, content_len);
  if (msg == NULL) {
    res = ERROR_MEMORY;
    goto err;
  }
  
  if (msg->mime == NULL) {
#if 0
      r = mailmessage_get_bodystructure(msg, &mime);
#endif
    if (check_security) {
      r = etpan_msg_get_bodystructure(app, folder, msg, &mime);
    }
    else {
      /*
        don't use etpan_msg_get_bodystructure because it is not useful
        and to avoid loops due to security part
      */
      r = mailmessage_get_bodystructure(msg, &mime);
    }
    if (r != MAIL_NO_ERROR) {
      res = r;
      goto free_msg;
    }
    mime = msg->mime;
  }
  else {
    mime = msg->mime;
  }

  r = recursive_replace_single_parts(app, mime);
  if (r != NO_ERROR) {
    res = r;
    goto clear_mime;
  }
  
  etpan_message_data_detach_mime(msg);
  etpan_msg_flush(app, folder, msg);
  mailmessage_free(msg);
  
  * result_mime = mime;
  
  return NO_ERROR;
  
 clear_mime:
  etpan_message_mime_clear(mime);
 free_msg:
  mailmessage_flush(msg);
  mailmessage_free(msg);
 err:
  return res;
}
#endif


struct mailmime *
etpan_mime_new_file_part(struct etpan_app * app,
    int use_filename, char * filename,
    char * default_content_type, int default_encoding)
{
  char basename_buf[PATH_MAX];
  char * name;
  struct mailmime_disposition * disposition;
  char * disposition_name;
  struct etpan_mime_info * mime_info;
  struct mailmime_mechanism * encoding;
  struct mailmime_content * content;
  struct mailmime * mime;
  size_t cur_token;
  int r;
  char * dup_filename;
  struct mailmime_fields * mime_fields;
  int encoding_type;
  char * content_type_str;
  int do_encoding;
  
  if (filename != NULL) {
    strncpy(basename_buf, filename, PATH_MAX);
    name = basename(basename_buf);
  }
  else {
    name = NULL;
  }
  
  disposition = NULL;
  if (use_filename) {
    disposition_name = strdup(name);
    if (disposition_name == NULL) {
      goto err;
    }
    
    disposition =
      mailmime_disposition_new_with_data(MAILMIME_DISPOSITION_TYPE_ATTACHMENT,
          disposition_name, NULL, NULL, NULL, (size_t) -1);
    if (disposition == NULL) {
      free(disposition_name);
      goto err;
    }
  }
  
  content = NULL;
  encoding = NULL;
  
  if (name != NULL) {
    mime_info = etpan_get_mime_by_name(app->config.mime_config, name);
    if (mime_info != NULL) {
      /* Content-Type */
      if (mime_info->type != NULL) {
        content = mailmime_content_new_with_str(mime_info->type);
        /* ignore if failed */
      }
    
      /* Content-Transfer-Encoding */
      cur_token = 0;
      r = mailmime_encoding_parse(mime_info->encoding,
          strlen(mime_info->encoding), &cur_token, &encoding);
      /* ignore if failed */
    }
  }
  
  /* default content-type */
  if (content == NULL) {
    if (default_content_type == NULL)
      content_type_str = "application/octet-stream";
    else
      content_type_str = default_content_type;
    
    content = mailmime_content_new_with_str(content_type_str);
    if (content == NULL) {
      goto free_content;
    }
    
    /* particular case of text/plain */
    if (strcasecmp(content_type_str, "text/plain") == 0) {
      struct mailmime_parameter * param;
      
      param = mailmime_param_new_with_data("charset",
          app->config.global_config->editor_charset);
      if (param == NULL) {
        goto free_content;
      }
      
      r = clist_append(content->ct_parameters, param);
      if (r < 0) {
        mailmime_parameter_free(param);
        goto free_content;
      }
    }
  }
  
  do_encoding = 1;
  if (content->ct_type->tp_type == MAILMIME_TYPE_COMPOSITE_TYPE) {
    struct mailmime_composite_type * composite;
    
    composite = content->ct_type->tp_data.tp_composite_type;
    
    switch (composite->ct_type) {
    case MAILMIME_COMPOSITE_TYPE_MESSAGE:
      if (strcasecmp(content->ct_subtype, "rfc822") == 0)
        do_encoding = 0;
      break;

    case MAILMIME_COMPOSITE_TYPE_MULTIPART:
      do_encoding = 0;
      break;
    }
  }
  
  if (do_encoding && (encoding == NULL)) {
    if (default_encoding == -1)
      encoding_type = MAILMIME_MECHANISM_BASE64;
    else
      encoding_type = default_encoding;
    
    /* default Content-Transfer-Encoding */
    encoding = mailmime_mechanism_new(encoding_type, NULL);
    if (encoding == NULL) {
      goto free_content;
    }
  }
  
  mime_fields = mailmime_fields_new_with_data(encoding,
      NULL, NULL, disposition, NULL);
  if (mime_fields == NULL) {
    goto free_content;
  }
  
  mime = mailmime_new_empty(content, mime_fields);
  if (mime == NULL) {
    goto free_mime_fields;
  }
  
  if ((filename != NULL) && (mime->mm_type == MAILMIME_SINGLE)) {
    /*
      duplicates the file so that the file can be deleted when
      the MIME part is done
    */
    dup_filename = etpan_dup_file(filename);
    if (dup_filename == NULL) {
      goto free_mime;
    }
  
    r = mailmime_set_body_file(mime, dup_filename);
    if (r != MAILIMF_NO_ERROR) {
      free(dup_filename);
      goto free_mime;
    }
  }
  
  return mime;
  
 free_mime:
  mailmime_free(mime);
  goto err;
 free_mime_fields:
  mailmime_fields_free(mime_fields);
  mailmime_content_free(content);
  goto err;
 free_content:
  if (encoding != NULL)
    mailmime_mechanism_free(encoding);
  if (content != NULL)
    mailmime_content_free(content);
  if (disposition != NULL)
    mailmime_disposition_free(disposition);
 err:
  return NULL;
}





static inline int to_be_quoted(char * word, size_t size)
{
  int do_quote;
  char * cur;
  size_t i;

  do_quote = 0;
  cur = word;
  for(i = 0 ; i < size ; i ++) {
    switch (* cur) {
    case ',':
    case ':':
    case '!':
    case '"':
    case '#':
    case '$':
    case '@':
    case '[':
    case '\\':
    case ']':
    case '^':
    case '`':
    case '{':
    case '|':
    case '}':
    case '~':
    case '=':
    case '?':
    case '_':
      do_quote = 1;
      break;
    default:
      if (((unsigned char) * cur) >= 128)
        do_quote = 1;
      break;
    }
    cur ++;
  }

  return do_quote;
}

#define MAX_IMF_LINE 72

static inline int quote_word(char * display_charset,
    MMAPString * mmapstr, char * word, size_t size)
{
  char * cur;
  size_t i;
  char hex[4];
  int col;
  
  if (mmap_string_append(mmapstr, "=?") == NULL)
    return ERROR_MEMORY;
  if (mmap_string_append(mmapstr, display_charset) == NULL)
    return ERROR_MEMORY;
  if (mmap_string_append(mmapstr, "?Q?") == NULL)
    return ERROR_MEMORY;
  
  col = mmapstr->len;
  
  cur = word;
  for(i = 0 ; i < size ; i ++) {
    int do_quote_char;

    if (col + 2 /* size of "?=" */
        + 3 /* max size of newly added character */
        + 1 /* minimum column of string in a
               folded header */ >= MAX_IMF_LINE) {
      int old_pos;
      /* adds a concatened encoded word */
      
      if (mmap_string_append(mmapstr, "?=") == NULL)
        return ERROR_MEMORY;
      
      if (mmap_string_append(mmapstr, " ") == NULL)
        return ERROR_MEMORY;
      
      old_pos = mmapstr->len;
      
      if (mmap_string_append(mmapstr, "=?") == NULL)
        return ERROR_MEMORY;
      if (mmap_string_append(mmapstr, display_charset) == NULL)
        return ERROR_MEMORY;
      if (mmap_string_append(mmapstr, "?Q?") == NULL)
        return ERROR_MEMORY;
      
      col = mmapstr->len - old_pos;
    }
    
    do_quote_char = 0;
    switch (* cur) {
    case ',':
    case ':':
    case '!':
    case '"':
    case '#':
    case '$':
    case '@':
    case '[':
    case '\\':
    case ']':
    case '^':
    case '`':
    case '{':
    case '|':
    case '}':
    case '~':
    case '=':
    case '?':
    case '_':
      do_quote_char = 1;
      break;

    default:
      if (((unsigned char) * cur) >= 128)
        do_quote_char = 1;
      break;
    }

    if (do_quote_char) {
      snprintf(hex, 4, "=%2.2X", (unsigned char) * cur);
      if (mmap_string_append(mmapstr, hex) == NULL)
        return ERROR_MEMORY;
      col += 3;
    }
    else {
      if (* cur == ' ') {
        if (mmap_string_append_c(mmapstr, '_') == NULL)
          return ERROR_MEMORY;
      }
      else {
        if (mmap_string_append_c(mmapstr, * cur) == NULL)
          return ERROR_MEMORY;
      }
      col += 3;
    }
    cur ++;
  }

  if (mmap_string_append(mmapstr, "?=") == NULL)
    return ERROR_MEMORY;

  return NO_ERROR;
}

static inline void get_word(char * begin, char ** pend, int * pto_be_quoted)
{
  char * cur;
  
  cur = begin;

  while ((* cur != ' ') && (* cur != '\t') && (* cur != '\0')) {
    cur ++;
  }

  if (cur - begin +
      1  /* minimum column of string in a
            folded header */ > MAX_IMF_LINE)
    * pto_be_quoted = 1;
  else
    * pto_be_quoted = to_be_quoted(begin, cur - begin);
  
  * pend = cur;
}

static char * etpan_make_quoted_printable(char * display_charset,
    char * phrase)
{
  char * str;
  char * cur;
  MMAPString * mmapstr;

  mmapstr = mmap_string_new("");
  if (mmapstr == NULL)
    goto err;

  cur = phrase;
  while (* cur != '\0') {
    char * begin;
    char * end;
    int r;
    int do_quote;
    int quote_words;

    begin = cur;
    end = begin;
    quote_words = 0;
    do_quote = 1;

    while (* cur != '\0') {
      get_word(cur, &cur, &do_quote);
      if (do_quote) {
        quote_words = 1;
        end = cur;
      }
      else
        break;
      if (* cur != '\0')
        cur ++;
    }

    if (quote_words) {
      r = quote_word(display_charset, mmapstr, begin, end - begin);
      if (r != NO_ERROR)
        goto free_mmap;

      if ((* end == ' ') || (* end == '\t')) {
        if (mmap_string_append_c(mmapstr, * end) == 0)
          goto free_mmap;
        end ++;
      }

      if (* end != '\0') {
        if (mmap_string_append_len(mmapstr, end, cur - end) == NULL)
          goto free_mmap;
      }
    }
    else {
      if (mmap_string_append_len(mmapstr, begin, cur - begin) == NULL)
        goto free_mmap;
    }

    if ((* cur == ' ') || (* cur == '\t')) {
      if (mmap_string_append_c(mmapstr, * cur) == 0)
        goto free_mmap;
      cur ++;
    }
  }

  str = strdup(mmapstr->str);
  if (str == NULL)
    goto free_mmap;

  mmap_string_free(mmapstr);
  
  return str;

 free_mmap:
  mmap_string_free(mmapstr);
 err:
  return NULL;
}



/* encode decode of MIME header values */

char * etpan_encode_mime_header(struct etpan_app * app,
    char * phrase)
{
  return
    etpan_make_quoted_printable(app->config.global_config->display_charset,
        phrase);
}


char * etpan_decode_mime_header(struct etpan_app * app,
    char * phrase)
{
  int r;
  size_t cur_token;
  char * decoded;

  cur_token = 0;
  r = mailmime_encoded_phrase_parse(app->config.global_config->message_charset,
      phrase, strlen(phrase),
      &cur_token, app->config.global_config->display_charset,
      &decoded);
  
  if (r != MAILIMF_NO_ERROR)
    return NULL;
  
  return decoded;
}


/* ************ decode / encode MIME in headers ****************** */

int etpan_mailbox_encode(struct etpan_app * app,
    struct mailimf_mailbox * mb)
{
  char * decoded;
  
  if (mb->mb_display_name == NULL)
    return NO_ERROR;
  
  decoded = etpan_encode_mime_header(app, mb->mb_display_name);
  if (decoded == NULL)
    return ERROR_MEMORY;
  
  /* replace display name */
  free(mb->mb_display_name);
  mb->mb_display_name = decoded;
  
  return NO_ERROR;
}

int etpan_mailbox_list_encode(struct etpan_app * app,
    struct mailimf_mailbox_list * mb_list)
{
  clistiter * cur;
  
  for(cur = clist_begin(mb_list->mb_list) ; cur != NULL ;
      cur = clist_next(cur)) {
    struct mailimf_mailbox * mb;
    
    mb = clist_content(cur);
    etpan_mailbox_encode(app, mb);
  }
  
  return NO_ERROR;
}

int etpan_group_encode(struct etpan_app * app,
    struct mailimf_group * group)
{
  char * decoded;
  
  decoded = etpan_encode_mime_header(app, group->grp_display_name);
  if (decoded != NULL) {
    free(group->grp_display_name);
    group->grp_display_name = decoded;
  }
  
  if (group->grp_mb_list != NULL)
    etpan_mailbox_list_encode(app, group->grp_mb_list);
  
  return NO_ERROR;
}

int etpan_addr_encode(struct etpan_app * app, 
    struct mailimf_address * addr)
{
  switch (addr->ad_type) {
  case MAILIMF_ADDRESS_MAILBOX:
    etpan_mailbox_encode(app, addr->ad_data.ad_mailbox);
    break;
  case MAILIMF_ADDRESS_GROUP:
    etpan_group_encode(app, addr->ad_data.ad_group);
    break;
  }
  
  return NO_ERROR;
}

int etpan_addr_list_encode(struct etpan_app * app,
    struct mailimf_address_list * addr_list)
{
  clistiter * cur;
  
  for(cur = clist_begin(addr_list->ad_list) ; cur != NULL ;
      cur = clist_next(cur)) {
    struct mailimf_address * addr;
    
    addr = clist_content(cur);
    etpan_addr_encode(app, addr);
  }
  
  return NO_ERROR;
}

int etpan_mailbox_decode(struct etpan_app * app,
    struct mailimf_mailbox * mb)
{
  char * decoded;
#if 0
  int r;
  size_t cur_token;
#endif
  
  if (mb->mb_display_name == NULL)
    return NO_ERROR;
  
  decoded =  etpan_decode_mime_header(app, mb->mb_display_name);
  if (decoded == NULL)
    return ERROR_INVAL;
#if 0
  cur_token = 0;
  r = mailmime_encoded_phrase_parse(app->global_config->message_charset,
      mb->display_name, strlen(mb->display_name),
      &cur_token, app->global_config->display_charset,
      &decoded);
  if (r != MAILIMF_NO_ERROR)
    return ERROR_INVAL;
#endif
  
  /* replace display name */
  free(mb->mb_display_name);
  mb->mb_display_name = decoded;
  
  return NO_ERROR;
}

int etpan_mailbox_list_decode(struct etpan_app * app,
    struct mailimf_mailbox_list * mb_list)
{
  clistiter * cur;
  
  for(cur = clist_begin(mb_list->mb_list) ; cur != NULL ;
      cur = clist_next(cur)) {
    struct mailimf_mailbox * mb;
    
    mb = clist_content(cur);
    etpan_mailbox_decode(app, mb);
  }
  
  return NO_ERROR;
}

int etpan_group_decode(struct etpan_app * app,
    struct mailimf_group * group)
{
  char * decoded;
#if 0
  int r;
  size_t cur_token;
#endif
  
  decoded = etpan_decode_mime_header(app, group->grp_display_name);
  if (decoded != NULL) {
    free(group->grp_display_name);
    group->grp_display_name = decoded;
  }
#if 0
  cur_token = 0;
  r = mailmime_encoded_phrase_parse(app->global_config->message_charset,
      group->display_name, strlen(group->display_name),
      &cur_token, app->global_config->display_charset,
      &decoded);
  if (r == MAILIMF_NO_ERROR) {
    free(group->display_name);
    group->display_name = decoded;
  }
#endif
  
  if (group->grp_mb_list != NULL)
    etpan_mailbox_list_decode(app, group->grp_mb_list);
  
  return NO_ERROR;
}

int etpan_addr_decode(struct etpan_app * app, 
    struct mailimf_address * addr)
{
  switch (addr->ad_type) {
  case MAILIMF_ADDRESS_MAILBOX:
    etpan_mailbox_decode(app, addr->ad_data.ad_mailbox);
    break;
  case MAILIMF_ADDRESS_GROUP:
    etpan_group_decode(app, addr->ad_data.ad_group);
    break;
  }
  
  return NO_ERROR;
}

int etpan_addr_list_decode(struct etpan_app * app,
    struct mailimf_address_list * addr_list)
{
  clistiter * cur;
  
  for(cur = clist_begin(addr_list->ad_list) ; cur != NULL ;
      cur = clist_next(cur)) {
    struct mailimf_address * addr;
    
    addr = clist_content(cur);
    etpan_addr_decode(app, addr);
  }
  
  return NO_ERROR;
}


static char * get_discrete_type(struct mailmime_discrete_type * discrete_type)
{
  switch (discrete_type->dt_type) {
  case MAILMIME_DISCRETE_TYPE_TEXT:
    return "text";

  case MAILMIME_DISCRETE_TYPE_IMAGE:
    return "image";

  case MAILMIME_DISCRETE_TYPE_AUDIO:
    return "audio";

  case MAILMIME_DISCRETE_TYPE_VIDEO:
    return "video";

  case MAILMIME_DISCRETE_TYPE_APPLICATION:
    return "application";

  case MAILMIME_DISCRETE_TYPE_EXTENSION:
    return discrete_type->dt_extension;
  }

  return NULL;
}

static char *
get_composite_type(struct mailmime_composite_type * composite_type)
{
  switch (composite_type->ct_type) {
  case MAILMIME_COMPOSITE_TYPE_MESSAGE:
    return "message";

  case MAILMIME_COMPOSITE_TYPE_MULTIPART:
    return "multipart";

  case MAILMIME_COMPOSITE_TYPE_EXTENSION:
    return composite_type->ct_token;
  }

  return NULL;
}


char * etpan_get_content_type_str(struct mailmime_content * content)
{
  char * str;
  char * result;

  str = "unknown";

  switch (content->ct_type->tp_type) {
  case MAILMIME_TYPE_DISCRETE_TYPE:
    str = get_discrete_type(content->ct_type->tp_data.tp_discrete_type);
    break;

  case MAILMIME_TYPE_COMPOSITE_TYPE:
    str = get_composite_type(content->ct_type->tp_data.tp_composite_type);
    break;
  }

  if (str == NULL)
    return NULL;

  result = malloc(strlen(str) + strlen(content->ct_subtype) + 2);
  if (result == NULL)
    return NULL;

  strcpy(result, str);
  strcat(result, "/");
  strcat(result, content->ct_subtype);

  return result;
}


#if 0
struct mailmime_parameter *
mailmime_param_new_with_data(char * name, char * value)
{
  char * param_name;
  char * param_value;
  struct mailmime_parameter * param;

  param_name = strdup(name);
  if (param_name == NULL)
    goto err;
  
  param_value = strdup(value);
  if (param_value == NULL)
    goto free_name;
  
  param = mailmime_parameter_new(param_name, param_value);
  if (param == NULL)
    goto free_value;
  
  return param;
  
 free_value:
  free(param_value);
 free_name:
  free(param_name);
 err:
  return NULL;
}
#endif

int etpan_mime_is_text(struct mailmime * build_info)
{
  if (build_info->mm_type == MAILMIME_SINGLE) {
    if (build_info->mm_content_type != NULL) {
      if (build_info->mm_content_type->ct_type->tp_type ==
          MAILMIME_TYPE_DISCRETE_TYPE) {
        if (build_info->mm_content_type->ct_type->tp_data.tp_discrete_type->dt_type ==
            MAILMIME_DISCRETE_TYPE_TEXT)
          return TRUE;
      }
    }
    else
      return TRUE;
  }

  return FALSE;
}



struct mailimf_mailbox_list *
etpan_get_from_field(struct etpan_app * app,
    struct etpan_account_info * account)
{
  struct mailimf_mailbox * mb;
  struct mailimf_mailbox_list * mb_list;
  clist * list;
  char * name;
  char * addr;
  int r;

  if (account->addr == NULL)
    goto err;

  addr = strdup(account->addr);
  if (addr == NULL)
    goto err;

  name = NULL;
  if (account->name != NULL) {
#if 0
    name = etpan_encode_mime_header(app, account->name);
#endif
    name = strdup(account->name);
    if (name == NULL)
      goto free_addr;
  }

  mb = mailimf_mailbox_new(name, addr);
  if (mb == NULL) {
    goto free_name;
  }
    
  list = clist_new();
  if (list == NULL) {
    mailimf_mailbox_free(mb);
    goto free_name;
  }

  r = clist_append(list, mb);
  if (r < 0) {
    mailimf_mailbox_free(mb);
    goto free_list;
  }

  mb_list = mailimf_mailbox_list_new(list);
  if (mb_list == NULL)
    goto free_list;

  return mb_list;

 free_list:
  clist_foreach(list, (clist_func) mailimf_mailbox_free, NULL);
  clist_free(list);
 free_name:
  if (name != NULL)
    free(name);
 free_addr:
  free(addr);
 err:
  return NULL;
}


int etpan_address_list_write(FILE * f, int * col,
    struct etpan_app * app, struct mailimf_address_list * addr_list)
{
  struct mailimf_address_list * decoded_addr_list;
  int r;

  decoded_addr_list = etpan_dup_address_list(addr_list);
  if (decoded_addr_list == NULL)
    return ERROR_MEMORY;
  
  etpan_addr_list_decode(app, decoded_addr_list);
  
  r = mailimf_address_list_write(f, col, decoded_addr_list);
  mailimf_address_list_free(decoded_addr_list);
  if (r != MAILIMF_NO_ERROR)
    return ERROR_FILE;
  
  return NO_ERROR;
}

int etpan_mailbox_list_write(FILE * f, int * col,
    struct etpan_app * app, struct mailimf_mailbox_list * mb_list)
{
  struct mailimf_mailbox_list * decoded_mb_list;
  int r;

  decoded_mb_list = etpan_dup_mailbox_list(mb_list);
  if (decoded_mb_list == NULL)
    return ERROR_MEMORY;
  
  etpan_mailbox_list_decode(app, decoded_mb_list);
  
  r = mailimf_mailbox_list_write(f, col, decoded_mb_list);
  mailimf_mailbox_list_free(decoded_mb_list);
  if (r != MAILIMF_NO_ERROR)
    return ERROR_FILE;
  
  return NO_ERROR;
}

#if 0
struct mailmime_parameter *
mailmime_parameter_dup(struct mailmime_parameter * param)
{
  char * name;
  char * value;
  struct mailmime_parameter * dup_param;

  name = strdup(param->name);
  if (name == NULL)
    goto err;
  
  value = strdup(param->value);
  if (value == NULL)
    goto free_name;
  
  dup_param = mailmime_parameter_new(name, value);
  if (dup_param == NULL)
    goto free_value;
  
  return dup_param;
  
 free_value:
  free(value);
 free_name:
  free(name);
 err:
  return NULL;
}

struct mailmime_composite_type *
mailmime_composite_type_dup(struct mailmime_composite_type * composite_type)
{
  struct mailmime_composite_type * dup_composite;
  char * token;
  
  token = NULL;
  if (composite_type->token != NULL) {
    token = strdup(composite_type->token);
    if (token == NULL)
      goto err;
  }
  
  dup_composite = mailmime_composite_type_new(composite_type->type, token);
  if (dup_composite == NULL)
    goto free_token;
  
  return dup_composite;
  
 free_token:
  if (token != NULL)
    free(token);
 err:
  return NULL;
}

struct mailmime_discrete_type *
mailmime_discrete_type_dup(struct mailmime_discrete_type * discrete_type)
{
  struct mailmime_discrete_type * dup_discrete;
  char * extension;
  
  extension = NULL;
  if (discrete_type->extension != NULL) {
    extension = strdup(discrete_type->extension);
    if (extension == NULL)
      goto err;
  }
  
  dup_discrete = mailmime_discrete_type_new(discrete_type->type, extension);
  if (dup_discrete == NULL)
    goto free_extension;
  
  return dup_discrete;
  
 free_extension:
  if (extension != NULL)
    free(extension);
 err:
  return NULL;
}

struct mailmime_type * mailmime_type_dup(struct mailmime_type * type)
{
  struct mailmime_type * dup_type;
  struct mailmime_discrete_type * discrete_type;
  struct mailmime_composite_type * composite_type;
  
  discrete_type = NULL;
  if (type->discrete_type != NULL) {
    discrete_type = mailmime_discrete_type_dup(type->discrete_type);
    if (discrete_type == NULL)
      goto err;
  }
  
  composite_type = NULL;
  if (type->composite_type != NULL) {
    composite_type = mailmime_composite_type_dup(type->composite_type);
    if (composite_type == NULL)
      goto free_discrete;
  }
  
  dup_type = mailmime_type_new(type->type, discrete_type, composite_type);
  if (dup_type == NULL)
    goto free_composite;
  
  return dup_type;
  
 free_composite:
  if (composite_type != NULL)
    mailmime_composite_type_free(composite_type);
 free_discrete:
  if (discrete_type != NULL)
    mailmime_discrete_type_free(discrete_type);
 err:
  return NULL;
}

struct mailmime_content *
mailmime_content_dup(struct mailmime_content * content)
{
  clist * list;
  struct mailmime_type * type;
  int r;
  struct mailmime_content * dup_content;
  char * subtype;

  type = mailmime_type_dup(content->type);
  if (type == NULL)
    goto err;
  
  subtype = strdup(content->subtype);
  if (subtype == NULL)
    goto free_type;
  
  list = clist_new();
  if (list == NULL)
    goto free_subtype;
  
  if (content->parameters != NULL) {
    clistiter * cur;
    
    for(cur = clist_begin(content->parameters) ;
        cur != NULL ; cur = clist_next(cur)) {
      struct mailmime_parameter * param;
      
      param = mailmime_parameter_dup(cur->data);
      if (param == NULL)
        goto free_list;
      
      r = clist_append(list, param);
      if (r < 0) {
        mailmime_parameter_free(param);
        goto free_list;
      }
    }
  }
  
  dup_content = mailmime_content_new(type, subtype, list);
  if (dup_content == NULL)
    goto free_list;

  return dup_content;
  
 free_list:
  clist_foreach(list, (clist_func) mailmime_parameter_free, NULL);
 free_subtype:
  free(subtype);
 free_type:
  mailmime_type_free(type);
 err:
  return NULL;
}
#endif

/* TODO : better function to duplicate mime fields, currenly such inelegant */

struct mailmime_fields *
mailmime_fields_dup(struct mailmime_fields * mime_fields)
{
  FILE * f;
  char tmp_file[PATH_MAX];
  int col;
  int r;
  struct mailmime_fields * dup_mime_fields;
  int fd;
  char * mapping;
  struct stat stat_info;
  struct mailimf_fields * fields;
  size_t cur_token;
  
  f = etpan_get_tmp_file(tmp_file, sizeof(tmp_file));
  if (f == NULL)
    goto err;
  
  col = 0;
  r = mailmime_fields_write(f, &col, mime_fields);
  if (r != MAILIMF_NO_ERROR)
    goto unlink;
  
  fflush(f);
  
  fd = fileno(f);
  if (fd == -1)
    goto unlink;
  
  r = fstat(fd, &stat_info);
  if (r < 0)
    goto unlink;
  
  mapping = mmap(NULL, stat_info.st_size, PROT_READ, MAP_PRIVATE, fd, 0);
  if (mapping == MAP_FAILED)
    goto unlink;
  
  cur_token = 0;
  r = mailimf_optional_fields_parse(mapping, stat_info.st_size,
      &cur_token, &fields);
  if (r != MAILIMF_NO_ERROR)
    goto unmap;
  
  r = mailmime_fields_parse(fields, &dup_mime_fields);
  mailimf_fields_free(fields);
  if (r != MAILIMF_NO_ERROR)
    goto unmap;
  
  munmap(mapping, stat_info.st_size);
  fclose(f);
  unlink(tmp_file);

  return dup_mime_fields;

 unmap:
  munmap(mapping, stat_info.st_size);
 unlink:
  fclose(f);
  unlink(tmp_file);
 err:
  return NULL;
}

#if 0
int mailmime_substitute(struct mailmime * old_mime,
    struct mailmime * new_mime)
{
  struct mailmime * parent;
  
  parent = old_mime->parent;
  if (parent == NULL)
    return ERROR_INVAL;
  
  if (old_mime->parent_type == MAILMIME_MESSAGE)
    parent->msg_mime = new_mime;
  else /* MAILMIME_MULTIPLE */
    old_mime->multipart_pos->data = new_mime;
  new_mime->parent = parent;
  new_mime->parent_type = old_mime->parent_type;
  
  /* detach old_mime */
  old_mime->parent = NULL;
  old_mime->parent_type = MAILMIME_NONE;
  
  return NO_ERROR;
}
#endif


/* write mime headers and body to a file, CR LF fixed */

#if 0
int etpan_fetch_mime_body_to_file(struct etpan_app * app,
    char * filename, size_t size, struct mailfolder * folder,
    mailmessage * msg, struct mailmime * mime)
{
  int r;
  int res;
  FILE * f;
  char * content;
  size_t content_len;
  int col;
  
  if (mime->parent_type == MAILMIME_NONE) {
    res = ERROR_INVAL;
    goto err;
  }
  
  f = etpan_get_tmp_file(filename, size);
  if (f == NULL) {
    res = ERROR_FETCH;
    goto err;
  }
  
  r = etpan_msg_fetch_section_mime(app, folder, msg, mime,
      &content, &content_len);
  if (r != MAIL_NO_ERROR) {
    res = ERROR_FETCH;
    goto close;
  }
  
  col = 0;
  r = mailimf_string_write(f, &col, content, content_len);
  etpan_msg_fetch_result_free(app, folder, msg, content);
  if (r != MAILIMF_NO_ERROR) {
    res = r;
    goto close;
  }
  
  r = etpan_msg_fetch_section(app, folder, msg, mime, &content, &content_len);
  if (r != MAIL_NO_ERROR) {
    res = ERROR_FETCH;
    goto close;
  }

  r = mailimf_string_write(f, &col, content, content_len);
  etpan_msg_fetch_result_free(app, folder, msg, content);
  if (r != MAILIMF_NO_ERROR) {
    res = r;
    goto close;
  }
  
  fclose(f);
  
  return NO_ERROR;
  
 close:
  fclose(f);
  unlink(filename);
 err:
  return res;
}
#endif

#if 0
int etpan_get_part_from_file(struct etpan_app * app,
    struct mailfolder * folder,
    char * filename,
    int check_security,
    struct mailmime ** result_mime)
{
  int fd;
  struct mailmime * mime;
  int r;
  struct stat stat_info;
  int res;
  char * mapping;

  fd = open(filename, O_RDONLY);
  if (fd < 0) {
    res = ERROR_FILE;
    goto err;
  }

  r = fstat(fd, &stat_info);
  if (r < 0) {
    res = ERROR_FILE;
    goto close;
  }

  mapping = mmap(NULL, stat_info.st_size, PROT_READ, MAP_PRIVATE, fd, 0);
  if (mapping == MAP_FAILED) {
    res = ERROR_FILE;
    goto close;
  }

#if 0  
  if (mapping == NULL) {
    res = ERROR_FILE;
    goto unmap;
  }
#endif
  
  /* check recursive parts if check_security is set */
#if 0
  r = etpan_get_mime(app, folder, mapping, stat_info.st_size,
      check_security, &mime);
#endif
  if (check_security)
    r = mailprivacy_get_mime(app->privacy, mapping, stat_info.st_size, &mime);
  else
    r = mailprivacy_get_mime(NULL, mapping, stat_info.st_size, &mime);
  if (r != MAIL_NO_ERROR) {
    res =  r;
    goto unmap;
  }

  if (mime->type == MAILMIME_MESSAGE) {
    struct mailmime * submime;
    
    submime = mime->msg_mime;
    if (mime->msg_mime != NULL) {
      mailmime_remove_part(submime);
      mailmime_free(mime);
      
      mime = submime;
    }
  }

  munmap(mapping, stat_info.st_size);
  
  close(fd);

  * result_mime = mime;

  return NO_ERROR;
  
 unmap:
  munmap(mapping, stat_info.st_size);
 close:
  close(fd);
 err:
  return res;
}
#endif
