/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libmac.so
 *
 *   File: mac_disklabel.h
 */

#ifndef _MAC_DISKLABEL_H
#define _MAC_DISKLABEL_H 1


/*
  mac disk labeling follows these rules:
 
  If you will be creating a MacOS compatibility disk then you need to keep several partitions 
  for MacOS.
  
  order         type                    name            length                  system            
  -----------------------------------------------------------------------------------------
  1             Apple_partition_map     Apple           63 (1st track ?)        partition map
  2             Apple_Driver43          Macintosh       54                      driver 4.3
  3             Apple_Driver43          Macintosh       74                      driver 4.3
  4             Apple_Driver_IOKit      Macintosh       512                     unknown
  5             Apple_Patches           Patch Partition 512                     unknown
  6             Apple_Bootstrap         bootstrap       1600                    unknown
  7             Apple_Free              Extra           xxxx                    free space for linux partitions
  8             Apple_HFS               untitled        xxxx                    HFS ... min of 160 meg
  
  Mac partitions have both a type and a name and both are strings!
       
  NewWorld Macs are those brightly colored machines.     
                   
  For NewWorld PowerMac systems you must have an 800k Apple_Bootstrap partition appearing before any MacOS
  partitions.                   
  
  You could create an Apple_UNIX_SRV2 partition in the free space and use it as Linux native.
  
  Linux partitions should always appear before the MacOS or MacOSX partitions.
  
  In the mac disklabel ... a block == a hardsector ... 512 bytes.
  
  The partition name does not matter EXCEPT for swap ... which should be named swap. using mount points as
  the partition name is a good idea.
    
  MacOSX uses UFS partitions created by Drive Setup as a placeholder.  It deletes them and creates two
  bootstrap partitions ... then recreates a UFS partition with the remainder of the room.
            
  Creating more than 15 partitions is not advised.
  
*/


#define MAC_DISK_MAGIC   0x4552               

typedef struct mac_driver_desc {
	u_int16_t	signature;	        // MAC_DRIVER_MAGIC 
	u_int16_t	block_size;             // usually 512 ... could be some multiple of 2 ?
	u_int32_t       block_count;            // size of driver in blocks
        u_int16_t       dev_type;               // ?
        u_int16_t       dev_id;                 // ?
        u_int32_t       data;                   // ?
        u_int16_t       driver_count;           // number of driver descriptors
        u_int32_t       start;                  // block number of start of useable space on disk
        u_int32_t       size;                   // size of disk in blocks
        u_int16_t       os_type;                // operating system
        char            sector_padding[486];    // remainder of sector is reserved for driver
} mac_disklabel_t;


#define MAC_PARTITION_MAGIC       0x504d 
#define MAC_STATUS_BOOTABLE	  8	        // means partition is bootable 
#define PMAP_STRING_SIZE          32            // maximum string size in a mac partition record

typedef struct mac_partition {
	u_int16_t       signature;	        // MAC_PARTITION_MAGIC 
	u_int16_t       res1;
	u_int32_t       map_count;	        // # blocks in partition map 
	u_int32_t       start_block;	        // absolute starting block # of partition 
	u_int32_t       block_count;	        // number of blocks in partition 
	char            name[PMAP_STRING_SIZE];	// partition name 
	char            type[PMAP_STRING_SIZE];	// string type description 
	u_int32_t       data_start;	        // rel block # of first data block 
	u_int32_t       data_count;	        // number of data blocks 
	u_int32_t       status;		        // partition status bits 
	u_int32_t	boot_start;
	u_int32_t	boot_size;
	u_int32_t	boot_load;
	u_int32_t	boot_load2;
	u_int32_t	boot_entry;
	u_int32_t	boot_entry2;
	u_int32_t	boot_cksum;
	char	        processor[16];	        // identifies ISA of boot, 680x0 x=0,2,3,4
	char            sector_padding[376];
} mac_partition_t;

#define MAC_PMAP_STRING         "Apple_partition_map"   // partition contains mac partition map
#define MAC_DRIVER_STRING       "Apple_Driver"          // partition contains a device driver
#define MAC_DRIVER43_STRING     "Apple_Driver43"        // partition contains scsi mgr 4.3 driver
#define MAC_HFS_STRING          "Apple_HFS"             // hfs or hfs extd file system
#define MAC_MFS_STRING          "Apple_MFS"             // original macintosh file system
#define MAC_SCRATCH_STRING      "Apple_Scratch"         // empty partition
#define MAC_PRODOS_STRING       "Apple_PRODOS"          // proDOS file system
#define MAC_FREE_STRING         "Apple_Free"            // free space partition
#define MAC_SWAP_STRING         "Apple_Swap"
#define MAC_AUX_STRING          "Apple_Aux"
#define MAC_MSDOS_STRING        "Apple_MSDOS"
#define MAC_MINIX_STRING        "Apple_Minix"
#define MAC_AFFS_STRING         "Apple_AFFS"
#define MAC_EXT2_STRING         "Apple_EXT2"
#define MAC_UNIX_SVR2           "Apple_Unix_SVR2"       // some kind of Unix partition ... Linux


#define MAC_STATUS_MASK         0x7F                    // mask for partition record status field 
#define MAC_STATUS_VALID        0x01                    // partition record is valid
#define MAC_STATUS_ALLOCATED    0x02                    // partition is allocated
#define MAC_STATUS_INUSE        0x04                    // partition in-use ... cleared with reboot
#define MAC_STATUS_BOOTINFO     0x08                    // partition has boot info
#define MAC_STATUS_READ         0x10                    // partition is readable
#define MAC_STATUS_WRITE        0x20                    // partition is writable
#define MAC_STATUS_PINNED       0x40                    // partition boot code is position independent

#define MAC_MAX_PARTITIONS      32                      // max number of MAC partition records in the PMAP


static inline void string_to_upper_case(char *str)
{
	int i;
	for(i=0;i<strlen(str); i++){
	      if (str[i] >= 'a'  &&  str[i] <= 'z') { 
		      str[i] -= 0x20;
	      }
	}
}


static inline boolean isa_mac_pmap_partition( mac_partition_t *p )
{
        if ( strncmp(p->type,MAC_PMAP_STRING,PMAP_STRING_SIZE) == 0 ) {		
                return TRUE;
        }
        
        return FALSE;
}

static inline boolean isa_mac_freespace_partition( mac_partition_t *p )
{
        if ( strncmp(p->type,MAC_FREE_STRING,PMAP_STRING_SIZE) == 0 ) {	
                return TRUE;
        }
        
        return FALSE;
}

static inline boolean isa_mac_scratch_partition( mac_partition_t *p )
{
        if ( strncmp(p->type,MAC_SCRATCH_STRING,PMAP_STRING_SIZE) == 0 ) {		
                return TRUE;
        }
        
        return FALSE;
}

static inline boolean isa_mac_null_partition( mac_partition_t *p )
{
        if ( strlen(p->type)==0 && strlen(p->name)==0 ) {		
                return TRUE;
        }
        
        return FALSE;
}

static inline boolean isa_mac_linux_partition( mac_partition_t *p )
{
	char  p_name[PMAP_STRING_SIZE+1];

        if ( strncmp(p->type,MAC_UNIX_SVR2,PMAP_STRING_SIZE) == 0 ) {

		strncpy(p_name, p->name, PMAP_STRING_SIZE);
		string_to_upper_case(p_name);

		if ( strncmp(p_name,"LINUX",PMAP_STRING_SIZE) == 0 ||		     
		     strncmp(p_name,"SWAP", PMAP_STRING_SIZE) == 0 ) {		
			return TRUE;
		}

        }
        
        return FALSE;
}

static inline boolean isa_mac_data_partition( mac_partition_t *p )
{
	if ( isa_mac_pmap_partition( p )      == FALSE &&
	     isa_mac_freespace_partition( p ) == FALSE &&
	     isa_mac_scratch_partition( p )   == FALSE &&
	     isa_mac_null_partition( p )      == FALSE ) {
                return TRUE;
        }
        
        return FALSE;
}

#endif
