/* 

                          Firewall Builder

                 Copyright (C) 2003 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@fwbuilder.org

  $Id: RuleSetView.cpp,v 1.85 2004/07/25 05:51:55 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/




#include "config.h"
#include "global.h"
#include "utils.h"
#include "FWWindow.h"
#include "RuleSetView.h"
#include "ObjectManipulator.h"
#include "ObjectEditor.h"
#include "platforms.h"
#include "inplaceComboBox.h"
#include "FWObjectDrag.h"
#include "FWObjectClipboard.h"
#include "findDialog.h"
#include "ColorLabelMenuItem.h"
#include "FWBSettings.h"
#include "SimpleTextEditor.h"
#include "FWObjectPropertiesFactory.h"

#include "askrulenumberdialog_q.h"

#include <iostream>

#include "fwbuilder/Firewall.h"
#include "fwbuilder/Resources.h"
#include "fwbuilder/Policy.h"
#include "fwbuilder/InterfacePolicy.h"
#include "fwbuilder/NAT.h"
#include "fwbuilder/RuleElement.h"
#include "fwbuilder/Interface.h"

#include <qapplication.h>
#include <qpainter.h>
#include <qrect.h>
#include <qtextedit.h>
#include <qpopupmenu.h>
#include <qmessagebox.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qpixmap.h>
#include <qcombobox.h>
#include <qdragobject.h>
#include <qspinbox.h>
#include <qaction.h>
#include <qtooltip.h>
#include <qcursor.h>

using namespace libfwbuilder;
using namespace std;


class mouseEventFilter : public QObject
{
    protected:
    bool eventFilter( QObject *object, QEvent *event)
    {
        if (event->type() == QEvent::MouseButtonPress )
        {
            cerr << "event type=" << event->type() << endl;
            return true;
        }
        else 
            return false;
    }
};

mouseEventFilter mef;

bool headerMouseEventInterceptor::eventFilter( QObject *object, QEvent *event)
{
    if (event->type() == QEvent::ContextMenu )
    {
        QContextMenuEvent *e = (QContextMenuEvent*)(event);
        
        int row = rsv->rowAt( rsv->contentsY()+e->pos().y() );
        rsv->contextMenu(row, -1, e->globalPos());

//        rsv->headerMousePressEvent(e->globalPos());
    }
    return false;
}

RuleObjToolTip::RuleObjToolTip(RuleSetView *w) : QToolTip(w->viewport(),0)
{
    rsv=w;
}

void RuleObjToolTip::maybeTip(const QPoint &pos)
{
    if (st->getObjTooltips())
    {
        int cx,cy;

        rsv->viewportToContents(pos.x(),pos.y(),cx,cy);

        int row = rsv->rowAt(cy);
        int col = rsv->columnAt(cx);

        QRect   cr;
        FWObject *obj = rsv->getObj(row,col,cy,&cr);

        if (obj==NULL) return;

        cr.moveTopLeft( rsv->contentsToViewport( cr.topLeft() ) );

        tip(cr,
          FWObjectPropertiesFactory::getObjectPropertiesDetailed(obj,true,true));
    }
}


RuleSetView::RuleSetView( int r, int c ) : QTable( r, c ), hme(this)
{
    kbdGoingUp = false;
    RuleElementSpacing=4;
    changingSelection = false;
    dragging = false;

//    verticalHeader()->setLabel(0, "0");
//    horizontalHeader()->setLabel(0, "0");

    new RuleObjToolTip(this);

    setDragEnabled(true);

    ncols=c;
    selectedObject = NULL;

    setFocusPolicy( StrongFocus );

    items.setAutoDelete( TRUE );
    widgets.setAutoDelete( TRUE );

    setSelectionMode( MultiRow );

    setAcceptDrops( TRUE );

    setCaption( tr( "A Rule Set" ) );
    setLeftMargin( fontMetrics().width( "W999W" ) );

    horizontalHeader()->setStretchEnabled(false);
    verticalHeader()->setStretchEnabled(false);

    horizontalHeader()->setResizeEnabled(true);
    verticalHeader()->setResizeEnabled(false);

    horizontalHeader()->setClickEnabled(false);
    verticalHeader()->setClickEnabled(true);

    horizontalHeader()->setMovingEnabled(true);
/*
    connect( horizontalHeader(), SIGNAL( clicked(int) ),
             this, SLOT( horizontalHeaderClicked(int) ) );

    connect( verticalHeader(), SIGNAL( clicked(int) ),
             this, SLOT( verticalHeaderClicked(int) ) );
*/

    connect( this, SIGNAL( contextMenuRequested(int,int,const QPoint&) ),
             this, SLOT( contextMenu(int,int,const QPoint&) ) );

    connect( this, SIGNAL( doubleClicked(int,int,int,const QPoint&) ),
             this, SLOT( doubleClicked(int,int,int,const QPoint&) ) );

    connect( this, SIGNAL( selectionChanged() ),
             this, SLOT( selectionChanged() ) );

    verticalHeader()->installEventFilter( &hme );
}

RuleSetView::~RuleSetView()
{
}

void RuleSetView::hideEvent(QHideEvent *ev)
{
    QString k = settingsKey();
    QString v;

    for (int col=0; col<ncols; col++)  
        v = v + QString("%1 ").arg( columnWidth(col) );
/* disabled because we need to remember the width for columns in
 * association with their names, not just sequentially. Since
 * different firewall types have different number of columns, doing it
 * just by numbers causes the last column to disappear when user
 * switches from a firewall with 6 columns to firewall with 7 columns 
 * (e.g. from pf to iptables)
 */
//    st->setStr(k,v);
}


QString RuleSetView::settingsKey()
{
    return QString("/RuleSets/") + name() + "_Columns";
}

void RuleSetView::setRuleNumber(int row, libfwbuilder::Rule *rule)
{
    QIconSet icn;

    if (rule!=NULL && rule->isDisabled())
        icn = QPixmap::fromMimeSource( Resources::global_res->getResourceStr(
                               "/FWBuilderResources/UI/Icons/neg").c_str());
    else
        icn = QIconSet();

    int s = verticalHeader()->sectionSize(row);
    verticalHeader()->setLabel( row, icn, QString::number(row) );
    verticalHeader()->resizeSection( row , s );
}

void RuleSetView::init()
{
    QApplication::setOverrideCursor( QCursor( Qt::WaitCursor) );

    horizontalHeader()->adjustHeaderSize();

    int          row=0;
    map<int,int> colW;
    bool         userColWidth=false;

    QPainter p(this);

    QString k = settingsKey();
    QString v = st->getStr(k);
    if (!v.isEmpty())
    {
        userColWidth=true;
        for (int col=0; col<ncols; col++)  
            colW[col]=v.section(' ',col,col).toInt();
    } else
    {
        for (int col=0; col<ncols; col++)
        {
            QString lbl = horizontalHeader()->label(col);
            QRect br=p.boundingRect(0, 0, 1000, 1000,
                                    Qt::AlignLeft|Qt::AlignVCenter,
                                    lbl );
            colW[col]=br.width() + 10;
        }
    }

    for (FWObject::iterator i=ruleset->begin(); i!=ruleset->end(); i++,row++)
    {
        ruleIndex[row] = *i;
        dirtyRows[row] = 1;

        setRuleNumber(row, Rule::cast( *i ));

//        adjustRow(row);

        int h=20;
        for (int col=0; col<ncols; col++)
        {
            QRect cr = calculateCellSize(row,col);
            h = QMAX(h, cr.height());
            if (!userColWidth)
                colW[col]=QMAX(colW[col],cr.width());
        }
        adjustRow_int(row,h);
    }

    for (int col=0; col<ncols; col++) setColumnWidth(col,colW[col]);

    updateContents();

    QApplication::restoreOverrideCursor();
}

void RuleSetView::iinit()
{
    setNumRows( ruleset->size() );

    QPixmap  pm = QPixmap::fromMimeSource( "accept.png" );
    pixmap_h    = pm.height();
    pixmap_w    = pm.width();

    QPainter p(this);
    QRect br = p.boundingRect(0, 0, 1000, 1000,
                              Qt::AlignLeft|Qt::AlignVCenter,"WMWM" );
    text_h   = br.height();
    item_h   = ( (pixmap_h>text_h)?pixmap_h:text_h ) + RuleElementSpacing;

    FWObject *f=ruleset;
    while (f!=NULL && !Firewall::isA(f)) f=f->getParent();
    assert(f!=NULL);

// f is a pointer at firewall object
    supports_logging      =false;
    supports_rule_options =false;
    supports_time         =false;

    try {
        supports_logging=
            Resources::getTargetCapabilityBool(f->getStr("platform"),
                                               "logging_in_policy");
        supports_rule_options=
            Resources::getTargetCapabilityBool(f->getStr("platform"),
                                               "options_in_policy");
        supports_time=
            Resources::getTargetCapabilityBool(f->getStr("platform"),
                                               "supports_time");
    } catch (FWException &ex)    {    }

    return;
}

void RuleSetView::clear()
{


}

void RuleSetView::adjustColumn( int col )
{
//    int w = horizontalHeader()->sectionSize(col);
    QString lbl = horizontalHeader()->label(col);
    QPainter p(this);
    QRect br=p.boundingRect(0, 0, 1000, 1000,
                            Qt::AlignLeft|Qt::AlignVCenter,
                            lbl );

    int     w = br.width() + 10;
    
    int row=0;
    for (FWObject::iterator i=ruleset->begin(); i!=ruleset->end(); i++,row++)
    {
        QRect cr = calculateCellSize(row,col);
        w=QMAX(w,cr.width());
    }
    setColumnWidth(col,w);
}

void RuleSetView::adjustRow_int( int row, int h )
{

/* make sure the row is no smaller than a label in the left header,
 * and no smaller than the "strut" (the minimal size of the gui
 * element as defined in QApplication)
 */
    QHeader * leftHeader = verticalHeader();
    h = QMAX(h, leftHeader->fontMetrics().height() + 2);
    h = QMAX(h, QApplication::globalStrut().height());

/* setRowHeight causes redraw. Beware of loops 'cause we call adjustRow from
 * cellPaint! */
    setRowHeight(row, h);

    dirtyRows[row]=0;
}

void RuleSetView::adjustRow( int row )
{
    int h = 20;

    for (int col=0; col<ncols; col++)
    {
        QRect cr = calculateCellSize(row,col);

        h = QMAX(h, cr.height());
        int w = cr.width();

        if (columnWidth(col)<w) setColumnWidth(col,w);
    }

    adjustRow_int(row,h);
}

QRect RuleSetView::calculateCellSize( int row, int col )
{
    int   h = 20;

    QPainter p(this);

    Rule *rule = Rule::cast( ruleIndex[row] );

    int hc=0;
    int wc=0;
    switch (getColType(col))
    {
    case Time:
    {
        RuleElement *re = getRE(rule,col);
        if (re==NULL)
        {
/* broken rule element, fix it */
            FWObject *nre=mw->db()->create("When",true);
            assert(nre!=NULL);
            rule->add(nre);
        }
    }
    case Object:
    {
        RuleElement *re = getRE(rule,col);
        if (re==NULL) return QRect(0,0,0,0);

        if (re->isAny())
        {
            QRect br=p.boundingRect(0, 0, 1000, 1000,
                                    Qt::AlignLeft|Qt::AlignVCenter,
                                    "WWWW" );
            hc += item_h;
            int itmW = RuleElementSpacing/2 + pixmap_w + 
                RuleElementSpacing + br.width();
            wc  = QMAX(wc, itmW);
        } else
        {
            for (FWObject::iterator j=re->begin(); j!=re->end(); j++)
            {
                FWObject *o1= *j;
                if (FWReference::cast(o1)!=NULL)
                    o1=FWReference::cast(o1)->getPointer();

                QRect br=p.boundingRect(0, 0, 1000, 1000,
                                        Qt::AlignLeft|Qt::AlignVCenter,
                                        QString::fromUtf8(o1->getName().c_str()) );
                hc += item_h;
                int itmW = RuleElementSpacing/2 + pixmap_w + 
                    RuleElementSpacing + br.width();
                wc  = QMAX(wc, itmW);
            }
        }
        break;
    }
    case Action:
    {
/* possible actions: "Accept", "Deny", "Reject", "Accounting" */
        QRect br=p.boundingRect(0, 0, 1000, 1000,
                                Qt::AlignLeft|Qt::AlignVCenter,tr("Accounting ") );
        hc = item_h;
        wc = RuleElementSpacing/2 + pixmap_w + RuleElementSpacing + br.width();
        break;
    }
    case Direction:
    {
        PolicyRule::Direction dir = PolicyRule::cast(rule)->getDirection();
/* broken direction, fix it */
        if (dir==PolicyRule::Undefined)
            PolicyRule::cast(rule)->setDirection(PolicyRule::Both);

/* possible directions: "Inbound", "Outbound" , "Both" */
        QRect br=p.boundingRect(0, 0, 1000, 1000,
                                Qt::AlignLeft|Qt::AlignVCenter,tr("Outbound ") );
        hc = item_h;
        wc = RuleElementSpacing/2 + pixmap_w + RuleElementSpacing + br.width();
        break;
    }
    case Options:
        hc = item_h;
        wc = RuleElementSpacing/2 + pixmap_w + RuleElementSpacing + pixmap_w;
        break;
    case Comment:
    {
        QRect br=p.boundingRect(0, 0, 1000, 1000,
                                Qt::AlignLeft|Qt::AlignVCenter,
                                QString::fromUtf8(rule->getComment().c_str()) );
        hc = br.height() + RuleElementSpacing;
        wc = RuleElementSpacing/2 + br.width();
        break;
    }
    default:
        break;
    }

    h = QMAX(h, hc);

    wc = QMAX(wc, QApplication::globalStrut().width());
    wc += RuleElementSpacing/2;  // some padding

    return QRect(0,0,wc,h);
}

void RuleSetView::info()
{
    mw->info(selectedObject);
}

QPixmap RuleSetView::getPixmap(FWObject *obj, PixmapAttr pmattr) const
{
    QPixmap pm;
    string icn = "icon";
    if (pmattr == Neg)  icn="icon-neg";
    if (pmattr == Ref)  icn="icon-ref";
    if (pmattr == Tree) icn="icon-tree";

    return QPixmap::fromMimeSource( 
        Resources::global_res->getObjResourceStr(obj, icn).c_str() );
}

/*
 *  insertWidget and cellwidget are only used when widgets are put in
 *  cells, which only happens if we enable cell editing. Which we do not.
 */
void RuleSetView::insertWidget( int r, int c, QWidget *w )
{
    widgets.replace( indexOf( r, c ), w );
}

QWidget* RuleSetView::cellWidget( int r, int c ) const
{
    return widgets.find( indexOf( r, c ) );
}

QString RuleSetView::objectText(RuleElement *re,FWObject *obj)
{
    if (re->isAny() &&
        (
            RuleElementTSrc::isA(re) ||
            RuleElementTDst::isA(re) ||
            RuleElementTSrv::isA(re)
        ) ) return QString(tr("Original"));

    if (Interface::isA(obj))
    {
        QString lbl= Interface::cast(obj)->getLabel().c_str();
        if ( !lbl.isEmpty() ) return lbl;
    }
    return QString::fromUtf8(obj->getName().c_str());
}

void RuleSetView::paintCell(QPainter *pntr,
                            int row,
                            int col,
                            const QRect &cr,
                            bool selected,
                            const QColorGroup &cg)
{
/* row may point at an empty row where there is no rule yet. This
 * happens if this method is called to redraw the table when we call
 * setNumRows
 */
    if (ruleIndex.count(row)==0) return;

    if (dirtyRows[row]!=0) 
    { 
        if (fwbdebug) qDebug("RuleSetView::paintCell dirty row %d",row);

        dirtyRows[row]=0;
        adjustRow(row); //    this causes repaint 
        return;
    }

    QString     rclr;
    Rule *rule = Rule::cast( ruleIndex[row] );
    if (rule!=NULL)
    {
        FWOptions  *ropt = rule->getOptionsObject();
        assert(ropt!=NULL);
        rclr = ropt->getStr("color").c_str();
    }

    QPixmap bufferpixmap;
    bufferpixmap.resize( cr.width() , cr.height() );
    bufferpixmap.fill( cg.base() );

    QPainter p( &bufferpixmap );

    QRect r = cellRect(row,col);

    int x  = r.left() + RuleElementSpacing/2;
    int y  = r.top();

//    QColor penColor;

//    p.fillRect( 0 , 0 , cr.width()-1, cr.height()-1, cg.base());

    if (!rclr.isEmpty())
    {
        QRect rect(0, y, cr.width(), cr.height() );
        p.fillRect(rect, QColor(rclr));
    }

    p.drawLine( cr.width()-1, 0, cr.width()-1, cr.height()-1 );
    p.drawLine( 0, cr.height()-1, cr.width()-1, cr.height()-1 );

    p.drawLine( cr.width(), 1, cr.width(), cr.height() );
    p.drawLine( 1, cr.height(), cr.width(), cr.height() );

//    if (selected)   penColor=cg.highlightedText();
//    else            penColor=cg.text();

    switch (getColType(col))
    {
    case Object:
    case Time:
    {
        RuleElement *re = getRE(row,col);
        if (re==NULL) return;

        bool  sel = (row==currentRow() && col==currentColumn());
        for (FWObject::iterator i=re->begin(); i!=re->end(); i++)
        {
            FWObject *o1= *i;
            if (FWReference::cast(o1)!=NULL)
                o1=FWReference::cast(o1)->getPointer();
            if (sel && o1==selectedObject)
            {
                QRect rect(0, y, cr.width(), item_h );
                p.fillRect(rect, cg.highlight());
                p.setPen( cg.highlightedText() );
            } else
            {
                p.setPen( cg.text() );
            }
            x = r.left()+1;

            QPixmap pm = getPixmap(o1 , re->getNeg()?Neg:Normal );

            p.drawPixmap( x, y + RuleElementSpacing/2, pm );

            x += pm.width()+1;

            p.drawText( x, y + RuleElementSpacing/2,
                         cr.width()-pm.width()-1, item_h,
                         Qt::AlignLeft|Qt::AlignVCenter, objectText(re,o1) );

            y += item_h;
        }
        break;
    }
    case Action:
    {
        PolicyRule *rule = PolicyRule::cast( ruleIndex[row] );
        if (rule==NULL) return;

        string act = rule->getActionAsString();
        string icn = Resources::global_res->getResourceStr("/FWBuilderResources/UI/Icons/"+ act );
        assert(icn!="");
        QPixmap pm = QPixmap::fromMimeSource( icn.c_str() );
        p.drawPixmap( x,y,pm );
        x+=pm.width()+1;
        QRect br=p.boundingRect(x, y, 1000, 1000,
                                 Qt::AlignLeft|Qt::AlignVCenter,
				act.c_str() );
        p.drawText( x, y, br.width(), pm.height(),
		    Qt::AlignLeft|Qt::AlignVCenter, act.c_str() );
        break;
    }
    case Direction:
    {
        PolicyRule *rule = PolicyRule::cast( ruleIndex[row] );
        if (rule==NULL) return;

        string dir = rule->getDirectionAsString();
        string icn = Resources::global_res->getResourceStr("/FWBuilderResources/UI/Icons/"+ dir );
        assert(icn!="");
        QPixmap pm = QPixmap::fromMimeSource( icn.c_str() );
        p.drawPixmap( x,y,pm );
        x+=pm.width()+1;
        QRect br=p.boundingRect(x, y, 1000, 1000,
                                 Qt::AlignLeft|Qt::AlignVCenter,dir.c_str() );
        p.drawText( x, y, br.width(), pm.height(),
		    Qt::AlignLeft|Qt::AlignVCenter, dir.c_str() );
        break;
    }
    case Options:
    {
        PolicyRule *rule = PolicyRule::cast( ruleIndex[row] );
        if (rule==NULL) return;

        if (rule->getLogging())
        {
            string icn = Resources::global_res->getResourceStr("/FWBuilderResources/UI/Icons/Log" );
            assert(icn!="");
            QPixmap pm = QPixmap::fromMimeSource( icn.c_str() );
            p.drawPixmap( x,y,pm );
            x+=pm.width()+2;
        }
        if (! isDefaultOptions( rule->getOptionsObject() ))
        {
            string icn = Resources::global_res->getResourceStr("/FWBuilderResources/UI/Icons/Options" );
            assert(icn!="");
            QPixmap pm = QPixmap::fromMimeSource( icn.c_str() );
            p.drawPixmap( x,y,pm );            
        }
        break;
    }
    case Comment:
    {
/* comments are found in both policy and nat rules, so we cast to Rule here */
        Rule *rule = Rule::cast( ruleIndex[row] );
        if (rule==NULL) return;

        QRect br=p.boundingRect(x, y, 1000, 1000,
                                 Qt::AlignLeft|Qt::AlignVCenter,
                                QString::fromUtf8(rule->getComment().c_str()) );
        p.drawText( x, y, br.width(), br.height(),
		    Qt::AlignLeft|Qt::AlignVCenter,
                    QString::fromUtf8(rule->getComment().c_str()) );

        break;
    }
    default:
        break;
    }  // switch

    p.end();

    pntr->drawPixmap( 0, 0, bufferpixmap );

    return;
}

RuleSetView::REType  RuleSetView::getColType(int col) const
{
    map<int,REType>::const_iterator i = colTypes.find(col);
    return i->second;
}

/***********************************************************
 *    I do not use in-place editing anymore,
 */
QWidget *RuleSetView::createEditor( int row, int col, bool initFromCell ) const
{
    return NULL;
}

void RuleSetView::selectRE( int row, int col)
{
    mw->selectRules();

    if (row!=currentRow() || col!=currentColumn())
    {
        selectedObject=NULL;
        updateCell(currentRow(),currentColumn());
    }
}

void RuleSetView::selectRE(libfwbuilder::FWReference *ref)
{
    mw->selectRules();

    selectedObject = ref->getPointer();

/* need to find row and column this object is in and show it */
    FWObject *re = ref->getParent();
    Rule     *r  = Rule::cast(re->getParent());
    assert(r!=NULL);

    int row      = r->getPosition();
    int col;
    for (col=0; col<ncols; ++col)
        if (re==getRE(r,col))
        {
            setCurrentCell(row,col);
            ensureCellVisible(row,col);
            updateCell(row,col);
            break;
        }
}

void RuleSetView::unselect()
{
//    selectedObject=NULL;
    setCurrentCell(-1,-1);
    updateCell(currentRow(),currentColumn());
}

bool RuleSetView::isSelected()
{
    return (selectedObject!=NULL);
}

void RuleSetView::horizontalHeaderClicked(int col)
{
}

void RuleSetView::verticalHeaderClicked(int row)
{
}

void RuleSetView::doubleClicked(int row,int col,int btn,const QPoint &pos)
{
    if (row<0) return;

    switch (getColType(col))
    {
    case Comment:
        editComment();
        break;

    default:
    {
        FWObject *obj = getObj(row,col,pos.y());
        if (obj==NULL)
        {
//        QTable::doubleClicked(row,col,btn,pos);
            return;
        }
        om->openObject(obj);
        om->edit(obj);
    }
    }
}

/*
 *  pos is in coordinates relative to the header
 */
void RuleSetView::headerMousePressEvent(const QPoint &pos)
{
    int row = rowAt( pos.y() );
    contextMenu(row, -1, pos);
}

void RuleSetView::selectionChanged()
{
    if (fwbdebug)
    {
        qDebug("RuleSetView::selectionChanged()");
        qDebug("changingSelection   =%d",changingSelection);
        qDebug("numSelections       =%d",numSelections());
    }

    if (changingSelection || numSelections()==0) return;

    int firstSelectedRule=-1;
    int lastSelectedRule=-1;

/*
 * we get multiple selections when user uses Ctrl+mouse click to
 * select several rows. Reset multiple selections so we end up with a
 * single one. In effect, Ctrl-click should behave like normal click,
 * that is, it should select a rule the user clicked on with or
 * without Ctrl.  Multiple rules can be selected with
 * Shift-click. Shift-click creates one large selection.
 */
    while (numSelections()>1)
    {
        for (int i=0; i<numSelections(); ++i)
        {
            QTableSelection sel=selection(i);
            if (sel.isActive())
            {
                if (fwbdebug)
                {
                    qDebug("sel %d: tRow=%d",i,sel.topRow());
                    qDebug("        bRow=%d",sel.bottomRow());
                }

                if (i==0)
                {
                    removeSelection(sel);
                    break;
                }
            }
        }
    }

    QTableSelection sel=selection(0);

    if (fwbdebug)
    {
        qDebug("Reset selection:");
        qDebug("      sel.isActive()=%d",sel.isActive());
        qDebug("              topRow=%d",sel.topRow());
        qDebug("           bottomRow=%d",sel.bottomRow());
        qDebug("             leftCol=%d",sel.leftCol());
        qDebug("            rightCol=%d",sel.rightCol());
    }

    if (sel.isActive())
    {
        firstSelectedRule=sel.topRow();
        lastSelectedRule=sel.bottomRow();

        int selectionSize=lastSelectedRule-firstSelectedRule+1;

        mw->copyRuleAction->setEnabled( selectionSize==1 );
        mw->cutRuleAction->setEnabled( selectionSize==1 );
        mw->pasteRuleAboveAction->setEnabled( selectionSize==1 );
        mw->pasteRuleBelowAction->setEnabled( selectionSize==1 );
    }
}

void RuleSetView::contextMenu(int row, int col, const QPoint &pos)
{
    QTableSelection sel=selection(0);

    if (fwbdebug)
    {
        qDebug("RuleSetView::contextMenu()");
        qDebug("Selection:  isActive=%d",sel.isActive());
        qDebug("              topRow=%d",sel.topRow());
        qDebug("           bottomRow=%d",sel.bottomRow());
        qDebug("             leftCol=%d",sel.leftCol());
        qDebug("            rightCol=%d",sel.rightCol());
    }

    int firstSelectedRule=-1;
    int lastSelectedRule=-1;

    if (sel.isActive())
    {
/* if we have an active selection but user called context menu
 * outside selected rows, reset selection. Otherwise work with existing
 * selection which may include several rows.
 */
        if (row<sel.topRow() || row>sel.bottomRow())
        {
            clearSelection();
            sel= QTableSelection(row,0,row,ncols-1);
            addSelection( sel );
            setCurrentCell(row,0);
        }
    } else
    {
        clearSelection();
        sel= QTableSelection(row,0,row,ncols-1);
        addSelection( sel );
        setCurrentCell(row,0);
    }

    firstSelectedRule=sel.topRow();
    lastSelectedRule=sel.bottomRow();

    if (row<0 && ruleset->size()==0)
    {
        QPopupMenu *popup=new QPopupMenu(this);
        addPopupMenuItem( this, popup, "", tr("Insert Rule"),
                          SLOT( insertRule() ) );
        addPopupMenuItem( this, popup, "", tr("Paste Rule"),
                          SLOT( pasteRuleAbove() ) );
        popup->exec( pos );
        delete popup;
        return;
    }

    if (row<0 && ruleset->size()!=0)
    {
/* this is when user clicks under the last rule */

        setCurrentCell(ruleset->size()-1,0);

        QPopupMenu *popup=new QPopupMenu(this);
        addPopupMenuItem( this, popup, "", tr("Paste Rule"),
                          SLOT( pasteRuleBelow() ) );
        popup->exec( pos );
        delete popup;
        return;
    }

//    QPoint   rp = mapFromGlobal( pos);
//    QHeader *hh = horizontalHeader();
//    QHeader *vh = verticalHeader();    
//    int nx = rp.x()-vh->width()-1;
//    int ny = rp.y()-hh->height()-1;
//    objectClicked(row,col,0,QPoint(nx,ny));

    lastPopupMenuAction=None;

    QPopupMenu *popup=new QPopupMenu(this);

    switch (getColType(col))
    {
    case Action:
    {
        addPopupMenuItem( this, popup, "/FWBuilderResources/UI/Icons/Accept",     "Accept",     SLOT( changeActionToAccept()     ) );
        addPopupMenuItem( this, popup, "/FWBuilderResources/UI/Icons/Deny",       "Deny",       SLOT( changeActionToDeny()       ) );
        addPopupMenuItem( this, popup, "/FWBuilderResources/UI/Icons/Reject",     "Reject",     SLOT( changeActionToReject()     ) );
        addPopupMenuItem( this, popup, "/FWBuilderResources/UI/Icons/Accounting", "Accounting", SLOT( changeActionToAccounting() ) );

        break;
    }
    case Direction:
    {
        addPopupMenuItem( this, popup, "/FWBuilderResources/UI/Icons/Inbound",  "Inbound",    SLOT( changeDirectionToIn()      ) );
        addPopupMenuItem( this, popup, "/FWBuilderResources/UI/Icons/Outbound", "Outbound",   SLOT( changeDirectionToOut()     ) );
        addPopupMenuItem( this, popup, "/FWBuilderResources/UI/Icons/Both",     "Both",       SLOT( changeDirectionToBoth()    ) );

        break;
    }
    case Options:
    {
        addPopupMenuItem( this, popup, "/FWBuilderResources/UI/Icons/Options","Rule Options", SLOT( editRuleOptions()          ) );
        addPopupMenuItem( this, popup, "/FWBuilderResources/UI/Icons/Log",    "Logging On",   SLOT( changeLogToOn()            ) );
        addPopupMenuItem( this, popup, "/FWBuilderResources/UI/Icons/Blank",  "Logging Off",   SLOT( changeLogToOff()          ) );

        break;
    }
    case Object:
    case Time:
    {
        RuleElement *re = getRE(row,col);
        if(re==NULL) return;

        int editID = popup->insertItem( tr("Edit")   , this , SLOT( editSelectedObject() ) );
        popup->insertSeparator();
        int copyID = popup->insertItem( tr("Copy")   , this , SLOT( copySelectedObject() ) );
        int cutID  = popup->insertItem( tr("Cut")    , this , SLOT( cutSelectedObject() ) );
        popup->insertItem( tr("Paste")  , this , SLOT( pasteObject() ) );
//        popup->insertSeparator();
        int delID  =popup->insertItem( tr("Delete") ,  this , SLOT( deleteSelectedObject() ) );
        popup->insertSeparator();
        int negID  = popup->insertItem( tr("Negate") , this , SLOT( negateRE() ) );

        if (selectedObject==NULL || re->isAny())
            popup->setItemEnabled(editID, false);
        popup->setItemEnabled(copyID, !re->isAny());
        popup->setItemEnabled(cutID,  !re->isAny());
        popup->setItemEnabled(delID,  !re->isAny());

        string cap_name;
        if (InterfacePolicy::cast(ruleset)!=NULL) cap_name="negation_in_interface_policy";
        if (Policy::cast(ruleset)!=NULL)          cap_name="negation_in_policy";
        if (NAT::cast(ruleset)!=NULL)             cap_name="negation_in_nat";

        FWObject *o = ruleset;
        while (o!=NULL && Firewall::cast(o)==NULL) o=o->getParent();
        assert(o!=NULL);
        Firewall *f=Firewall::cast(o);

        bool supports_neg=false;
        try  {
            supports_neg=Resources::getTargetCapabilityBool(f->getStr("platform"),
                                                            cap_name);
        } catch (FWException &ex)
        {
            QMessageBox::critical( NULL , "Firewall Builder",
                                   ex.toString().c_str(),
                                   QString::null,QString::null);
        }
        popup->setItemEnabled(negID, supports_neg &&  !re->isAny());

        break;
    }

    case RuleOp:
    {
//        setCurrentCell(row,0);
        setFocus();

        Rule *rule = Rule::cast(ruleIndex[row]);
        if (rule==NULL)
        {
            addPopupMenuItem( this, popup, "", tr("Insert Rule"),       SLOT( insertRule()          ) );
        } else
        {
            int rn = rule->getPosition();
            int selectionSize=lastSelectedRule-firstSelectedRule+1;
            QLabel *l;
            if (selectionSize!=1)
            {
                l=new QLabel(QString("  ")+tr("Rules: %1-%2").arg(firstSelectedRule).arg(lastSelectedRule), popup);
            } else
            {
                l=new QLabel(QString("  ")+tr("Rule: %1").arg(rn), popup);
            }
            l->setAlignment( Qt::AlignHCenter );
            popup->insertItem( l );
            popup->insertSeparator();
            l=new QLabel(QString("  ")+tr("Color Label:"), popup);
            popup->insertItem( l );
            ColorLabelMenuItem *cl = new ColorLabelMenuItem(popup);
            popup->insertItem(cl);

            connect( cl,   SIGNAL( returnColor(const QString&) ),
                     this, SLOT( setRuleColor(const QString&) ) );

            popup->insertSeparator();

            QString itemLbl;

            addPopupMenuItem( this, popup, "", tr("Insert Rule"),
                              SLOT( insertRule() ) );
            addPopupMenuItem( this, popup, "", tr("Add Rule Below"),
                              SLOT( addRuleAfterCurrent() ) );

            if (selectionSize==1) itemLbl=tr("Remove Rule");
            else                  itemLbl=tr("Remove Rules");
            addPopupMenuItem( this, popup, "", itemLbl,
                              SLOT( removeRule()));
            if (selectionSize==1) itemLbl=tr("Move Rule");
            else                  itemLbl=tr("Move Rules");
            addPopupMenuItem( this, popup, "", itemLbl,
                              SLOT( moveRule()));

            popup->insertSeparator();

            addPopupMenuItem( this, popup, "", tr("Copy Rule"),
                              SLOT( copyRule() ) );
            addPopupMenuItem( this, popup, "", tr("Cut Rule"),
                              SLOT( cutRule() ) );
            addPopupMenuItem( this, popup, "", tr("Paste Rule Above"),
                              SLOT( pasteRuleAbove() ) );
            addPopupMenuItem( this, popup, "", tr("Paste Rule Below"),
                              SLOT( pasteRuleBelow() ) );

            popup->insertSeparator();
            Rule *r = Rule::cast( ruleIndex[row] );
            if (r->isDisabled())
            {
                if (selectionSize==1) itemLbl=tr("Enable Rule");
                else                  itemLbl=tr("Enable Rules");
                addPopupMenuItem( this, popup, "", itemLbl,
                                  SLOT( enableRule() ) );
            }else{
                if (selectionSize==1) itemLbl=tr("Disable Rule");
                else                  itemLbl=tr("Disable Rules");
                addPopupMenuItem( this, popup, "", itemLbl,
                                  SLOT( disableRule() ) );
            }
        }
        break;
    }

    case Comment:
        popup->insertItem( tr("Edit")   , this , SLOT( editComment() ) );
        break;

    default:
        popup->insertItem( tr("Edit")   , this , SLOT( editRE() ) );
        break;
    }

    

    popup->exec( pos );

    delete popup;
}

void RuleSetView::setRuleColor(const QString &c)
{
    if (!isTreeReadWrite(this,ruleset)) return;

    int firstSelectedRule=-1;
    int lastSelectedRule=-1;

    QTableSelection sel=selection(0);
    if (sel.isActive())
    {
        firstSelectedRule=sel.topRow();
        lastSelectedRule=sel.bottomRow();
//        removeSelection(0);
//        verticalHeader()->update();
    } else
    {
        firstSelectedRule=currentRow();
        lastSelectedRule=currentRow();
    }

    if ( firstSelectedRule!=-1 )
    {
        for (int i=firstSelectedRule; i<=lastSelectedRule; ++i)
        {
            Rule *rule = Rule::cast( ruleIndex[i] );
            FWOptions *ropt = rule->getOptionsObject();
            ropt->setStr("color",c.latin1());
            dirtyRows[i]=1;
            adjustRow(i);   // this causes repaint
        }
    }
}

void RuleSetView::changeActionToAccept()
{
    if (!isTreeReadWrite(this,ruleset)) return;

    if ( currentRow()!=-1 && currentColumn()!=-1 )
    {
        PolicyRule *rule = PolicyRule::cast( ruleIndex[currentRow()] );
        rule->setAction( PolicyRule::Accept );
    }
}

void RuleSetView::changeActionToDeny()
{
    if (!isTreeReadWrite(this,ruleset)) return;

    if ( currentRow()!=-1 && currentColumn()!=-1 )
    {
        PolicyRule *rule = PolicyRule::cast( ruleIndex[currentRow()] );
        rule->setAction( PolicyRule::Deny );
    }
}

void RuleSetView::changeActionToReject()
{
    if (!isTreeReadWrite(this,ruleset)) return;

    if ( currentRow()!=-1 && currentColumn()!=-1 )
    {
        PolicyRule *rule = PolicyRule::cast( ruleIndex[currentRow()] );
        rule->setAction( PolicyRule::Reject );
    }
}

void RuleSetView::changeActionToAccounting()
{
    if (!isTreeReadWrite(this,ruleset)) return;

    if ( currentRow()!=-1 && currentColumn()!=-1 )
    {
        PolicyRule *rule = PolicyRule::cast( ruleIndex[currentRow()] );
        rule->setAction( PolicyRule::Accounting );
    }
}

void RuleSetView::changeDirectionToIn()
{
    if (!isTreeReadWrite(this,ruleset)) return;

    if ( currentRow()!=-1 && currentColumn()!=-1 )
    {
        PolicyRule *rule = PolicyRule::cast( ruleIndex[currentRow()] );
        rule->setDirection( PolicyRule::Inbound );
    }
}

void RuleSetView::changeDirectionToOut()
{
    if (!isTreeReadWrite(this,ruleset)) return;

    if ( currentRow()!=-1 && currentColumn()!=-1 )
    {
        PolicyRule *rule = PolicyRule::cast( ruleIndex[currentRow()] );
        rule->setDirection( PolicyRule::Outbound );
    }
}

void RuleSetView::changeDirectionToBoth()
{
    if (!isTreeReadWrite(this,ruleset)) return;

    if ( currentRow()!=-1 && currentColumn()!=-1 )
    {
        PolicyRule *rule = PolicyRule::cast( ruleIndex[currentRow()] );
        rule->setDirection( PolicyRule::Both );
    }
}

void RuleSetView::editRuleOptions()
{
//    if (!isTreeReadWrite(this,ruleset)) return;

    if ( currentRow()!=-1 && currentColumn()!=-1 )
    {
        PolicyRule *rule = PolicyRule::cast( ruleIndex[currentRow()] );

        oe->open(rule);
        if (!oe->isVisible())
        {
//            st->restoreGeometry(oe, QRect(960,200,180,600) );
            oe->show();
        }
    }
}

void RuleSetView::updateCurrentCell()
{
//    setCurrentCell( currentRow(), currentColumn() );
    updateCell( currentRow(),currentColumn());
}

void RuleSetView::updateAll()
{
    int r=0;
    for (FWObject::iterator i=ruleset->begin(); i!=ruleset->end(); i++,r++)
        dirtyRows[r] = 1;

    updateContents();
}

void RuleSetView::changeLogToOn()
{
    if (!isTreeReadWrite(this,ruleset)) return;

    if ( currentRow()!=-1 && currentColumn()!=-1 )
    {
        PolicyRule *rule = PolicyRule::cast( ruleIndex[currentRow()] );
        rule->setLogging( true );
    }
}

void RuleSetView::changeLogToOff()
{
    if (!isTreeReadWrite(this,ruleset)) return;

    if ( currentRow()!=-1 && currentColumn()!=-1 )
    {
        PolicyRule *rule = PolicyRule::cast( ruleIndex[currentRow()] );
        rule->setLogging( false );
    }
}

void RuleSetView::editSelectedObject()
{
/* need to store pointer to selected object because opening it in the
 * object manipulator automatically unselects policy view (so
 * selectedObject becomes NULL)
 */

    FWObject *obj=selectedObject;
    if ( selectedObject!=NULL) om->openObject(selectedObject,true);
    om->edit(obj);
}

void RuleSetView::copySelectedObject()
{
    if ( selectedObject!=NULL)
    {
        FWObjectClipboard::obj_clipboard->clear();
        FWObjectClipboard::obj_clipboard->add( selectedObject );
    }
}

void RuleSetView::cutSelectedObject()
{
    if (!isTreeReadWrite(this,ruleset)) return;

    if ( selectedObject!=NULL)
    {
        FWObjectClipboard::obj_clipboard->clear();
        FWObjectClipboard::obj_clipboard->add( selectedObject );
        deleteSelectedObject();
    }
}

void RuleSetView::deleteSelectedObject()
{
    if (!isTreeReadWrite(this,ruleset)) return;

    if ( selectedObject!=NULL)
    {
        deleteObject(currentRow(),currentColumn(),selectedObject);
        selectedObject=NULL;
    }
}

void RuleSetView::deleteObject(int row, int col, FWObject *obj)
{
    RuleElement *re = getRE(row,col);
    if (re==NULL || re->isAny()) return;

    re->removeRef(obj);

    if (re->isAny()) re->setNeg(false);

    dirtyRows[row]=1;
    adjustColumn(col);
    adjustRow(row);
    updateCell(row,col);
}

bool RuleSetView::insertObject(int row, int col, FWObject *obj)
{
    if (!isTreeReadWrite(this,ruleset)) return false;

    if (getColType(col)!=Object && getColType(col)!=Time) return false;

    RuleElement *re = getRE(row,col);
    assert (re!=NULL);

    if (! re->validateChild(obj) ) return false;

    if (re->getAnyElementId()==obj->getId()) return false;

    if (! re->isAny())
    {
/* avoid duplicates */
        string cp_id=obj->getId();
        list<FWObject*>::iterator j;
        for(j=re->begin(); j!=re->end(); ++j)     
        {
            FWObject *o=*j;
            if(cp_id==o->getId()) return false;

            FWReference *ref;
            if( (ref=FWReference::cast(o))!=NULL &&
                cp_id==ref->getPointerId()) return false;
        }
    }

    re->addRef(obj);
    dirtyRows[row]=1;

    adjustColumn(col);
    adjustRow(row);
    updateCell(row,col);
    
    return true;
}

void RuleSetView::pasteObject()
{
    if (!isTreeReadWrite(this,ruleset)) return;

    vector<FWObject*>::iterator i;
    for (i= FWObjectClipboard::obj_clipboard->begin();
         i!=FWObjectClipboard::obj_clipboard->end(); ++i)
    {
        FWObject *co= *i;
        if (Rule::cast(co)!=NULL)  pasteRuleAbove();
        else
        {
            if (currentRow()>=0)
                insertObject(currentRow(),currentColumn(),co);
        }
    }

/*
    if (FWObjectClipboard::obj_clipboard->getObject()!=NULL)
        insertObject(currentRow(),currentColumn(),
                     FWObjectClipboard::obj_clipboard->getObject() );
*/
}

void RuleSetView::negateRE()
{
    if (!isTreeReadWrite(this,ruleset)) return;

    if ( currentRow()!=-1 && currentColumn()!=-1 )
    {
        RuleElement *re = getRE(currentRow(),currentColumn());
        if (re==NULL) return;
        re->toggleNeg();
        updateCell(currentRow(),currentColumn());
    }
}

void RuleSetView::editRE()
{
    if (!isTreeReadWrite(this,ruleset)) return;

    editCell(currentRow(),currentColumn());
}

void RuleSetView::editComment()
{
    if (!isTreeReadWrite(this,ruleset)) return;

    int rn=currentRow();
    Rule *r = Rule::cast( ruleIndex[rn] );

    SimpleTextEditor edt( QString::fromUtf8(r->getComment().c_str()),
                          false, tr("Comment Editor") );
    if ( edt.exec() == QDialog::Accepted )
    {
        r->setComment( string(edt.text().utf8()) );
        adjustColumn(currentColumn());
        adjustRow(currentRow());
        updateCell(currentRow(),currentColumn());
    }
}

FWObject* RuleSetView::getObj(int row, int col, int mouse_y_pos, QRect *objr)
{
    RuleElement *re = getRE(row,col);
    if (re==NULL) return NULL;

    QRect cr=cellGeometry(row,col);

/*
 * n     is the number of objects in the cell
 * y_rel is a distance of the mouse cursor from the top of the cell
 * h     is the cell height
 */
    int   y_rel = mouse_y_pos-cr.y();
    int   y_obj = cr.y();
    int   on=0;
    int   oy=0;
    FWObject *o1=NULL;
    FWObject *obj=NULL;
    FWObject *prev=NULL;
    for (FWObject::iterator i=re->begin(); i!=re->end(); i++,on++)
    {
        o1= *i;
        if (FWReference::cast(o1)!=NULL) o1=FWReference::cast(o1)->getPointer();
        if (y_rel>oy && y_rel<oy+item_h)
        {
            obj=o1;
            break;
        }
        oy+=item_h;
        y_obj+=item_h;
        prev=o1;
    }
    if (obj==NULL) obj=prev;

    cr.setY(y_obj);
    cr.setHeight(item_h);

    if (objr!=NULL) *objr=cr;

    return obj;
}

void RuleSetView::contentsMousePressEvent( QMouseEvent* ev )
{
    dragging = true;

    mw->selectRules();

    int row=rowAt(ev->y());
    int col=columnAt(ev->x());

    RuleElement *re = getRE(row,col);
    if (isEditing() || re==NULL)
    {
        selectedObject=NULL;
        QTable::contentsMousePressEvent(ev);
        return;
    }

    FWObject *obj=getObj(row,col,ev->y());

    bool needUpdate= (row==currentRow() && col==currentColumn() && selectedObject!=obj);
    selectedObject=obj;
    info();
    if (needUpdate) updateCell(row,col);

    QTable::contentsMousePressEvent(ev);
}

void RuleSetView::contentsMouseMoveEvent( QMouseEvent* ev )
{
    QTable::contentsMouseMoveEvent(ev);
    return;

    if (dragging)
    {
        QDragObject* d = dragObject();
        d->dragMove();
        dragging =false;
    }
}

void RuleSetView::keyPressEvent( QKeyEvent* ev )
{
    mw->selectRules();

    RuleElement *re;

    if (ev->key()==Qt::Key_Left || ev->key()==Qt::Key_Right)
    {
        int shift= (ev->key()==Qt::Key_Left) ? -1 : 1;

/* keyboard 'Left' or 'Right', switch to the object with the same
 * number in the cell to the left or to the right
 */
        int objno=0;
        re = getRE(currentRow(),currentColumn());
        if (re!=NULL)
        {
            for (FWObject::iterator i=re->begin(); i!=re->end(); ++i,++objno)
            {
                FWObject *o1= *i;
                if (FWReference::cast(o1)!=NULL) o1=FWReference::cast(o1)->getPointer();
                if (o1==selectedObject) break;
            }
        }
        selectedObject=NULL;
        re = getRE(currentRow(),currentColumn() + shift);
        if (re==NULL) { QTable::keyPressEvent(ev); return; }
        int n=0;
        for (FWObject::iterator i=re->begin(); i!=re->end(); ++i,++n)
        {
            FWObject *o1= *i;
            if (FWReference::cast(o1)!=NULL) o1=FWReference::cast(o1)->getPointer();
            if (n==objno)
            {
                selectedObject= o1;
                info();
                break;
            }
        }
        if (selectedObject==NULL)
        {
            FWObject *o1=re->back();
            if (FWReference::cast(o1)!=NULL) o1=FWReference::cast(o1)->getPointer();
            selectedObject=o1;
            info();
        }
        QTable::keyPressEvent(ev);
        return;
    }

    if (ev->key()==Qt::Key_Down || ev->key()==Qt::Key_Up)
    {
        re = getRE(currentRow(),currentColumn());
        if (re==NULL) { selectedObject=NULL; QTable::keyPressEvent(ev); return; }

        FWObject          *prev=NULL;
        FWObject          *o1  =NULL;
        FWObject::iterator i;
        for (i=re->begin(); i!=re->end(); ++i)
        {
            o1= *i;
            if (FWReference::cast(o1)!=NULL) o1=FWReference::cast(o1)->getPointer();
            if (ev->key()==Qt::Key_Up   && o1==selectedObject)   break;
            if (ev->key()==Qt::Key_Down && prev==selectedObject) break;
            prev=o1;
        }
        if (ev->key()==Qt::Key_Up && prev==NULL)
        {
/* keyboard 'Up',  switch to the last object in the cell above */
            if (currentRow()-1<0) return;

            re = getRE(currentRow()-1,currentColumn());
            if (re!=NULL)  // can be NULL if currentRow is 0
            {
                o1=re->back();
                if (FWReference::cast(o1)!=NULL) o1=FWReference::cast(o1)->getPointer();
                selectedObject=o1;
                info();
            }
            QTable::keyPressEvent(ev);
            return;
        }
        if (ev->key()==Qt::Key_Down && i==re->end())
        {
/* keyboard 'Down',  switch to the first object in the cell below */
            if (currentRow()+1>=int(ruleset->size())) return;

            re = getRE(currentRow()+1,currentColumn());
            if (re!=NULL)
            {
                o1=re->front();
                if (FWReference::cast(o1)!=NULL) o1=FWReference::cast(o1)->getPointer();
                selectedObject=o1;
                info();
            }
            QTable::keyPressEvent(ev);
            return;
        }
/* switching to another object in the same cell */
        selectedObject = (ev->key()==Qt::Key_Up) ? prev : o1;
        info();
        updateCell(currentRow(),currentColumn());
        ev->accept();
        return;
    }
    QTable::keyPressEvent(ev);
}

QDragObject* RuleSetView::dragObject()
{
    FWObject *obj = selectedObject;
    if (obj==NULL) return NULL;

    QString icn_filename =
        Resources::global_res->getObjResourceStr(obj, "icon").c_str();

    list<FWObject*> dragobj;
    dragobj.push_back(obj);

    FWObjectDrag    *drag = new FWObjectDrag(dragobj, this, NULL);

    QPixmap          pm   = QPixmap::fromMimeSource( icn_filename );
    drag->setPixmap( pm,
                     QPoint( pm.rect().width() / 2,
                             pm.rect().height() / 2 ) );

    return drag;
}

void RuleSetView::dragEnterEvent( QDragEnterEvent *ev)
{
    ev->acceptAction( QTextDrag::canDecode(ev) );
}

void RuleSetView::contentsDragEnterEvent( QDragEnterEvent *ev)
{
    ev->acceptAction( QTextDrag::canDecode(ev) );
}

void RuleSetView::dragMoveEvent( QDragMoveEvent *ev)
{
    if (FWObjectDrag::canDecode(ev) && !ruleset->isReadOnly())
    {
        QHeader *hh = horizontalHeader();
        QHeader *vh = verticalHeader();
    
        int  row = rowAt( ev->pos().y() + contentsY() - hh->height() );
        int  col = columnAt( ev->pos().x() + contentsX() - vh->width() );

        if (col<0 || ( getColType(col)!=Object && getColType(col)!=Time) )
        {
            ev->acceptAction(false);
            return;
        }

        RuleElement *re = getRE(row,col);
        if (re==NULL)
        {
            ev->acceptAction(false);
            return;
        }

        bool  acceptE = true;
        list<FWObject*> dragol;
        if (FWObjectDrag::decode(ev, dragol))
        {
            for (list<FWObject*>::iterator i=dragol.begin();
                 i!=dragol.end(); ++i)
            {
                FWObject *dragobj = *i;
                assert(dragobj!=NULL);
                
                acceptE &= re->validateChild(dragobj);
            }
            ev->acceptAction( acceptE );
            return;
        }
    }
    ev->accept(false);
}


void RuleSetView::dropEvent( QDropEvent *ev)
{
    if (!isTreeReadWrite(this,ruleset)) return;

    QHeader *hh = horizontalHeader();
    QHeader *vh = verticalHeader();

    int  row = rowAt( ev->pos().y() + contentsY() - hh->height() );
    int  col = columnAt( ev->pos().x() + contentsX() - vh->width() );

    if (row<0 || col<0) return;

/* without this check the user can drag and drop an object inside the
 * same rule element. This is bad because it is considered a change,
 * even though nothing really changes. With this check, we can not
 * drag and drop an object from the tree into a selected cell... 

    if (row==currentRow() && col==currentColumn()) return;
 */

    if (fwbdebug)
    {
        qDebug("RuleSetView::dropEvent  drop event mode=%d", ev->action());
        qDebug("                        src widget = %p", ev->source());
        qDebug("                              this = %p", this   );
    }

    list<FWObject*> dragol;
    if (FWObjectDrag::decode(ev, dragol))
    {
        for (list<FWObject*>::iterator i=dragol.begin();
             i!=dragol.end(); ++i)
        {
            FWObject *dragobj = *i;
            assert(dragobj!=NULL);

            if (ev->source()!=this)
            {
                insertObject(row,col,dragobj);
            } else
            {
                switch (ev->action())
                {
                case QDropEvent::Move:
                    if (insertObject(row,col,dragobj) )
                        deleteObject(currentRow(),currentColumn(),dragobj);
                    break;

                default:
                    insertObject(row,col,dragobj);
                    break;
                }
            }
        }
        ev->acceptAction();
    }
}

void RuleSetView::removeRule()
{
    if (!hasFocus()) return;
    if (!isTreeReadWrite(this,ruleset)) return;

    int firstSelectedRule=-1;
    int lastSelectedRule=-1;

    QTableSelection sel=selection(0);
    if (sel.isActive())
    {
        firstSelectedRule=sel.topRow();
        lastSelectedRule=sel.bottomRow();
        clearSelection();
//        removeSelection(0);
        verticalHeader()->update();
    } else
    {
        firstSelectedRule=currentRow();
        lastSelectedRule=currentRow();
    }

/* remove rules firstSelectedRule through lastSelectedRule */

    if ( firstSelectedRule!=-1 )
    {
        clearSelection();
        verticalHeader()->update();

        setUpdatesEnabled(false);
        for (int rn=lastSelectedRule; rn>=firstSelectedRule; --rn)
        {
            if ( ruleset->deleteRule(rn) )
            {
                int lastN=ruleIndex.size()-1;
                ruleIndex.erase(rn);

                for (int i=rn; i<lastN; ++i)   ruleIndex[i]=ruleIndex[i+1];

                removeRow(rn);

                for (int row=rn; row<lastN; ++row)
                    setRuleNumber(row, Rule::cast( ruleIndex[row] ));

            }
        }
        setUpdatesEnabled(true);

        setCurrentCell( firstSelectedRule, currentColumn() );
        updateContents();
    }
}

void RuleSetView::insertRule()
{
//    if (!hasFocus()) return;       // <-- can insert rule even if does not have focus
    if (!isTreeReadWrite(this,ruleset)) return;

    int firstSelectedRule=-1;
    int lastSelectedRule=-1;

    QTableSelection sel=selection(0);
    if (sel.isActive())
    {
        firstSelectedRule=sel.topRow();
        lastSelectedRule=sel.bottomRow();
//        removeSelection(0);
//        verticalHeader()->update();
    } else
    {
        firstSelectedRule=currentRow();
        lastSelectedRule=currentRow();
    }

    insertRule(firstSelectedRule,NULL);
}

void RuleSetView::addRuleAfterCurrent()
{
    if (!hasFocus()) return;
    if (!isTreeReadWrite(this,ruleset)) return;

    int firstSelectedRule=-1;
    int lastSelectedRule=-1;

    QTableSelection sel=selection(0);
    if (sel.isActive())
    {
        firstSelectedRule=sel.topRow();
        lastSelectedRule=sel.bottomRow();
//        removeSelection(0);
//        verticalHeader()->update();
    } else
    {
        firstSelectedRule=currentRow();
        lastSelectedRule=currentRow();
    }

    insertRule(lastSelectedRule+1,NULL);
}

Rule* RuleSetView::insertRule(int pos, FWObject *r)
{
    if (r!=NULL && 
        ruleset->getTypeName()==Policy::TYPENAME && 
        r->getTypeName()!=PolicyRule::TYPENAME)  return NULL;
    if (r!=NULL && 
        ruleset->getTypeName()==NAT::TYPENAME    && 
        r->getTypeName()!=NATRule::TYPENAME   )  return NULL;

    if (pos<0) pos=0;

//    insertRows( pos , 1 );

    Rule *newrule=NULL;
    if ( ruleset->getRuleSetSize()==0) newrule=ruleset->insertRuleAtTop();
    else
    {
        if (pos==ruleset->getRuleSetSize())
        {
            newrule=ruleset->appendRuleAtBottom();
        } else
            newrule=ruleset->insertRuleBefore(pos);
        assert(newrule!=NULL);
    }

    if (Policy::cast(ruleset) || InterfacePolicy::cast(ruleset))
    {
        (PolicyRule::cast(newrule))->setLogging(supports_logging);
        (PolicyRule::cast(newrule))->setAction(PolicyRule::Deny);
    }

    if (InterfacePolicy::cast(ruleset))
        PolicyRule::cast(newrule)->setDirection(PolicyRule::Both);

    if (r!=NULL)  copyRuleContent(newrule,Rule::cast(r));

//    if (InterfacePolicy::cast(ruleset) && 
//        (r==NULL || PolicyRule::cast(r)->getDirection()==PolicyRule::Undefined))
//        PolicyRule::cast(newrule)->setDirection(PolicyRule::Both);

    for (int i=ruleIndex.size(); i>pos; --i)  ruleIndex[i]=ruleIndex[i-1];
    ruleIndex[pos] = newrule;

   insertRows( pos , 1 );

    for (int i=ruleIndex.size(); i>=pos; --i)
        setRuleNumber(i, Rule::cast(ruleIndex[i]));

    dirtyRows[pos]=1;
////    adjustRow(pos);

    setCurrentCell( pos, currentColumn() );
    updateCell(pos,currentColumn());

    return newrule;
}

void RuleSetView::copyRuleContent(Rule *dst, Rule *src)
{
    string id=dst->getId();
    int     p=dst->getPosition();

    if ( src->isDisabled() ) dst->disable();
    else                     dst->enable();

    map<string, string>::const_iterator i;
    for(i=dst->dataBegin(); i!=dst->dataEnd(); ++i) {
        string f= (*i).first;
        dst->setStr(f, src->getStr(f) );
    }

    dst->setComment( src->getComment() );

    list<FWObject*>::iterator j;
    for(j=dst->begin(); j!=dst->end(); ++j) {
        string    dtype= (*j)->getTypeName();
        FWObject *selem= src->getFirstByType(dtype);
        if (selem!=NULL) 
            (*j)->duplicate(selem);
    }

    if (id!="") dst->setId(id);
    dst->setPosition(p);
}

void RuleSetView::moveRule()
{
    if (!hasFocus()) return;

    int firstSelectedRule=-1;
    int lastSelectedRule=-1;

    QTableSelection sel=selection(0);
    if (sel.isActive())
    {
        firstSelectedRule=sel.topRow();
        lastSelectedRule=sel.bottomRow();
    } else
    {
        firstSelectedRule=currentRow();
        lastSelectedRule=currentRow();
    }
    int selectionSize=lastSelectedRule-firstSelectedRule+1;

/* calculate acceptable range of rule numbers for the dialog */
    int minRN=0;
    int maxRN=ruleset->size()-selectionSize;

    askRuleNumberDialog_q d(this);
    d.newRuleNum->setMinValue(minRN);
    d.newRuleNum->setMaxValue(maxRN);
    
    if (d.exec()==QDialog::Accepted)
    {
        int newN = d.newRuleNum->value();
        int nn   = newN;
        if (firstSelectedRule==nn) return;

        clearSelection();
//        removeSelection(0);
        verticalHeader()->update();

        setUpdatesEnabled(false);

        if (firstSelectedRule>nn)
        {  // moving block of rules up
            for (int i=firstSelectedRule; i<=lastSelectedRule; i++)
            {
                int j=i;
                while (j!=nn)
                {
                    if (!ruleset->moveRuleUp(j)) return;

                    FWObject *r=ruleIndex[j];
                    ruleIndex[j]=ruleIndex[j-1];
                    ruleIndex[j-1]=r;

                    swapRows(j-1,j);

                    dirtyRows[j-1]=1;
                    dirtyRows[j]=1;

                    Rule *rule = Rule::cast( ruleIndex[j-1] );
                    setRuleNumber(j-1,rule);
                    rule = Rule::cast( ruleIndex[j] );
                    setRuleNumber(j,rule);

                    --j;
                }
                nn++;
            }
        } else 
        {   // moving block of rules down
            for (int i=lastSelectedRule; i>=firstSelectedRule; i--)
            {
                int j=i;
                while (j!=nn+selectionSize-1)
                {
                    if (!ruleset->moveRuleDown(j)) return;

                    FWObject *r=ruleIndex[j];
                    ruleIndex[j]=ruleIndex[j+1];
                    ruleIndex[j+1]=r;

                    swapRows(j+1,j);

                    dirtyRows[j+1]=1;
                    dirtyRows[j]=1;

                    Rule *rule = Rule::cast( ruleIndex[j+1] );
                    setRuleNumber(j+1,rule);
                    rule = Rule::cast( ruleIndex[j] );
                    setRuleNumber(j,rule);

                    ++j;
                }
                nn--;
            }
        }


        setUpdatesEnabled(true);

        setCurrentCell( newN, currentColumn() );
        selectRE( newN , currentColumn() );
        updateContents();
    }
}

void RuleSetView::moveRuleUp()
{
    if (!hasFocus()) return;
    int rn=currentRow();
/* swap rule rn and rn-1 */

    if (rn==0) return;

    if (ruleset->moveRuleUp(rn))
    {
        FWObject *r=ruleIndex[rn];
        ruleIndex[rn]=ruleIndex[rn-1];
        ruleIndex[rn-1]=r;

        swapRows(rn-1,rn);

        dirtyRows[rn-1]=1;
        dirtyRows[rn]=1;

        setCurrentCell( rn-1, currentColumn() );
        selectRE( rn-1 , currentColumn() );
    }
}

void RuleSetView::moveRuleDown()
{
    if (!hasFocus()) return;
    int rn=currentRow();
/* swap rule rn and rn+1 */

    if (rn==ruleset->getRuleSetSize()-1) return;

    if (ruleset->moveRuleDown(rn))
    {
        FWObject *r=ruleIndex[rn];
        ruleIndex[rn]=ruleIndex[rn+1];
        ruleIndex[rn+1]=r;

        swapRows(rn+1,rn);

        dirtyRows[rn+1]=1;
        dirtyRows[rn]=1;

        setCurrentCell( rn+1, currentColumn() );
        selectRE( rn+1 , currentColumn() );
    }
}


void RuleSetView::copyRule()
{
    if (!hasFocus()) return;

    int firstSelectedRule=-1;
    int lastSelectedRule=-1;

    QTableSelection sel=selection(0);
    if (sel.isActive())
    {
        firstSelectedRule=sel.topRow();
        lastSelectedRule=sel.bottomRow();
//        removeSelection(0);
//        verticalHeader()->update();
    } else
    {
        firstSelectedRule=currentRow();
        lastSelectedRule=currentRow();
    }

    if ( firstSelectedRule!=-1 )
    {
        FWObjectClipboard::obj_clipboard->clear();
        for (int i=firstSelectedRule; i<=lastSelectedRule; ++i)
        {
            FWObject *rule = ruleIndex[i];
            FWObjectClipboard::obj_clipboard->add( rule );
        }
    }

#if 0
    if (!hasFocus()) return;
    FWObject *rule=ruleIndex[ currentRow() ];

    if ( rule!=NULL)
    {
        FWObjectClipboard::obj_clipboard->clear();
        FWObjectClipboard::obj_clipboard->add( rule );
    }
#endif
}

void RuleSetView::cutRule()
{
    copyRule();
    removeRule();
}

void RuleSetView::pasteRuleAbove()
{
    if (!isTreeReadWrite(this,ruleset)) return;

    int firstSelectedRule=-1;
    int lastSelectedRule=-1;

    QTableSelection sel=selection(0);
    if (sel.isActive())
    {
        firstSelectedRule=sel.topRow();
        lastSelectedRule=sel.bottomRow();
//        removeSelection(0);
//        verticalHeader()->update();
    } else
    {
        firstSelectedRule=currentRow();
        lastSelectedRule=currentRow();
    }

    int n=0;
    vector<FWObject*>::iterator i;
    for (i= FWObjectClipboard::obj_clipboard->begin();
         i!=FWObjectClipboard::obj_clipboard->end(); ++i,++n)
    {
        FWObject *co= *i;
        if (!Rule::cast(co)) continue;
        insertRule( firstSelectedRule+n, co);
    }

//    if (FWObjectClipboard::obj_clipboard->getObject()!=NULL)
//        insertRule( rn, Rule::cast(FWObjectClipboard::obj_clipboard->getObject()) );
}

void RuleSetView::pasteRuleBelow()
{    
    if (!isTreeReadWrite(this,ruleset)) return;

    int firstSelectedRule=-1;
    int lastSelectedRule=-1;

    QTableSelection sel=selection(0);
    if (sel.isActive())
    {
        firstSelectedRule=sel.topRow();
        lastSelectedRule=sel.bottomRow();
//        removeSelection(0);
//        verticalHeader()->update();
    } else
    {
        firstSelectedRule=currentRow();
        lastSelectedRule=currentRow();
    }

    int n=0;
    vector<FWObject*>::iterator i;
    for (i= FWObjectClipboard::obj_clipboard->begin();
         i!=FWObjectClipboard::obj_clipboard->end(); ++i,++n)
    {
        FWObject *co= *i;
        if (!Rule::cast(co)) continue;
        insertRule( lastSelectedRule+1+n, co);
    }

//    if (FWObjectClipboard::obj_clipboard->getObject()!=NULL)
//        insertRule( rn+1, Rule::cast(FWObjectClipboard::obj_clipboard->getObject()) );
}

void RuleSetView::enableRule()
{
    if (!isTreeReadWrite(this,ruleset)) return;

    int firstSelectedRule=-1;
    int lastSelectedRule=-1;

    QTableSelection sel=selection(0);
    if (sel.isActive())
    {
        firstSelectedRule=sel.topRow();
        lastSelectedRule=sel.bottomRow();
    } else
    {
        firstSelectedRule=currentRow();
        lastSelectedRule=currentRow();
    }

    if ( firstSelectedRule!=-1 )
    {
        for (int rn=lastSelectedRule; rn>=firstSelectedRule; --rn)
        {
            Rule *r = Rule::cast( ruleIndex[rn] );
            r->enable();
            setRuleNumber(rn,r);
        }
    }
}

void RuleSetView::disableRule()
{
    if (!isTreeReadWrite(this,ruleset)) return;

    int firstSelectedRule=-1;
    int lastSelectedRule=-1;

    QTableSelection sel=selection(0);
    if (sel.isActive())
    {
        firstSelectedRule=sel.topRow();
        lastSelectedRule=sel.bottomRow();
    } else
    {
        firstSelectedRule=currentRow();
        lastSelectedRule=currentRow();
    }

    if ( firstSelectedRule!=-1 )
    {
        for (int rn=lastSelectedRule; rn>=firstSelectedRule; --rn)
        {
            Rule *r = Rule::cast( ruleIndex[rn] );
            r->disable();
            setRuleNumber(rn,r);
        }
    }
}












PolicyView::PolicyView(Policy *p) : RuleSetView(1, 5)
{
    setName("PolicyView");
    ruleset=p;
    iinit();
    init();
}

void PolicyView::init()
{
    ncols=5 +
        ((supports_time)?1:0) + 
        ((supports_logging && supports_rule_options)?1:0);

    setNumCols(ncols);

    colTypes[-1]=RuleOp;

    int col=0;
    horizontalHeader()->setLabel( col, tr( "Source" ) );      // 0
    colTypes[col++]=Object;

    horizontalHeader()->setLabel( col, tr( "Destination" ) ); // 1
    colTypes[col++]=Object;

    horizontalHeader()->setLabel( col, tr( "Service" ) );     // 2
    colTypes[col++]=Object;

    horizontalHeader()->setLabel( col, tr( "Action" ) );      // 3
    colTypes[col++]=Action;

    if (supports_time)
    {
        horizontalHeader()->setLabel( col, tr( "Time" ) );    // 4
        colTypes[col++]=Time;
    }

    if (supports_logging && supports_rule_options)
    {
        horizontalHeader()->setLabel( col, tr( "Options" ) );
        colTypes[col++]=Options;
    }

    horizontalHeader()->setLabel( col, tr( "Comment" ) );
    colTypes[col]=Comment;
//    setColumnStretchable(col, true);

    RuleSetView::init();
}

RuleElement* PolicyView::getRE( int row, int col ) 
{
    if (row<0) return NULL;

    if (ruleIndex.count(row)==0) return NULL;
    PolicyRule *r = PolicyRule::cast( ruleIndex[row] );
    if(r==NULL) return NULL;
    return getRE(r, col);
}

RuleElement* PolicyView::getRE( Rule* r, int col ) 
{
    string ret;

    switch (getColType(col))
    {
    case Object:
        switch (col)
        {
        case 0: ret=RuleElementSrc::TYPENAME; break;
        case 1: ret=RuleElementDst::TYPENAME; break;
        case 2: ret=RuleElementSrv::TYPENAME; break;
        }
        break;
    case Time:
        ret=RuleElementInterval::TYPENAME; break;
    default: return NULL;
    }

    return RuleElement::cast( r->getFirstByType(ret) );
}

InterfacePolicyView::InterfacePolicyView(InterfacePolicy *p) : RuleSetView(1,7)
{
    setName("InterfacePolicyView");
    ruleset=p;
    iinit();
    init();
}

void InterfacePolicyView::init()
{
    ncols=6 +
        ((supports_time)?1:0) + 
        ((supports_logging && supports_rule_options)?1:0);

    setNumCols(ncols);

    colTypes[-1]=RuleOp;

    int col=0;
    horizontalHeader()->setLabel( col, tr( "Source" ) );      // 0
    colTypes[col++]=Object;

    horizontalHeader()->setLabel( col, tr( "Destination" ) ); // 1
    colTypes[col++]=Object;

    horizontalHeader()->setLabel( col, tr( "Service" ) );     // 2
    colTypes[col++]=Object;

    horizontalHeader()->setLabel( col, tr( "Direction" ) );   // 3
    colTypes[col++]=Direction;

    horizontalHeader()->setLabel( col, tr( "Action" ) );      // 4
    colTypes[col++]=Action;

    if (supports_time)
    {
        horizontalHeader()->setLabel( col, tr( "Time" ) );    // 5
        colTypes[col++]=Time;
    }

    if (supports_logging && supports_rule_options)
    {
        horizontalHeader()->setLabel( col, tr( "Options" ) );
        colTypes[col++]=Options;
    }

    horizontalHeader()->setLabel( col, tr( "Comment" ) );
    colTypes[col]=Comment;
//    setColumnStretchable(col, true);

//    ncols=col;

    RuleSetView::init();
}

RuleElement* InterfacePolicyView::getRE( int row, int col ) 
{
    if (row<0) return NULL;
    PolicyRule *r = PolicyRule::cast( ruleIndex[row] );
    assert(r!=NULL);
    return getRE(r,col);
}

RuleElement* InterfacePolicyView::getRE( Rule *r, int col ) 
{
    string ret;

    switch (getColType(col))
    {
    case Object:
        switch (col)
        {
        case 0: ret=RuleElementSrc::TYPENAME; break;
        case 1: ret=RuleElementDst::TYPENAME; break;
        case 2: ret=RuleElementSrv::TYPENAME; break;
        }
        break;
    case Time:
        ret=RuleElementInterval::TYPENAME; break;
    default: return NULL;
    }

    return RuleElement::cast( r->getFirstByType(ret) );
}





NATView::NATView(NAT *p) : RuleSetView(1,7)
{
    setName("NATView");
    ruleset=p;
    iinit();
    init();
}

void NATView::init()
{
    colTypes[-1]=RuleOp;

    ncols=7;

    int col=0;
    horizontalHeader()->setLabel( col, tr( "Original Src" ) );
    colTypes[col++]=Object;

    horizontalHeader()->setLabel( col, tr( "Original Dst" ) );
    colTypes[col++]=Object;

    horizontalHeader()->setLabel( col, tr( "Original Srv" ) );
    colTypes[col++]=Object;

    horizontalHeader()->setLabel( col, tr( "Translated Src" ) );
    colTypes[col++]=Object;

    horizontalHeader()->setLabel( col, tr( "Translated Dst" ) );
    colTypes[col++]=Object;

    horizontalHeader()->setLabel( col, tr( "Translated Srv" ) );
    colTypes[col++]=Object;

    horizontalHeader()->setLabel( col, tr( "Comment" ) );
    colTypes[col]=Comment;
//    setColumnStretchable(col, true);

//    ncols=col;

    RuleSetView::init();
}

RuleElement* NATView::getRE( int row, int col ) 
{
    if (row<0) return NULL;
    NATRule *r = NATRule::cast( ruleIndex[row] );
    assert(r!=NULL);
    return getRE(r,col);
}

RuleElement* NATView::getRE( Rule *r, int col ) 
{
    string ret;

    switch (getColType(col))
    {
    case Object:
        switch (col)
        {
        case 0: ret=RuleElementOSrc::TYPENAME; break;
        case 1: ret=RuleElementODst::TYPENAME; break;
        case 2: ret=RuleElementOSrv::TYPENAME; break;
        case 3: ret=RuleElementTSrc::TYPENAME; break;
        case 4: ret=RuleElementTDst::TYPENAME; break;
        case 5: ret=RuleElementTSrv::TYPENAME; break;
        }
        break;
    default: return NULL;
    }

    return RuleElement::cast( r->getFirstByType(ret) );
}

