/* 

                          Firewall Builder

                 Copyright (C) 2003 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@fwbuilder.org

  $Id: findDialog.cpp,v 1.8 2004/07/01 03:19:41 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include "config.h"
#include "global.h"
#include "utils.h"

#include "findDialog.h"
#include "ObjectManipulator.h"
#include "FWWindow.h"
#include "FWBTree.h"
#include "FWBSettings.h"

#include "fwbuilder/FWObjectDatabase.h"
#include "fwbuilder/FWReference.h"
#include "fwbuilder/RuleSet.h"
#include "fwbuilder/RuleElement.h"
#include "fwbuilder/Firewall.h"

#include <qcombobox.h>
#include <qcheckbox.h>
#include <qapplication.h>
#include <qcursor.h>
#include <qregexp.h>
#include <qlineedit.h>
#include <qpushbutton.h>
#include <qmessagebox.h>
#include <qtimer.h>

#include <iostream>

using namespace libfwbuilder;
using namespace std;

#define MAX_SEARCH_ITEMS_COUNT 10

findDialog::findDialog(QWidget *p) : findDialog_q(p), treeSeeker()
{
    lastFound=NULL;
    lastSearch=""; 

//    connect( findText->lineEdit() , SIGNAL( textChanged(const QString&) ),
//             this, SLOT( findTextChanged(const QString&) ) );

    findText->setFocus();
}

void findDialog::setObject(FWObject *o)
{
    reset();
    findText->setCurrentText( QString::fromUtf8(o->getName().c_str()) );
}

void findDialog::reset()
{
    lastFound=NULL;
    lastSearch="";
    treeSeeker=mw->db()->tree_begin();
}

void findDialog::findTextChanged(const QString &ns)
{
    if (ns!=lastSearch)  reset();
    lastSearch=ns;
}

void findDialog::find()
{
    if (findText->currentText() != findText->text(0))
    {
        if (findText->count()>=MAX_SEARCH_ITEMS_COUNT)
            findText->removeItem(MAX_SEARCH_ITEMS_COUNT-1);

        findText->insertItem( findText->currentText() , 0 );

        if (fwbdebug)
        {
            qDebug("findDialog::find() : findText->text(0)=%s",
                   findText->text(0).latin1());
        }
    }

    findNext();
}

bool findDialog::match(const QString &name)
{
    QString s=findText->currentText();

    bool res=false;

    if (useRegexp->isChecked()) res= ( name.find( QRegExp(s) )!=-1 );
    else                        res= ( name.find(s)!=-1 );

    return res;
}

void findDialog::findNext()
{
    QString s=findText->currentText();

    if (s.isEmpty()) return;

    if (findText->count()>10)  findText->removeItem(0);

    FWObject *o=NULL;

loop:
    QApplication::setOverrideCursor( QCursor( Qt::WaitCursor) );

    for (; treeSeeker!=mw->db()->tree_end(); ++treeSeeker)
    {
        o = *treeSeeker;

        if( RuleElement::cast(o->getParent())!=NULL)
        {
            if (! searchInRules->isChecked()) continue;
        } else
        {
/* if not in rules, then in the tree. */
            if (! searchInTree->isChecked()) continue;
        }

        if (FWReference::cast(o)!=NULL)
        {
            FWReference *r=FWReference::cast(o);
            if ( match( r->getPointerId().c_str() )) break;
            if ( match( QString::fromUtf8(r->getPointer()->getName().c_str()) ) ) break;
        } else
        {
            if (match( o->getId().c_str() )) break;
            if (match( QString::fromUtf8(o->getName().c_str()))) break;
        }
    }

    QApplication::restoreOverrideCursor();

    if (treeSeeker==mw->db()->tree_end())
    {
        reset();

        if ( QMessageBox::warning(
              this,"Firewall Builder", 
              tr("Search hit the end of the object tree."),
              "&Continue at top", "&Stop", QString::null, 0, 1 )==0 ) goto loop;

        return;
    }
    assert(o!=NULL);

/* found object. Shift iterator so it does not return the same object
 * when user hits 'find next'
 */
   
    ++treeSeeker;

    if (FWReference::cast(o)!=NULL && RuleElement::cast(o->getParent())!=NULL)
    {
        mw->ensureObjectVisible( FWReference::cast(o) );
        QTimer::singleShot(200, this, SLOT(makeActive()) );
        return;
    } 

    if (Group::cast(o->getParent())!=NULL && 
        !FWBTree::isSystem(o->getParent()))
    {
        om->openObject( o->getParent() );
        om->edit( o->getParent() );
        QTimer::singleShot(200, this, SLOT(makeActive()) );
        return;
    }

    om->openObject( o );
    om->edit( o );

    QTimer::singleShot(200, this, SLOT(makeActive()) );
}

void findDialog::makeActive()
{
    setActiveWindow();
}

void findDialog::showEvent( QShowEvent *ev)
{
    st->restoreGeometry(this, QRect(200,100,330,140) );
    QDialog::showEvent(ev);
    
    useRegexp->setChecked( st->getBool("Search/useRegexp") );
    searchInTree->setChecked(  st->getBool("Search/findInTree" ) );
    searchInRules->setChecked( st->getBool("Search/findInRules") );

    findText->setFocus();
}

void findDialog::hideEvent( QHideEvent *ev)
{
    st->saveGeometry(this);
    QDialog::hideEvent(ev);

    st->setBool("Search/useRegexp",    useRegexp->isChecked() );
    st->setBool("Search/findInTree",  searchInTree->isChecked() );
    st->setBool("Search/findInRules", searchInRules->isChecked() );
}
    
