/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <fcntl.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>

#include "gdis.h"
#include "coords.h"
#include "edit.h"
#include "file.h"
#include "graph.h"
#include "model.h"
#include "parse.h"
#include "task.h"
#include "matrix.h"
#include "surface.h"
#include "spatial.h"
#include "gtkshorts.h"
#include "interface.h"
#include "dialog.h"
#include "opengl.h"

extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

GtkWidget *phonon_freq, *phonon_ir, *phonon_raman, *gp_dump_file, *gp_temp_file;

/*******************/
/* run a gulp file */
/*******************/
#define DEBUG_EXEC_GULP 0
gint exec_gulp(const gchar *input, const gchar *output)
{
gchar *cmd;

/* checks */
if (!sysenv.gulp_path)
  return(-1);

/* stop gulp renaming existing output files */
unlink(output);

cmd = g_strdup_printf("%s < %s > %s", sysenv.gulp_path, input, output);

#if DEBUG_EXEC_GULP
printf("executing: [%s]\n",cmd);
#endif

task_sync(cmd);

/* done */
g_free(cmd);
return(0);
}

/*****************************/
/* execute a gulp run (task) */
/*****************************/
#define DEBUG_EXEC_GULP_TASK 0
void exec_gulp_task(gpointer *ptr)
{
gchar *inpfile, *outfile;
gchar **buff;
struct model_pak *data;

g_return_if_fail(ptr != NULL);
data = (struct model_pak *) ptr;

/* construct input and output filenames */
inpfile = g_build_filename(sysenv.cwd, data->gulp.temp_file, NULL);

/* TODO - put this in a routine (change_extension?) */
buff = g_strsplit(data->gulp.temp_file, ".gin", 0);
/* copy first token as the base name */
outfile = g_malloc((5+strlen(*buff))*sizeof(gchar));
strcpy(outfile, *buff);
strcat(outfile, ".got");
g_strfreev(buff);

#if DEBUG_EXEC_GULP_TASK
printf(" input file: %s\n", inpfile);
printf("output file: %s\n", outfile);
#endif

/* create the input file */
write_gulp(inpfile, data);

/* delete the old file to be sure output is only data from current run */
unlink(outfile);

/* are we supposed to execute GULP? */
if (data->gulp.no_exec)
  {
#if DEBUG_EXEC_GULP_TASK
printf("Skipping GULP execution on user request.\n");
#endif
  return;
  }

exec_gulp(data->gulp.temp_file, outfile);
}

/*****************************/
/* phonon selection handlers */
/*****************************/
void update_phonon_range(struct model_pak *model)
{
/* checks */
g_assert(model != NULL);
if (!model->phonon_slider)
  return;

/* update the slider */
if (model->num_phonons > 0)
  gtk_range_set_range(GTK_RANGE(model->phonon_slider), 1, model->num_phonons);
}

/*****************************/
/* process a gulp run (task) */
/*****************************/
#define DEBUG_PROC_GULP 0
void proc_gulp_task(gpointer *ptr)
{
gchar *inp, *res, *out;
GString *line;
struct model_pak *dest, *data;

/* TODO - model locking (moldel_ptr RO/RW etc) to prevent screw ups */
g_return_if_fail(ptr != NULL);
data = (struct model_pak *) ptr;

/* don't attempt to process if gulp execution was turned off */
if (data->gulp.no_exec)
  return;

/* FIXME - if the user has moved directories since submitting */
/* the gulp job then cwd will have changed and this will fail */
inp = g_build_filename(sysenv.cwd, data->gulp.temp_file, NULL);
res = g_build_filename(sysenv.cwd, data->gulp.dump_file, NULL);
out = g_build_filename(sysenv.cwd, data->gulp.out_file, NULL);

/* NEW - needs testing */
if (data->gulp.dock)
  {
  free_core_list(data);

  read_gulp(res, data);

  init_objs(REDO_COORDS, data);
  calc_bonds(data);
  calc_mols(data);
  }
else
switch (data->gulp.run)
  {
  case E_SINGLE:
/* same model (ie with current energetics dialog) so update */
    read_gulp_output(out, data);
/* update energy (TODO - only if successful) */
    line = g_string_new(NULL);
    if (data->gulp.free)
      g_string_sprintf(line,"%f (free energy)",data->gulp.energy);
    else
      g_string_sprintf(line,"%f",data->gulp.energy);

/* is there a dialog entry to be updated? */
    if (GTK_IS_ENTRY(data->gulp.energy_entry))
      gtk_entry_set_text(GTK_ENTRY(data->gulp.energy_entry), line->str);
    if (data->periodic == 2)
      {
/* update surface dipole dialog entry */
      g_string_sprintf(line,"%f e.Angs",data->gulp.sdipole);
      if (GTK_IS_ENTRY(data->gulp.sdipole_entry))
        gtk_entry_set_text(GTK_ENTRY(data->gulp.sdipole_entry), line->str);

/* update surface energy dialog entry */
      g_string_sprintf(line,"%f    %s",data->gulp.esurf[0], data->gulp.esurf_units);
      if (GTK_IS_ENTRY(data->gulp.esurf_entry))
        gtk_entry_set_text(GTK_ENTRY(data->gulp.esurf_entry), line->str);

/* update attachment energy dialog entry */
      g_string_sprintf(line,"%f    %s",data->gulp.eatt[0], data->gulp.eatt_units);
      if (GTK_IS_ENTRY(data->gulp.eatt_entry))
        gtk_entry_set_text(GTK_ENTRY(data->gulp.eatt_entry), line->str);
      }

    update_phonon_range(data);

    g_string_free(line, TRUE);
    break;

  case E_OPTIMIZE:
/* TODO - make it possile to get dialog data by request */
/* so that we can check if a dialog exsits to be updated */
/* get new coords */
/* create new model for the minimized result */
    dest = model_new();
    g_return_if_fail(dest != NULL);

/* read main data from the res file (correct charges etc.) */
    read_gulp(res, dest);
/* graft to the model tree, so subsequent GULP read doesn't replace coords */
    tree_model_add(dest);
/* get the output energy/phonons etc. */
    read_gulp_output(out, dest);
    break;

/* MD */
  default:
    break;
  }

g_free(inp);
g_free(res);
g_free(out);

redraw_canvas(ALL);
return;
}

/***********************************************/
/* controls the use of GULP to optimise coords */
/***********************************************/
#define DEBUG_RUN_GULP 0
void run_gulp(GtkWidget *w, struct model_pak *data)
{
gchar **buff;

/* checks */
g_return_if_fail(data != NULL);
if (!sysenv.gulp_path)
  {
  show_text(ERROR, "GULP executable was not found.\n");
  return;
  }

/* checks */
if (data->gulp.dock)
  {
  if (!data->selection)
    {
    show_text(WARNING, "Nothing selected for docking, exiting.\n");
    return;
    }
  }

/* set up output file to use */
g_free(data->gulp.out_file);

/* TODO - put this in a routine */
buff = g_strsplit(data->gulp.temp_file, ".gin", 0);
/* copy first token as the base name */
data->gulp.out_file = g_malloc((5+strlen(*buff))*sizeof(gchar));
strcpy(data->gulp.out_file, *buff);
strcat(data->gulp.out_file, ".got");
g_strfreev(buff);

#if DEBUG_RUN_GULP
printf("output file: %s\n", data->gulp.out_file);
#endif

task_new("Gulp", &exec_gulp_task, data, &proc_gulp_task, data, data);
}

/*************************/
/* GULP run type toggles */
/*************************/
void gulp_run_single(struct model_pak *data)
{
data->gulp.run = E_SINGLE;
}
void gulp_run_optimize(struct model_pak *data)
{
data->gulp.run = E_OPTIMIZE;
}
void gulp_run_dynamics(struct model_pak *data)
{
data->gulp.run = MD;
}

/************************/
/* GULP keyword toggles */
/************************/
gint cb_gulp_keyword(GtkWidget *w, gpointer *obj)
{
gint keyword;
struct model_pak *data;

keyword = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(obj), "key"));

data = (struct model_pak *) g_object_get_data(G_OBJECT(obj), "ptr");
g_return_val_if_fail(data != NULL, FALSE);

switch(keyword)
  {
  case NO_EATT:
    data->gulp.no_eatt ^= 1;
    break;
  case NO_ESURF:
    data->gulp.no_esurf ^= 1;
    break;
  case NVE:
  case NVT:
  case NPT:
    if (data->gulp.run != MD)
      {
      show_text(WARNING, "Ignoring ensemble as it's not a dynamics run.\n");
      return(TRUE);
      }
    data->gulp.ensemble = keyword;
    break;
  case CONP:
  case CONV:
    data->gulp.method = keyword;
    break;
  case BFGS_OPT:
  case CONJ_OPT:
  case RFO_OPT:
    data->gulp.optimiser = keyword;
    break;
  case MOLE:
  case MOLMEC:
  case MOLQ:
  case NOBUILD:
    data->gulp.coulomb = keyword;
    break;
  case TEMPERATURE:
    data->gulp.temp = (gdouble) GTK_ADJUSTMENT(obj)->value;
    break;
  case DIPOLE_TOLERANCE:
    data->gulp.sdipole_tolerance = (gdouble) GTK_ADJUSTMENT(obj)->value;
    break;
  case MAXCYC:
    data->gulp.maxcyc = (gint) GTK_ADJUSTMENT(obj)->value;
    break;
  case TEMP_FILE:
    g_free(data->gulp.temp_file);
    data->gulp.temp_file = g_strdup(gtk_entry_get_text(GTK_ENTRY(obj)));
    break; 
  case DUMP_FILE:
    g_free(data->gulp.dump_file);
    data->gulp.dump_file = g_strdup(gtk_entry_get_text(GTK_ENTRY(obj)));
    break;
  case LIB_FILE:
    g_free(data->gulp.libfile);
    data->gulp.libfile = g_strdup(gtk_entry_get_text(GTK_ENTRY(obj)));
    break;
/* NEW */
  case SBULK_ENERGY:
    data->gulp.sbulkenergy = str_to_float(gtk_entry_get_text(GTK_ENTRY(obj)));
    break;
  }

return(FALSE);
}

/*******************************************************/
/* alternate event parser (overlap in optimiser stuff) */
/*******************************************************/
gint switch_keyword(GtkWidget *w, gpointer *obj)
{
gint keyword;
struct model_pak *data;

keyword = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(obj), "key"));

data = (struct model_pak *) g_object_get_data(G_OBJECT(obj), "ptr");
g_return_val_if_fail(data != NULL, FALSE);

switch(keyword)
  {
  case SWITCH_OFF:
  case BFGS_OPT:
  case CONJ_OPT:
  case RFO_OPT:
    data->gulp.optimiser2 = keyword;
    break;
  case CYCLE:
  case GNORM:
    data->gulp.switch_type = keyword;
    break;
  }
return(FALSE);
}

/*********************************/
/* change optimiser switch value */
/*********************************/
gint switch_value_changed(gpointer *entry)
{
struct model_pak *data;

/* retrieve the appropriate model */
g_return_val_if_fail(entry != NULL, FALSE);
data = (struct model_pak *) g_object_get_data(G_OBJECT(entry), "ptr");
g_return_val_if_fail(data != NULL, FALSE);

data->gulp.switch_value = str_to_float(gtk_entry_get_text(GTK_ENTRY(entry)));
return(FALSE);
}

/***********************************/
/* register structure name changes */
/***********************************/
void name_entry_changed(GtkWidget *w, struct model_pak *data)
{
g_assert(w != NULL);
g_assert(data != NULL);

/* de-alloc */
g_free(data->basename);
g_free(data->gulp.dump_file);
g_free(data->gulp.temp_file);

/* set the new basename & related info */
data->basename = g_strdup(gtk_entry_get_text(GTK_ENTRY(w)));

data->gulp.dump_file = g_strdup_printf("%s.res", data->basename);
gtk_entry_set_text(GTK_ENTRY(gp_dump_file), data->gulp.dump_file);

data->gulp.temp_file = g_strdup_printf("%s.gin", data->basename);
gtk_entry_set_text(GTK_ENTRY(gp_temp_file), data->gulp.temp_file);

/* update the model tree */
tree_model_add(data);
}

/**************************/
/* phonon scaling handler */
/**************************/
void phonon_scaling_changed(GtkWidget *w, gpointer *ptr)
{
sysenv.render.phonon_scaling = GTK_ADJUSTMENT(w)->value;
}

#define DEBUG_PHONON_DISPLAY 0

/************************************/
/* vibrational mode display handler */
/************************************/
void phonon_mode_hide(GtkWidget *w, struct model_pak *data)
{
gint i;

/* get rid of previous */
/* FIXME - only delete phonon related spatials??? */
i = g_slist_length(data->spatial);
while (i)
  delete_spatial(--i, data);

init_objs(REDO_COORDS, data);
redraw_canvas(SINGLE);
}

/************************************/
/* vibrational mode display handler */
/************************************/
void phonon_mode_show(GtkWidget *w, struct model_pak *data)
{
gint i;
gdouble x1[3], x2[3];
GSList *list;
gpointer *freq;
struct vec_pak *vec1, *vec2;
struct spatial_pak *spatial;
struct core_pak *core;

/* get rid of previous */
/* FIXME - only delete phonon related spatials??? */
i = g_slist_length(data->spatial);
while (i)
  delete_spatial(--i, data);

/* find and display current */
freq = g_slist_nth_data(data->phonons, data->current_phonon-1);
if (freq)
  {
/* create & init the spatial object */
  spatial = g_malloc(sizeof(struct spatial_pak));
  spatial->type = SPATIAL_VECTORS;
  spatial->data = NULL;

/* get eigenvectors from all atoms */
  for (list=data->cores ; list; list=g_slist_next(list))
    {
    core = (struct core_pak *) list->data;

/* atom coords */
    ARR3SET(x1, core->x);

/* vibrational eigenvector */
    x2[0] = *((gdouble *) g_slist_nth_data(core->vibx_list, data->current_phonon-1));
    x2[1] = *((gdouble *) g_slist_nth_data(core->viby_list, data->current_phonon-1));
    x2[2] = *((gdouble *) g_slist_nth_data(core->vibz_list, data->current_phonon-1));
 
#if DEBUG_PHONON_DISPLAY
P3VEC("vec: ", x2);
#endif

/* scaling */
    VEC3MUL(x2, sysenv.render.phonon_scaling);
    vecmat(data->ilatmat, x2);
/* construct the vector */
    ARR3ADD(x2, x1);
    vec1 = g_malloc(sizeof(struct vec_pak));
    vec2 = g_malloc(sizeof(struct vec_pak));
    ARR3SET(vec1->x, x1);
    ARR3SET(vec2->x, x2);
/* append eigenvector for this atom */
    spatial->data = g_slist_append(spatial->data, vec1);
    spatial->data = g_slist_append(spatial->data, vec2);
    }

  data->spatial = g_slist_append(data->spatial, spatial);
/* drawing update */
  init_objs(REDO_COORDS, data);
  redraw_canvas(SINGLE);
  }
}

/********************************/
/* eigenvector show/hide toggle */
/********************************/
void phonon_mode_toggle(GtkWidget *w, struct model_pak *model)
{
if (model->show_eigenvectors)
  phonon_mode_show(w, model);
else
  phonon_mode_hide(w, model);
}

/***********************************/
/* cleanup a vibrational animation */
/***********************************/
void phonon_timeout_cleanup(struct model_pak *model)
{
GSList *list;
struct core_pak *core;

g_assert(model != NULL);

for (list=model->cores ; list; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;

  VEC3SET(core->offset, 0.0, 0.0, 0.0);
  }
}

/************************************/
/* timeout to control the animation */
/************************************/
/* NB: lots of sanity checks that return with FALSE (ie stop timeout) */
/* so if the model is deleted during an animation we don't segfault */
#define MAX_PULSE_COUNT 10.0
gint phonon_mode_timeout(gint n)
{
gdouble f;
gpointer ptr;
GSList *list;
struct core_pak *core;
struct model_pak *model;

/* checks */
model = model_ptr(n, RECALL);
if (!model)
  return(FALSE);
ptr = g_slist_nth_data(model->phonons, model->current_phonon-1);
if (!ptr)
  {
  phonon_timeout_cleanup(model);
  return(FALSE);
  }
if (!model->pulse_direction)
  return(FALSE);

/* setup scaling for this step */
model->pulse_count += model->pulse_direction;
if (model->pulse_count <= -MAX_PULSE_COUNT)
  {
  model->pulse_count = -MAX_PULSE_COUNT;
  model->pulse_direction = 1;
  }
if (model->pulse_count >= MAX_PULSE_COUNT)
  {
  model->pulse_count = MAX_PULSE_COUNT;
  model->pulse_direction = -1;
  }

f = sysenv.render.phonon_scaling;
f *= (gdouble) model->pulse_count;
f /= MAX_PULSE_COUNT;

/* get eigenvectors from all atoms */
for (list=model->cores ; list; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;

/* vibrational eigenvector */
  ptr = g_slist_nth_data(core->vibx_list, model->current_phonon-1);
  if (!ptr)
    {
    phonon_timeout_cleanup(model);
    return(FALSE);
    }
  core->offset[0] = *((gdouble *) ptr);

  ptr = g_slist_nth_data(core->viby_list, model->current_phonon-1);
  if (!ptr)
    {
    phonon_timeout_cleanup(model);
    return(FALSE);
    }
  core->offset[1] = *((gdouble *) ptr);

  ptr = g_slist_nth_data(core->vibz_list, model->current_phonon-1);
  if (!ptr)
    {
    phonon_timeout_cleanup(model);
    return(FALSE);
    }
  core->offset[2] = *((gdouble *) ptr);

/* pulse offset scaling */
  VEC3MUL(core->offset, f);

vecmat(model->ilatmat, core->offset);

/* TODO - shell also */

  }

/* redraw */
calc_coords(REFRESH, model);
redraw_canvas(SINGLE);
return(TRUE);
}

/****************************/
/* animate the current mode */
/****************************/
void phonon_mode_animate(GtkWidget *w, struct model_pak *model)
{
g_assert(model != NULL);

model->pulse_count = 0;
model->pulse_direction = 1;

g_timeout_add(25, (GSourceFunc) phonon_mode_timeout,
              GINT_TO_POINTER(model->number));
}

/******************************/
/* stop phonon mode animation */
/*****************************/
void phonon_mode_stop(GtkWidget *w, struct model_pak *model)
{
/* reset */
model->pulse_direction = 0;
model->pulse_count = 0;
phonon_timeout_cleanup(model);

/* redraw */
calc_coords(REFRESH, model);
redraw_canvas(SINGLE);
}

/**************************/
/* kpoint change handlers */
/**************************/
void kpoint_toggle(struct model_pak *model)
{
model->gulp.num_kpoints ^= 1;
}

/*********************************/
/* phonon slider changed updates */
/*********************************/
void update_phonon_frequency(GtkWidget *w, struct model_pak *model)
{
gpointer freq, ir, raman;

g_assert(model != NULL);

freq = g_slist_nth_data(model->phonons, model->current_phonon-1);
if (freq)
  {
  ir = g_slist_nth_data(model->ir_list, model->current_phonon-1);
  raman = g_slist_nth_data(model->raman_list, model->current_phonon-1);

/* update displayed frequency */
  gtk_entry_set_text(GTK_ENTRY(phonon_freq), (gchar *) freq);
  gtk_entry_set_text(GTK_ENTRY(phonon_ir), (gchar *) ir);
  gtk_entry_set_text(GTK_ENTRY(phonon_raman), (gchar *) raman);


/* update displayed vectors */
  if (model->show_eigenvectors)
    {
    phonon_mode_hide(NULL, model);
    phonon_mode_show(NULL, model);
    }
  }
}

/*****************************/
/* phonon intensity plotting */
/*****************************/
void phonon_graph(GSList *mode, struct model_pak *model)
{
gint n;
gdouble *spectrum;
gpointer graph;
GSList *list;

g_assert(model != NULL);

spectrum = g_malloc(model->num_phonons * sizeof(gdouble));

/* extract data into array */
n=0;
for (list=mode ; list ; list=g_slist_next(list))
  {
g_assert(n < model->num_phonons);

  spectrum[n] = str_to_float(list->data);
  n++;
  }

/* create a new graph */
if (mode == model->ir_list)
  graph = graph_new("IR spectrum", model);
else
  graph = graph_new("Raman spectrum", model);

graph_add_data(model->num_phonons, spectrum, 1, model->num_phonons, graph);
graph_set_yticks(FALSE, 2, graph);
tree_model_add(model);

g_free(spectrum);
}

/***************************************/
/* phonon intensity plotting callbacks */
/***************************************/
void cb_graph_ir(GtkWidget *w, struct model_pak *model)
{
g_assert(model != NULL);
phonon_graph(model->ir_list, model);
}

void cb_graph_raman(GtkWidget *w, struct model_pak *model)
{
g_assert(model != NULL);
phonon_graph(model->raman_list, model);
}


/* CURRENT - invoke test module */
/*
void phonon_module_invoke(void)
{
struct model_pak *model;

model = sysenv.active_model;

module_function_invoke("test", model);
}
*/


/****************/
/* phonon page  */
/****************/
void phonon_display_box(struct model_pak *data, GtkWidget *box)
{
gint i;
GtkWidget *table, *frame, *hbox, *hbox2, *vbox, *label;

frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(box), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
vbox = gtk_vbox_new(FALSE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);

gtksh_direct_check("Compute vibrational modes", &data->gulp.phonon, NULL, NULL, vbox);
gtksh_direct_check("Compute eigenvectors", &data->gulp.eigen, NULL, NULL, vbox);

if (data->periodic)
  {
/* TODO - more than one kpoint */
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  gtksh_direct_check("Specify kpoint", &data->gulp.num_kpoints, NULL, NULL, hbox);

  for (i=0 ; i<data->periodic ; i++)
    {
    switch (i)
      {
      case 0:
        gtksh_direct_spin(NULL, &data->gulp.kpoints[0], 0.0, 1.0, 0.1, NULL, NULL, hbox);
        break;
      case 1:
        gtksh_direct_spin(NULL, &data->gulp.kpoints[1], 0.0, 1.0, 0.1, NULL, NULL, hbox);
        break;
      case 2:
        gtksh_direct_spin(NULL, &data->gulp.kpoints[2], 0.0, 1.0, 0.1, NULL, NULL, hbox);
        break;
      }
    }
  }

/* phonon slide selector */
frame = gtk_frame_new("Eigenvectors");
gtk_box_pack_start(GTK_BOX(box), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
vbox = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
label = gtk_label_new("Number  ");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
data->phonon_slider = gtksh_direct_hscale(0, 1, 1, &data->current_phonon, 
                                    update_phonon_frequency, data, hbox);

/* phonon info */
table = gtk_table_new(2, 3, TRUE);
gtk_box_pack_start(GTK_BOX(vbox), table, TRUE, TRUE, 0);

label = gtk_label_new("Frequency");
gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 0, 1);

label = gtksh_button("IR intensity", cb_graph_ir, data, NULL, 0);
gtk_table_attach_defaults(GTK_TABLE(table), label, 1, 2, 0, 1);

label = gtksh_button("Raman intensity", cb_graph_raman, data, NULL, 0);
gtk_table_attach_defaults(GTK_TABLE(table), label, 2, 3, 0, 1);

phonon_freq = gtk_entry_new_with_max_length(LINELEN);
gtk_entry_set_text(GTK_ENTRY(phonon_freq), " ");
gtk_entry_set_editable(GTK_ENTRY(phonon_freq), FALSE);
gtk_table_attach_defaults(GTK_TABLE(table), phonon_freq, 0, 1, 1, 2);

phonon_ir = gtk_entry_new_with_max_length(LINELEN);
gtk_entry_set_text(GTK_ENTRY(phonon_ir), " ");
gtk_entry_set_editable(GTK_ENTRY(phonon_ir), FALSE);
gtk_table_attach_defaults(GTK_TABLE(table), phonon_ir, 1, 2, 1, 2);

phonon_raman = gtk_entry_new_with_max_length(LINELEN);
gtk_entry_set_text(GTK_ENTRY(phonon_raman), " ");
gtk_entry_set_editable(GTK_ENTRY(phonon_raman), FALSE);
gtk_table_attach_defaults(GTK_TABLE(table), phonon_raman, 2, 3, 1, 2);

/* phonon vector scaling */
gtksh_direct_spin("Scaling", &sysenv.render.phonon_scaling, 0.1, 9.9, 0.1,
                  update_phonon_frequency, data, vbox);

gtksh_direct_check("Display vectors", &data->show_eigenvectors, 
                   phonon_mode_toggle, data, vbox);

/* animation */
hbox2 = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox2, TRUE, FALSE, 0);



/* NEW - module list */
/*
if (module_function_exists("test"))
  {
  hbox = gtk_hbox_new(TRUE, 0);
  gtk_box_pack_start(GTK_BOX(hbox2), hbox, FALSE, FALSE, 0);
  gtksh_button_x(NULL, phonon_module_invoke, NULL, hbox);
  }
*/


hbox = gtk_hbox_new(TRUE, PANEL_SPACING);
gtk_box_pack_end(GTK_BOX(hbox2), hbox, FALSE, FALSE, 0);

gtksh_icon_button(GTK_STOCK_GO_FORWARD, NULL,
                  phonon_mode_animate, (gpointer) data, hbox);

gtksh_icon_button("GDIS_STOP", NULL,
                  phonon_mode_stop, (gpointer) data, hbox);


gtk_widget_show_all(box);

update_phonon_range(data);
}

/*******************************/
/* GULP job setup/results page */
/*******************************/
void gulp_widget(GtkWidget *box, struct model_pak *data)
{
GtkWidget *hbox, *vbox, *vbox1, *vbox2, *page;
GtkWidget *frame, *button, *label, *entry, *notebook;
GSList *keyword[5];
GString *line;

/* string manipulation scratchpad */
line = g_string_new(NULL);

/* print method blurb */
vbox = gtk_vbox_new(TRUE, 0);
gtk_box_pack_start(GTK_BOX(box), vbox, FALSE, FALSE, 0);
g_string_sprintf(line,"\nMolecular mechanics method\nGULP GUI\n");
label = gtk_label_new(line->str);
gtk_container_add(GTK_CONTAINER(vbox), label);

/* NEW - setup some gulp file name defaults */
if (g_ascii_strncasecmp(data->gulp.temp_file, "none", 4) == 0)
  {
/* FIXME - what if temp_file = filename? (may not want to overwrite) */
  g_string_sprintf(line,"%s.gin",g_strstrip(data->basename));
  g_free(data->gulp.temp_file);
  data->gulp.temp_file = g_strdup(line->str); 
  }
if (g_ascii_strncasecmp(data->gulp.dump_file, "none", 4) == 0)
  {
  g_string_sprintf(line,"%s.res",g_strstrip(data->basename));
  g_free(data->gulp.dump_file);
  data->gulp.dump_file = g_strdup(line->str); 
  }

/* create notebook */
notebook = gtk_notebook_new();
gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_TOP);
gtk_container_add(GTK_CONTAINER(box), notebook);
gtk_notebook_set_show_border(GTK_NOTEBOOK(notebook), FALSE);

/* run type page */
page = gtk_vbox_new(FALSE, 0);
label = gtk_label_new (" Control ");
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), page, label);
hbox = gtk_hbox_new(FALSE, 0);
gtk_container_add(GTK_CONTAINER(page), hbox);

/* split panel */
vbox1 = gtk_vbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(hbox), vbox1, TRUE, TRUE, 0);
vbox2 = gtk_vbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(hbox), vbox2, TRUE, TRUE, 0);

/* frame */
frame = gtk_frame_new("Run type");
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);

/* do the radio buttons */
new_radio_group(0, vbox, TT);
button = add_radio_button("Single point", (gpointer) gulp_run_single, data);
if (data->gulp.run == E_SINGLE)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

button = add_radio_button("Optimize", (gpointer) gulp_run_optimize, data);
if (data->gulp.run == E_OPTIMIZE)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

button = add_radio_button("Dynamics", (gpointer) gulp_run_dynamics, data);
if (data->gulp.run == MD)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

gtksh_direct_check("Dock selection", &data->gulp.dock, NULL, NULL, vbox);


/* method constraint */
frame = gtk_frame_new(" Constraint ");
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame),vbox);

/* do the first (DEFAULT MODE) radio button */
button = gtk_radio_button_new_with_label (NULL, "Constant pressure");
gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, TRUE, 0);
g_signal_connect(GTK_OBJECT(button), "clicked",
                 GTK_SIGNAL_FUNC (cb_gulp_keyword), (gpointer) button);
g_object_set_data(G_OBJECT(button), "key", (gpointer) CONP);
g_object_set_data(G_OBJECT(button), "ptr", (gpointer) data);
if (data->gulp.method == CONP)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* make a radio group */
keyword[1] = gtk_radio_button_group (GTK_RADIO_BUTTON (button));
/* do the next button */
button = gtk_radio_button_new_with_label (keyword[1], "Constant volume");
gtk_box_pack_start(GTK_BOX (vbox), button, TRUE, TRUE, 0);
g_signal_connect(GTK_OBJECT(button), "clicked",
                 GTK_SIGNAL_FUNC (cb_gulp_keyword), (gpointer) button);
g_object_set_data(G_OBJECT(button), "key", (gpointer) CONV);
g_object_set_data(G_OBJECT(button), "ptr", (gpointer) data);
if (data->gulp.method == CONV)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);


/* frame */
frame = gtk_frame_new(" Molecule options ");
gtk_box_pack_start(GTK_BOX(vbox2), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame),vbox);
/* do the first radio button */
button = gtk_radio_button_new_with_label(NULL,
       "Coulomb subtract all intramolecular");
gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, TRUE, 0);
g_signal_connect(GTK_OBJECT (button), "clicked",
                 GTK_SIGNAL_FUNC (cb_gulp_keyword), (gpointer) button);
g_object_set_data(G_OBJECT(button), "key", (gpointer) MOLE);
g_object_set_data(G_OBJECT(button), "ptr", (gpointer) data);
if (data->gulp.coulomb == MOLE)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* make a radio group */
keyword[2] = gtk_radio_button_group (GTK_RADIO_BUTTON (button));
/* do the next button */
button = gtk_radio_button_new_with_label(keyword[2], 
       "Coulomb subtract 1-2 and 1-3 intramolecular");
gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, TRUE, 0);
g_signal_connect(GTK_OBJECT (button), "clicked",
                 GTK_SIGNAL_FUNC (cb_gulp_keyword), (gpointer) button);
g_object_set_data(G_OBJECT(button), "key", (gpointer) MOLMEC);
g_object_set_data(G_OBJECT(button), "ptr", (gpointer) data);
if (data->gulp.coulomb == MOLMEC)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* subsequent button... */
button = gtk_radio_button_new_with_label(gtk_radio_button_group
       (GTK_RADIO_BUTTON (button)), "Build, but retain coulomb interactions");
gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, TRUE, 0);
g_signal_connect(GTK_OBJECT (button), "clicked",
                 GTK_SIGNAL_FUNC (cb_gulp_keyword), (gpointer) button);
g_object_set_data(G_OBJECT(button), "key", (gpointer) MOLQ);
g_object_set_data(G_OBJECT(button), "ptr", (gpointer) data);
if (data->gulp.coulomb == MOLQ)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* subsequent button... */
button = gtk_radio_button_new_with_label(gtk_radio_button_group 
          (GTK_RADIO_BUTTON (button)), "Molecule building off");
gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, TRUE, 0);
g_signal_connect(GTK_OBJECT (button), "clicked",
                 GTK_SIGNAL_FUNC (cb_gulp_keyword), (gpointer) button);
g_object_set_data(G_OBJECT(button), "key", (gpointer) NOBUILD);
g_object_set_data(G_OBJECT(button), "ptr", (gpointer) data);
if (data->gulp.coulomb == NOBUILD)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* frame for keyword toggles */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox2), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame),vbox);
gtksh_direct_check("QEq Electronegativity equalisation", &data->gulp.qeq, NULL, NULL, vbox);
gtksh_direct_check("Fix the initial connectivity", &data->gulp.fix, NULL, NULL, vbox);
gtksh_direct_check("Create input file then stop", &data->gulp.no_exec, NULL, NULL, vbox);

/* minimize page */
page = gtk_vbox_new(FALSE,0);
label = gtk_label_new (" Optimisation ");
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), page, label);
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(page), hbox, FALSE, FALSE, 0);

/* optimiser to use */
frame = gtk_frame_new("Primary optimiser");
gtk_box_pack_start(GTK_BOX(hbox), frame, TRUE, TRUE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);

/* do the first (DEFAULT MODE) radio button */
button = gtk_radio_button_new_with_label (NULL, "bfgs");
gtk_box_pack_start(GTK_BOX (vbox), button, TRUE, TRUE, 0);
g_signal_connect(GTK_OBJECT (button), "clicked",
                 GTK_SIGNAL_FUNC (cb_gulp_keyword), (gpointer) button);
g_object_set_data(G_OBJECT(button), "key", (gpointer) BFGS_OPT);
g_object_set_data(G_OBJECT(button), "ptr", (gpointer) data);
if (data->gulp.optimiser == BFGS_OPT || data->gulp.optimiser == -1)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* make a radio group */
keyword[1] = gtk_radio_button_group (GTK_RADIO_BUTTON (button));
/* do the next button */
button = gtk_radio_button_new_with_label (keyword[1], "conj");
gtk_box_pack_start(GTK_BOX (vbox), button, TRUE, TRUE, 0);
g_signal_connect(GTK_OBJECT (button), "clicked",
                 GTK_SIGNAL_FUNC (cb_gulp_keyword), (gpointer) button);
g_object_set_data(G_OBJECT(button), "key", (gpointer) CONJ_OPT);
g_object_set_data(G_OBJECT(button), "ptr", (gpointer) data);
if (data->gulp.optimiser == CONJ_OPT)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* subsequent button... */
button = gtk_radio_button_new_with_label(gtk_radio_button_group
       (GTK_RADIO_BUTTON (button)), "rfo");
gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, TRUE, 0);
g_signal_connect(GTK_OBJECT (button), "clicked",
                 GTK_SIGNAL_FUNC (cb_gulp_keyword), (gpointer) button);
g_object_set_data(G_OBJECT(button), "key", (gpointer) RFO_OPT);
g_object_set_data(G_OBJECT(button), "ptr", (gpointer) data);
if (data->gulp.optimiser == RFO_OPT)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);


/* optimiser to use */
frame = gtk_frame_new("Secondary optimiser");
gtk_box_pack_start(GTK_BOX(hbox), frame, TRUE, TRUE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);

/* do the first (DEFAULT MODE) radio button */
button = gtk_radio_button_new_with_label (NULL, "none");
gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 0);
g_signal_connect(GTK_OBJECT(button), "clicked",
                 GTK_SIGNAL_FUNC (switch_keyword), (gpointer) button);
g_object_set_data(G_OBJECT(button), "key", (gpointer) SWITCH_OFF);
g_object_set_data(G_OBJECT(button), "ptr", (gpointer) data);
if (data->gulp.optimiser2 == SWITCH_OFF || data->gulp.optimiser2 == -1)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* make a radio group */
keyword[1] = gtk_radio_button_group(GTK_RADIO_BUTTON(button));
/* do the next button */
button = gtk_radio_button_new_with_label (keyword[1], "bfgs");
gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 0);
g_signal_connect(GTK_OBJECT(button), "clicked",
                 GTK_SIGNAL_FUNC (switch_keyword), (gpointer) button);
g_object_set_data(G_OBJECT(button), "key", (gpointer) BFGS_OPT);
g_object_set_data(G_OBJECT(button), "ptr", (gpointer) data);
if (data->gulp.optimiser2 == BFGS_OPT)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* subsequent button... */
button = gtk_radio_button_new_with_label(gtk_radio_button_group
                                        (GTK_RADIO_BUTTON(button)), "conj");
gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 0);
g_signal_connect(GTK_OBJECT(button), "clicked",
                 GTK_SIGNAL_FUNC (switch_keyword), (gpointer) button);
g_object_set_data(G_OBJECT(button), "key", (gpointer) CONJ_OPT);
g_object_set_data(G_OBJECT(button), "ptr", (gpointer) data);
if (data->gulp.optimiser2 == CONJ_OPT)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* subsequent button... */
button = gtk_radio_button_new_with_label(gtk_radio_button_group
                                        (GTK_RADIO_BUTTON(button)), "rfo");
gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 0);
g_signal_connect(GTK_OBJECT(button), "clicked",
                 GTK_SIGNAL_FUNC(switch_keyword), (gpointer) button);
g_object_set_data(G_OBJECT(button), "key", (gpointer) RFO_OPT);
g_object_set_data(G_OBJECT(button), "ptr", (gpointer) data);
if (data->gulp.optimiser2 == RFO_OPT)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);


/* optimiser to use */
frame = gtk_frame_new("Switching critera");
gtk_box_pack_start(GTK_BOX(hbox), frame, TRUE, TRUE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
vbox = gtk_vbox_new(FALSE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);

/* do the first (DEFAULT MODE) radio button */
button = gtk_radio_button_new_with_label(NULL, "cycle");
gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 0);
g_signal_connect(GTK_OBJECT(button), "clicked",
                 GTK_SIGNAL_FUNC (switch_keyword), (gpointer) button);
g_object_set_data(G_OBJECT(button), "key", (gpointer) CYCLE);
g_object_set_data(G_OBJECT(button), "ptr", (gpointer) data);
if (data->gulp.switch_type == CYCLE)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* make a radio group */
keyword[1] = gtk_radio_button_group(GTK_RADIO_BUTTON(button));
/* do the next button */
button = gtk_radio_button_new_with_label(keyword[1], "gnorm");
gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 0);
g_signal_connect(GTK_OBJECT(button), "clicked",
                 GTK_SIGNAL_FUNC (switch_keyword), (gpointer) button);
g_object_set_data(G_OBJECT(button), "key", (gpointer) GNORM);
g_object_set_data(G_OBJECT(button), "ptr", (gpointer) data);
if (data->gulp.switch_type == GNORM || data->gulp.switch_type < 0)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

entry = gtk_entry_new();
gtk_entry_set_text(GTK_ENTRY(entry), g_strdup_printf("%f", data->gulp.switch_value));
gtk_box_pack_end(GTK_BOX(vbox), entry, TRUE, FALSE, 0);

g_signal_connect(GTK_OBJECT(entry), "changed",
                 GTK_SIGNAL_FUNC(switch_value_changed), (gpointer) entry);
g_object_set_data(G_OBJECT(entry), "ptr", (gpointer) data);

/* frame */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(page), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
hbox = gtk_hbox_new(FALSE, 0);
gtk_container_add (GTK_CONTAINER(frame),hbox);
gtk_container_set_border_width(GTK_CONTAINER(hbox), PANEL_SPACING);

/* optimization cycles */
label = gtk_label_new(" Maximum cycles ");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
gtksh_direct_spin(NULL, &data->gulp.maxcyc, 0, 500, 10, NULL, NULL, hbox);

/* dynamics page */
page = gtk_vbox_new(FALSE,0);
label = gtk_label_new (" Dynamics ");
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), page, label);
gtk_container_set_border_width (GTK_CONTAINER(vbox), 10);

/* dynamics ensemble */
frame = gtk_frame_new("Ensemble");
gtk_box_pack_start(GTK_BOX (page), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame),vbox);

/* do the first (DEFAULT MODE) radio button */
button = gtk_radio_button_new_with_label (NULL, "NVE");
gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, TRUE, 0);
g_signal_connect(GTK_OBJECT (button), "clicked",
                 GTK_SIGNAL_FUNC (cb_gulp_keyword), (gpointer) button);
g_object_set_data(G_OBJECT(button), "key", (gpointer) NVE);
g_object_set_data(G_OBJECT(button), "ptr", (gpointer) data);
if (data->gulp.ensemble == NVE)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* make a radio group */
keyword[1] = gtk_radio_button_group (GTK_RADIO_BUTTON (button));
/* do the next button */
button = gtk_radio_button_new_with_label (keyword[1], "NVT");
gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, TRUE, 0);
g_signal_connect(GTK_OBJECT (button), "clicked",
                 GTK_SIGNAL_FUNC (cb_gulp_keyword), (gpointer) button);
g_object_set_data(G_OBJECT(button), "key", (gpointer) NVT);
g_object_set_data(G_OBJECT(button), "ptr", (gpointer) data);
/* NEW */
if (data->gulp.ensemble == NVT)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* subsequent button... */
button = gtk_radio_button_new_with_label(gtk_radio_button_group
       (GTK_RADIO_BUTTON (button)), "NPT");
gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, TRUE, 0);
g_signal_connect(GTK_OBJECT (button), "clicked",
                 GTK_SIGNAL_FUNC (cb_gulp_keyword), (gpointer) button);
g_object_set_data(G_OBJECT(button), "key", (gpointer) NPT);
g_object_set_data(G_OBJECT(button), "ptr", (gpointer) data);
if (data->gulp.ensemble == NPT)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* TODO */
/* step size/run length, trajectory file etc. */

/* vibrational eigenvector display page */
page = gtk_vbox_new(FALSE,0);
label = gtk_label_new (" Vibrational ");
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), page, label);
phonon_display_box(data, page);


/* create misc page */
hbox = gtk_hbox_new(FALSE,0);
label = gtk_label_new (" Misc ");
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), hbox, label);

page = gtk_vbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(hbox), page, FALSE, FALSE, 0);

frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(page), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame),vbox);

/* allow this for all models */
/* as some surface calcs are done on 2D models */
button = gtk_check_button_new_with_label("No attachment energy calculation");
gtk_box_pack_start(GTK_BOX(vbox), button, TRUE, TRUE, 0);
g_signal_connect(GTK_OBJECT (button), "clicked",
                 GTK_SIGNAL_FUNC (cb_gulp_keyword), (gpointer) button);
g_object_set_data(G_OBJECT(button), "key", (gpointer) NO_EATT);
g_object_set_data(G_OBJECT(button), "ptr", (gpointer) data);

/* frame */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(page), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
hbox = gtk_hbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame), hbox);
gtk_container_set_border_width(GTK_CONTAINER(hbox), PANEL_SPACING);
gtksh_direct_spin("Temperature", &data->gulp.temp, 0, 500, 10, NULL, NULL, hbox);

/* frame */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(page), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
hbox = gtk_hbox_new(TRUE, 0);
gtk_container_add (GTK_CONTAINER(frame),hbox);
gtk_container_set_border_width(GTK_CONTAINER(hbox), PANEL_SPACING);
gtksh_direct_spin("Surface dipole cutoff", &data->gulp.sdipole_tolerance,
                                    0.001, 0.1, 0.001, NULL, NULL, hbox);

/* setup the filename entry part */
frame = gtk_frame_new("Supplied data");
gtk_box_pack_start(GTK_BOX(page),frame,FALSE,FALSE,0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
hbox = gtk_hbox_new(FALSE, 0);
gtk_container_add (GTK_CONTAINER(frame),hbox);
gtk_container_set_border_width(GTK_CONTAINER(hbox), PANEL_SPACING);

/* left vbox */
vbox = gtk_vbox_new(TRUE, 2);
gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);

/* GULP template data */
/* TODO - ability to edit (via another dialog) */
label = gtk_label_new("Potentials");
gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, FALSE, 0);
gtk_misc_set_alignment(GTK_MISC(label), 0.0f, 0.0f);
label = gtk_label_new("Species data");
gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, FALSE, 0);
gtk_misc_set_alignment(GTK_MISC(label), 0.0f, 0.0f);
label = gtk_label_new("Element data");
gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, FALSE, 0);
gtk_misc_set_alignment(GTK_MISC(label), 0.0f, 0.0f);

/* right vbox */
vbox = gtk_vbox_new(TRUE, 2);
gtk_box_pack_start (GTK_BOX (hbox), vbox, FALSE, FALSE, 0);

g_string_sprintf(line,"%d", g_slist_length(data->gulp.ff_list));
label = gtk_label_new(line->str);
gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);
g_string_sprintf(line,"%d", g_slist_length(data->gulp.species_list));
label = gtk_label_new(line->str);
gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);
g_string_sprintf(line,"%d", g_slist_length(data->gulp.element_list));
label = gtk_label_new(line->str);
gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);

/* setup the filename entry */
frame = gtk_frame_new("Files");
gtk_box_pack_start(GTK_BOX(box),frame,FALSE,FALSE,0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
hbox = gtk_hbox_new(FALSE, 0);
gtk_container_add (GTK_CONTAINER(frame),hbox);
gtk_container_set_border_width(GTK_CONTAINER(hbox), PANEL_SPACING);

/* left vbox */
vbox = gtk_vbox_new(TRUE, 2);
gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);

/* temporary GULP input filename */
label = gtk_label_new(" Job input file ");
gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);
label = gtk_label_new(" Dump file ");
gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);
label = gtk_label_new(" Potential library ");
gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);

/* right vbox */
vbox = gtk_vbox_new(TRUE, 2);
gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);

/* temporary input */
gp_temp_file = gtk_entry_new_with_max_length(LINELEN);
gtk_entry_set_text(GTK_ENTRY(gp_temp_file), data->gulp.temp_file);
gtk_box_pack_start(GTK_BOX(vbox), gp_temp_file, TRUE, TRUE, 0);
gtk_entry_set_editable(GTK_ENTRY(gp_temp_file), FALSE);

/* dump file */
gp_dump_file = gtk_entry_new_with_max_length(LINELEN);
gtk_entry_set_text(GTK_ENTRY(gp_dump_file), data->gulp.dump_file);
gtk_box_pack_start(GTK_BOX(vbox), gp_dump_file, TRUE, TRUE, 0);
gtk_entry_set_editable(GTK_ENTRY(gp_dump_file), FALSE);

/* potential library */
entry = gtk_entry_new_with_max_length(LINELEN);
gtk_entry_set_text(GTK_ENTRY(entry), data->gulp.libfile);
gtk_box_pack_start (GTK_BOX (vbox), entry, TRUE, TRUE, 0);
/* event */
g_signal_connect(GTK_OBJECT (entry), "changed",
                 GTK_SIGNAL_FUNC (cb_gulp_keyword), (gpointer) entry);
g_object_set_data(G_OBJECT(entry), "key", (gpointer) LIB_FILE);
g_object_set_data(G_OBJECT(entry), "ptr", (gpointer) data);


/* setup the filename entry */
frame = gtk_frame_new("Details");
gtk_box_pack_start(GTK_BOX(box),frame,FALSE,FALSE,0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
hbox = gtk_hbox_new(FALSE, 0);
gtk_container_add (GTK_CONTAINER(frame),hbox);

/* left vbox */
vbox = gtk_vbox_new(TRUE, 2);
gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);

/* temporary GULP input filename */
label = gtk_label_new("Structure name");
gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);
label = gtk_label_new("Total energy (eV)");
gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);
if (data->periodic == 2)
  {
  label = gtk_label_new("Surface bulk energy (eV)");
  gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);
  label = gtk_label_new("Surface dipole");
  gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);
  label = gtk_label_new("Surface energy");
  gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);
  label = gtk_label_new("Attachment energy");
  gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);
  }

/* right vbox */
vbox = gtk_vbox_new(TRUE, 2);
gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);

entry = gtk_entry_new_with_max_length(LINELEN);
gtk_entry_set_text(GTK_ENTRY(entry), data->basename);
gtk_box_pack_start(GTK_BOX(vbox), entry, TRUE, TRUE, 0);
g_signal_connect(GTK_OBJECT(entry), "changed",
                 GTK_SIGNAL_FUNC(name_entry_changed), (gpointer) data);

data->gulp.energy_entry = gtk_entry_new_with_max_length(LINELEN);
if (data->gulp.free)
  g_string_sprintf(line,"%f (free energy)",data->gulp.energy);
else
  g_string_sprintf(line,"%f",data->gulp.energy);

gtk_entry_set_text(GTK_ENTRY(data->gulp.energy_entry),line->str);
gtk_box_pack_start (GTK_BOX (vbox), data->gulp.energy_entry, TRUE, TRUE, 0);
gtk_entry_set_editable(GTK_ENTRY(data->gulp.energy_entry), FALSE);

if (data->periodic == 2)
  {
  data->gulp.sbe_entry = gtk_entry_new_with_max_length(LINELEN);
  g_string_sprintf(line,"%f",data->gulp.sbulkenergy);
  gtk_entry_set_text(GTK_ENTRY(data->gulp.sbe_entry),line->str);
  gtk_box_pack_start(GTK_BOX(vbox), data->gulp.sbe_entry, TRUE, TRUE, 0);

/* NEW - sbulkenergy editable */
gtk_entry_set_editable(GTK_ENTRY(data->gulp.sbe_entry), TRUE);
g_signal_connect(GTK_OBJECT(data->gulp.sbe_entry), "changed",
                 GTK_SIGNAL_FUNC(cb_gulp_keyword), (gpointer) entry);
g_object_set_data(G_OBJECT(data->gulp.sbe_entry), "key", (gpointer) SBULK_ENERGY);
g_object_set_data(G_OBJECT(data->gulp.sbe_entry), "ptr", (gpointer) data);

/* surface dipole */
  data->gulp.sdipole_entry = gtk_entry_new_with_max_length(LINELEN);
  g_string_sprintf(line,"%f e.Angs", data->gulp.sdipole);
  gtk_entry_set_text(GTK_ENTRY(data->gulp.sdipole_entry), line->str);
  gtk_box_pack_start(GTK_BOX(vbox), data->gulp.sdipole_entry, TRUE, TRUE, 0);
  gtk_entry_set_editable(GTK_ENTRY(data->gulp.sdipole_entry), FALSE);
/* surface energy */
  data->gulp.esurf_entry = gtk_entry_new_with_max_length(LINELEN);
  if (data->gulp.esurf[1] == 0.0)
     g_string_sprintf(line,"%f    %s",
                            data->gulp.esurf[0], data->gulp.esurf_units);
  else
     g_string_sprintf(line,"%f    %s",
                            data->gulp.esurf[1], data->gulp.esurf_units);

  gtk_entry_set_text(GTK_ENTRY(data->gulp.esurf_entry), line->str);
  gtk_box_pack_start(GTK_BOX(vbox), data->gulp.esurf_entry, TRUE, TRUE, 0);
  gtk_entry_set_editable(GTK_ENTRY(data->gulp.esurf_entry), FALSE);
/* attachment energy */
  data->gulp.eatt_entry = gtk_entry_new_with_max_length(LINELEN);
  if (data->gulp.eatt[1] == 0.0)
     g_string_sprintf(line,"%f    %s",
                            data->gulp.eatt[0], data->gulp.eatt_units);
  else
     g_string_sprintf(line,"%f    %s",
                            data->gulp.eatt[1], data->gulp.eatt_units);
  gtk_entry_set_text(GTK_ENTRY(data->gulp.eatt_entry), line->str);
  gtk_box_pack_start(GTK_BOX(vbox), data->gulp.eatt_entry, TRUE, TRUE, 0);
  gtk_entry_set_editable(GTK_ENTRY(data->gulp.eatt_entry), FALSE);
  }

/* execution button */
/*
hbox = gtk_hbox_new(TRUE, 0);
gtk_box_pack_end(GTK_BOX(box), hbox, FALSE, FALSE, PANEL_SPACING);
gtksh_stock_button(GTK_STOCK_EXECUTE, run_gulp, data, hbox);
*/

/* done */
gtk_widget_show_all(box);

g_string_free(line, TRUE);
pick_model(data);
}

/****************************************/
/* run the approriate energetics method */
/****************************************/
void energy_execute(GtkWidget *w, GtkNotebook *notebook)
{
gint n;
struct model_pak *model;

model = sysenv.active_model;
if (!model)
  return;

n = gtk_notebook_get_current_page(notebook);
switch (n)
  {
  case 0:
    run_gulp(NULL, model);
    break;
  case 1:
    run_gamess(NULL, model);
    break;
  }
}

/*********************************/
/* cleanup for energetics dialog */
/*********************************/
void energetics_cleanup(struct model_pak *model)
{
g_assert(model != NULL);

model->gulp.energy_entry = NULL;
model->gulp.sdipole_entry = NULL;
model->gulp.esurf_entry = NULL;
model->gulp.eatt_entry = NULL;
}

/*****************************/
/* unified energetics dialog */
/*****************************/
void energetics_dialog(void)
{
gpointer dialog;
GtkWidget *window, *page;
GtkWidget *frame, *label, *notebook;
struct model_pak *model;

model = sysenv.active_model;
if (!model)
  return;

/* request an energetics dialog */
dialog = dialog_request(ENERGETICS, "Energetics", NULL, energetics_cleanup, model);
if (!dialog)
  return;
window = dialog_window(dialog);

/* notebook frame */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(window)->vbox), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);

/* create notebook */
notebook = gtk_notebook_new();
gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_TOP);
gtk_container_add(GTK_CONTAINER(frame), notebook);
gtk_notebook_set_show_border(GTK_NOTEBOOK(notebook), FALSE);

/* page 1 */
page = gtk_vbox_new(FALSE,0);
label = gtk_label_new (" GULP ");
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), page, label);
gtk_container_set_border_width(GTK_CONTAINER(page), PANEL_SPACING);
gulp_widget(page, model);

/* page 2 */
page = gtk_vbox_new(FALSE,0);
label = gtk_label_new (" GAMESS ");
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), page, label);
gtk_container_set_border_width(GTK_CONTAINER(page), PANEL_SPACING);
gamess_widget(page, model);

/* terminating button */
gtksh_stock_button(GTK_STOCK_EXECUTE, energy_execute, notebook,
                   GTK_DIALOG(window)->action_area);

gtksh_stock_button(GTK_STOCK_CLOSE, dialog_destroy, dialog,
                   GTK_DIALOG(window)->action_area);

gtk_widget_show_all(window);
}

