/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>

#ifndef __WIN32
#include <sys/times.h>
#endif

#include "gdis.h"
#include "coords.h"
#include "edit.h"
#include "matrix.h"
#include "opengl.h"
#include "render.h"
#include "select.h"
#include "gtkshorts.h"
#include "interface.h"
#include "dialog.h"
#include "measure.h"

extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

gint selected_label = -1;
GtkWidget *label_list, *start_spin, *stop_spin;
GtkWidget *match1, *match2, *match3, *match4, *search;
GtkWidget *meas_tv;
GtkTreeStore *meas_ts=NULL;

enum {MEAS_NAME, MEAS_ATOMS, MEAS_VALUE, MEAS_MODEL, MEAS_POINTER, MEAS_NCOLS};

/************************/
/* measurement printout */
/************************/
void meas_dump_all(void)
{
struct model_pak *model;

model = sysenv.active_model;
if (model)
  measure_dump_all(model);
}

/*****************************/
/* safely acquire tree model */
/*****************************/
GtkTreeModel *meas_treemodel(void)
{
if (!meas_tv)
  return(NULL);
if (!GTK_IS_TREE_VIEW(meas_tv))
  return(NULL);
return(gtk_tree_view_get_model(GTK_TREE_VIEW(meas_tv)));
}

/*********************************************/
/* free data pointers assoc with an iterator */
/*********************************************/
void meas_free_iter_data(GtkTreeIter *iter)
{
gchar *text;
GtkTreeModel *treemodel;

treemodel = meas_treemodel();
if (!treemodel)
  return;

gtk_tree_model_get(treemodel, iter, MEAS_NAME, &text, -1);
g_free(text);
gtk_tree_model_get(treemodel, iter, MEAS_ATOMS, &text, -1);
g_free(text);
gtk_tree_model_get(treemodel, iter, MEAS_VALUE, &text, -1);
g_free(text);
}

/************************/
/* graft model iterator */
/************************/
gint meas_model_iter(GtkTreeIter *iter, struct model_pak *model)
{
GtkTreeModel *treemodel;
struct model_pak *target=NULL;

/* checks */
g_assert(model != NULL);
if (!meas_ts)
  return(0);
treemodel = meas_treemodel();
if (!treemodel)
  return(0);

/* get the first parent iterator */
if (gtk_tree_model_get_iter_first(treemodel, iter))
  {
/* search for target model, until we run out of iterators */
  gtk_tree_model_get(treemodel, iter, MEAS_MODEL, &target, -1);

  while (target != model)
    {
    if (gtk_tree_model_iter_next(treemodel, iter))
      gtk_tree_model_get(treemodel, iter, MEAS_MODEL, &target, -1);
    else
      break;
    }
  }

/* append new if model is not on the tree */
if (target != model)
  {
  gtk_tree_store_append(meas_ts, iter, NULL);
  gtk_tree_store_set(meas_ts, iter, MEAS_NAME, model->basename,
                                    MEAS_MODEL, model,
                                    MEAS_POINTER, NULL,
                                    -1);
  }
return(1);
}

/******************************/
/* graft a single measurement */
/******************************/
void meas_graft_single(gpointer measure, struct model_pak *model)
{
gchar *info[3];
GtkTreeIter parent, iter;

/* checks */
g_assert(model != NULL);
g_assert(measure != NULL);

/* update dialog (if it exists) */
if (!meas_ts)
  return;
if (!meas_model_iter(&parent, model))
  return;

/* set child iterator data */
gtk_tree_store_append(meas_ts, &iter, &parent);

info[0] = measure_type_label_create(measure);
info[1] = measure_constituents_create(measure);
info[2] = g_strdup(measure_value_get(measure));

/* add the info */
gtk_tree_store_set(meas_ts, &iter, MEAS_NAME, info[0],
                                   MEAS_ATOMS, info[1],
                                   MEAS_VALUE, info[2],
                                   MEAS_MODEL, model,
                                   MEAS_POINTER, measure,
                                   -1);
g_free(info[0]);
g_free(info[1]);
g_free(info[2]);

gtk_tree_view_expand_all(GTK_TREE_VIEW(meas_tv));
}

/***********************************************************/
/* remove a model's measurements, leaving the model's iter */
/***********************************************************/
void meas_prune_all(struct model_pak *model)
{
GtkTreeIter parent, child;
GtkTreeModel *treemodel;

/* checks */
g_assert(model != NULL);
if (!meas_ts)
  return;
treemodel = meas_treemodel();
if (!treemodel)
  return;

if (meas_model_iter(&parent, model))
  {
/* remove all child iterators and free their data */ 
  while (gtk_tree_model_iter_children(treemodel, &child, &parent))
    {
    meas_free_iter_data(&child);
    gtk_tree_store_remove(meas_ts, &child); 
    }
  }
}

/**********************************/
/* graft measurements to the tree */
/**********************************/
void meas_graft_all(struct model_pak *model)
{
GSList *list;
GtkTreeModel *treemodel;
GtkTreeIter parent;
GtkTreePath *path;

/* return */
g_assert(model != NULL);
if (!meas_ts)
  return;
if (!meas_tv)
  return;

/* get model's iterator (if any) */
if (!meas_model_iter(&parent, model))
  return;

if (!g_slist_length(model->measure_list))
  {
/* no measurements - remove parent iterator */
  gtk_tree_store_remove(meas_ts, &parent); 
  }
else
  {
/* add all measurements */
  for (list=model->measure_list ; list ; list=g_slist_next(list))
    meas_graft_single(list->data, model);

/* expand model's iterator row */
  treemodel = meas_treemodel();
  if (treemodel)
    {
    path = gtk_tree_model_get_path(treemodel, &parent);
    gtk_tree_view_expand_row(GTK_TREE_VIEW(meas_tv), path, FALSE);
    gtk_tree_path_free(path);
    }
  }
}

/***********************************/
/* update measurements for a model */
/***********************************/
void meas_graft_model(struct model_pak *model)
{
g_assert(model != NULL);

/* update underlying measurements */
measure_update_global(model);
meas_prune_all(model);
meas_graft_all(model);
}

/*************************************/
/* remove model and its measurements */
/*************************************/
void meas_prune_model(struct model_pak *model)
{
GtkTreeIter parent, child;
GtkTreeModel *treemodel;

/* checks */
g_assert(model != NULL);
if (!meas_ts)
  return;
treemodel = meas_treemodel();
if (!treemodel)
  return;

if (meas_model_iter(&parent, model))
  {
/* free all child iterator data (iterators themselves are auto freed after) */
  if (gtk_tree_model_iter_children(treemodel, &child, &parent))
    {
    do
      {
      meas_free_iter_data(&child);
      }
    while (gtk_tree_model_iter_next(treemodel, &child));
    }
/* remove parent (auto removes all child iterators) */
  gtk_tree_store_remove(meas_ts, &parent); 
  }
}

/***************************/
/* geometry label deletion */
/***************************/
void meas_prune_selected(void)
{
GtkTreeIter iter;
GtkTreeModel *treemodel;
GtkTreeSelection *selection;
gpointer measurement=NULL;
struct model_pak *model=NULL;

/* checks */
treemodel = meas_treemodel();
if (!treemodel)
  return;

/* get selected iterator */
if (!(selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(meas_tv))))
  return;
if (!gtk_tree_selection_get_selected(selection, &treemodel, &iter))
  return;

gtk_tree_model_get(treemodel, &iter, MEAS_MODEL, &model, -1);
gtk_tree_model_get(treemodel, &iter, MEAS_POINTER, &measurement, -1);

g_assert(model != NULL);

/* if measurement selected - delete single, else delete all */
if (measurement)
  {
/* free data */
  meas_free_iter_data(&iter);
  gtk_tree_store_remove(meas_ts, &iter); 
  measure_free(measurement, model);
  }
else
  {
  meas_prune_model(model);
  measure_free_all(model);
  }

redraw_canvas(SINGLE);
}

/*****************/
/* info on bonds */
/*****************/
void info_bond(GtkWidget *w, gint x, gint y, struct model_pak *model)
{
gpointer measure;
GSList *list;
struct core_pak *core[2];
struct bond_pak *bond;

/* seek */
list = gl_seek_bond(w, x, y, model);
if (!list)
  return;
else
  bond = list->data;

/* FIXME - have to ignore hydrogen bonds to stop dodgy bond measurements being drawn */
if (bond->type == BOND_HBOND)
  return;

core[0] = bond->atom1;
core[1] = bond->atom2;

measure = measure_bond_test(core, 0.0, 0.0, model);
if (measure)
  {
  measure_update_single(measure, model);
  meas_graft_single(measure, model);
  }
redraw_canvas(SINGLE);
}

/*****************/
/* info on dists */
/*****************/
void info_dist(gint x, gint y, struct model_pak *model)
{
gdouble r[3];
gpointer measure;
static struct core_pak *core[2];

/* attempt to find core */
pixel_coord_map(x, y, r, model);
if (!(core[model->state] = seek_coord3d(r, model)))
  return;

/* are we looking at the 1st or the 2nd call? */
switch (model->state)
  {
  case 0:
/* select */
    select_add_core(core[0], model);
    model->state++;
    break;

  case 1:
/* remove from selection */
    select_del_core(core[0], model);
    select_del_core(core[1], model);

/* create the measurement (if it doesn't exist) */
    measure = measure_distance_test(MEASURE_DISTANCE, core, 0.0, 0.0, model);
    if (measure)
      {
      measure_update_single(measure, model);
      meas_graft_single(measure, model);
      }

    model->state--;
    break;

  default:
    g_assert_not_reached();
    model->state=0;
    return;
  }
redraw_canvas(SINGLE);
}

/******************/
/* info on angles */
/******************/
void info_angle(gint x, gint y, struct model_pak *model)
{
gdouble r[3];
gpointer measure;
static struct core_pak *core[3];

/* attempt to find core */
pixel_coord_map(x, y, r, model);
if (!(core[model->state] = seek_coord3d(r, model)))
  return;

/* are we looking at the 1st or the 2nd call? */
switch (model->state)
  {
  case 0:
  case 1:
    select_add_core(core[model->state], model);
    model->state++;
    break;

  case 2:
/* remove highlighting */
    select_del_core(core[0], model);
    select_del_core(core[1], model);
    select_del_core(core[2], model);

/* create the measurement (if it doesn't exist) */
    measure = measure_angle_test(core, 0.0, 180.0, model);
    if (measure)
      {
      measure_update_single(measure, model);
      meas_graft_single(measure, model);
      }

    model->state=0;
    break;

  default:
    g_assert_not_reached();
    model->state=0;
    return;
  }
redraw_canvas(SINGLE);
}

/****************************/
/* info on torsional angles */
/****************************/
void info_torsion(gint x, gint y, struct model_pak *model)
{
gdouble r[3];
gpointer measure;
static struct core_pak *core[4];

/* attempt to find core */
pixel_coord_map(x, y, r, model);
if (!(core[model->state] = seek_coord3d(r, model)))
  return;

/* what stage are we at? */
switch (model->state)
  {
  case 0:
  case 1:
  case 2:
    select_add_core(core[model->state], model);
    model->state++;
    break;

  case 3:
/* remove highlighting */
    select_del_core(core[0], model);
    select_del_core(core[1], model);
    select_del_core(core[2], model);
    select_del_core(core[3], model);

/* create the measurement (if it doesn't exist) */
    measure = measure_torsion_test(core, 0.0, 180.0, model);
    if (measure)
      {
      measure_update_single(measure, model);
      meas_graft_single(measure, model);
      }

    model->state=0;
    break;

  default:
    g_assert_not_reached();
  }
redraw_canvas(SINGLE);
}

/************************/
/* match pairs of atoms */
/************************/
#define DEBUG_PAIR_MATCH 0
gint pair_match(const gchar *label1, const gchar *label2,
                struct core_pak *core1, struct core_pak *core2)
{
gint a, b, i, j, mask;

#if DEBUG_PAIR_MATCH
printf("[%s,%s] : [%s,%s]\n", label1, label2, core1->label, core2->label);
#endif

/* if a or b = 0 => any i or j is accepted */
/* otherwise it must match either i or j */
/* if a and b are non zero, both i & j must match both a & b */
a = elem_symbol_test(label1);
b = elem_symbol_test(label2);
i = core1->atom_code;
j = core2->atom_code;

/* fill out the mask */
mask = 0;
if (i == a)
  {
/* if input label doesn't match the element symbol length - it means the */
/* user has put in something like H1 - compare this with the atom label */
  if (g_ascii_strcasecmp(label1, elements[i].symbol) != 0)
    {
    if (g_ascii_strcasecmp(core1->label, label1) == 0)
      mask |= 1;
    }
  else
    mask |= 1; 
  }

if (j == a)
  {
  if (g_ascii_strcasecmp(label1, elements[j].symbol) != 0)
    {
    if (g_ascii_strcasecmp(core2->label, label1) == 0)
      mask |= 2;
    }
  else
    mask |= 2; 
  }

if (i == b)
  {
  if (g_ascii_strcasecmp(label2, elements[i].symbol) != 0)
    {
    if (g_ascii_strcasecmp(core1->label, label2) == 0)
      mask |= 4;
    }
  else
    mask |= 4; 
  }

if (j == b)
  {
  if (g_ascii_strcasecmp(label2, elements[j].symbol) != 0)
    {
    if (g_ascii_strcasecmp(core2->label, label2) == 0)
      mask |= 8;
    }
  else
    mask |= 8; 
  }

#if DEBUG_PAIR_MATCH
printf("mask = %d\n", mask);
#endif

/* if both types must match - only two possibilities (a,b) or (b,a) */
/* but we can get further matches (than the required 2) when labels are compared */
if (a && b)
  {
  switch(mask)
    {
/* single valid pair match */
    case 6:
    case 9:
/* valid pair match plus one extra */
    case 7:
    case 11:
    case 13:
    case 14:
/* pair match in all possible combinations */
    case 15:
      break;
/* bad match - exit */
    default:
      return(0);
    }
  }

/* if only one type to match - any match at all will do */
if (a || b)
  if (!mask)
    return(0);

#if DEBUG_PAIR_MATCH
printf("accepted [%d][%d] as a match.\n",i,j);
#endif

return(1);
}

/***************************/
/* geometry search - bonds */
/***************************/
#define DEBUG_BOND_SEARCH 0
void cb_bond_search(gint type, gdouble r1, gdouble r2, struct model_pak *model)
{
const gchar *label[2];

g_assert(model != NULL);

label[0] = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(match1)->entry));
label[1] = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(match2)->entry));

measure_bond_search(label, r1, r2, model);
meas_graft_model(model);
}

/*******************************/
/* geometry search - distances */
/*******************************/
#define DEBUG_DIST_SEARCH 0
void cb_dist_search(gint type, gdouble r1, gdouble r2, gint inter, struct model_pak *model)
{
const gchar *label[2];

/* checks */
g_assert(model != NULL);

/* get pattern matching entries */
label[0] = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(match1)->entry));
label[1] = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(match2)->entry));

if (inter)
  measure_inter_search(label, r1, r2, model);
else
  measure_dist_search(label, r1, r2, model);
meas_graft_model(model);
}

/****************************/
/* geometry search - angles */
/****************************/
#define DEBUG_ANGLE_SEARCH 0
void cb_angle_search(gint type, gdouble a1, gdouble a2, struct model_pak *model)
{
const gchar *label[3];

g_assert(model != NULL);

/* get pattern matching entries */
label[0] = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(match1)->entry));
label[1] = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(match2)->entry));
label[2] = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(match3)->entry));

measure_angle_search(label, a1, a2, model);
meas_graft_model(model);
}

/******************************/
/* geometry search - torsions */
/******************************/
void cb_dihedral_search(struct model_pak *data, gdouble a1, gdouble a2, gint type)
{
type &= 15;
printf("Not implemented yet!\n");
}

/********************************************/
/* geometry search callback - general setup */
/********************************************/
void geom_search(void)
{
gint match_mask=0;
gdouble x1, x2;
const gchar *tmp;
struct model_pak *data;

data = sysenv.active_model;
if (!data)
  return;

/* and if switches back to dist/bond search, change to 2.0 (ie angs) */
x1 = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(start_spin));
x2 = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(stop_spin));

/* setup the atom matching requirements */
tmp = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(match1)->entry));
if (g_ascii_strncasecmp(tmp, "Select", 6) == 0)
  match_mask |= 1;
tmp = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(match2)->entry));
if (g_ascii_strncasecmp(tmp, "Select", 6) == 0)
  match_mask |= 2;
tmp = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(match3)->entry));
if (g_ascii_strncasecmp(tmp, "Select", 6) == 0)
  match_mask |= 4;

/* get the measurement type (ensure match mask ignores irrelevant bits) */
tmp = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(search)->entry));
if (g_ascii_strncasecmp(tmp, "Bonds", 5) == 0)
  cb_bond_search(match_mask, x1, x2, data);
if (g_ascii_strncasecmp(tmp, "Distance", 8) == 0)
  cb_dist_search(match_mask, x1, x2, 0, data);
if (g_ascii_strncasecmp(tmp, "Intermolecular", 14) == 0)
  cb_dist_search(match_mask, x1, x2, 1, data);
if (g_ascii_strncasecmp(tmp, "Angles", 6) == 0)
  cb_angle_search(match_mask, x1, x2, data);

redraw_canvas(SINGLE);
}

/********************/
/* clean up on exit */
/********************/
void meas_cleanup(void)
{
/* NB: since the tree store is independant of the model's geom list */
/* it must be completely removed (and then restored) with the dialog */
gtk_tree_store_clear(meas_ts);
meas_tv = NULL;
meas_ts = NULL;
}

/*****************************/
/* manually add measurements */
/*****************************/
void meas_manual_page(GtkWidget *box)
{
GtkWidget *frame, *vbox;

/* Frame - type */
frame = gtk_frame_new (NULL);
gtk_box_pack_start(GTK_BOX(box),frame,FALSE,FALSE,0); 
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);

vbox = gtk_vbox_new(TRUE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);

gtksh_button_x("Measure distances",
               gtk_switch_mode, GINT_TO_POINTER(DIST_INFO), vbox);
gtksh_button_x("Measure bonds",
               gtk_switch_mode, GINT_TO_POINTER(BOND_INFO), vbox);
gtksh_button_x("Measure angles",
               gtk_switch_mode, GINT_TO_POINTER(ANGLE_INFO), vbox);
gtksh_button_x("Measure torsions",
               gtk_switch_mode, GINT_TO_POINTER(DIHEDRAL_INFO), vbox);
}

/*******************************************/
/* search mode changed - update dist range */
/*******************************************/
void search_type_changed(GtkWidget *w, gpointer *data)
{
const gchar *tmp;

tmp = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(search)->entry));
if (g_ascii_strncasecmp(tmp, "Angles", 6) == 0)
  {
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(stop_spin), 180.0);
  gtk_widget_set_sensitive(GTK_WIDGET(match3), TRUE);
  }
else
  {
/* TODO - if previous mode was ANGLE change, else leave alone */
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(stop_spin), 2.0);
  gtk_widget_set_sensitive(GTK_WIDGET(match3), FALSE);
  }
}

/***************/
/* search page */
/***************/
void meas_search_page(GtkWidget *box)
{
GtkWidget *frame, *vbox, *hbox, *table, *button, *label;
GtkAdjustment *adj;
GList *match_list=NULL, *search_list=NULL;

/* Frame - type */
frame = gtk_frame_new (NULL);
gtk_box_pack_start(GTK_BOX(box),frame,FALSE,FALSE,0); 
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);

vbox = gtk_vbox_new(FALSE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);

/* atom matching options */
table = gtk_table_new(3, 4, FALSE);
gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, 0);

/* labels for top row */
label = gtk_label_new("Atom 1");
gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,0,1);
label = gtk_label_new("Atom 2");
gtk_table_attach_defaults(GTK_TABLE(table),label,1,2,0,1);
label = gtk_label_new("Atom 3");
gtk_table_attach_defaults(GTK_TABLE(table),label,2,3,0,1);

/* construct combo options */
match_list = g_list_append(match_list, "Any");
match_list = g_list_append(match_list, "Selected");
search_list = g_list_append(search_list, "Bonds");
search_list = g_list_append(search_list, "Distances");
search_list = g_list_append(search_list, "Intermolecular");
search_list = g_list_append(search_list, "Angles");

/* atom match combo boxes */
match1 = gtk_combo_new();
gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(match1)->entry), "Any");
gtk_combo_set_popdown_strings(GTK_COMBO(match1), match_list);
gtk_table_attach_defaults(GTK_TABLE(table),match1,0,1,1,2);

match2 = gtk_combo_new();
gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(match2)->entry), "Any");
gtk_combo_set_popdown_strings(GTK_COMBO(match2), match_list);
gtk_table_attach_defaults(GTK_TABLE(table),match2,1,2,1,2);

match3 = gtk_combo_new();
gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(match3)->entry), "Any");
gtk_combo_set_popdown_strings(GTK_COMBO(match3), match_list);
gtk_table_attach_defaults(GTK_TABLE(table),match3,2,3,1,2);

/* labels for second row */
label = gtk_label_new("Search type");
gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,2,3);
label = gtk_label_new("Search range");
gtk_table_attach_defaults(GTK_TABLE(table),label,1,2,2,3);

/* search type combo box */
search = gtk_combo_new();
gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(search)->entry), "Bonds");
gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(search)->entry), FALSE);
gtk_combo_set_popdown_strings(GTK_COMBO(search), search_list);
gtk_table_attach_defaults(GTK_TABLE(table),search,0,1,3,4);

/* CURRENT */
g_signal_connect(GTK_OBJECT(GTK_COMBO(search)->entry), "changed", 
                 GTK_SIGNAL_FUNC(search_type_changed), (gpointer *) search);


/* stop distance spinner */
hbox = gtk_hbox_new(TRUE, 0);

/* start value */
adj = (GtkAdjustment *) gtk_adjustment_new
      (0.1, 0.0, 360.0, 0.1, 1.0, 0);
start_spin = gtk_spin_button_new (adj, 0.01, 2);
gtk_spin_button_set_wrap(GTK_SPIN_BUTTON (start_spin), FALSE);
gtk_box_pack_start(GTK_BOX(hbox), start_spin, TRUE, TRUE, 0);

/* stop value */
adj = (GtkAdjustment *) gtk_adjustment_new
      (2.0, 0.0, 360.0, 0.1, 1.0, 0);
stop_spin = gtk_spin_button_new (adj, 0.01, 2);
gtk_spin_button_set_wrap(GTK_SPIN_BUTTON (stop_spin), FALSE);
gtk_box_pack_start(GTK_BOX(hbox), stop_spin, TRUE, TRUE, 0);
gtk_table_attach_defaults(GTK_TABLE(table),hbox,1,2,3,4);

/* search button */
button = gtksh_stock_button(GTK_STOCK_FIND, geom_search, NULL, NULL);
gtk_table_attach_defaults(GTK_TABLE(table),button,2,3,3,4);

/* I don't like doing this, but the default size is */
/* much larger than it needs to be, wasting too much space */
gtk_widget_set_size_request(match1, 13*sysenv.gtk_fontsize, -1);
gtk_widget_set_size_request(match2, 13*sysenv.gtk_fontsize, -1);
gtk_widget_set_size_request(match3, 13*sysenv.gtk_fontsize, -1);
gtk_widget_set_size_request(search, 13*sysenv.gtk_fontsize, -1);

gtk_table_set_row_spacings(GTK_TABLE(table), PANEL_SPACING);
gtk_table_set_col_spacings(GTK_TABLE(table), PANEL_SPACING);

/* default measurement is bonds */
gtk_widget_set_sensitive(GTK_WIDGET(match3), FALSE);
}

/***********************************/
/* select a particular measurement */
/***********************************/
void measure_select(gpointer measurement, struct model_pak *model)
{
GSList *list;

for (list=model->measure_list ; list ; list=g_slist_next(list))
  {
  if (measurement == list->data)
    {
/* selected measurement */
    measure_colour_set(1.0, 1.0, 0.0, list->data);
    }
  else
    {
/* reset other measurements */
    measure_colour_set(0.8, 0.8, 0.8, list->data);
    }
  }
redraw_canvas(SINGLE);
}

/**********************************/
/* measurement selection callback */
/**********************************/
void measure_select_changed(GtkTreeSelection *selection, gpointer data)
{
GtkTreeModel *treemodel;
GtkTreeIter iter;
gpointer measure;
struct model_pak *model;

if (gtk_tree_selection_get_selected(selection, &treemodel, &iter))
  {
  gtk_tree_model_get(treemodel, &iter, MEAS_MODEL, &model, -1);
  gtk_tree_model_get(treemodel, &iter, MEAS_POINTER, &measure, -1);
  if (measure && model)
    measure_select(measure, model);
  }
}

/**************************/
/* geometric measurements */
/**************************/
void geom_info(void)
{
gint i;
gpointer dialog;
gchar *titles[] = {"  Name  ", " Constituent atoms ", " Value "};
GSList *list;
GtkWidget *window, *swin, *notebook;
GtkWidget *frame, *vbox, *hbox, *label;
GtkTreeSelection *select;
GtkCellRenderer *renderer;
GtkTreeViewColumn *column;
struct model_pak *model;

/* request dialog */
dialog = dialog_request(GEOMETRY, "Measurements", NULL, meas_cleanup, NULL);
if (!dialog)
  return;
window = dialog_window(dialog);
gtk_window_set_default_size(GTK_WINDOW(window), 250, 450);

/* notebook */
notebook = gtk_notebook_new();
gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_TOP);
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(window)->vbox),notebook,FALSE,FALSE,0);
gtk_notebook_set_show_border(GTK_NOTEBOOK(notebook), TRUE);

/* manual measurement */
vbox = gtk_vbox_new(FALSE, PANEL_SPACING);
label = gtk_label_new(" Manual ");
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox, label);
meas_manual_page(vbox);

/* searching */
vbox = gtk_vbox_new(FALSE, PANEL_SPACING);
label = gtk_label_new(" Search ");
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox, label);
meas_search_page(vbox);

/* measurements viewing pane */
frame = gtk_frame_new ("Label list");
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(window)->vbox),frame,TRUE,TRUE,0); 
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
vbox = gtk_vbox_new(FALSE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(GTK_BOX(vbox)), PANEL_SPACING);
swin = gtk_scrolled_window_new (NULL, NULL);
gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(swin),
                               GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
gtk_box_pack_start(GTK_BOX(vbox), swin, TRUE, TRUE, 0);

/* NEW -  tree view */
/* underlying data storage */
/* 3 strings - data, 1 pointer - the measurement itself */
/* NB: model assoc. is passed with the selection callback */
if (!meas_ts)
  meas_ts = gtk_tree_store_new(MEAS_NCOLS, 
                               G_TYPE_STRING,
                               G_TYPE_STRING,
                               G_TYPE_STRING,
                               G_TYPE_POINTER,
                               G_TYPE_POINTER);

meas_tv = gtk_tree_view_new_with_model(GTK_TREE_MODEL(meas_ts));
gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(swin), meas_tv);

select = gtk_tree_view_get_selection(GTK_TREE_VIEW(meas_tv));
g_signal_connect(G_OBJECT(select), "changed",
                 G_CALLBACK(measure_select_changed), NULL);

/* set up column renderers */
for (i=0 ; i<=MEAS_VALUE ; i++)
  {
  renderer = gtk_cell_renderer_text_new();
  column = gtk_tree_view_column_new_with_attributes
              (titles[i], renderer, "text", i, NULL);
  gtk_tree_view_append_column(GTK_TREE_VIEW(meas_tv), column);
  }

/* list modification buttons */
hbox = gtk_hbox_new(TRUE, 10);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, PANEL_SPACING);

gtksh_button(" Select ", measure_select_all, NULL, hbox, TT);
gtksh_button(" Delete ", meas_prune_selected, NULL, hbox, TT);
gtksh_button(" Dump ", meas_dump_all, NULL, hbox, TT);

/* terminating button */
gtksh_stock_button(GTK_STOCK_CLOSE, dialog_destroy, dialog,
                   GTK_DIALOG(window)->action_area);

/* done */
gtk_widget_show_all(window);

/* refresh all measurements */
for (list=sysenv.mal ; list ; list=g_slist_next(list))
  {
  model = list->data;
  meas_graft_model(model);
  }
gtk_tree_view_expand_all(GTK_TREE_VIEW(meas_tv));
}

