/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <GL/gl.h>

#include "gdis.h"
#include "file.h"
#include "coords.h"
#include "matrix.h"
#include "molsurf.h"
#include "molsurf_data.h"
#include "spatial.h"
#include "surface.h"
#include "task.h"
#include "interface.h"
#include "opengl.h"
#include "zone.h"

/* main pak structures */
extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

/* molsurf globals */
struct model_pak *ms_data=NULL;
gdouble ms_dsize;
gint ms_zone=FALSE;
extern gint ms_epot_autoscale;
extern GtkWidget *surf_epot_min, *surf_epot_max, *surf_epot_div;
/* TODO - lock this if in thread */
GSList *ms_points=NULL;

/************************************************/
/* gaussian based distance to molecular surface */
/************************************************/
/* r is position, a is probe radius, list is core list */
gdouble ms_dist(gdouble *r, GSList *core_list)
{
gint i, j, k, m, ni, p, q;
gdouble a, sum, x[3];
GSList *list;
struct core_pak *core;

/* periodic images  */
ni = pow(3, ms_data->periodic);
p = ms_data->periodic/2;
if (p)
  q = ms_data->periodic - 2;
else
  q = 0;

sum = 0.0;
for (m=ni ; m-- ; )
  {
  i = (m % 3) - p;
  j = (m / 3) - p;
  k = (m / 9) - q;

  for (list=core_list ; list ; list=g_slist_next(list))
    {
    core = (struct core_pak *) list->data;

    a = elements[core->atom_code].vdw;

    ARR3SET(x, core->x);
    VEC3ADD(x, i, j, k);
    vecmat(ms_data->latmat, x);
    ARR3SUB(x, r);

    sum += exp(-(VEC3MAG(x) - a)/ms_dsize);
    }
  }
return(-ms_dsize*log(sum));
}

/*********************************************/
/* gaussian based molecular surface gradient */
/*********************************************/
struct core_pak *ms_grad(struct smv_pak *point, GSList *core_list)
{
gint i, j, k, m, ni, p, q, touch;
gdouble a, e, dx, sum1[3], sum2[3], x[3], tmp[3];
GSList *list;
struct core_pak *core, *touch_core;

VEC3SET(sum1, 0.0, 0.0, 0.0);
VEC3SET(sum2, 0.0, 0.0, 0.0);

if (ms_zone)
  {
  m = zone_index(point->rx, ms_data);
  core_list = zone_area_cores(1, ms_data->zones[m], ms_data);
  }

/* periodic images */
ni = pow(3, ms_data->periodic);
p = ms_data->periodic/2;
if (p)
  q = ms_data->periodic - 2;
else
  q = 0;

touch = 0;
touch_core = NULL;
for (list=core_list ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;

  a = elements[core->atom_code].vdw;

  for (m=ni ; m-- ; )
    {
    i = (m % 3) - p;
    j = (m / 3) - p;
    k = (m / 9) - q;

/* get separation vector */
    ARR3SET(x, core->x);

    VEC3ADD(x, i, j, k);
    vecmat(ms_data->latmat, x);

    ARR3SET(tmp, x);
    ARR3SUB(tmp, point->rx);
    dx = VEC3MAG(tmp);

/* vector components of the gradient */
    e = exp(-(dx - a)/ms_dsize);
    VEC3ADD(sum1, e, e, e);
    e /= (dx * ms_dsize);

    sum2[0] += e * (x[0] - point->rx[0]);
    sum2[1] += e * (x[1] - point->rx[1]);
    sum2[2] += e * (x[2] - point->rx[2]);

    if (dx < 1.1*a)
      {
      touch_core = core;
      touch++;
      }
    }
  }

/* compute overall normal */
for (i=3 ; i-- ; )
  point->nx[i] = sum2[i]/sum1[i];

normalize(point->nx, 3);

/* only return a core if it was the only one touching the supplied point */
if (touch == 1)
  return(touch_core);
return(NULL);
}

/***************************************/
/* create a triangular surface segment */
/***************************************/
struct smt_pak *new_triangle(struct smv_pak *p1,
                             struct smv_pak *p2,
                             struct smv_pak *p3)
{
struct smt_pak *triangle;

triangle = g_malloc(sizeof(struct smt_pak));

/* point references */
triangle->point[0] = p1;
triangle->point[1] = p2;
triangle->point[2] = p3;

/* adjacencies */
p1->adj = g_slist_prepend(p1->adj, p2);
p1->adj = g_slist_prepend(p1->adj, p3);
p2->adj = g_slist_prepend(p2->adj, p1);
p2->adj = g_slist_prepend(p2->adj, p3);
p3->adj = g_slist_prepend(p3->adj, p1);
p3->adj = g_slist_prepend(p3->adj, p2);

return(triangle);
}

/*******************************/
/* allocate for a new midpoint */
/*******************************/
struct smv_pak *new_point(gdouble *x)
{
struct smv_pak *p;

p = g_malloc(sizeof(struct smv_pak));
ms_points = g_slist_prepend(ms_points, p);

ARR3SET(p->rx, x);
p->adj = NULL;
p->ignore = FALSE;

return(p);
}

/*************************************************/
/* compute the real area of a triangular spatial */
/*************************************************/
gdouble triangle_area(struct smt_pak *triangle)
{
gdouble n[3];

calc_norm(n, triangle->point[0]->rx,
             triangle->point[1]->rx,
             triangle->point[2]->rx); 
return(0.5*VEC3MAG(n));
}

/****************************************/
/* calculate new molecular surface area */
/****************************************/
gdouble molsurf_area(GSList *tlist)
{
gdouble area=0.0;
GSList *item;
struct smt_pak *triangle;

for (item=tlist ; item ; item=g_slist_next(item))
  {
  triangle = (struct smt_pak *) item->data;

  area += triangle_area(triangle);
  }
return(area);
}

/**************************/
/* common colour routines */
/**************************/
void ms_afm_colour(gdouble *colour, gdouble value, struct model_pak *data)
{
/* TODO - need AFM min/max stored in data */
}

void ms_hfs_colour(gdouble *colour, gdouble value, struct model_pak *data)
{
VEC3SET(colour, 1.0, 1.0, 1.0);
VEC3MUL(colour, value);
}

void ms_epot_colour(gdouble *colour, gdouble value, struct model_pak *model)
{
gdouble r, g, b;

r = g = b = 1.0;

if (value < 0.0)
  {
  b *= 1.0 - value/model->gulp.epot_min;
  g *= 1.0 - value/model->gulp.epot_min;
  }
else
  {
  r *= 1.0 - value/model->gulp.epot_max;
  g *= 1.0 - value/model->gulp.epot_max;
  }

VEC3SET(colour, r, g, b);
}

/****************************/
/* min/max point calculator */
/****************************/
void ms_get_zlimits(gdouble *min, gdouble *max, GSList *list)
{
GSList *item;
struct smv_pak *point;

g_assert(list != NULL);

/* init */
point = list->data;
*min = *max = point->rx[2];

/* loop */
for (item=list ; item ; item=g_slist_next(item))
  {
  point = (struct smv_pak *) item->data;

  if (point->rx[2] < *min)
    *min = point->rx[2];
  if (point->rx[2] > *max)
    *max = point->rx[2];
  }
}

/*****************************************/
/* uses GULP's new epot list calc method */
/*****************************************/
gint ms_get_epot(GSList *list, struct model_pak *model)
{
gint run;
gchar *inp, *out, *full_inp, *full_out;
GSList *item;
struct vec_pak *vec;
struct smv_pak *point;

/* checks */
g_assert(model != NULL);

/* setup the site coordinate list */
if (model->gulp.epot_vecs)
  free_slist(model->gulp.epot_vecs);
model->gulp.epot_vecs = NULL;

/* fill out the list */
for (item=list ; item ; item=g_slist_next(item))
  {
  point = (struct smv_pak *) item->data;
  if (!point->adj)
    continue;

  vec = g_malloc(sizeof(struct vec_pak));

  ARR3SET(vec->rx, point->rx);

  model->gulp.epot_vecs = g_slist_prepend(model->gulp.epot_vecs, vec);
  }
model->gulp.epot_vecs = g_slist_reverse(model->gulp.epot_vecs);

/* setup filenames (NB: GULP will end up being run in sysenv.cwd) */
inp = gun("gin");
out = gun("got");
full_inp =  g_build_filename(sysenv.cwd, inp, NULL);
full_out =  g_build_filename(sysenv.cwd, out, NULL);

/* create a single point GULP input file */
run = model->gulp.run;
model->gulp.run = E_SINGLE;
write_gulp(full_inp, model);
model->gulp.run = run;

/* run the GULP calc & get the results */
/* NB: don't use inp/out here as gulp is run in sysenv.cwd */
/* and it casues a channel error */
if (exec_gulp(inp, out))
  return(1);
if (read_gulp_output(full_out, model))
  return(1);

g_free(full_inp);
g_free(full_out);
g_free(inp);
g_free(out);
return(0);
}

/******************************/
/* colour a list of triangles */
/******************************/
void ms_colour_by_touch(GSList *points)
{
GSList *list;
struct smv_pak *p;
struct core_pak *core;

for (list=points ; list ; list=g_slist_next(list))
  {
  p = list->data;
  if (!p->adj)
    continue;

  core = ms_grad(p, ms_data->cores);

  if (core)
    {
    ARR3SET(p->colour, core->colour);
    VEC3MUL(p->colour, 1.0/65535.0);
    }
  else
    {
    ARR3SET(p->colour, sysenv.render.rsurf_colour);
    }
  }
}

/*****************************************************/
/* colour triangles by cartesian z value (AFM style) */
/*****************************************************/
void ms_colour_by_height(GSList *points)
{
gdouble z, zmin, zmax;
GSList *list;
struct smv_pak *p;

/* determine the range */
zmin = G_MAXDOUBLE;
zmax = -G_MAXDOUBLE;
for (list=points ; list ; list=g_slist_next(list))
  {
  p = list->data;
  if (!p->adj)
    continue;

  if (p->rx[2] < zmin)
    zmin = p->rx[2];

  if (p->rx[2] > zmax)
    zmax = p->rx[2];
  }

/* colour the points */
for (list=points ; list ; list=g_slist_next(list))
  {
  p = list->data;
  if (!p->adj)
    continue;
  ms_grad(p, ms_data->cores);

  z = p->rx[2] - zmin;
  z *= 1.0/(zmax - zmin);
  VEC3SET(p->colour, 0.7*z+0.2, 0.8*z, 0.5*z*z*z*z);
  }
}

/*************************************/
/* colour by electrostatic potential */
/*************************************/
void ms_colour_by_potential(GSList *points)
{
gdouble *q;
GSList *list1, *list2;
struct smv_pak *p;
struct model_pak *model;

model = sysenv.active_model;
g_assert(model != NULL);

/* get potential for point list */
ms_get_epot(points, model);
model->ms_colour_scale = TRUE;

if (model->gulp.epot_autoscale)
  model->gulp.epot_divisions = 11;

/* point colouring */
list2 = model->gulp.epot_vals;
for (list1 = points ; list1 ; list1=g_slist_next(list1))
  {
  p = list1->data;
  if (!p->adj)
    continue;

  ms_grad(p, ms_data->cores);
  if (list2)
    {
    q = (gdouble *) list2->data;
    ms_epot_colour(p->colour, *q, model);
    list2 = g_slist_next(list2);
    }
   else
    {
/* default colour (ie electrostatic calc has failed) */
    VEC3SET(p->colour, 0.5, 0.5, 0.5);
    }
  }
}

/*****************************************************/
/* colour by the weight function (Hirshfeld surface) */
/*****************************************************/
void ms_colour_by_weight(GSList *points)
{
}

/***********************************************/
/* create triangles for surface representation */
/***********************************************/
void ms_create_triangles(GSList *tlist, struct model_pak *model)
{
gint i, invert;
gdouble n[3];
GSList *item1;
struct vec_pak *vec;
struct smt_pak *triangle;
struct spatial_pak *spatial;

g_assert(model != NULL);

/* NB: best to calc the normals here - so it's indep. of the colouring method */
for (item1=tlist ; item1 ; item1=g_slist_next(item1))
  {
  triangle = (struct smt_pak *) item1->data;

/* init a new spatial */
  spatial = g_malloc(sizeof(struct spatial_pak));
  spatial->type = SPATIAL_MOLSURF;
  spatial->method = GL_TRIANGLES;
  spatial->data = NULL;

/* use triangle normal to test point ordering */
  calc_norm(n, triangle->point[0]->rx,
               triangle->point[1]->rx,
               triangle->point[2]->rx); 
  invert = 0;

/* fill out the vertex list */
  for (i=3 ; i-- ; )
    {
    vec = g_malloc(sizeof(struct vec_pak));

/* copy vertex colour, coords & normal */
    ARR3SET(vec->colour, triangle->point[i]->colour);
    ARR3SET(vec->x, triangle->point[i]->rx);
    ARR3SET(vec->n, triangle->point[i]->nx);
    VEC3MUL(vec->n, -1.0);

/* normal test */
    if (via(n, vec->n, 3) > PI/2.0)
      invert = 1;

/* cartesian */
    vecmat(model->ilatmat, vec->n);
    vecmat(model->ilatmat, vec->x);

/* DEBUG - draw the normal */
/*
fn_make_spatial(SPATIAL_VECTOR, vec->x, vec->n, model);
*/

    spatial->data = g_slist_prepend(spatial->data, vec);
    }

/* vertex ordering */
  if (invert)
    spatial->data = g_slist_reverse(spatial->data);

  model->spatial = g_slist_prepend(model->spatial, spatial);
  }
}

/**********************************************/
/* triangulation post process and improvement */
/**********************************************/
void ms_process(void)
{
/* remove all points that are not part of the final triangulation */
/* too slow & unecessary (just skip points with NULL adj) */
/*
list = ms_points;
while (list)
  {
  p = list->data;
  list = g_slist_next(list);
  if (!p->adj)
    {
    ms_points = g_slist_remove(ms_points, p);
    g_free(p);
    }
  }
*/
}

/******************************************************************/
/* Linearly interpolate the position where an isosurface cuts     */
/* an edge between two vertices, each with their own scalar value */
/******************************************************************/
struct smv_pak *ms_interpolate(gdouble isolevel, struct smv_pak *p1, struct smv_pak *p2)
{
gdouble mu, valp1, valp2, x[3];
struct smv_pak *p;

valp1 = p1->value;
valp2 = p2->value;

if (fabs(isolevel-valp1) < 0.00001)
  return(p1);
if (fabs(isolevel-valp2) < 0.00001)
  return(p2);
if (fabs(valp1-valp2) < 0.00001)
  return(p1);

mu = (isolevel - valp1) / (valp2 - valp1);
x[0] = p1->rx[0] + mu * (p2->rx[0] - p1->rx[0]);
x[1] = p1->rx[1] + mu * (p2->rx[1] - p1->rx[1]);
x[2] = p1->rx[2] + mu * (p2->rx[2] - p1->rx[2]);

p = new_point(x);

return(p);
}

/*******************************************/
/* compute the triangulation for the given */
/* cube at the supplied isosurface value   */
/*******************************************/
GSList *ms_triangulate(gdouble isolevel, GSList *cube)
{
gint i, cubeindex;
struct smv_pak *p[8], **v, *v1, *v2, *v3;
struct smt_pak *t;
GSList *list;

g_assert(g_slist_length(cube) == 8);

i=0;
for (list=cube ; list ; list=g_slist_next(list))
  p[i++] = list->data;

cubeindex = 0;
if (!p[0]->ignore)
  if (p[0]->value < isolevel)
    cubeindex |= 1;
if (!p[1]->ignore)
  if (p[1]->value < isolevel)
    cubeindex |= 2;
if (!p[2]->ignore)
  if (p[2]->value < isolevel)
    cubeindex |= 4;
if (!p[3]->ignore)
  if (p[3]->value < isolevel)
    cubeindex |= 8;
if (!p[4]->ignore)
  if (p[4]->value < isolevel)
    cubeindex |= 16;
if (!p[5]->ignore)
  if (p[5]->value < isolevel)
    cubeindex |= 32;
if (!p[6]->ignore)
  if (p[6]->value < isolevel)
    cubeindex |= 64;
if (!p[7]->ignore)
  if (p[7]->value < isolevel)
    cubeindex |= 128;

/* cube is entirely in/out of the surface */
if (!ms_edge_table[cubeindex])
  return(NULL);

v = g_malloc(12 * sizeof(struct smv_pak *));
for (i=12 ; i-- ; )
  v[i] = NULL;

/* find the vertices where the surface intersects the cube */
if (ms_edge_table[cubeindex] & 1)
  v[0] = ms_interpolate(isolevel,p[0],p[1]);
if (ms_edge_table[cubeindex] & 2)
  v[1] = ms_interpolate(isolevel,p[1],p[2]);
if (ms_edge_table[cubeindex] & 4)
  v[2] = ms_interpolate(isolevel,p[2],p[3]);
if (ms_edge_table[cubeindex] & 8)
  v[3] = ms_interpolate(isolevel,p[3],p[0]);
if (ms_edge_table[cubeindex] & 16)
  v[4] = ms_interpolate(isolevel,p[4],p[5]);
if (ms_edge_table[cubeindex] & 32)
  v[5] = ms_interpolate(isolevel,p[5],p[6]);
if (ms_edge_table[cubeindex] & 64)
  v[6] = ms_interpolate(isolevel,p[6],p[7]);
if (ms_edge_table[cubeindex] & 128)
  v[7] = ms_interpolate(isolevel,p[7],p[4]);
if (ms_edge_table[cubeindex] & 256)
  v[8] = ms_interpolate(isolevel,p[0],p[4]);
if (ms_edge_table[cubeindex] & 512)
  v[9] = ms_interpolate(isolevel,p[1],p[5]);
if (ms_edge_table[cubeindex] & 1024)
  v[10] = ms_interpolate(isolevel,p[2],p[6]);
if (ms_edge_table[cubeindex] & 2048)
  v[11] = ms_interpolate(isolevel,p[3],p[7]);

/* create the triangulation */
list = NULL;
for (i=0 ; ms_tri_table[cubeindex][i]!=-1 ; i+=3)
  {
v1 = v[ms_tri_table[cubeindex][i]];
v2 = v[ms_tri_table[cubeindex][i+1]];
v3 = v[ms_tri_table[cubeindex][i+2]];

g_assert(v1 != NULL);
g_assert(v2 != NULL);
g_assert(v3 != NULL);

  t = new_triangle(v1, v2, v3);

  list = g_slist_prepend(list, t);
  }

return(list);
}

gint ms_index(gint i, gint j, gint k, gint *g)
{
return(i*g[1]*g[2] + j*g[2] + k);
}

/**********************************/
/* cube partitioning surface calc */
/**********************************/
#define DEBUG_MS_CUBE 0
void ms_cube(gdouble blur, gint type, gint method, struct model_pak *model)
{
gint i, j, k, z, index, eval, grid[3];
gdouble dx, dy, dz;
gdouble x[3], min[3], max[3];
GSList *plist, *tlist, *list;
struct smv_pak **p;

g_assert(model != NULL);

ms_data = model;
ms_points = NULL;

/* NEW - use this to approximate probe radius */
/* avoids problems with the fact that we want the molecular surf */
/* but the gaussian routines really give the accessible surf */
/* and subtracting off the normal can crunch the trianglulation */
/*
ms_dsize = isolevel / G_PI;
*/
ms_dsize = blur;

cor_calc_xlimits(min, max, model->cores);

/* cartesian setup */
for (i=model->periodic ; i<3 ; i++)
  {
/* buffer - TODO - factor in prad/max vdw radius */
  min[i] -= 5.0;
  max[i] += 5.0;
  }

/* fractional setup */
for (i=model->periodic ; i-- ; )
  {
  min[i] = 0.0;
  max[i] = 1.0;
  }

/* special case for surfaces - stop halfway between min & max */
/*
if (model->periodic == 2)
  {
  min[2] = 0.5 * (max[2] + min[2]);
  }
*/

/* TODO - more intelligent grid partitioning */
/* NB: zone_init() should be made more general */
/* so it can be tailored to the BLOCK_SIZE */
#define BLOCK_SIZE 0.5

for (i=model->periodic ; i<3 ; i++)
  grid[i] = 1 + (max[i] - min[i]) / BLOCK_SIZE;

for (i=model->periodic ; i-- ; )
  grid[i] = 1 + model->pbc[i] / BLOCK_SIZE;

for (i=3 ; i-- ; )
  if (grid[i] < 2)
    grid[i] = 2;

#if DEBUG_MS_CUBE
P3VEC(" min: ", min);
P3VEC(" max: ", max);
printf("grid: %d %d %d\n", grid[0], grid[1], grid[2]);
#endif

dx = (max[0] - min[0]) / (grid[0]-1);
dy = (max[1] - min[1]) / (grid[1]-1);
dz = (max[2] - min[2]) / (grid[2]-1);

/* CURRENT */
if (!model->periodic && g_slist_length(model->cores) > 10)
  ms_zone = TRUE;
else
  ms_zone = FALSE;

#if DEBUG_MS_CUBE
printf("discretizing... (%d)\n", ms_zone);
#endif

/* create grid points */
p = g_malloc(grid[0]*grid[1]*grid[2]*sizeof(struct smv_pak *));
for (i=grid[0] ; i-- ; )
  {
  for (j=grid[1] ; j-- ; )
    {
    eval = TRUE;
    for (k=grid[2] ; k-- ; )
      {
      x[0] = min[0] + i*dx;
      x[1] = min[1] + j*dy;
      x[2] = min[2] + k*dz;
      vecmat(model->latmat, x);
      index = ms_index(i, j, k, grid);
      p[index] = new_point(x);

/* zone core list checking */
      if (ms_zone)
        {
        z = zone_index(x, model);
        list = zone_area_cores(1, model->zones[z], model);
        }
      else
        list = model->cores;

/* evaluation */
      if (list)
        {
        if (eval)
          p[index]->value = ms_dist(x, list);
        else
          p[index]->value = -1000.0;
        }
      else
        p[index]->ignore = TRUE;

/* surface exception - stop evaluating after first contact */
      if (model->periodic == 2)
        if (p[index]->value < 0.0)
          eval = FALSE;
      }
    }
  }

#if DEBUG_MS_CUBE
printf("polygonizing...\n");
#endif

/* polygonize each cube */
tlist = NULL;
for (i=grid[0]-1 ; i-- ; )
  {
  for (j=grid[1]-1 ; j-- ; )
    {
    for (k=grid[2]-1 ; k-- ; )
      {
      plist = NULL;

      index = ms_index(i, j, k, grid);
      plist = g_slist_prepend(plist, p[index]);
      index = ms_index(i+1, j, k, grid);
      plist = g_slist_prepend(plist, p[index]);
      index = ms_index(i+1, j+1, k, grid);
      plist = g_slist_prepend(plist, p[index]);
      index = ms_index(i, j+1, k, grid);
      plist = g_slist_prepend(plist, p[index]);
      index = ms_index(i, j, k+1, grid);
      plist = g_slist_prepend(plist, p[index]);
      index = ms_index(i+1, j, k+1, grid);
      plist = g_slist_prepend(plist, p[index]);
      index = ms_index(i+1, j+1, k+1, grid);
      plist = g_slist_prepend(plist, p[index]);
      index = ms_index(i, j+1, k+1, grid);
      plist = g_slist_prepend(plist, p[index]);

      plist = g_slist_reverse(plist);

      list = ms_triangulate(0.0, plist);

      if (list)
        tlist = g_slist_concat(tlist, list);

      g_slist_free(plist);
      }
    }
  }


#if DEBUG_MS_CUBE
printf("processing...\n");
#endif

/* process the point list (remove points not part of the molsurf) */
/* TODO - enhancement phase (point merging/divide triangles etc.) */
ms_process();

#if DEBUG_MS_CUBE
printf("colouring...\n");
#endif

/* colour the resulting points */
switch (method)
  {
  case MS_TOUCH:
    ms_colour_by_touch(ms_points);
    break;

  case MS_AFM:
    ms_colour_by_height(ms_points);
    break;

  case MS_EPOT:
    ms_colour_by_potential(ms_points);
    break;

  case MS_HIRSHFELD:
    ms_colour_by_weight(ms_points);
    break;

  default:
    printf("Not implemented yet.\n");
  }

#if DEBUG_MS_CUBE
printf("exporting...\n");
#endif

/* create the display triangles */
ms_create_triangles(tlist, model);

/* TODO - free triangle/point data */
g_slist_free(tlist);
g_slist_free(ms_points);
}

