/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <stdlib.h>
#include <math.h>
#include <GL/gl.h>

#include "gdis.h"
#include "coords.h"
#include "interface.h"
#include "gtkshorts.h"
#include "matrix.h"
#include "spatial.h"
#include "numeric.h"
#include "morph.h"
#include "opengl.h"
#include "select.h"
#include "zone.h"

/* data structures */
extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

/**********************************/
/* spatial partitioning structure */
/**********************************/
struct zone_pak
{
gint grid[3];
GSList *cores;
GSList *shells;
};

/********************************/
/* free space partitioning data */
/********************************/
void zone_free(struct model_pak *model)
{
gint i;
struct zone_pak *zone;

/* free zone array core lists */
for (i=0 ; i<model->num_zones ; i++)
  {
  zone = model->zones[i];

  g_slist_free(zone->cores);
  g_slist_free(zone->shells);
  g_free(zone);
  }

/* free the zone array */
if (model->zones)
  g_free(model->zones);

model->zones = NULL;
model->num_zones = 0;
}

/*********************************/
/* initialize space partitioning */
/*********************************/
/* after this distance - connectivity is no longer considered (worst case) */
#define ZONE_SIZE 2.0
#define DEBUG_ZONE_INIT 0
void zone_init(struct model_pak *model)
{
gint i, j, k, zi, num_zones;
gint mad_core_flag = FALSE, mad_shel_flag = FALSE;
gint div[3];
gdouble tmp, dx[3], min[3], max[3];
GSList *list;
struct core_pak *core;
struct shel_pak *shel;
struct zone_pak *zone;

#if DEBUG_ZONE_INIT
printf("zone_init(%s) [%d D]\n", model->filename, model->periodic);
#endif

zone_free(model);

/* FIXME - will there be a problem for periodic models */
/* if atoms are not constrained ie in the range [0-1) */
/* TODO - allow zones "outside" pbc, but when comparing -> clamp to within */

/* get the (possibly mixed) frac/cart coord limits */
VEC3SET(min, 0.0, 0.0, 0.0);
VEC3SET(max, 0.0, 0.0, 0.0);
cor_calc_xlimits(min, max, model->cores);

/* set number of divisions - CARTESIAN PART ONLY */
for (i=model->periodic ; i<3 ; i++)
  {
/* safety margin */
  max[i] += 0.1;
  min[i] -= 0.1;
  tmp = dx[i] = max[i] - min[i];
  tmp /= ZONE_SIZE;
  tmp += 0.5;
  div[i] = (gint) tmp;
  }

/* set number of divisions - FRACTIONAL PART ONLY */
for (i=model->periodic ; i-- ; )
  {
  min[i] = 0.0;
  max[i] = 1.0;
  dx[i] = 1.0;
  tmp = model->pbc[i] / ZONE_SIZE;
  tmp += 0.5;
  div[i] =  (gint) tmp;
  }

/* division sizes */
for (i=3; i--; )
  {
  if (!div[i])
    div[i] = 1;
  dx[i] /=  div[i];
  }

#if DEBUG_ZONE_INIT
for (i=0 ; i<3 ; i++)
  printf("[%d] : %11.4f - %11.4f  (%f)(%d)\n", i, min[i], max[i], dx[i], div[i]);
#endif

/* TODO - if periodic - extend zone by 1 in all directions (unfragment) */
/* ie div +=2 , min -= dx, max += dx */
/* TODO - OR (better) if we can just constrain to get the equivalent existing zone */

ARR3SET(model->zone_min, min);
ARR3SET(model->zone_max, max);
ARR3SET(model->zone_div, div);
ARR3SET(model->zone_dx, dx);

num_zones = div[0] * div[1] * div[2];
model->num_zones = num_zones;

#if DEBUG_ZONE_INIT
printf("Allocating %dx%dx%d = %d zones...\n", div[0], div[1], div[2], num_zones);
#endif

model->zones = g_malloc(num_zones * sizeof(struct zone_pak *));

#if DEBUG_ZONE_INIT
printf("Initializing %d zones...\n", num_zones);
#endif

for (k=0 ; k<div[2] ; k++)
  {
  for (j=0 ; j<div[1] ; j++)
    {
    for (i=0 ; i<div[0] ; i++)
      {
      zi = k*div[1]*div[0]; 
      zi += j*div[0]; 
      zi += i;

      model->zones[zi] = g_malloc(sizeof(struct zone_pak));

      zone = (struct zone_pak *) model->zones[zi];
      zone->cores = NULL;
      zone->shells = NULL;
      VEC3SET(zone->grid, i, j, k);
      }
    }
  }

/* TODO - check if no zonal division is required */
/* NB: only for isolated molecules, as periodic will */
/* always have zones to make unfragmenting easier */

#if DEBUG_ZONE_INIT
printf("Partioning %d cores, %d shells...\n", g_slist_length(model->cores),
                                              g_slist_length(model->shels));
#endif

/* core assignment loop */
for (list=model->cores ; list ; list=g_slist_next(list))
  {
  core = list->data;

/* zone index */
  zi = zone_index(core->x, model);
  if (zi < 0 || zi >= model->num_zones)
    {
#if DEBUG_ZONE_INIT
printf("[z=%d] [%s]", zi, core->label);
P3VEC(" : ", core->x);
#endif
/* usually caused by very large (corrupted?) core coordinates */
/* can also be caused during surface initialization as we allow */
/* some molecules to be outside the cell in order to avoid loosing */
/* some at the top or bottom slice boundaries - don't think it's a problem */
    mad_core_flag = TRUE;
    zi = 0;
    }

/* add to the list */
  zone = (struct zone_pak *) model->zones[zi];
  zone->cores = g_slist_prepend(zone->cores, core);

/* use region colouring to show zones */
/*
core->region = zi;
printf("grid coords: %d %d %d  ", grid[0], grid[1], grid[2]);
printf("region: %d\n", zi); 
*/
  }

/*
if (mad_core_flag)
  show_text(ERROR, "Your model has caught mad core disease.\n");
*/

/* shell assignment loop */
for (list=model->shels ; list ; list=g_slist_next(list))
  {
  shel = list->data;

/* zone index */
  zi = zone_index(shel->x, model);
  if (zi < 0 || zi >= model->num_zones)
    {
/* mad core disease - usually caused by very large (corrupted?) core coordinates */
    mad_shel_flag = TRUE;
    zi = 0;
    }

/* add to the list */
  zone = (struct zone_pak *) model->zones[zi];
  zone->shells = g_slist_prepend(zone->shells, shel);

/* use region colouring to show zones */
/*
core->region = zi;
printf("grid coords: %d %d %d  ", grid[0], grid[1], grid[2]);
printf("region: %d\n", zi); 
*/
  }

/*
if (mad_shel_flag)
  show_text(ERROR, "Your model has caught mad shell disease.\n");
*/

#if DEBUG_ZONE_INIT
for (i=0 ; i<num_zones ; i++)
  {
  zone = (struct zone_pak *) model->zones[i];
  j = g_slist_length(zone->cores);
  k = g_slist_length(zone->shells);
  if (j || k)
    printf("zone %d (%p): %d cores, %d shells.\n", i, zone, j, k);
  }
#endif
}

/******************/
/* zone debugging */
/******************/
void zone_info(struct model_pak *model)
{
gint i, j, k;
struct zone_pak *zone;

printf("con_info_zones(%s)\n", model->filename);

for (i=0 ; i<3 ; i++)
  printf("[%d] : %11.4f - %11.4f  (%f)(%d)\n",
          i, model->zone_min[i], model->zone_max[i],
          model->zone_dx[i], model->zone_div[i]);

for (i=0 ; i<model->num_zones ; i++)
  {
  zone = (struct zone_pak *) model->zones[i];
  j = g_slist_length(zone->cores);
  k = g_slist_length(zone->shells);
  if (j || k)
    printf("zone %d: %d cores, %d shells.\n", i, j, k);
  }
}

/*********************/
/* extract zone data */
/*********************/
GSList *zone_cores(gpointer ptr_zone)
{
struct zone_pak *zone = ptr_zone;
return(zone->cores);
}
GSList *zone_shells(gpointer ptr_zone)
{
struct zone_pak *zone = ptr_zone;
return(zone->shells);
}
gint *zone_grid(gpointer ptr_zone)
{
struct zone_pak *zone = ptr_zone;
return(zone->grid);
}

/**************************************/
/* get the zone index of the position */
/**************************************/
/* FIXME - if out of bounds -> need a re-zone & then return the index as well */
#define DEBUG_GET_ZI 0
gint zone_index(gdouble *position, struct model_pak *model)
{
gint i, zi, grid[3];
gdouble x[3];

/* pbc constrain */
ARR3SET(x, position);

#if DEBUG_GET_ZI
printf("x1: %.20f %.20f %.20f\n", x[0], x[1], x[2]);
#endif

/* NB: grid is just used as a dummy variable here */
fractional_clamp(x, grid, model->periodic);

#if DEBUG_GET_ZI
printf("x2: %.20f %.20f %.20f\n", x[0], x[1], x[2]);
#endif

/* get the grid index */
for (i=0 ; i<3 ; i++)
  grid[i] = (x[i]-model->zone_min[i])/model->zone_dx[i];

/* NEW - clamp to enforce bounds */
for (i=3 ; i-- ; )
  grid[i] = CLAMP(grid[i], 0, model->zone_div[i]-1); 

#if DEBUG_GET_ZI
printf(" g: %d %d %d\n", grid[0], grid[1], grid[2]);
#endif

/* get the zone index */
zi = grid[2]*model->zone_div[1]*model->zone_div[0]; 
zi += grid[1]*model->zone_div[0]; 
zi += grid[0];

return(zi);
}

/***************************/
/* get the core's zone ptr */
/***************************/
/* TODO - can we eliminate the need for external routines to call this? */
gpointer zone_ptr(struct core_pak *core, struct model_pak *model)
{
gint zi;

if (!model->zones)
  return(NULL);

/* if out of bounds -> re-zoning needed */
zi = zone_index(core->x, model);
if (zi < 0 || zi >= model->num_zones)
  {
  zone_init(model);
  zi = zone_index(core->x, model);
  if (zi < 0 || zi >= model->num_zones)
    {
/* mad core again - print warning? */
    zi = 0;
    }
  }
return(model->zones[zi]);
}

/*****************************/
/* incremental zone updating */
/*****************************/
void zone_append_core(gpointer ptr_zone, struct core_pak *core)
{
struct zone_pak *zone = ptr_zone;

zone->cores = g_slist_prepend(zone->cores, core);
}

void zone_remove_core(gpointer ptr_zone, struct core_pak *core)
{
struct zone_pak *zone = ptr_zone;

zone->cores = g_slist_remove(zone->cores, core);
}

void zone_append_shell(gpointer ptr_zone, struct shel_pak *shell)
{
struct zone_pak *zone = ptr_zone;

zone->shells = g_slist_prepend(zone->shells, shell);
}

void zone_remove_shell(gpointer ptr_zone, struct shel_pak *shell)
{
struct zone_pak *zone = ptr_zone;

zone->shells = g_slist_remove(zone->shells, shell);
}

/**********************************************/
/* construct core list from surrounding zones */
/**********************************************/
#define DEBUG_CORE_LIST 0
GSList *zone_area_cores(gint buffer, gpointer ptr_zone, struct model_pak *model)
{
guint i;
gint a, b, c, zi;
gint g[3], start[3], stop[3];
GSList *list;
struct zone_pak *zone = ptr_zone, *buffer_zone;

g_assert(zone != NULL);
g_assert(model != NULL);
g_assert(buffer >= 0);

/*
printf("zone_area_cores(%p\n", zone);
*/

#if DEBUG_CORE_LIST
printf("[%2d %2d %2d]\n", zone->grid[0], zone->grid[1], zone->grid[2]);
#endif

/* setup scan limits */
ARR3SET(start, zone->grid);
ARR3SET(stop, zone->grid);

/* add buffering zones around the edge */
VEC3SUB(start, buffer, buffer, buffer);
VEC3ADD(stop, buffer, buffer, buffer);

/*
for (i=3 ; i-- ; )
  {
  if (start[i] < 0)
    start[i] = 0;
  if (stop[i] >= model->zone_div[i])
    stop[i] = model->zone_div[i]-1;
  }
*/

/*
#if DEBUG_CORE_LIST
printf("start: %d %d %d\n", start[0], start[1], start[2]);
printf(" stop: %d %d %d\n", stop[0], stop[1], stop[2]);
#endif
*/

g_assert(stop[0] < 1000);
g_assert(stop[1] < 1000);
g_assert(stop[2] < 1000);

g_assert(start[0] > -1000);
g_assert(start[1] > -1000);
g_assert(start[2] > -1000);

/* zone sweep */
list=NULL;
for (c=start[2] ; c<=stop[2] ; c++)
  {
  for (b=start[1] ; b<=stop[1] ; b++)
    {
    for (a=start[0] ; a<=stop[0] ; a++)
      {

/* constrain zone indexed */
      VEC3SET(g, a, b, c);
      for (i=3 ; i-- ; )
        while (g[i] < 0)
          g[i] += model->zone_div[i];
      for (i=3 ; i-- ; )
        while (g[i] >= model->zone_div[i])
          g[i] -= model->zone_div[i];

#if DEBUG_CORE_LIST
printf(" > [%2d %2d %2d]\n", g[0], g[1], g[2]);
#endif

/* convert to a valid zone index */
      zi = g[2]*model->zone_div[1]*model->zone_div[0]; 
      zi += g[1]*model->zone_div[0]; 
      zi += g[0];

/* this is the actual zone + required xlat that gives the current zone */
/*
#if DEBUG_CORE_LIST
printf("[%2d %2d %2d] (%9.4f %9.4f %9.4f) : zone = %2d \n",
        g[0], g[1], g[2], pic[0], pic[1], pic[2], zi);
#endif
*/

g_assert(zi >= 0);
g_assert(zi < model->num_zones);

/* loop over cores in the corresponding zone */
      buffer_zone = model->zones[zi];
      g_assert(buffer_zone != NULL);

/* NB: only add if core clist is non-NULL */
      if (buffer_zone->cores)
        list = g_slist_concat(list, g_slist_copy(buffer_zone->cores));
      }
    }
  }
return(list);
}

