/*  GFC-UI: GTK+ Foundation Classes (User Interface Library)
 *  Copyright (C) 2002-2004 The GFC Development Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
/// @file gfc/pango/types.hh
/// @brief Pango types C++ wrapper interface.
///
/// Provides common typedefs, enumerations and utility classes.

#ifndef GFC_PANGO_TYPES_HH
#define GFC_PANGO_TYPES_HH

#ifndef __PANGO_TYPES_H__
#include <pango/pango-types.h>
#endif

namespace GFC {

namespace G {
class Unichar;
}

namespace Pango {
	
/// Glyph is a convenient typedef for PangoGlyph. Glyph structure represents a single glyph
/// in the output form of a glyph string.

typedef PangoGlyph Glyph;

/// @enum Pango::Direction
/// The Direction type represents the direction of writing for unicode bidirectional text.
///
/// <B>Note:</B> Not every value in this enumeration makes sense for every usage of Pango::Direction.

enum Direction
{
	DIRECTION_LTR = PANGO_DIRECTION_LTR,
	///< The text is written left-to-right.

	DIRECTION_RTL = PANGO_DIRECTION_RTL,
	///< The text is written right-to-left.

	DIRECTION_TTB_LTR = PANGO_DIRECTION_TTB_LTR,
	///< The text is written vertically top-to-bottom, with the rows ordered from left to right.
	
	DIRECTION_TTB_RTL = PANGO_DIRECTION_TTB_RTL,
	///< The text is written vertically top-to-bottom, with the rows ordered from right to left.

	DIRECTION_WEAK_LTR = PANGO_DIRECTION_WEAK_LTR,
	///< A weak left-to-right direction.

	DIRECTION_WEAK_RTL = PANGO_DIRECTION_WEAK_RTL,
	///< A weak right-to-left direction .

	DIRECTION_NEUTRAL = PANGO_DIRECTION_NEUTRAL
	///< No direction specified.
};

/// @name Direction Methods
/// @{

Direction unichar_direction(G::Unichar ch);
///< Determines the direction of a character; either Pango::DIRECTION_LTR, 
///< Pango::DIRECTION_LTR, or Pango::DIRECTION_NEUTRAL. 
///< @param ch The character to examine. 
///< @return The direction of a the character, as used in the Unicode bidirectional algorithm.

Direction find_base_dir(const char *text, int length = -1);
///< Searches a string for the first character that has a strong direction, 
///< according to the Unicode bidirectional algorithm.
///< @param text The text to process. 
///< @param length The length of text in bytes, or -1 if text is null-terminated. 
///< @return The direction corresponding to the first strong character. 
///<
///< <BR>If no such character is found, then Pango::DIRECTION_NEUTRAL is returned.

/// @}

/// @class Rectangle types.hh gfc/pango/types.hh
/// @brief A PangoRectangle C++ wrapper class.
///
/// Rectangle represents a rectangle. It is frequently used to represent the logical or ink extents
/// of a single glyph or section of text (see, for instance, Pango::Font::get_glyph_extents()).

class Rectangle
{
	PangoRectangle rectangle_;

public:
/// @name Constructors
/// @{

	Rectangle();
	///< Construct a with its x, y, width and height set to 0;

	Rectangle(int x, int y, int width, int height);
	///< Construct a new rectangle.
	///< @param x The X coordinate of the rectangle.
	///< @param y The Y coordinate of the rectangle.
	///< @param width The width of the rectangle.
	///< @param height The height of the rectangle.
	
	Rectangle(const PangoRectangle& rectangle);
	///< Construct a new rectangle from an existing PangoRectangle.
	///< @param rectangle A PangoRectangle.

/// @}
/// @name Accessors
/// @{

	PangoRectangle* pango_rectangle() const;
	///< Get a pointer to the PangoRectangle structure.

	operator PangoRectangle* () const;
	///< Conversion operator; safely converts a Rectangle to a PangoRectangle pointer.

	int x() const;
	///< Returns the X coordinate of the top-left corner of the rectangle.

	int y() const;
	///< Returns the Y coordinate of the top-left corner of the rectangle.

	int width() const;
	///< Returns the width of the rectangle.
	
	int height() const;
	///< Returns the height of the rectangle.
	
	int ascent() const;
	///< Extracts the ascent from a Pango::Rectangle representing glyph extents.
	///< The ascent is the distance from the baseline to the highest point of 
	///< the character. This is positive if the glyph ascends above the baseline.

	int descent() const;
	///< Extracts the descent from a Pango::Rectangle representing glyph extents.
	///< The descent is the distance from the baseline to the lowest point of 
	///< the character. This is positive if the glyph descends below the baseline. 

	int lbearing() const;
	///< Extracts the left bearing from a Pango::Rectangle representing glyph extents.
	///< The left bearing is the distance from the horizontal origin to the farthest
	///< left point of the character. This is positive for characters drawn completely
	///< to the right of the glyph origin.

	int rbearing() const;
	///< Extracts the right bearing from a Pango::Rectangle representing glyph extents.
	///< The right bearing is the distance from the horizontal origin to the farthest
	///< right point of the character. This is positive except for characters drawn 
	///< completely to the left of the horizontal origin.

/// @}
/// @name Methods
/// @{

	void set(int x, int y, int width, int height);
	///< Sets the position and size of the rectangle.
	///< @param x The new X coordinate of the rectangle.
	///< @param y The new Y coordinate of the rectangle.
	///< @param width The new width of the rectangle.
	///< @param height The new height of the rectangle.
	
/// @}
};

} // namespace Pango

} // namespace GFC

#include <gfc/pango/inline/types.inl>

#endif // GFC_PANGO_TYPES_HH

