with Types; use Types;
with Iirs; use Iirs;

package Xrefs is
   type Xref_Kind is
     (
      --  Declaration of an identifier.
      Xref_Decl,

      --  Use of a named entity.
      Xref_Ref,

      --  Identifier after the 'end' keyword.
      Xref_End,

      --  Body of a declaration (for package, subprograms or protected type).
      Xref_Body
     );

   --  Initialize the xref table.
   --  Must be called once.
   procedure Init;

   --  Low level xref addition.
   --  An entity at LOC references REF with the KIND way.
   procedure Add_Xref (Loc : Location_Type; Ref : Iir; Kind : Xref_Kind);

   --  Add a declaration of an identifier.
   --  This is somewhat a self-reference.
   procedure Xref_Decl (Decl : Iir);
   pragma Inline (Xref_Decl);

   --  NAME refers to DECL.
   procedure Xref_Ref (Name : Iir; Decl : Iir);
   pragma Inline (Xref_Ref);

   --  BODy refers to SPEC.
   procedure Xref_Body (Bod : Iir; Spec : Iir);
   pragma Inline (Xref_Body);

   --  Just resolved NAME refers to its named entity.
   procedure Xref_Name (Name : Iir);
   pragma Inline (Xref_Name);

   --  LOC is the location of the simple_name after 'end' for DECL.
   procedure Xref_End (Loc : Location_Type; Decl : Iir);
   pragma Inline (Xref_End);

   --  Sort the xref table by location.  This is required before searching with
   --  Find.
   procedure Sort_By_Location;

   --  Sort the xref table by location of the nodes.
   procedure Sort_By_Node_Location;

   subtype Xref is Natural;

   --  A bad xref.
   --  May be returned by Find.
   Bad_Xref : constant Xref := 0;

   --  First xref.
   --  May be used to size a table.
   First_Xref : constant Xref := 1;

   --  Find a reference by location.
   --  The table must already be sorted with Sort_By_Location.
   --  Returns BAD_REF is does not exist.
   function Find (Loc : Location_Type) return Xref;

   --  Convert subprogram_body nodes used by end_xrefs into declaration nodes.
   --  This is used to that every node has a name.
   procedure Fix_End_Xrefs;

   --  Get the last possible xref available.
   --  May be used to size tables.
   function Get_Last_Xref return Xref;

   --  Get the location of N, ie where a name (or operator) appears.
   function Get_Xref_Location (N : Xref) return Location_Type;
   pragma Inline (Get_Xref_Location);

   --  Get the kind of cross-reference.
   function Get_Xref_Kind (N : Xref) return Xref_Kind;
   pragma Inline (Get_Xref_Kind);

   --  Get the node referenced by the name.
   function Get_Xref_Node (N : Xref) return Iir;
   pragma Inline (Get_Xref_Node);
end Xrefs;
