#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <signal.h>
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <string.h>
#include <netinet/in.h>
#include <stdlib.h>

#include "config.h"

#include "world.h"
#include "settings.h"

#define FAILED '1'
#define UNKNOWN_HOST '2'
#define SUCCESS '3'

void do_disconnect(world *w);
void abnormal_disconnect(world *w, int err);

int
waitline (int sok, char *buf, int bufsize)
{
   int i = 0;

   while (1)
   {
      if (read (sok, &buf[i], 1) < 1)
         return -1;
      if (buf[i] == '\n')
      {
         buf[i] = 0;
         return i;  
      }
      i++;
      if (i == (bufsize - 1))
         return 0;
   }
}   

void server_cleanup(server_t *server) {
    if(server->iotag != -1) {
        gdk_input_remove(server->iotag);
        server->iotag = -1;
    }
    close(server->childwrite);
    close(server->childread);
    waitpid(server->childpid, NULL, 0);
    server->connecting = FALSE;
}

int check_connecting(server_t *server) {
    if(server->connecting) {
        kill(server->childpid, SIGKILL);
        server_cleanup(server);
        close(server->socket);
        return TRUE;
    }
    return FALSE;
}

server_t *init_server() {
    server_t *server = g_malloc(sizeof(server_t));
    server->connecting = FALSE;
    server->connected = FALSE;
    server->socket = -1;
    server->iotag = -1;
    server->childpid = -1;
    server->childread = -1;
    server->childwrite = -1;
    return server;
}

void read_data(world *w, int socket) {
    int err, len;
    char lbuf[2050];
 
    len = recv(socket, lbuf, sizeof(lbuf) - 2, 0);
    if(len < 1) {
        if(len < 0) {
            if(errno == EAGAIN || errno == EWOULDBLOCK)
                return;                                  
            err = errno;
        } else {
            err = 0;
        }
        abnormal_disconnect(w, err);
        if(settings->reconnect)
            gm_net_connect(w, w->p->hostname, w->p->port);
        return;
    } else {
        gm_world_handle_output_data(w, lbuf, len);
    }   
}
          


void connecting_data(world *w, int socket) {
    char tbuf[128];
    char *s;

    waitline(socket, tbuf, sizeof(tbuf));
    switch(tbuf[0]) {
    case UNKNOWN_HOST:
        if(debug) printf("NETHANDLER: Got command: UNKNOWN HOST\n");
        server_cleanup(w->server);
        gm_world_println(w, "\x1b[31m% Connecting failed: "
                         "hostname not found\x1b[0m");
        close(socket);
        gm_world_disconnected(w);
        break;
    case FAILED:
        if(debug) printf("NETHANDLER: Got command: FAILED\n");
        waitline(w->server->childread, tbuf, sizeof(tbuf));
        server_cleanup(w->server);
        s = g_strdup_printf("\x1b[31m%% Connecting failed: %s (error %s)\x1b[0m",
                            strerror(atoi(tbuf)), tbuf);
        gm_world_println(w, s);
        g_free(s);
        close(socket);
        gm_world_disconnected(w);
        break;
    case SUCCESS:
        if(debug) printf("NETHANDLER: Got command: SUCCESS\n");
        server_cleanup(w->server);
        w->server->connected = TRUE;
        gm_world_println(w, "\x1b[32m% Connected\x1b[0m");
        gm_world_connected(w);
        w->server->iotag = gdk_input_add(w->server->socket,
                                         GDK_INPUT_READ|GDK_INPUT_EXCEPTION,
                                         (GdkInputFunction) read_data, w);
    }
}

void gm_net_connect(world *w, const char *hostname, int port) {
    int sok, sw, pid, read_des[2];
    char *s;
    struct hostent *HostAddr;
    struct sockaddr_in SAddr;
    
    if(debug) printf("NET: Connecting...\n");
    if(!hostname || !hostname[0] || port < 1)
        return;

    if(w->server) {
        if(debug) printf("NET: Disconnecting current\n");
        do_disconnect(w);
    } else {
        if(debug) printf("NET: Creating new server\n");
        w->server = init_server();
    }
    
    if(debug) printf("NET: Opening socket\n");
    sok = socket(AF_INET, SOCK_STREAM, 0);
    if(sok == -1)
        return;

    s = g_strdup_printf("\x1b[32m%% Connecting to %s:%d...\x1b[0m",
                        hostname, port);
    gm_world_println(w, s);
    g_free(s);

    sw = 1;
    setsockopt(sok, SOL_SOCKET, SO_REUSEADDR, (char *) &sw, sizeof (sw));
    sw = 1;
    setsockopt(sok, SOL_SOCKET, SO_KEEPALIVE, (char *) &sw, sizeof (sw));

    w->server->connecting = TRUE;
    w->server->socket = sok;
    
    if(debug) printf("NET: Creating pipe\n");
    if(pipe(read_des) < 0)
        return;

    gm_world_connecting(w);
    
    if(debug) printf("NET: Forking\n");
    switch(pid = fork()) {
    case -1:
        return;
    case 0:
        dup2(read_des[1], 1);  /* make the pipe our stdout */
        setuid(getuid());

        if((HostAddr = gethostbyname(hostname))) {
            memset(&SAddr, 0, sizeof(SAddr));
            SAddr.sin_port = htons(port);
            SAddr.sin_family = AF_INET;   
            memcpy((void *)&SAddr.sin_addr, HostAddr->h_addr, HostAddr->h_length);

            if(connect(sok, (struct sockaddr *) &SAddr, sizeof (SAddr)) < 0)
                printf("%c\n%d\n", FAILED, errno);
            else
                printf ("%c\n", SUCCESS);
        } else {
            printf ("%c\n", UNKNOWN_HOST);
        }
        fflush(stdout);
        _exit(0);
    }

    if(debug) printf("NET: Forked, connecting handler\n");
    w->server->childpid = pid;
    w->server->iotag = gdk_input_add(read_des[0], GDK_INPUT_READ,
                                     (GdkInputFunction) connecting_data, w);
    w->server->childread = read_des[0];
    w->server->childwrite = read_des[1];
    if(debug) printf("NET: Done (iotag == %d)\n", w->server->iotag);
}

void do_disconnect(world *w) {
    if(!w->server)
        return;
    if(check_connecting(w->server))
        return;
    if(w->server->connected) {
        if(w->server->iotag != -1) {
            gdk_input_remove(w->server->iotag);
            w->server->iotag = -1;
        }
        close(w->server->socket);
        w->server->connected = FALSE;
    }
}

void gm_net_disconnect(world *w) {
    do_disconnect(w);
    gm_world_println(w, "\x1b[32m% Disconnected\x1b[0m");
    gm_world_disconnected(w);
}

void abnormal_disconnect(world *w, int err) {
    char *s;
    do_disconnect(w);
    if(err) {
        s = g_strdup_printf("\x1b[31m%% Connection lost: %s (error %d)\x1b[0m",
                         strerror(err), err);
        gm_world_println(w, s);
        g_free(s);
    } else {
        gm_world_println(w, "\x1b[31m% Connection lost\x1b[0m");
    }
    gm_world_disconnected(w);
}


void gm_net_send(world *w, const char *text, int length) {
    if(w->server) {
        if(w->server->connected) {
            send(w->server->socket, text, length, 0);
            return;
        }
        if(w->server->connecting) {
            gm_world_println(w, "% Still connecting ...");
            return;
        }
    }
    gm_world_println(w, "\x1b[31m% Not connected !\x1b[0m");
}

