/******************************************************************************\
 gnofin/history.c   $Revision: 1.5 $
 Copyright (C) 1999-2000 Darin Fisher

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\******************************************************************************/

//#define ENABLE_DEBUG_TRACE

#include "common.h"
#include <stdio.h>
#include "history.h"

typedef struct {
  HistorySignals *signals;
  gpointer *data;
} HistoryItem;

struct _HistoryBatchItem {
  GList *head;
  GList *tail;
};

#define FIRE_CALLBACK(pdat,item,func) \
  G_STMT_START { \
    if ((item) && (item)->signals->func) \
      (item)->signals->func (pdat, (item)->data); \
  } G_STMT_END

/******************************************************************************/

static void
batch_mode_undo (gpointer pdat, HistoryBatchItem *batch)
{
  GList *it;

  trace ("");

  for (it=batch->tail; it; it=it->prev)
  {
    HistoryItem *item = LIST_DEREF (HistoryItem, it);
    FIRE_CALLBACK (pdat, item, undo_item);
  }
}

static void
batch_mode_redo (gpointer pdat, HistoryBatchItem *batch)
{
  GList *it;

  trace ("");

  for (it=batch->head; it; it=it->next)
  {
    HistoryItem *item = LIST_DEREF (HistoryItem, it);
    FIRE_CALLBACK (pdat, item, redo_item);
  }
}

static void
batch_mode_free (gpointer pdat, HistoryBatchItem *batch)
{
  GList *it;
  
  trace ("");

  for (it=batch->head; it; it=it->next)
  {
    HistoryItem *item = LIST_DEREF (HistoryItem, it);
    FIRE_CALLBACK (pdat, item, free_item);
  }
  g_list_free (batch->head);
  g_free (batch);
}

static void
batch_mode_dump (gpointer pdat, HistoryBatchItem *batch)
{
  g_print ("    I - %p {BATCH_MODE, %d items}\n", batch, g_list_length (batch->head));
}

static HistorySignals batch_mode_signals = 
{
  (HistoryFunc) batch_mode_undo,
  (HistoryFunc) batch_mode_redo,
  (HistoryFunc) batch_mode_free,
  (HistoryFunc) batch_mode_dump
};

/******************************************************************************/

History *
history_new (gpointer parent_data)
{
  History *history;
  
  trace ("");
  
  history = g_new0 (History, 1);
  history_init (history, parent_data);

  return history;
}

void
history_init (History *history, gpointer parent_data)
{
  trace ("");
  g_return_if_fail (history);

  history->parent_data = parent_data;
}

void
history_undo (History *history)
{
  trace ("");
  
  g_return_if_fail (history);

  if (history->can_undo)
  {
    HistoryItem *item;

    /* save a pointer to the current history item */
    item = LIST_DEREF (HistoryItem, history->current);

    /* make the previous history item current */
    history->current = history->current->prev;
    history->can_undo = (history->current != NULL);
    history->can_redo = 1;

    /* fire the undo callback, if it exists */
    FIRE_CALLBACK (history->parent_data, item, undo_item);
  }
}

void
history_redo (History *history)
{
  trace ("");
  
  g_return_if_fail (history);

  if (history->can_redo)
  {
    HistoryItem *item;

    /* it doesn't make sense to redo if there aren't any remembered items */
    g_assert (history->items);

    /* if we can't undo (history->current == NULL), then we have to just 
     * pick the first remembered item to redo.
     * we also must adjust the current undo item accordingly. */
    if (history->current)
    {
      item = LIST_DEREF (HistoryItem, history->current->next);
      history->current = history->current->next;
    }
    else
    {
      item = LIST_DEREF (HistoryItem, history->items);
      history->current = history->items;
    }

    history->can_redo = (history->current->next != NULL);
    history->can_undo = 1;

    /* fire the redo callback, if it exists */
    FIRE_CALLBACK (history->parent_data, item, redo_item);
  }
}

void
history_remember (History *history, HistorySignals *signals, gpointer data)
{
  GList *it, *head;
  HistoryItem *item;

  trace ("");
  
  g_return_if_fail (history);

  /* free all above current */
  if (history->current)
  {
    head = history->current->next;
    history->current->next = NULL;
  }
  else
  {
    head = history->items;
    history->items = NULL;
  }
  if (head)
  {
    for (it=head; it; it=it->next)
    {
      item = LIST_DEREF (HistoryItem, it);

      FIRE_CALLBACK (history->parent_data, item, free_item);

      /* precautionary */
      it->data = NULL;
    }
    g_list_free (head);
  }

  if (history->current)
    g_assert (history->current->next == NULL);

  /* Create the new item structure */
  item = g_new0 (HistoryItem, 1);
  item->data = data;
  item->signals = signals;

  if (history->batch_mode)
  {
    g_assert (history->batch_item != NULL);

    /* append to batch */
    history->batch_item->tail = g_list_append (history->batch_item->tail, item);

    if (history->batch_item->head == NULL)
      history->batch_item->head = history->batch_item->tail;
    else
      history->batch_item->tail = g_list_last (history->batch_item->tail);
  }
  else
  {
    /* append to history -- could be more efficient */
    history->items = g_list_first (g_list_append (history->current, item));
    g_assert (history->items->prev == NULL);
    history->current = g_list_last (history->items);

    /* set flags */
    history->can_redo = 0;
    history->can_undo = 1;
  }
}

void
history_destroy (History *history)
{
  trace ("");

  history_clear (history);
  g_free (history);
}

void
history_clear (History *history)
{
  HistoryItem *item;
  GList *it;

  trace ("");
  g_return_if_fail (history);

  for (it=history->items; it; it=it->next)
  {
    item = LIST_DEREF (HistoryItem, it);

    FIRE_CALLBACK (history->parent_data, item, free_item);
    g_free (item);
  }
  g_list_free (history->items);

  history->items = NULL;
  history->current = NULL;
  history->can_undo = 0;
  history->can_redo = 0;
}

void
history_dump (History *history)
{
  GList *it, *head;

  g_print ("  H - %p {len=%u,cu=%u,cr=%u}\n",
           history,
	   g_list_length (history->items),
	   history->can_undo,
	   history->can_redo);
 
  g_print ("   undo items:\n");
  head = history->current;
  for (it=head; it; it=it->prev)
  {
    HistoryItem *item = LIST_DEREF (HistoryItem, it);
    FIRE_CALLBACK (history->parent_data, item, dump_item);
  }

  g_print ("   redo items:\n");
  head = history->current ? history->current->next : history->items;
  for (it=head; it; it=it->next)
  {
    HistoryItem *item = LIST_DEREF (HistoryItem, it);
    FIRE_CALLBACK (history->parent_data, item, dump_item);
  }
}

void
history_enter_batch_mode (History *history)
{
  trace ("");
  g_return_if_fail (history);

  if (++history->batch_mode == 1)
  {
    g_assert (history->batch_item == NULL);

    history->batch_item = g_new0 (HistoryBatchItem, 1);
  }
}

void
history_leave_batch_mode (History *history)
{
  trace ("");
  g_return_if_fail (history);

  if (--history->batch_mode == 0)
  {
    g_assert (history->batch_item != NULL);

    history_remember (history, &batch_mode_signals, history->batch_item);
    history->batch_item = NULL;
  }
}

// vim: ts=8 sw=2
