/******************************************************************************\
 gnofin/xml-io.c   $Revision: 1.14 $
 Copyright (C) 1999-2000 Darin Fisher

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\******************************************************************************/

/*
 * XML IO: Read/write bankbook as XML using libxml
 *
 * Author:
 *   Darin Fisher (dfisher@jagger.me.berkeley.edu)
 */

//#define ENABLE_DEBUG_TRACE

#include "common.h"
#include <gnome-xml/tree.h>
#include <gnome-xml/parser.h>
#include <libgnome/gnome-config.h>
#include <stdlib.h>
#include <string.h>
#include "data-if.h"
#include "xml-io.h"
#include "dialogs.h"
#include "config-saver.h"
#include "numeric-parser.h"
#include "gnofin-defaults.h"


#define NS_HREF "http://gnofin.sourceforge.net/"


/******************************************************************************
 * Configuration
 */

typedef struct {
  gint compression_level;
} Config;

#define CAT  "XML"
#define KEY  "/" PACKAGE "/" CAT "/"

static void
load_config (Config *config)
{
  trace ("");
  config->compression_level = gnome_config_get_int (KEY "compression_level=9");
}

static void
save_config (const Config *config)
{
  trace ("");
  gnome_config_set_int (KEY "compression_level", config->compression_level);
}

static Config *
get_config (void)
{
  static Config config = {0};
  static gboolean init = FALSE;

  if (!init)
  {
    load_config (&config);
    config_saver_register (CAT, (ConfigSaveFunc) save_config, &config);
    init = TRUE;
  }
  return &config;
}


/****************************************************************************** 
 * File version
 */

/* if the major version is incremented it means that the file format has been
 * fundamentally changed.  such that older version of xml-io would be unable
 * to read those files generated by the later version.  the minor version is
 * incremented whenever a change takes place, but only for changes that do not
 * effect backwards compatibility of the file format. */

/* revision history:
 *
 *   1.0   - initial xml file format
 *   1.1   - added io extensions (8-feb-00)
 *   1.2   - foreign tag (9-feb-00)
 *   1.3   - ???
 *   1.4   - added category support (18-feb-00)
 *   2.0   - reorganized with gnofin_data as the root tag
 *   2.1   - support for 64-bit money_t value (13-mar-00)
 */

#define XML_IO_VERSION         "2.1"
#define XML_IO_MAJOR_VERSION    2
#define XML_IO_MINOR_VERSION    1

static gboolean
parse_version (const gchar *ver, guint *major, guint *minor)
{
  guint maj, min;
  gchar buf[32], *c;

  trace ("");
  g_return_val_if_fail (ver, FALSE);

  if (!strncpy (buf, ver, sizeof buf))
    return FALSE;

  c = strchr (buf, '.');
  if (!c)
    return FALSE;
  *c = '\0';
  c++;

  if (!uint_parse (buf, &maj))
    return FALSE;
  if (!uint_parse (c, &min))
    return FALSE;

  if (major) *major = maj;
  if (minor) *minor = min;
  return TRUE;
}


/****************************************************************************** 
 * Error types
 */

#define fail(errorcode) \
  G_STMT_START { \
    error=(errorcode); \
    goto fail_; \
  } G_STMT_END

enum {
  ERROR_INVALID_FORMAT = 1,
  ERROR_NOT_XML,
  ERROR_WRONG_NS,
  ERROR_TOO_NEW,
  ERROR_UNKNOWN
};

static const gchar *
xml_io_strerror (guint err)
{
  switch (err)
  {
  case ERROR_INVALID_FORMAT:
    return _("Invalid file format");
  case ERROR_NOT_XML:
    return _("Not a valid XML (version 1.0) file");
  case ERROR_WRONG_NS:
    return _("XML namespace does not match");
  case ERROR_TOO_NEW:
    return _("File is from a more recent version of Gnofin with incompatible structure");
  case ERROR_UNKNOWN:
  default:
    return _("Unknown");
  }
}


/******************************************************************************
 * XML io extensions
 */

static GSList *extensions = NULL;

static gint
match_extension_name (const XmlIOExtension *ext, const gchar *name)
{
  return g_strcasecmp (ext->name, name);
}

static inline XmlIOExtension *
lookup_extension (const gchar *name)
{
  trace ("");
  return (XmlIOExtension *) g_slist_find_custom (extensions, (gpointer) name,
					        (GCompareFunc) match_extension_name);
}

static inline gboolean
read_extension (GtkWindow *win, xmlNodePtr node)
{
  XmlIOExtension *ext;

  trace ("%s", node ? node->name : NULL);
  g_return_val_if_fail (node, FALSE);

  ext = lookup_extension (node->name);
  if (ext == NULL)
    return FALSE;
  else
    return ext->read (win, node, ext->private_data);
}

static xmlNodePtr
write_extensions (GtkWindow *win, xmlDocPtr doc, xmlNsPtr ns)
{
  xmlNodePtr node;
  xmlNodePtr child;
  XmlIOExtension *ext;
  GSList *it;

  trace ("");
  g_return_val_if_fail (doc, NULL);
  g_return_val_if_fail (ns, NULL);
  g_return_val_if_fail (extensions, NULL);

  node = xmlNewDocNode (doc, ns, "extensions", NULL);
  if (!node)
    return NULL;

  for (it=extensions; it; it=it->next)
  {
    ext = LIST_DEREF (XmlIOExtension, it);

    child = ext->write (win, doc, ns, ext->private_data);
    if (!child)
    {
      xmlFreeNode (node);
      return NULL;
    }
    xmlAddChild (node, child);
  }
  return node;
}


/****************************************************************************** 
 * Probing
 */

gboolean
xml_io_probe (const gchar *filename)
{
  xmlDocPtr doc;
  xmlNsPtr fin;

  trace ("");
  g_return_val_if_fail (filename, FALSE);

  doc = xmlParseFile (filename);
  if (!doc)
    return FALSE;
  
  if (!doc->root)
    goto fail_;

  fin = xmlSearchNsByHref (doc, doc->root, NS_HREF);
  if (!fin)
    goto fail_;

  if (!doc->root->name)
    goto fail_;
  if (g_strcasecmp (doc->root->name, "bankbook") &&
      g_strcasecmp (doc->root->name, "gnofin_data"))
    goto fail_;

  xmlFreeDoc (doc);
  return TRUE;

fail_:
  xmlFreeDoc (doc);
  return FALSE;
}


/****************************************************************************** 
 * Opening
 */

static xmlNodePtr
get_next_node (xmlNodePtr node)
{
  /* Skip to next element node (ie. ignore comments) */
  while (node && (node->type != XML_ELEMENT_NODE))
    node = node->next;
  return node;
}

static xmlNodePtr
get_next_node_of_type (xmlNodePtr node, xmlElementType type)
{
  /* Skip to next element node of specified type */
  while (node && (node->type != type))
    node = node->next;
  return node;
}

static gboolean
read_record_type (xmlNodePtr node, Bankbook * book)
{
  RecordTypeInfo typ = {0};
  xmlAttrPtr attr;
  gboolean result;

  trace ("");
  g_return_val_if_fail (node, FALSE);
  g_return_val_if_fail (book, FALSE);
  g_return_val_if_fail (node->name, FALSE);
  g_return_val_if_fail (g_strcasecmp (node->name, "record_type") == 0, FALSE);

  attr = node->properties;
  while (attr)
  {
    if (attr->val)
    {
      if (g_strcasecmp (attr->name, "name") == 0)
      {
        if (!typ.name && attr->val->content)
	  typ.name = g_strdup (attr->val->content);
      }
      else if (g_strcasecmp (attr->name, "description") == 0)
      {
        if (!typ.description && attr->val->content)
	  typ.description = g_strdup (attr->val->content);
      }
      else if (g_strcasecmp (attr->name, "numbered") == 0)
      {
        if (attr->val->content && (attr->val->content[0] == '1'))
	  typ.numbered = 1;
      }
      else if (g_strcasecmp (attr->name, "linked") == 0)
      {
        if (attr->val->content && (attr->val->content[0] == '1'))
	  typ.linked = 1;
      }
      else if (g_strcasecmp (attr->name, "sign") == 0)
      {
        if (attr->val->content && (attr->val->content[0] == '+'))
	  typ.sign = RECORD_TYPE_SIGN_POS;
        else if (attr->val->content && (attr->val->content[0] == '-'))
	  typ.sign = RECORD_TYPE_SIGN_NEG;
      }
    }
    attr = attr->next;
  }

  if (typ.name == NULL)
  {
    trace ("record_type::name field not set -- fatal");
    result = FALSE;
  }
  else
    result = (if_bankbook_insert_record_type (book, &typ) != NULL);

  g_free (typ.name);
  g_free (typ.description);

  return result;
}

static gboolean
read_record_types (xmlNodePtr root, Bankbook *book)
{
  xmlNodePtr node;

  trace ("");
  g_return_val_if_fail (root, FALSE);
  g_return_val_if_fail (book, FALSE);
  g_return_val_if_fail (root->name, FALSE);
  g_return_val_if_fail (g_strcasecmp (root->name, "record_types") == 0, FALSE);

  node = get_next_node (root->childs);
  if (g_strcasecmp (node->name, "record_type") == 0)
  {
    while (node)
    {
      if (!read_record_type (node, book))
        return FALSE;
      node = get_next_node (node->next);
    }
    return TRUE;
  }
  else
  {
    trace ("did not get \"record_type\" tag");
    return FALSE;
  }
}

static gboolean
read_record (xmlNodePtr node, Bankbook *book, Account *account)
{
  RecordInfo rec = {0};
  xmlAttrPtr attr;
  gboolean result = TRUE;
  GDateDay   day   = 0;
  GDateMonth month = G_DATE_BAD_MONTH;
  GDateYear  year  = 0;
  gboolean link_broken = FALSE;

  trace ("");
  g_return_val_if_fail (node, FALSE);
  g_return_val_if_fail (book, FALSE);
  g_return_val_if_fail (node->name, FALSE);
  g_return_val_if_fail (g_strcasecmp (node->name, "record") == 0, FALSE);

  attr = node->properties;
  while (attr)
  {
    if (attr->val)
    {
      if (g_strcasecmp (attr->name, "day") == 0)
      {
        if (attr->val->content)
	  day = (GDateDay) atoi (attr->val->content);
      }
      else if (g_strcasecmp (attr->name, "month") == 0)
      {
        if (attr->val->content)
	  month = (GDateMonth) atoi (attr->val->content);
      }
      else if (g_strcasecmp (attr->name, "year") == 0)
      {
        if (attr->val->content)
	  year = (GDateYear) atoi (attr->val->content);
      }
      else if (g_strcasecmp (attr->name, "type") == 0)
      {
        if (attr->val->content)
	  rec.type = if_bankbook_get_record_type_by_name (book, attr->val->content);
      }
      else if (g_strcasecmp (attr->name, "number") == 0)
      {
        if (attr->val->content)
	  rec.number = (guint) atoi (attr->val->content);
      }
      else if (g_strcasecmp (attr->name, "link_broken") == 0)
      {
        if (attr->val->content)
	{
	  guint n;
	  uint_parse (attr->val->content, &n);
	  link_broken = (n != 0);
        }
      }
      else if (g_strcasecmp (attr->name, "linked_account") == 0)
      {
        if (!rec.linked_acc_name && attr->val->content)
	  rec.linked_acc_name = g_strdup (attr->val->content);
      }
      else if (g_strcasecmp (attr->name, "category") == 0)
      {
        if (!rec.category && attr->val->content)
          rec.category = g_strdup (attr->val->content);
      }
      else if (g_strcasecmp (attr->name, "payee") == 0)
      {
        if (!rec.payee && attr->val->content)
	  rec.payee = g_strdup (attr->val->content);
      }
      else if (g_strcasecmp (attr->name, "memo") == 0)
      {
        if (!rec.memo && attr->val->content)
	  rec.memo = g_strdup (attr->val->content);
      }
      else if (g_strcasecmp (attr->name, "amount") == 0)
      {
        if (attr->val->content)
	  money_parse_raw (attr->val->content, &rec.amount);
      }
      else if (g_strcasecmp (attr->name, "cleared") == 0)
      {
        if (attr->val->content)
	{
	  guint n;
	  uint_parse (attr->val->content, &n);
	  rec.cleared = (n != 0);
        }
      }
      else if (g_strcasecmp (attr->name, "exchange_rate") == 0)
      {
        if (attr->val->content)
	  float_parse (attr->val->content, &rec.exchange_rate);
      }
    }
    attr = attr->next;
  }

  /* If linked but link not broken verify that linked account exists.
   * If not, then skip to end */
  if (!link_broken && rec.linked_acc_name)
  {
    Account *acc = if_bankbook_get_account_by_name (book, rec.linked_acc_name);
    if (acc == NULL)
      goto end;
  }

  /* Set date */
  if (!g_date_valid_dmy (day, month, year))
    goto end;
  g_date_clear (&rec.date, 1);
  g_date_set_dmy (&rec.date, day, month, year);

  /* Older file versions didn't encode an exchange rate */
  if (rec.exchange_rate == 0.0)
    rec.exchange_rate = 1.0;

  result = (if_account_insert_record (account, &rec) != NULL);

end:
  g_free (rec.category);
  g_free (rec.payee);
  g_free (rec.memo);

  return result;
}

static gboolean
read_records (xmlNodePtr root, Bankbook * book, Account * account)
{
  xmlNodePtr node;

  trace ("");
  g_return_val_if_fail (root, FALSE);
  g_return_val_if_fail (book, FALSE);
  g_return_val_if_fail (root->name, FALSE);
  g_return_val_if_fail (g_strcasecmp (root->name, "records") == 0, FALSE);

  node = get_next_node (root->childs);
  if (node && (g_strcasecmp (node->name, "record") == 0))
  {
    while (node)
    {
      if (!read_record (node, book, account))
        return FALSE;
      node = get_next_node (node->next);
    }
  }
  return TRUE;
}

static gboolean
read_account (xmlNodePtr node, Bankbook *book)
{
  AccountInfo acc = {0};
  Account *account = NULL;
  xmlAttrPtr attr;
  gboolean result = TRUE;

  trace ("");
  g_return_val_if_fail (node, FALSE);
  g_return_val_if_fail (book, FALSE);
  g_return_val_if_fail (node->name, FALSE);
  g_return_val_if_fail (g_strcasecmp (node->name, "account") == 0, FALSE);

  attr = node->properties;
  while (attr)
  {
    if (attr->val)
    {
      if (g_strcasecmp (attr->name, "name") == 0)
      {
        if (!acc.name && attr->val->content)
	  acc.name = g_strdup (attr->val->content);
      }
      else if (g_strcasecmp (attr->name, "foreign") == 0)
      {
        if (attr->val->content)
	  acc.foreign = (attr->val->content[0] == '1');
      }
    }
    attr = attr->next;
  }

  if (acc.name == NULL)
  {
    trace ("account::name field not set -- fatal");
    result = FALSE;
  }
  else 
  {
    node = get_next_node (node->childs);
    if (node && (g_strcasecmp (node->name, "notes") == 0))
    {
      xmlNodePtr notes = get_next_node_of_type (node->childs, XML_TEXT_NODE);

      if (notes && notes->content)
        acc.notes = g_strdup (notes->content);
    }
    if (acc.notes == NULL)
      trace ("account::notes field was empty");

    /* Now we can insert the account */
    if ((account = if_bankbook_insert_account (book, &acc)) == NULL)
      result = FALSE;
    else
    {
      node = get_next_node (node->next);
      if (node && (g_strcasecmp (node->name, "records") == 0))
        result = read_records (node, book, account);
      else
	trace ("account::records field was empty");
    }
  }  

  g_free (acc.name);
  g_free (acc.notes);

  return result;
}

static gboolean
read_accounts (xmlNodePtr root, Bankbook *book)
{
  xmlNodePtr node;

  trace ("");
  g_return_val_if_fail (root, FALSE);
  g_return_val_if_fail (book, FALSE);
  g_return_val_if_fail (root->name, FALSE);
  g_return_val_if_fail (g_strcasecmp (root->name, "accounts") == 0, FALSE);

  node = get_next_node (root->childs);
  if (node && (g_strcasecmp (node->name, "account") == 0))
  {
    while (node)
    {
      if (!read_account (node, book))
        return FALSE;
      node = get_next_node (node->next);
    }
  }
  return TRUE;
}

static gboolean
read_category (xmlNodePtr node, Bankbook *book)
{
  xmlAttrPtr attr;
  gchar *name = NULL;
  
  trace ("");
  g_return_val_if_fail (node, FALSE);
  g_return_val_if_fail (book, FALSE);
  g_return_val_if_fail (node->name, FALSE);
  g_return_val_if_fail (g_strcasecmp (node->name, "category") == 0, FALSE);

  attr = node->properties;
  while (attr)
  {
    if (attr->val)
    {
      if (g_strcasecmp (attr->name, "name") == 0)
      {
        if (!name && attr->val->content)
	{
	  name = attr->val->content;
	  if_bankbook_insert_category (book, name);
	}
      }
    }
    attr = attr->next;
  }
  return TRUE;
}

static gboolean
read_categories (xmlNodePtr root, Bankbook *book)
{
  xmlNodePtr node;

  trace ("");
  g_return_val_if_fail (root, FALSE);
  g_return_val_if_fail (book, FALSE);
  g_return_val_if_fail (root->name, FALSE);
  g_return_val_if_fail (g_strcasecmp (root->name, "categories") == 0, FALSE);

  node = get_next_node (root->childs);
  while (node)
  {
    if (g_strcasecmp (node->name, "category") == 0)
    {
      if (!read_category (node, book))
        return FALSE;
    }
    node = get_next_node (node->next);
  }
  return TRUE;
}

static gboolean
read_bankbook (xmlNodePtr root, Bankbook *book)
{
  xmlNodePtr node;

  trace ("");
  g_return_val_if_fail (root, FALSE);
  g_return_val_if_fail (book, FALSE);
  g_return_val_if_fail (root->name, FALSE);
  g_return_val_if_fail (g_strcasecmp (root->name, "bankbook") == 0, FALSE);

  node = get_next_node (root->childs);
  if (node && (g_strcasecmp (node->name, "record_types") == 0))
  {
    if (!read_record_types (node, book))
      return FALSE;
  }

  node = get_next_node (node->next);
  if (node && (g_strcasecmp (node->name, "accounts") == 0))
  {
    if (!read_accounts (node, book))
      return FALSE;
  }

  node = get_next_node (node->next);
  if (node && (g_strcasecmp (node->name, "categories") == 0))
  {
    if (!read_categories (node, book))
      return FALSE;
  }

  return TRUE;
}

static gboolean
read_gnofin_data (GtkWindow *win, xmlNodePtr root, Bankbook *book)
{
  xmlNodePtr node;

  trace ("");

  node = get_next_node (root->childs);
  if (node && (g_strcasecmp (node->name, "bankbook") == 0))
  {
    if (!read_bankbook (node, book))
      return FALSE;
  }

  node = get_next_node (node->next);
  if (node && (g_strcasecmp (node->name, "extensions") == 0))
  {
    for (node = node->childs; node; node = get_next_node (node->next))
      read_extension (win, node);
  }

  return TRUE;
}

gboolean
xml_io_load (GtkWindow *win, const gchar *filename, Bankbook *book)
{
  xmlDocPtr doc;
  xmlNodePtr root;
  xmlNsPtr ns;
  xmlAttrPtr attr;
  guint major = (guint) -1;
  guint error = 0;

  trace ("");
  g_return_val_if_fail (filename, FALSE);
  g_return_val_if_fail (book, FALSE);

  /* Load the file */
  doc = xmlParseFile (filename);
  if (!doc)
    fail (ERROR_NOT_XML);
  if (!doc->root)
    fail (ERROR_INVALID_FORMAT);

  /* Check namespace */
  ns = xmlSearchNsByHref (doc, doc->root, NS_HREF);
  if (!ns)
    fail (ERROR_WRONG_NS);

  /* Verify name of top node */
  root = get_next_node (doc->root);
  if (!root || !root->name || (g_strcasecmp (root->name, "bankbook") &&    // v1
  			       g_strcasecmp (root->name, "gnofin_data")))  // v2
    fail (ERROR_INVALID_FORMAT);

  /* Search attributes for a version property.  Gnofin 0.7.1 did not write
   * a version property.  So, if not defined, we assume a file version of 1.0
   * We verify only that the major version number is valid.  We should still
   * be able to read the file despite a difference b/w minor version numbers. */
  attr = root->properties;
  while (attr)
  {
    if (attr->val)
    {
      if (g_strcasecmp (attr->name, "version") == 0)
	parse_version (attr->val->content, &major, NULL);
    }
    attr = attr->next;
  }
  if (major == (guint) -1)
    major = 1;
  if (major > XML_IO_MAJOR_VERSION)
    fail (ERROR_TOO_NEW);

  /* Read bankbook tree */
  if (major == 1 && !read_bankbook (root, book))
    fail (ERROR_INVALID_FORMAT);
  else
  if (major == 2 && !read_gnofin_data (win, root, book))
    fail (ERROR_INVALID_FORMAT);

  if (major == 1)
    install_default_categories (book); /* Users upgrading will want these */

  xmlFreeDoc (doc);
  return TRUE;

fail_:
  if (doc)
    xmlFreeDoc (doc);
  if (error)
    dialog_error (win, _("Error loading file: %s\n[%s]"),
		  filename, xml_io_strerror (error));
  return FALSE;
}


/****************************************************************************** 
 * Saving
 */

static xmlNodePtr
write_record_type (xmlDocPtr doc, xmlNsPtr ns, const RecordType *record_type)
{
  xmlNodePtr node;
  RecordTypeInfo typ = {0};

  node = xmlNewDocNode (doc, ns, "record_type", NULL);

  if_record_type_get_info (record_type, 0, &typ);

  xmlNewProp (node, "name", typ.name);
  xmlNewProp (node, "description", typ.description);

  if (typ.numbered)
    xmlNewProp (node, "numbered", "1");
  else
    xmlNewProp (node, "numbered", "0");

  if (typ.linked)
    xmlNewProp (node, "linked", "1");
  else
    xmlNewProp (node, "linked", "0");

  switch (typ.sign)
  {
  case RECORD_TYPE_SIGN_POS:
    xmlNewProp (node, "sign", "+");
    break;
  case RECORD_TYPE_SIGN_NEG:
    xmlNewProp (node, "sign", "-");
    break;
  case RECORD_TYPE_SIGN_ANY:
    xmlNewProp (node, "sign", "");
    break;
  }

  return node;
}

static xmlNodePtr
write_record_types (xmlDocPtr doc, xmlNsPtr ns, const GList *record_types)
{
  xmlNodePtr node;
  xmlNodePtr child;

  trace ("");
  g_return_val_if_fail (doc, NULL);
  g_return_val_if_fail (ns, NULL);
  g_return_val_if_fail (record_types, NULL);

  node = xmlNewDocNode (doc, ns, "record_types", NULL);

  while (record_types)
  {
    child = write_record_type (doc, ns, LIST_DEREF (const RecordType, record_types));
    if (!child)
    {
      xmlFreeNode (node);
      return NULL;
    }
    xmlAddChild (node, child);
    record_types = record_types->next;
  }
  return node;
}

static xmlNodePtr
write_record (xmlDocPtr doc, xmlNsPtr ns, const Record *record)
{
  xmlNodePtr node;
  RecordInfo rec;
  gchar buf[256];

  trace ("");
  g_return_val_if_fail (doc, NULL);
  g_return_val_if_fail (ns, NULL);
  g_return_val_if_fail (record, NULL);

  if_record_get_info (record, 0, &rec);

  node = xmlNewDocNode (doc, ns, "record", NULL);

  /* Write date */
  g_snprintf (buf, sizeof buf, "%d", rec.date.day);
  xmlNewProp (node, "day", buf);

  g_snprintf (buf, sizeof buf, "%d", rec.date.month);
  xmlNewProp (node, "month", buf);

  g_snprintf (buf, sizeof buf, "%d", rec.date.year);
  xmlNewProp (node, "year", buf);

  /* Write type */
  xmlNewProp (node, "type", if_record_type_get_name (rec.type));

  /* Write link */
  if (rec.linked_acc_name)
  {
    /* The link_broken tag must be written first */
    xmlNewProp (node, "link_broken", rec.link_broken ? "1" : "0");
    xmlNewProp (node, "linked_account", rec.linked_acc_name);
  }
  
  /* Write number */
  g_snprintf (buf, sizeof buf, "%d", rec.number);
  xmlNewProp (node, "number", buf);

  /* Write strings */
  xmlNewProp (node, "category", rec.category);
  xmlNewProp (node, "payee", rec.payee);
  xmlNewProp (node, "memo", rec.memo);

  /* Write amount */
  money_stringize_raw (buf, sizeof buf, rec.amount);
  xmlNewProp (node, "amount", buf);

  /* Write cleared flag */
  if (rec.cleared)
    xmlNewProp (node, "cleared", "1");
  else
    xmlNewProp (node, "cleared", "0");

  /* Write exchange rate */
  g_snprintf (buf, sizeof buf, "%12g", (double) rec.exchange_rate);
  xmlNewProp (node, "exchange_rate", buf);

  return node;
}

static xmlNodePtr
write_records (xmlDocPtr doc, xmlNsPtr ns, const GList *records)
{
  xmlNodePtr node;
  xmlNodePtr child;

  trace ("");
  g_return_val_if_fail (doc, NULL);
  g_return_val_if_fail (ns, NULL);

  node = xmlNewDocNode (doc, ns, "records", NULL);

  while (records)
  {
    child = write_record (doc, ns, LIST_DEREF (Record, records));
    if (!child)
    {
      xmlFreeNode (node);
      return NULL;
    }
    xmlAddChild (node, child);
    records = records->next;
  }
  return node;
}

static xmlNodePtr
write_account_notes (xmlDocPtr doc, xmlNsPtr ns, const gchar *notes)
{
  xmlNodePtr node;
  xmlNodePtr child;

  trace ("");
  g_return_val_if_fail (doc, NULL);
  g_return_val_if_fail (ns, NULL);
  g_return_val_if_fail (notes, NULL);

  node = xmlNewDocNode (doc, ns, "notes", NULL);

  child = xmlNewDocText (doc, notes);
  xmlAddChild (node, child);

  return node;
}

static xmlNodePtr
write_account (xmlDocPtr doc, xmlNsPtr ns, const Account *account)
{
  xmlNodePtr node;
  xmlNodePtr child;
  AccountInfo acc = {0};

  trace ("");
  g_return_val_if_fail (doc, NULL);
  g_return_val_if_fail (ns, NULL);
  g_return_val_if_fail (account, NULL);

  node = xmlNewDocNode (doc, ns, "account", NULL);

  if_account_get_info (account, 0, &acc);

  xmlNewProp (node, "name", acc.name);
  xmlNewProp (node, "foreign", acc.foreign ? "1" : "0");

  child = write_account_notes (doc, ns, acc.notes);
  if (!child)
  {
    xmlFreeNode (node);
    return NULL;
  }
  xmlAddChild (node, child);

  child = write_records (doc, ns, if_account_get_records (account));
  if (!child)
  {
    xmlFreeNode (node);
    return NULL;
  }
  xmlAddChild (node, child);

  return node;
}

static xmlNodePtr
write_accounts (xmlDocPtr doc, xmlNsPtr ns, const GList *accounts)
{
  xmlNodePtr node;
  xmlNodePtr child;

  trace ("");
  g_return_val_if_fail (doc, NULL);
  g_return_val_if_fail (ns, NULL);
  g_return_val_if_fail (accounts, NULL);

  node = xmlNewDocNode (doc, ns, "accounts", NULL);

  while (accounts)
  {
    child = write_account (doc, ns, LIST_DEREF (const Account, accounts));
    if (!child)
    {
      xmlFreeNode (node);
      return NULL;
    }
    xmlAddChild (node, child);
    accounts = accounts->next;
  }
  return node;
}

static xmlNodePtr
write_categories (xmlDocPtr doc, xmlNsPtr ns, const GList *cats)
{
  xmlNodePtr node;
  xmlNodePtr child;

  trace ("");
  g_return_val_if_fail (doc, NULL);
  g_return_val_if_fail (ns, NULL);
  g_return_val_if_fail (cats, NULL);

  node = xmlNewDocNode (doc, ns, "categories", NULL);

  while (cats)
  {
    child = xmlNewDocNode (doc, ns, "category", NULL);
    if (!child)
    {
      xmlFreeNode (node);
      return NULL;
    }
    xmlNewProp (child, "name", LIST_DEREF (const gchar, cats));
    xmlAddChild (node, child);
    cats = cats->next;
  }
  return node;
}

static xmlNodePtr
write_bankbook (xmlDocPtr doc, xmlNsPtr ns, const Bankbook *book, gboolean with_categories)
{
  xmlNodePtr node;
  xmlNodePtr child;
  GList *list;

  trace ("");
  g_return_val_if_fail (doc, NULL);
  g_return_val_if_fail (ns, NULL);
  g_return_val_if_fail (book, NULL);

  node = xmlNewDocNode (doc, ns, "bankbook", NULL);  

  list = (GList *) if_bankbook_get_record_types (book);
  if (list)
  {
    child = write_record_types (doc, ns, list);
    if (!child)
    {
      xmlFreeNode (node);
      return NULL;
    }
    xmlAddChild (node, child);
  }
  
  list = (GList *) if_bankbook_get_accounts (book);
  if (list)
  {
    child = write_accounts (doc, ns, list);
    if (!child)
    {
      xmlFreeNode (node);
      return NULL;
    }
    xmlAddChild (node, child);
  }

  if (with_categories)
  {
    /* Remember category list (because some might be unused and therefore
     * would be otherwise lost) */
    list = if_bankbook_get_category_strings (book);
    if (list)
    {
      child = write_categories (doc, ns, list);
      g_list_free (list);
      if (!child)
      {
	xmlFreeNode (node);
	return NULL;
      }
      xmlAddChild (node, child);
    }
  }

  return node;
}

static xmlNodePtr
write_gnofin_data (GtkWindow *win, xmlDocPtr doc, const Bankbook *book)
{
  xmlNsPtr ns;
  xmlNodePtr node;
  xmlNodePtr child;

  trace ("");
  g_return_val_if_fail (doc, NULL);
  g_return_val_if_fail (book, NULL);

  node = xmlNewDocNode (doc, NULL, "gnofin_data", NULL);  
  if (!node)
    return NULL;

  /* According to libxml-1.8.4 we must have a root node before
   * we can define a namespace.  xmlNewGlobalNs is deprecated. */
  ns = xmlNewNs (node, NS_HREF, "fin");
  if (!ns)
  {
    xmlFreeNode (node);
    return NULL;
  }
  xmlSetNs (node, ns);

  /* Write version property */
  xmlNewProp (node, "version", XML_IO_VERSION);

  /* Write bankbook */
  child = write_bankbook (doc, ns, book, TRUE);
  if (!child)
  {
    xmlFreeNode (node);
    return NULL;
  }
  xmlAddChild (node, child);

  /* Write extensions */
  if (extensions)
  {
    child = write_extensions (win, doc, ns);
    if (!child)
    {
      xmlFreeNode (node);
      return NULL;
    }
    xmlAddChild (node, child);
  }

  return node; 
}

gboolean
xml_io_save (GtkWindow      *win,
	     const gchar    *filename,
	     const Bankbook *book)
{
  xmlDocPtr doc;
  guint error = 0;

  trace ("");
  g_return_val_if_fail (filename, FALSE);
  g_return_val_if_fail (book, FALSE);

  doc = xmlNewDoc ("1.0");
  if (!doc)
  {
    trace ("failed to create document");
    fail (ERROR_UNKNOWN);
  }

  doc->root = write_gnofin_data (win, doc, book);
  if (!doc->root)
    fail (ERROR_UNKNOWN);

  xmlSetDocCompressMode (doc, get_config ()->compression_level);
  if (xmlSaveFile (filename, doc) < 0)
    fail (ERROR_UNKNOWN);

  xmlFreeDoc (doc);
  return TRUE;

fail_:
  if (doc)
    xmlFreeDoc (doc);
  if (error)
    dialog_error (win, _("Error writing file: %s\n[%s]"), filename, xml_io_strerror (error));
  return FALSE;
}


/****************************************************************************** 
 * Clips
 */

static xmlNodePtr
write_gnofin_clip (xmlDocPtr doc, const Bankbook *book)
{
  xmlNsPtr ns;
  xmlNodePtr node;
  xmlNodePtr child;

  trace ("");
  g_return_val_if_fail (doc, NULL);
  g_return_val_if_fail (book, NULL);

  node = xmlNewDocNode (doc, NULL, "gnofin_clip", NULL);
  if (!node)
    return FALSE;
  
  ns = xmlNewNs (node, NS_HREF, "fin");
  if (!ns)
  {
    xmlFreeNode (node);
    return FALSE;
  }
  xmlSetNs (node, ns);

  /* Write version property (required if we intend to share
   * our clipboard data with other programs) */
  xmlNewProp (node, "version", XML_IO_VERSION);

  child = write_bankbook (doc, ns, book, FALSE);
  if (!child)
  {
    xmlFreeNode (node);
    return FALSE;
  }
  xmlAddChild (node, child);

  return node;
}

gboolean
xml_io_clip_write (gchar **data, guint *size, const Bankbook *book)
{
  xmlDocPtr doc;

  trace ("");
  g_return_val_if_fail (book, FALSE);

  doc = xmlNewDoc ("1.0");
  if (!doc)
    return FALSE;
  
  doc->root = write_gnofin_clip (doc, book);
  if (!doc->root)
  {
    xmlFreeDoc (doc);
    return FALSE;
  }

  /* We make the cast to xmlChar... libxml could very well return
   * a UNICODE data stream, but for what we are going to use it
   * for it shouldn't matter. */
  xmlDocDumpMemory (doc, (xmlChar **) data, size);

  xmlFreeDoc (doc);
  return TRUE;
}

static gboolean
read_gnofin_clip (xmlNodePtr root, Bankbook *book)
{
  xmlNodePtr node;

  trace ("");
  g_return_val_if_fail (root, FALSE);
  g_return_val_if_fail (book, FALSE);

  node = get_next_node (root->childs);
  if (node && (g_strcasecmp (node->name, "bankbook") == 0))
  {
    if (!read_bankbook (node, book))
      return FALSE;
  }

  return TRUE;
}

gboolean
xml_io_clip_read  (gchar *data, guint size, Bankbook *book)
{
  xmlNsPtr ns;
  xmlDocPtr doc;
  xmlNodePtr node;
  xmlAttrPtr attr;
  guint major = (guint) -1;

  trace ("");
  g_return_val_if_fail (data, FALSE);
  g_return_val_if_fail (size, FALSE);
  g_return_val_if_fail (book, FALSE);

  doc = xmlParseMemory ((xmlChar *) data, size);
  if (!doc || !doc->root)
    return FALSE;

  ns = xmlSearchNsByHref (doc, doc->root, NS_HREF);
  if (!ns)
  {
    xmlFreeDoc (doc);
    return FALSE;
  }
  
  node = get_next_node (doc->root);
  if (!node || !node->name || g_strcasecmp (node->name, "gnofin_clip"))
  {
    xmlFreeDoc (doc);
    return FALSE;
  }

  /* Verify version */
  attr = node->properties;
  while (attr)
  {
    if (attr->val)
    {
      if (g_strcasecmp (attr->name, "version") == 0)
        parse_version (attr->val->content, &major, NULL);
    }
    attr = attr->next;
  }
  if (major > XML_IO_MAJOR_VERSION)
  {
    xmlFreeDoc (doc);
    return FALSE;
  }

  if (!read_gnofin_clip (node, book))
  {
    xmlFreeDoc (doc); 
    return FALSE;
  }

  xmlFreeDoc (doc); 
  return TRUE;
}


/****************************************************************************** 
 * Compression level
 */

void
xml_io_set_compression_level (gint level)
{
  get_config ()->compression_level = CLAMP (level, 0, 9);
}

gint
xml_io_get_compression_level (void)
{
  trace ("");
  return get_config ()->compression_level;
}


/******************************************************************************
 * XML io extensions: interface
 */

gboolean
xml_io_extension_register (XmlIOExtension *ext)
{
  trace ("");
  g_return_val_if_fail (ext, FALSE);
  g_return_val_if_fail (ext->name, FALSE);

  if (g_slist_find_custom (extensions, (gpointer) ext->name, (GCompareFunc) match_extension_name))
    return FALSE;  /* extension is already registered */

  extensions = g_slist_prepend (extensions, ext);
  return TRUE;
}

void
xml_io_extension_unregister (XmlIOExtension *ext)
{
  trace ("");
  g_return_if_fail (ext);

  extensions = g_slist_remove (extensions, ext);
}
