/*
    GNOME Commander - A GNOME based file manager 
    Copyright (C) 2001-2003 Marcus Bjurman

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/ 
#include <config.h>
#include <pthread.h>
#include "gnome-cmd-includes.h"
#include "widget-factory.h"
#include "gnome-cmd-file-props-dialog.h"
#include "gnome-cmd-file-funcs.h"
#include "gnome-cmd-dir-funcs.h"
#include "gnome-cmd-chown-component.h"
#include "gnome-cmd-chmod-component.h"
#include "gnome-cmd-data.h"
#include "utils.h"
#include "support.h"
#include "imageloader.h"


typedef struct {
	GtkWidget *dialog;
	GnomeCmdFile *finfo;
	
	/* Properties tab stuff */
	gboolean stop;
	GnomeVFSFileSize size;
	GnomeVFSURI *uri;
	GtkWidget *filename_entry;
	GtkWidget *size_label;
	GtkWidget *calc_button;
	pthread_t thread;
	
	/* Permissions tab stuff */
	GtkWidget *chown_component;
	GtkWidget *chmod_component;
	
} GnomeCmdFilePropsDialogPrivate;


static const gchar *
get_size_disp_string (GnomeVFSFileSize size)
{
	static gchar s[64];	
	GnomeCmdSizeDispMode mode = gnome_cmd_data_get_size_disp_mode ();
	
	if (mode == GNOME_CMD_SIZE_DISP_MODE_POWERED)
		return create_nice_size_str (size);

	snprintf (s, 64, _("%s bytes"), size2string (size, mode));
	return s;
}


static void
_calc_tree_size (GnomeCmdFilePropsDialogPrivate *data, GnomeVFSURI *uri)
{
	GnomeVFSFileInfoOptions infoOpts = 0;
	GnomeVFSResult result;
	GList *list = NULL, *tmp;
	gchar *uri_str;

	if (data->stop)
		pthread_exit (NULL);
	
	if (!uri) return;
	uri_str = gnome_vfs_uri_to_string (uri, 0);
	if (!uri_str) return;

	result = gnome_vfs_directory_list_load (
		&list, uri_str, infoOpts, NULL);

	if (result != GNOME_VFS_OK) return;	
	if (!list) return;

	tmp = list;
	while (tmp) {
		GnomeVFSFileInfo *info = (GnomeVFSFileInfo*)tmp->data;

		if (strcmp (info->name, ".") != 0 && strcmp (info->name, "..") != 0) {		
			if (info->type == GNOME_VFS_FILE_TYPE_DIRECTORY) {
				GnomeVFSURI *new_uri = gnome_vfs_uri_append_file_name (uri, info->name);
				_calc_tree_size (data, new_uri);
				gnome_vfs_uri_unref (new_uri);
			}
			else
				data->size += info->size;
		}

		tmp = tmp->next;
	}

	tmp = list;
	while (tmp) {
		GnomeVFSFileInfo *info = (GnomeVFSFileInfo*)tmp->data;		
		gnome_vfs_file_info_unref (info);		
		tmp = tmp->next;
	}

	g_list_free (list);
	g_free (uri_str);

	if (data->stop)
		pthread_exit (NULL);
	
	if (data->size_label) {
		gdk_threads_enter ();
		gtk_label_set_text (GTK_LABEL (data->size_label),
							get_size_disp_string (data->size));
		gdk_threads_leave ();
	}
}


static void
thread_func (GnomeCmdFilePropsDialogPrivate *data)
{
	_calc_tree_size (data, data->uri);
}


/*
 * Tells the thread to exit and then waits for it to do so.
 */
static gboolean
join_thread_func (GnomeCmdFilePropsDialogPrivate *data)
{
	pthread_join (data->thread, NULL);
	gnome_cmd_file_unref (data->finfo);
	gnome_cmd_dir_unref (data->finfo->dir);
	g_free (data);
	return FALSE;
}


static void
on_dialog_destroy (GtkDialog *dialog, GnomeCmdFilePropsDialogPrivate *data)
{
	data->stop = TRUE;
	
	if (!data->thread) {
		gnome_cmd_file_unref (data->finfo);
		gnome_cmd_dir_unref (data->finfo->dir);
		g_free (data);
	}
	else
		gtk_timeout_add (1, (GtkFunction)join_thread_func, data);
}


static void
on_dialog_ok (GtkButton *btn, GnomeCmdFilePropsDialogPrivate *data)
{
	GnomeVFSResult result = GNOME_VFS_OK;
	
	gchar *filename = gtk_entry_get_text (GTK_ENTRY (data->filename_entry));
	if (strcmp (filename, gnome_cmd_file_get_name (data->finfo)) != 0) {
		result = gnome_cmd_file_rename (data->finfo, filename);
	}

	if (result == GNOME_VFS_OK) {
		GnomeVFSFilePermissions perms = gnome_cmd_chmod_component_get_perms (
			GNOME_CMD_CHMOD_COMPONENT (data->chmod_component));
		if (perms != data->finfo->info->permissions) {
			result = gnome_cmd_file_chmod (data->finfo, perms);
		}
	}


	if (result == GNOME_VFS_OK) {
		uid_t uid = gnome_cmd_chown_component_get_owner (
			GNOME_CMD_CHOWN_COMPONENT (data->chown_component));
		gid_t gid = gnome_cmd_chown_component_get_group (
			GNOME_CMD_CHOWN_COMPONENT (data->chown_component));
		
		if (uid == data->finfo->info->uid)
			uid = -1;
		if (gid == data->finfo->info->gid)
			gid = -1;

		if (uid != -1 || gid != -1) {
			result = gnome_cmd_file_chown (data->finfo, uid, gid);
		}
	}
	
	if (result != GNOME_VFS_OK) {
		create_error_dialog (gnome_vfs_result_to_string (result));
		return;
	}

	gtk_widget_destroy (data->dialog);
}


static void
on_dialog_cancel (GtkButton *btn, GnomeCmdFilePropsDialogPrivate *data)
{
	gtk_widget_destroy (data->dialog);
}


static void
on_calc_tree_size (GtkButton *button, GnomeCmdFilePropsDialogPrivate *data)
{
	g_return_if_fail (data != NULL);

	data->stop = FALSE;
	data->size = 0;

	gtk_widget_set_sensitive (data->calc_button, FALSE);
	pthread_create (&data->thread, NULL, (PthreadFunc)thread_func, data);
}


static void
add_sep (GtkWidget *table, gint y)
{
	GtkWidget *hsep;

	hsep = create_hsep (table);
	gtk_table_attach (GTK_TABLE (table), hsep,
					  0, 2, y, y+1, GTK_FILL, GTK_FILL, 0, 0);
}


static GtkWidget*
create_properties_tab (GnomeCmdFilePropsDialogPrivate *data)
{
	gint y = 0;
	GtkWidget *dialog = data->dialog;
	GtkWidget *table;
	GtkWidget *label;
	GtkWidget *space_frame;

	space_frame = create_space_frame (dialog, 6);
	
	table = create_table (dialog, 6, 3);
	gtk_container_add (GTK_CONTAINER (space_frame), table);
	
	label = create_label (dialog, _("Filename:"));
	table_add (table, label, 0, y, GTK_FILL);

	data->filename_entry = create_entry (
		dialog, "filename_entry", gnome_cmd_file_get_name (data->finfo));
	table_add (table, data->filename_entry, 1, y++, GTK_FILL|GTK_EXPAND);
	gtk_editable_set_position (GTK_EDITABLE (data->filename_entry), 0);

	if (data->finfo->info->symlink_name) {
		label = create_label (dialog, _("Symlink target:"));
		table_add (table, label, 0, y, GTK_FILL);
	
		label = create_label (dialog, data->finfo->info->symlink_name);
		table_add (table, label, 1, y++, GTK_FILL);
	}

	add_sep (table, y++);
	
	label = create_label (dialog, _("Type:"));
	table_add (table, label, 0, y, GTK_FILL);
	
	label = create_label (dialog, gnome_cmd_file_get_mime_type_desc (data->finfo));
	table_add (table, label, 1, y++, GTK_FILL);


	label = create_label (dialog, _("MIME Type:"));
	table_add (table, label, 0, y, GTK_FILL);

	label = create_label (dialog, gnome_cmd_file_get_mime_type (data->finfo));
	table_add (table, label, 1, y++, GTK_FILL);
	

	add_sep (table, y++);
	
	
	label = create_label (dialog, _("Accessed:"));
	table_add (table, label, 0, y, GTK_FILL);

	label = create_label (dialog, gnome_cmd_file_get_adate (data->finfo, TRUE));
	table_add (table, label, 1, y++, GTK_FILL);


	label = create_label (dialog, _("Modified:"));
	table_add (table, label, 0, y, GTK_FILL);

	label = create_label (dialog, gnome_cmd_file_get_mdate (data->finfo, TRUE));
	table_add (table, label, 1, y++, GTK_FILL);


	add_sep (table, y++);
	
	
	label = create_label (dialog, _("Size:"));
	table_add (table, label, 0, y, GTK_FILL);

	label = create_label (dialog, get_size_disp_string (data->finfo->info->size));
	if (data->finfo->info->type == GNOME_VFS_FILE_TYPE_DIRECTORY) {
		GtkWidget *hbox;
		GtkWidget *button;
		
		hbox = create_hbox (dialog, FALSE, 12);
		gtk_container_set_border_width (GTK_CONTAINER (hbox), 0);
		table_add (table, hbox, 1, y++, GTK_FILL);
		
		gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
		
		button = create_button_with_data (dialog, _("_Count Total Size"),
													 on_calc_tree_size, data);
		gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);
		data->calc_button = button;
	}
	else {
		table_add (table, label, 1, y++, GTK_FILL);
	}
	data->size_label = label;



	return space_frame;
}


static GtkWidget*
create_permissions_tab (GnomeCmdFilePropsDialogPrivate *data)
{
	GtkWidget *vbox;
	GtkWidget *cat;
	GtkWidget *space_frame;

	space_frame = create_space_frame (data->dialog, 6);
	
	vbox = create_vbox (data->dialog, FALSE, 6);
	gtk_container_add (GTK_CONTAINER (space_frame), vbox);

	data->chown_component = gnome_cmd_chown_component_new ();	
	gtk_widget_ref (data->chown_component);
	gtk_object_set_data_full (GTK_OBJECT (data->dialog),
							  "chown_component", data->chown_component,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (data->chown_component);
	gnome_cmd_chown_component_set (
		GNOME_CMD_CHOWN_COMPONENT (data->chown_component),
		data->finfo->info->uid, data->finfo->info->gid);

	cat = create_category (data->dialog, data->chown_component, _("Owner and group"));
	gtk_box_pack_start (GTK_BOX (vbox), cat, TRUE, TRUE, 0);
	

	data->chmod_component = gnome_cmd_chmod_component_new (0);
	gtk_widget_ref (data->chmod_component);
	gtk_object_set_data_full (GTK_OBJECT (data->dialog),
							  "chmod_component", data->chmod_component,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (data->chmod_component);
	gnome_cmd_chmod_component_set_perms (
		GNOME_CMD_CHMOD_COMPONENT (data->chmod_component),
		data->finfo->info->permissions);

	cat = create_category (data->dialog, data->chmod_component, _("Access permissions"));
	gtk_box_pack_start (GTK_BOX (vbox), cat, TRUE, TRUE, 0);
	
	return space_frame;
}


GtkWidget*
gnome_cmd_file_props_dialog_create (GnomeCmdFile *finfo)
{
	GtkWidget *dialog;
	GtkWidget *notebook;
	GtkWidget *space_frame;
	GnomeCmdFilePropsDialogPrivate *data;
	
	g_return_val_if_fail (finfo != NULL, NULL);
	g_return_val_if_fail (finfo->info != NULL, NULL);

	if (strcmp (finfo->info->name, "..") == 0)
		return NULL;

	data = g_new (GnomeCmdFilePropsDialogPrivate, 1);
	data->thread = 0;

	dialog = create_ok_cancel_dialog (_("File Properties"),
									  on_dialog_ok, on_dialog_cancel, data);
	gtk_signal_connect (GTK_OBJECT (dialog), "destroy", (GtkSignalFunc)on_dialog_destroy, data);

	data->dialog = GTK_WIDGET (dialog);
	data->finfo = finfo;
	data->uri = gnome_cmd_file_get_uri (finfo);
	gnome_cmd_file_ref (finfo);
	gnome_cmd_dir_ref (finfo->dir);

	space_frame = create_space_frame (dialog, 0);
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), space_frame, TRUE, TRUE, 0);

	notebook = gtk_notebook_new ();
	gtk_widget_ref (notebook);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "notebook", notebook,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (notebook);
	gtk_container_add (GTK_CONTAINER (space_frame), notebook);

	gtk_container_add (GTK_CONTAINER (notebook),
					   create_properties_tab (data));
	gtk_container_add (GTK_CONTAINER (notebook),
					   create_permissions_tab (data));

	gtk_notebook_set_tab_label (
		GTK_NOTEBOOK (notebook),
		gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook), 0),
		gtk_label_new (_("Properties")));
	gtk_notebook_set_tab_label (
		GTK_NOTEBOOK (notebook),
		gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook), 1),
		gtk_label_new (_("Permissions")));
	
	return dialog;
}

