/* GNOME DB library
 * Copyright (C) 1999-2001 The Free Software Foundation
 *
 * Authors:
 *	Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gda-export.h>
#include "gnome-db-export.h"
#include "gnome-db-login.h"
#include "gnome-db-util.h"
#include <gtk/gtkclist.h>
#include <gtk/gtkmenuitem.h>
#include <gtk/gtkmenu.h>
#include <gtk/gtknotebook.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-stock.h>

struct _GnomeDbExportPrivate {
	GtkWidget *cnc_list;
	GtkWidget *notebook;
	GdaConnectionPool *pool;
	GdaExport *gda_export;

	GtkWidget *run_button;

	GtkWidget *options_tab;
	GtkWidget *filename_entry;
	GtkWidget *export_table_data;
	GtkWidget *compression_level;

	GtkWidget *objects_tab;
	GtkWidget *object_notebook;
	GtkWidget *tables_list;
	GtkWidget *selected_objects;

	GtkWidget *log_tab;
	GtkWidget *log_output;

	gchar *filename;
};

static void gnome_db_export_class_init (GnomeDbExportClass *klass);
static void gnome_db_export_init       (GnomeDbExport *exp);
static void gnome_db_export_destroy    (GtkObject *object);

static void connect_button_clicked_cb  (GtkButton *button, gpointer user_data);
static void connection_selected_cb     (GtkMenuItem *menu_item, gpointer user_data);
static void object_tab_changed_cb      (GtkNotebook *notebook,
					GtkNotebookPage *page,
					guint page_num,
					gpointer user_data);

enum {
	LAST_SIGNAL
};

static gint db_export_signals[LAST_SIGNAL] = { 0, };

/*
 * Private functions
 */
static void
add_pool_connection (GdaConnectionPool *pool,
		     GdaConnection *cnc,
		     const gchar *gda_name,
		     const gchar *username,
		     const gchar *password,
		     gpointer user_data)
{
	GtkWidget *menu_item;
	GtkWidget *menu = (GtkWidget *) user_data;
	GnomeDbExport *exp;

	g_return_if_fail (GDA_IS_CONNECTION (cnc));
	g_return_if_fail (GTK_IS_MENU (menu));

	exp = gtk_object_get_data (GTK_OBJECT (menu), "GNOME_DB_Export_ExportWidget");
	if (gda_name) {
		gchar *tmp_str = g_strdup (gda_name);

		menu_item = gtk_menu_item_new_with_label (tmp_str);
		gtk_object_set_data_full (GTK_OBJECT (menu_item),
					  "GNOME_DB_MenuItemLabel",
					  (gpointer) tmp_str,
					  (GtkDestroyNotify) g_free);
		gtk_object_set_data (GTK_OBJECT (menu_item),
				     "GNOME_DB_Export_GdaConnection", cnc);

		gtk_signal_connect (GTK_OBJECT (menu_item),
				    "activate",
				    GTK_SIGNAL_FUNC (connection_selected_cb),
				    (gpointer) exp);

		gtk_widget_show (menu_item);
		gtk_menu_append (GTK_MENU (menu), menu_item);
	}
}

static void
fill_connection_list (GnomeDbExport *exp)
{
	GtkWidget *menu;

	g_return_if_fail (GNOME_DB_IS_EXPORT (exp));

	if (GTK_IS_OPTION_MENU (exp->priv->cnc_list)) {
		gtk_option_menu_remove_menu (GTK_OPTION_MENU (exp->priv->cnc_list));
		menu = gtk_menu_new ();
		gtk_object_set_data (GTK_OBJECT (menu),
				     "GNOME_DB_Export_ExportWidget", exp);

		/* if we don't have a GdaConnectionPool, create one */
		if (!GDA_IS_CONNECTION_POOL (exp->priv->pool))
			exp->priv->pool = gda_connection_pool_new ();
		gda_connection_pool_foreach (
			exp->priv->pool,
			(GdaConnectionPoolForeachFunc) add_pool_connection,
			(gpointer) menu);

		gtk_option_menu_set_menu (GTK_OPTION_MENU (exp->priv->cnc_list), menu);
	}
}

static void
show_list_in_clist (GtkCList *clist, GList *list)
{
	GList *l;

	g_return_if_fail (GTK_IS_CLIST (clist));

	gnome_db_clear_clist (clist);
	gtk_clist_freeze (clist);
	for (l = g_list_first (list); l != NULL; l = g_list_next (l)) {
		gchar *row[1] = { (gchar *) l->data };
		gtk_clist_append (clist, row);
	}
	gtk_clist_thaw (clist);
}

/*
 * Callbacks
 */
static void
add_all_cb (GtkButton *button, gpointer user_data)
{
	GList *tables;
	GnomeDbExport *exp = (GnomeDbExport *) user_data;
	
	g_return_if_fail (GNOME_DB_IS_EXPORT (exp));

	tables = gda_export_get_tables (exp->priv->gda_export);
	gda_export_select_table_list (exp->priv->gda_export, tables);

	g_list_foreach (tables, (GFunc) g_free, NULL);
	g_list_free (tables);
}

static void
add_selection_cb (GtkButton *button, gpointer user_data)
{
	GtkWidget *object_list;
	GtkWidget *tab;
	GnomeDbExport *exp = (GnomeDbExport *) user_data;

	g_return_if_fail (GNOME_DB_IS_EXPORT (exp));

	tab = gtk_notebook_get_nth_page (
		GTK_NOTEBOOK (exp->priv->object_notebook),
		gtk_notebook_get_current_page (GTK_NOTEBOOK (exp->priv->object_notebook)));
	object_list = gtk_object_get_data (GTK_OBJECT (tab), "GNOME_DB_Export_ObjectList");
	if (GTK_IS_WIDGET (object_list)) {
		GList *selection;
		gchar *txt;
		GList *l;

		selection = GTK_CLIST (object_list)->selection;

		if (!selection)
			return;

		for (l = selection; l != NULL; l = g_list_next (l)) {
			gtk_clist_get_text (GTK_CLIST (object_list),
					    GPOINTER_TO_INT (l->data),
					    0, &txt);
			if (object_list == exp->priv->tables_list)
				gda_export_select_table (exp->priv->gda_export, txt);
		}
	}
}

static void
destroy_export_cb (GtkObject *object, gpointer user_data)
{
	GnomeDbExport *exp = (GnomeDbExport *) object;
	GdaConnection *cnc = (GdaConnection *) user_data;

	g_return_if_fail (GNOME_DB_IS_EXPORT (exp));
	g_return_if_fail (GDA_IS_CONNECTION (cnc));

	gda_connection_pool_close_connection (exp->priv->pool, cnc);
}

static void
connect_button_clicked_cb (GtkButton *button, gpointer user_data)
{
	GtkWidget *dialog;
	GtkWidget *login;
	GnomeDbExport *exp = (GnomeDbExport *) user_data;

	g_return_if_fail (GNOME_DB_IS_EXPORT (exp));

	/* create the dialog box */
	dialog = gnome_dialog_new (_("Open connection"),
				   GNOME_STOCK_BUTTON_OK,
				   GNOME_STOCK_BUTTON_CANCEL,
				   NULL);
	gnome_dialog_set_default (GNOME_DIALOG (dialog), 0);
	login = gnome_db_login_new (NULL);
	gtk_widget_show (login);
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), login, 1, 1, 0);

	/* run the dialog */
	if (!gnome_dialog_run (GNOME_DIALOG (dialog))) {
		GdaConnection *cnc;

		cnc = gda_connection_pool_open_connection (
			exp->priv->pool,
			gnome_db_login_get_gda_name (GNOME_DB_LOGIN (login)),
			gnome_db_login_get_username (GNOME_DB_LOGIN (login)),
			gnome_db_login_get_password (GNOME_DB_LOGIN (login)));
		if (GDA_IS_CONNECTION (cnc)) {
			gtk_signal_connect (GTK_OBJECT (exp),
					    "destroy",
					    GTK_SIGNAL_FUNC (destroy_export_cb),
					    cnc);
			fill_connection_list (exp);
		}
	}
	gnome_dialog_close (GNOME_DIALOG (dialog));
}

static void
connection_selected_cb (GtkMenuItem *menu_item, gpointer user_data)
{
	GdaConnection *cnc;
	GnomeDbExport *exp = (GnomeDbExport *) user_data;

	g_return_if_fail (GNOME_DB_IS_EXPORT (exp));

	cnc = gtk_object_get_data (GTK_OBJECT (menu_item),
				   "GNOME_DB_Export_GdaConnection");
	gda_export_set_connection (exp->priv->gda_export, cnc);

	/*
	 * ugly yes, but it does what we need, which is to refresh the
	 * object lists
	 */
	object_tab_changed_cb (GTK_NOTEBOOK (exp->priv->object_notebook),
			       NULL,
			       gtk_notebook_get_current_page (
				       GTK_NOTEBOOK (exp->priv->object_notebook)),
			       exp);
}

static void
export_cancelled_cb (GdaExport *gda_exp, gpointer user_data)
{
	GnomeDbExport *exp = (GnomeDbExport *) user_data;

	g_return_if_fail (GDA_IS_EXPORT (gda_exp));
	g_return_if_fail (GNOME_DB_IS_EXPORT (exp));

	gnome_db_show_error (_("The export process was cancelled due to an error."));
	gtk_widget_set_sensitive (exp->priv->run_button, TRUE);
}

static void
export_finished_cb (GdaExport *gda_exp, GdaXmlDatabase *xmldb, gpointer user_data)
{
	GnomeDbExport *exp = (GnomeDbExport *) user_data;

	g_return_if_fail (GDA_IS_EXPORT (gda_exp));
	g_return_if_fail (GNOME_DB_IS_EXPORT (exp));

	if (GDA_IS_XML_DATABASE (xmldb)) {
		if (gnome_db_yes_no_dialog (_("Would you like to view the database before saving it?"))) {
			GtkWidget *dialog;
			GtkWidget *designer;

			dialog = gnome_dialog_new (_("Database exported"),
						   GNOME_STOCK_BUTTON_CLOSE,
						   NULL);
			gtk_widget_set_usize (dialog, 400, 300);
			designer = gnome_db_designer_new_from_xml (xmldb);
			gtk_widget_show (designer);
			gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), designer, 1, 1, 0);
			gnome_dialog_run_and_close (GNOME_DIALOG (dialog));
		}

		/* save the database */
		if (gda_xml_database_save (xmldb, exp->priv->filename)) {
			gchar *msg;

			msg = g_strdup_printf (_("The database has been successfully saved to %s"),
					       exp->priv->filename);
			gnome_db_show_message (msg);
			g_free (msg);
		}
		else {
			/* FIXME: show a retry dialog */
		}
	}
	else
		gnome_db_show_error (_("No database was generated. There is a bug."));

	/* re-enable buttons */
	gtk_widget_set_sensitive (exp->priv->run_button, TRUE);
}

static void
object_tab_changed_cb (GtkNotebook *notebook,
		       GtkNotebookPage *page,
		       guint page_num,
		       gpointer user_data)
{
	GtkWidget *tab;
	GtkWidget *object_list;
	GList *list;
	GList *selection;
	GnomeDbExport *exp = (GnomeDbExport *) user_data;

	g_return_if_fail (GNOME_DB_IS_EXPORT (exp));

	tab = gtk_notebook_get_nth_page (notebook, page_num);
	object_list = gtk_object_get_data (GTK_OBJECT (tab), "GNOME_DB_Export_ObjectList");
	if (GTK_IS_WIDGET (object_list)) {
		if (object_list == exp->priv->tables_list) {
			list = gda_export_get_tables (exp->priv->gda_export);
			selection = gda_export_get_selected_tables (exp->priv->gda_export);
		}

		show_list_in_clist (GTK_CLIST (object_list), list);
		show_list_in_clist (GTK_CLIST (exp->priv->selected_objects), selection);

		g_list_foreach (list, (GFunc) g_free, NULL);
		g_list_free (list);
		g_list_foreach (selection, (GFunc) g_free, NULL);
		g_list_free (selection);
	}
}

static void
object_selection_changed_cb (GdaExport *gda_exp,
			     GDA_Connection_QType qtype,
			     const gchar *name,
			     gpointer user_data)
{
	GtkWidget *tab;
	GtkWidget *object_list;
	GnomeDbExport *exp = (GnomeDbExport *) user_data;

	g_return_if_fail (GNOME_DB_IS_EXPORT (exp));

	tab = gtk_notebook_get_nth_page (
		GTK_NOTEBOOK (exp->priv->object_notebook),
		gtk_notebook_get_current_page (GTK_NOTEBOOK (exp->priv->object_notebook)));
	object_list = gtk_object_get_data (GTK_OBJECT (tab), "GNOME_DB_Export_ObjectList");
	if (GTK_IS_WIDGET (object_list)) {
		if (object_list == exp->priv->tables_list &&
		    qtype == GDA_Connection_GDCN_SCHEMA_TABLES) {
			GList *selection;

			selection = gda_export_get_selected_tables (exp->priv->gda_export);
			show_list_in_clist (GTK_CLIST (exp->priv->selected_objects), selection);

			g_list_free (selection);
		}
	}
}

static void
run_button_clicked_cb (GtkButton *button, gpointer user_data)
{
	gchar *filename;
	GdaExportFlags flags = 0;
	GnomeDbExport *exp = (GnomeDbExport *) user_data;

	g_return_if_fail (GNOME_DB_IS_EXPORT (exp));

	filename = gtk_entry_get_text (
		GTK_ENTRY (gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (exp->priv->filename_entry))));
	if (!filename || !filename[0]) {
		gnome_db_show_error (_("You must specify a file name"));
		return;
	}

	if (exp->priv->filename)
		g_free (exp->priv->filename);
	exp->priv->filename = g_strdup (filename);

	/* set flags for export execution */
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (exp->priv->export_table_data)))
		flags |= GDA_EXPORT_FLAGS_TABLE_DATA;

	/* run the export */
	gtk_widget_set_sensitive (exp->priv->run_button, FALSE);
	gda_export_run (exp->priv->gda_export, flags);
}

/*
 * GnomeDbExport implementation
 */

static void
gnome_db_export_class_init (GnomeDbExportClass *klass)
{
	GtkObjectClass *object_class = GTK_OBJECT_CLASS (klass);

	object_class->destroy = gnome_db_export_destroy;
}

static void
gnome_db_export_init (GnomeDbExport *exp)
{
	GtkWidget *table;
	GtkWidget *label;
	GtkWidget *button;
	GtkWidget *scroll;
	GtkAdjustment *adj;

	exp->priv = g_new0 (GnomeDbExportPrivate, 1);
	exp->priv->gda_export = gda_export_new (NULL);
	gtk_signal_connect (GTK_OBJECT (exp->priv->gda_export),
			    "object_selected",
			    GTK_SIGNAL_FUNC (object_selection_changed_cb),
			    exp);
	gtk_signal_connect (GTK_OBJECT (exp->priv->gda_export),
			    "object_unselected",
			    GTK_SIGNAL_FUNC (object_selection_changed_cb),
			    exp);
	gtk_signal_connect (GTK_OBJECT (exp->priv->gda_export),
			    "finished",
			    GTK_SIGNAL_FUNC (export_finished_cb),
			    exp);
	gtk_signal_connect (GTK_OBJECT (exp->priv->gda_export),
			    "cancelled",
			    GTK_SIGNAL_FUNC (export_cancelled_cb),
			    exp);

	/* create the main container */
	table = gnome_db_new_table_widget (4, 4, FALSE);
	gtk_box_pack_start (GTK_BOX (exp), table, TRUE, TRUE, 0);

	/* create the information widgets */
	label = gnome_db_new_label_widget (_("Database"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
	exp->priv->cnc_list = gnome_db_new_option_menu_widget ();
	gtk_table_attach (GTK_TABLE (table), exp->priv->cnc_list, 1, 2, 0, 1,
			  GTK_FILL, GTK_FILL, 3, 3);

	button = gnome_db_new_button_widget_with_pixmap ("", GNOME_STOCK_MENU_CONVERT);
	gtk_signal_connect (GTK_OBJECT (button),
			    "clicked",
			    GTK_SIGNAL_FUNC (connect_button_clicked_cb),
			    (gpointer) exp);
	gtk_table_attach (GTK_TABLE (table), button, 2, 3, 0, 1, GTK_FILL, GTK_FILL, 3, 3);

	exp->priv->run_button = gnome_db_new_button_widget_with_pixmap ("", GNOME_STOCK_MENU_EXEC);
	gtk_signal_connect (GTK_OBJECT (exp->priv->run_button),
			    "clicked",
			    GTK_SIGNAL_FUNC (run_button_clicked_cb),
			    (gpointer) exp);
	gtk_table_attach (GTK_TABLE (table), exp->priv->run_button, 2, 3, 1, 2, GTK_FILL, GTK_FILL, 3, 3);

	exp->priv->notebook = gnome_db_new_notebook_widget ();
	gtk_table_attach (GTK_TABLE (table), exp->priv->notebook, 0, 4, 2, 4,
			  GTK_FILL | GTK_EXPAND | GTK_SHRINK,
			  GTK_FILL | GTK_EXPAND | GTK_SHRINK,
			  3, 3);

	/* create the 'Options' tab */
	exp->priv->options_tab = gnome_db_new_table_widget (6, 3, FALSE);
	gtk_notebook_append_page (GTK_NOTEBOOK (exp->priv->notebook),
				  exp->priv->options_tab,
				  gtk_label_new (_("Options")));

	label = gnome_db_new_label_widget (_("Output file"));
	gtk_table_attach (GTK_TABLE (exp->priv->options_tab), label, 0, 1, 0, 1,
			  GTK_FILL, GTK_FILL, 3, 3);
	exp->priv->filename_entry = gnome_db_new_file_entry_widget ("GNOME_DB_Export_OutputFile");
	gtk_table_attach (GTK_TABLE (exp->priv->options_tab), exp->priv->filename_entry,
			  1, 3, 0, 1, GTK_FILL, GTK_FILL, 3, 3);

	exp->priv->export_table_data =
		gnome_db_new_check_button_widget (_("Export table data"), TRUE);
	gtk_table_attach (GTK_TABLE (exp->priv->options_tab), exp->priv->export_table_data,
			  0, 2, 1, 2, GTK_FILL, GTK_FILL, 3, 3);

	label = gnome_db_new_label_widget (_("Compression level"));
	gtk_table_attach (GTK_TABLE (exp->priv->options_tab), label, 0, 1, 2, 3,
			  GTK_FILL, GTK_FILL, 3, 3);
	exp->priv->compression_level = gnome_db_new_spin_button_widget (0, 0, 5, 0);
	gtk_table_attach (GTK_TABLE (exp->priv->options_tab), exp->priv->compression_level,
			  1, 2, 2, 3, GTK_SHRINK, GTK_SHRINK, 3, 3);

	/* create the 'Objects' tab */
	exp->priv->objects_tab = gnome_db_new_table_widget (3, 6, FALSE);
	gtk_notebook_append_page (GTK_NOTEBOOK (exp->priv->notebook),
				  exp->priv->objects_tab,
				  gtk_label_new (_("Objects")));
	exp->priv->object_notebook = gnome_db_new_notebook_widget ();
	gtk_notebook_set_tab_pos (GTK_NOTEBOOK (exp->priv->object_notebook),
				  GTK_POS_BOTTOM);
	gtk_signal_connect (GTK_OBJECT (exp->priv->object_notebook),
			    "switch_page",
			    GTK_SIGNAL_FUNC (object_tab_changed_cb),
			    (gpointer) exp);
	gtk_table_attach (GTK_TABLE (exp->priv->objects_tab), exp->priv->object_notebook,
			  0, 1, 0, 6,
			  GTK_FILL | GTK_SHRINK | GTK_EXPAND,
			  GTK_FILL | GTK_SHRINK | GTK_EXPAND,
			  3, 3);

	scroll = gnome_db_new_scrolled_window_widget ();
	exp->priv->tables_list = gnome_db_new_clist_widget (NULL, 1);
	gtk_clist_set_selection_mode (GTK_CLIST (exp->priv->tables_list),
				      GTK_SELECTION_MULTIPLE);
	gtk_container_add (GTK_CONTAINER (scroll), exp->priv->tables_list);
	gtk_notebook_append_page (GTK_NOTEBOOK (exp->priv->object_notebook),
				  scroll,
				  gtk_label_new (_("Tables")));
	gtk_object_set_data (GTK_OBJECT (scroll),
			     "GNOME_DB_Export_ObjectList",
			     exp->priv->tables_list);

	/* create buttons */
	button = gnome_db_new_button_widget (">");
	gtk_signal_connect (GTK_OBJECT (button),
			    "clicked",
			    GTK_SIGNAL_FUNC (add_selection_cb),
			    exp);
	gtk_table_attach (GTK_TABLE (exp->priv->objects_tab), button, 1, 2, 0, 1,
			  GTK_FILL, GTK_FILL, 3, 3);

	button = gnome_db_new_button_widget (">>");
	gtk_signal_connect (GTK_OBJECT (button),
			    "clicked",
			    GTK_SIGNAL_FUNC (add_all_cb),
			    exp);
	gtk_table_attach (GTK_TABLE (exp->priv->objects_tab), button, 1, 2, 1, 2,
			  GTK_FILL, GTK_FILL, 3, 3);

	button = gnome_db_new_button_widget ("<<");
	gtk_table_attach (GTK_TABLE (exp->priv->objects_tab), button, 1, 2, 2, 3,
			  GTK_FILL, GTK_FILL, 3, 3);

	button = gnome_db_new_button_widget ("<");
	gtk_table_attach (GTK_TABLE (exp->priv->objects_tab), button, 1, 2, 3, 4,
			  GTK_FILL, GTK_FILL, 3, 3);

	/* create selection list */
	scroll = gnome_db_new_scrolled_window_widget ();
	gtk_table_attach (GTK_TABLE (exp->priv->objects_tab), scroll, 2, 3, 0, 6,
			  GTK_FILL | GTK_SHRINK | GTK_EXPAND,
			  GTK_FILL | GTK_SHRINK | GTK_EXPAND,
			  3, 3);
	exp->priv->selected_objects = gnome_db_new_clist_widget (NULL, 1);
	gtk_clist_set_selection_mode (GTK_CLIST (exp->priv->selected_objects),
				      GTK_SELECTION_MULTIPLE);
	gtk_container_add (GTK_CONTAINER (scroll), exp->priv->selected_objects);

	/* create the 'Log' tab */
	exp->priv->log_tab = gnome_db_new_scrolled_window_widget ();
	gtk_notebook_append_page (GTK_NOTEBOOK (exp->priv->notebook),
				  exp->priv->log_tab,
				  gtk_label_new (_("Log")));
	exp->priv->log_output = gnome_db_new_text_widget ();
	gtk_text_set_editable (GTK_TEXT (exp->priv->log_output), FALSE);
	gtk_container_add (GTK_CONTAINER (exp->priv->log_tab), exp->priv->log_output);
}

static void
gnome_db_export_destroy (GtkObject *object)
{
	GtkObjectClass *parent_class;
	GnomeDbExport *exp = (GnomeDbExport *) object;

	g_return_if_fail (GNOME_DB_IS_EXPORT (exp));

	if (GDA_IS_EXPORT (exp->priv->gda_export)) {
		gda_export_free (exp->priv->gda_export);
		exp->priv->gda_export = NULL;
	}
	if (GDA_IS_CONNECTION_POOL (exp->priv->pool)) {
		gda_connection_pool_free (exp->priv->pool);
		exp->priv->pool = NULL;
	}
	g_free (exp->priv->filename);

	g_free ((gpointer) exp->priv);

	parent_class = gtk_type_class (gtk_vbox_get_type ());
	if (parent_class && parent_class->destroy)
		parent_class->destroy (object);
}

/**
 * gnome_db_export_get_type
 */
GtkType
gnome_db_export_get_type (void)
{
	static GtkType type = 0;

	if (!type) {
		GtkTypeInfo info = {
			"GnomeDbExport",
			sizeof (GnomeDbExport),
			sizeof (GnomeDbExportClass),
			(GtkClassInitFunc) gnome_db_export_class_init,
			(GtkObjectInitFunc) gnome_db_export_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};
		type = gtk_type_unique (gtk_vbox_get_type (), &info);
	}

	return type;
}

/**
 * gnome_db_export_new
 *
 * Create a new #GnomeDbExport widget. This widget is just a frontend
 * to the #GdaExport object provided in libgda, which lets you easily
 * add database exporting to your GNOME applications.
 *
 * Returns: the newly created widget
 */
GtkWidget *
gnome_db_export_new (void)
{
	GnomeDbExport *exp;

	exp = GNOME_DB_EXPORT (gtk_type_new (gnome_db_export_get_type ()));
	return GTK_WIDGET (exp);
}

/**
 * gnome_db_export_get_pool
 * @exp: a #GnomeDbExport widget
 *
 * Return the #GdaConnectionPool object being used by the given
 * #GnomeDbExport widget
 *
 * Returns: a #GdaConnectionPool object
 */
GdaConnectionPool *
gnome_db_export_get_pool (GnomeDbExport *exp)
{
	g_return_val_if_fail (GNOME_DB_IS_EXPORT (exp), NULL);
	return exp->priv->pool;
}

/**
 * gnome_db_export_set_pool
 * @exp: a #GnomeDbExport widget
 * @pool: a #GdaConnectionPool object
 *
 * Associate a #GdaConnectionPool object with the given #GnomeDbExport
 * widget. By doing this, you force the #GnomeDbExport widget to
 * use that connection pool to get the connections it needs, and
 * also reusing the ones that are already open on the connection
 * pool
 */
void
gnome_db_export_set_pool (GnomeDbExport *exp, GdaConnectionPool *pool)
{
	g_return_if_fail (GNOME_DB_IS_EXPORT (exp));

	if (GDA_IS_CONNECTION_POOL (exp->priv->pool))
		gda_connection_pool_free (exp->priv->pool);

	exp->priv->pool = pool;
	if (GDA_IS_CONNECTION_POOL (exp->priv->pool))
		gtk_object_ref (GTK_OBJECT (exp->priv->pool));

	fill_connection_list (exp);
}
