#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2005 Free Software Foundation
#
# FILE:
# ResultSet.py
#
# DESCRIPTION:
#
# NOTES:
#
# $Id :$

__all__ = ['ResultSet']

from gnue.common.datasources import GConditions, Exceptions
import string

from RecordSet import RecordSet

###########################################################
#
#
#
###########################################################
class ResultSet:

  _recordSetClass = RecordSet

  def __init__(self, dataObject, cursor=None,defaultValues={},masterRecordSet=None):
     self._dataObject = dataObject
     self._cursor = cursor
     self._cachedRecords = []
     self._currentRecord = -1
     self._masterRecordSet = masterRecordSet
     self._readonly = False
     self._recordCount = 0
     self._postingRecord = None

     self._defaultValues = {}
     self._defaultValues.update(defaultValues)

     self.__listeners = []

     self.current = None

     if masterRecordSet:
       masterRecordSet.addDetailResultSet(self)

  # Since we are overriding __len__
  def __nonzero__(self):
    return 1

  # Return the # of records
  def __len__(self):
    return self.getRecordCount()

  def __getitem__(self, index):
    rs = self.getRecord(index)
    if not rs:
      raise IndexError
    else:
      return rs


  # Returns whether this result set is read only or not
  def isReadOnly(self):
    return self._readonly


  # Returns 1=At first record, 0=Not first record
  def isFirstRecord(self):
    return (self._currentRecord == 0)


  # Returns 1=At last record, 0=Not last record
  def isLastRecord(self):
    if self._currentRecord < len(self._cachedRecords) - 1 or \
       self._cacheNextRecord():
      return False
    else:
      return True


  # returns -1=No records in memory, #=Current record #
  def getRecordNumber(self):
    return self._currentRecord


  # returns # of records currently loaded
  def getCacheCount(self):
    return len(self._cachedRecords)

  # returns # of records the
  def getRecordCount(self):
    gEnter (8)
    result = self._recordCount > 0 and self._recordCount or self.getCacheCount()
    return gLeave (8, result)

  # Get a specific record (0=based)
  def getRecord(self, record):
    while (record + 1 > len(self._cachedRecords)) and self._cacheNextRecord():
      pass

    if record + 1 > len(self._cachedRecords):
      return None
    else:
      return self._cachedRecords[record]


  # Move to a record number already in cache
  # -1 sets the current record to None
  def __move (self, record):
    if record != self._currentRecord \
         or (self._currentRecord >= 0 \
             and self.current != self._cachedRecords [self._currentRecord]):
      self._currentRecord = record
      if record == -1:
        self.current = None
      else:
        self.current = self._cachedRecords [self._currentRecord]
      self.notifyDetailObjects ()
      self.__notifyListeners ()


  # move to record #, returns record or None if record number out of range
  def setRecord(self, record):

    while (record > len(self._cachedRecords) -1) and self._cacheNextRecord():
      pass

    if record >= len(self._cachedRecords):
      return None
    else:
      self.__move (record)
      return self.current


  # returns next record, or None if no more records
  def nextRecord(self):
    if self._currentRecord + 1 == len(self._cachedRecords):
      if not self._cacheNextRecord():
        return None

    self.__move (self._currentRecord + 1)
    return self.current


  # returns 1=New record loaded, 0=At first record
  def prevRecord(self):
    if self._currentRecord < 1:
      return None
    else:
      self.__move (self._currentRecord - 1)
      return self.current


  # returns 1=at first record, 0=No records loaded
  def firstRecord(self):
    if self._currentRecord < 0:
      if not self._cacheNextRecord():
        return None

    self.__move (0)
    return self.current


  # returns 1=at last record, 0=No records loaded
  def lastRecord(self):
    if self._currentRecord == -1:
      return None
    else:
      while self._cacheNextRecord():
        pass
      self.__move (len (self._cachedRecords) - 1)
      return self.current



  # Insert a blank record after the current record
  def insertRecord(self):
    if self.isReadOnly():
      # Provide better feedback??
      tmsg =  _("Attempted to insert into a read only datasource")
      raise Exceptions.ReadOnlyError, tmsg
    else:
      gDebug (8,'Inserting a blank record')
      self._currentRecord += 1
      self._cachedRecords.insert(self._currentRecord, self._createEmptyRecord())
      self._recordCount += 1
      self.current = self._cachedRecords[self._currentRecord]

      # Set any dataobject-wide default values
      for field in self._dataObject._defaultValues.keys():
        gDebug (8, "DataObject-Wide default for %s" % field)
        self.current.setField(field, self._dataObject._defaultValues[field],False)

      # Set any resultset specific values
      for field in self._defaultValues.keys():
        gDebug (8, "ResultSet-specific default for %s" % field)
        self.current.setField(field, self._defaultValues[field],False)

      # Pull any primary keys from a master record set
      if self._masterRecordSet != None and hasattr(self._dataObject, '_masterfields'):
        i = 0
        for field in self._dataObject._masterfields:
          gDebug (8, "Value for %s to %s" % (field, \
                     self._masterRecordSet.getField (field)))
          self.current.setField(self._dataObject._detailfields[i],self._masterRecordSet.getField(field),False)
          i += 1

      self.notifyDetailObjects()
      self.__notifyListeners ()
      return self.current

  # ---------------------------------------------------------------------------
  # Find a record by field values
  # ---------------------------------------------------------------------------

  def findRecord (self, fieldValues):
    """
    This function searches through the already loaded records and sets the
    current record to the first record to match the given fieldValues
    dictionary (in the form {fieldname: value}).
    If no match is found, then the record pointer is set to -1.
    """

    # Make sure that all records are cached
    while self._cacheNextRecord():
      pass

    # Find match
    for i in range (len (self._cachedRecords)):
      record = self._cachedRecords [i]
      found = True
      for (key, value) in fieldValues.items ():
        if record [key] != value:
          found = False
          continue
      if found:
        self.__move (i)
        return self.current

    # No match found
    self.__move (-1)
    return self.current

  # ---------------------------------------------------------------------------
  # Get data as array
  # ---------------------------------------------------------------------------

  def getArray (self, fields):
    """
    This function returns the values of the given fields for all records as a
    2-dimensional list. The record pointer is *not* moved.
    """

    # First, load all records into the cache
    while self._cacheNextRecord():
      pass

    # Now build up the array
    result = []
    for record in self._cachedRecords:
      line = []
      for field in fields:
        line.append (record [field])
      result.append (line)
    return result

  # ---------------------------------------------------------------------------
  # Close the result set
  # ---------------------------------------------------------------------------

  def close (self):
    """
    This function should be called, if a result set is no longer needed. It
    breaks up reference cycles so garbage collection can do it's job.
    """

    for item in self._cachedRecords:
      item._parent = None

    if self._dataObject._dataSource:
      self._dataObject._dataSource.close ()

    self._dataObject    = None
    self._cachedRecords = []
    self.__listeners    = []


  def duplicateRecord(self, exclude=(), include=()):
    current = self.current
    inserted = self.insertRecord()
    if not inserted:
      return None

    # If include= is specified, then that is our base list.
    # Otherwise, get the base list as the fields in the table
    if include:
      fields = list(include)
    else:
      fields = list(current._fields.keys())

    # Exclude all the fields in exclude=
    for field in exclude:
      try:
        fields.pop(exclude)
      except:
        pass

    # Do not duplicate the primary key field,
    # unless it was named in the include= parameter
    try:
      for field in self._dataObject._primaryIdField:
        if field not in include:
          try:
            fields.pop(field)
          except:
            pass
    except AttributeError:
      pass

    # Copy the fields over
    for field in fields:
      inserted[field] = current[field]

    return inserted

  # Returns 1=DataObject, or a detail resultset, has uncommitted changes
  def isPending(self):
    for rec in (self._cachedRecords):
      if rec.isPending ():
        return True
    return False


  # Returns 1=DataObject has uncommitted changes
  def isRecordPending(self):
    return self.current.isPending()


  def getPostingRecordset(self):
    global postingRecordset
    return postingRecordset

  # Post changes to the database
  def post(self, foreign_keys={}):
    global postingRecordset
    # post our changes
    self._update_cursor = self._dataObject._dataConnection.cursor()

    recordPosition = 0
    while recordPosition < len(self._cachedRecords):
      self._postingRecord = self._cachedRecords[recordPosition]
      postingRecordset = self._postingRecord
      delete = self._postingRecord.isEmpty() or self._postingRecord.isDeleted()
      if not delete:
        # Flip the flag for 'default' values to true so that hidden
        # default fields are included in insert statements
        if self._postingRecord.isPending():
          for field in self._dataObject._defaultValues.keys():
            self._postingRecord._modifiedFlags[field] = True

        for field in foreign_keys.keys():
          self._postingRecord._fields[field] = foreign_keys[field]
          # Some DBs will throw an exception if you update a Primary Key
          # (even if you are updating to the same value)
          if self._postingRecord.isInserted():
            self._postingRecord._modifiedFlags[field] = True

        recordPosition += 1
      else:
        # Adjust the current record if a preceding record
        # or the current record is deleted
        if recordPosition <= self._currentRecord:
          self._currentRecord -= 1
        self._cachedRecords.pop(recordPosition)
        self._recordCount -= 1
     
      self._postingRecord.post(recordPosition)

    # Move to record 0 if all preceding records were deleted
    # (or set to -1 if all records were deleted)
    if self._currentRecord < 0:
      if len(self._cachedRecords):
        self._currentRecord = 0
      else:
        self._currentRecord = -1
# TODO: I don't think we need this anymore
#    if self._currentRecord >= self._recordCount:
#      self._currentRecord = self._recordCount - 1

  def notifyDetailObjects(self):
    gDebug (8,'Master record changed; Notifying Detail Objects')
    for detail in self._dataObject._detailObjects:
      if detail[1]:
        detail[1].masterResultSetChanged(self,
                                         detail[0]._masterRecordChanged(self))


  # Other objects can register here. Their currentRecordMoved method will then
  # get called whenever the current record of this ResultSet changes.
  def registerListener (self, listener):
    if listener not in self.__listeners:
      self.__listeners.append (listener)
    # Inform new listener about current record. This happens whenever the
    # *Resultset* (not the current record but the whole resultset) changes.
    if self._currentRecord >= 0:
      listener.currentRecordMoved ()


  # This gets called whenever the ResultSet gets a new current recordset,
  # that happens in case of record navigation and in case of inserting a new
  # record.
  def __notifyListeners (self):
    for listener in self.__listeners:
      listener.currentRecordMoved ()


  # Returns 1=Field is bound to a database field
  def isFieldBound(self, fieldName):

    # TODO: until the case problem get's fixed in common we do the following
    # workaround.
    for item in self._dataObject._fieldReferences.keys ():
      if item.lower () == fieldName.lower ():
        return True

    return False


  # Load cacheCount number of new records
  def _cacheNextRecord(self):
    rs = self._loadNextRecord()
    if rs:
      self._dataObject._dataSource._onRecordLoaded(self._cachedRecords[-1])
    return rs


  # ---------------------------------------------------------------------------
  # Nice string representation
  # ---------------------------------------------------------------------------

  def __repr__ (self):
    do = self._dataObject
    if hasattr (do, 'table'):
      return "<ResultSet for %s>" % do.table
    else:
      return "<NIL-Table RecordSet>"


  ###
  ### Methods below should be overridden by Vendor Specific functions
  ### (_createEmptyRecord may not need to be overridden in all cases)
  ###

  # Load cacheCount number of new records
  def _loadNextRecord(self):
    return False

  # Create an empty recordset
  def _createEmptyRecord(self):
    return self._recordSetClass(self)

  # Iterator support (Python 2.2+)
  def __iter__(self):
    return _ResultSetIter(self)



# A simple resultset iterator
# Lets you use ResultSets as:
#
#   for record in myResultSet:
#      blah
#
# NOTE: Python 2.2+  (but it won't get called in
#    Python 2.1 or below, so not a problem)
#
class _ResultSetIter:
  def __init__(self, resultset):
    self.resultset = resultset
    self.used = False
    self.done = False

  def __iter__(self):
    return self

  def next(self):
    if self.done:
      raise StopIteration
    if not self.used:
      rs = self.resultset.firstRecord()
      self.used = True
    else:
      rs = self.resultset.nextRecord()

    if not rs:
      self.done = True
      raise StopIteration
    else:
      return rs


# TODO: wtf?
postingRecordset = None

