#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2005 Free Software Foundation
#
#
# FILE:
# NamespaceCore.py
#
# DESCRIPTION:
# Provides the basic classes needed by the generic trigger system
#
# NOTES:
#
import sys
import types
import string
import copy
from gnue.common.definitions.GObjects import GObj
from gnue.common.apps import GDebug

from gnue.common.formatting import GTypecast
from xml.sax import saxutils
from gnue.common.definitions.GParserHelpers import GContent


#######################################################################
#
# Trigger/Function namespace classes
#
# Classes used in implementing the trigger/function namespaces
#
#
# GObjNamespace
#
# Manager class for a namespace.  Passed a GObj based
# tree at __init__ which it uses to construct a new
# NamespaceElement based tree
#
class GObjNamespace(GObj):
  def __init__(self,objectTree = None, rootName="root"):
    self._globalNamespace = {'True' : 1,
                             'False': 0,
                             }

    self._rname = rootName

    if objectTree:
      self._globalNamespace[self._rname] = self.constructTriggerObject(objectTree)
    else:
      GDebug.printMesg(1,'GObjNamespace was passed an empty object tree')

  #
  # constructTriggerObject
  #
  # Travels down thru a GObj based tree and builds a set
  # of NamespaceElements that will implement the namespace
  # inside triggers/formulas.
  #
  def constructTriggerObject(self, gobjObject, triggerParent=None):
    triggerObject = None

    # Some items in a GObj tree may not be GObj based (GContent for instance)

    if isinstance(gobjObject,GObj) and hasattr(gobjObject,'name'):
      triggerObject = NamespaceElement(triggerParent)
      triggerObject._object = gobjObject

      # Add this triggerObject to global namespace if the GObj requests it
      if gobjObject._triggerGlobal:
        self._globalNamespace[gobjObject.name] = triggerObject

      # setup get and set functions when they exist in the GObj
      triggerObject._triggerSet = gobjObject._triggerSet
      triggerObject._triggerGet = gobjObject._triggerGet

      # Add any trigger methods defined by GObj
      if len(gobjObject._triggerFunctions):
        for item in gobjObject._triggerFunctions.keys():

          if type(gobjObject._triggerFunctions[item]['function']) == types.MethodType:
            object = NamespaceFunction(item,gobjObject._triggerFunctions[item]['function'])
            triggerObject.__dict__[item] = object
            # Add this function to global namespace if the GObj requests it
            if gobjObject._triggerFunctions[item].get('global',0):
              self._globalNamespace[item] = object

          else:
            raise 'Only functions are supported in an objects _triggerFunctions (%s %s)' % (gobjObject,item)

            sys.exit()

      # Load the defined __properties__ into this object's
      # NamespaceElementProperties instance
      if len(gobjObject._triggerProperties):
        for item in gobjObject._triggerProperties.keys():
          if gobjObject._triggerProperties[item].has_key('set'):
            setFunc = gobjObject._triggerProperties[item]['set']
          else:
            setFunc = None
          triggerObject._triggerProperties.addProperty(item,gobjObject._triggerProperties[item]['get'], setFunc)

      # Process the children of this Gobj
      if len(gobjObject._children):
        for child in gobjObject._children:
          object = self.constructTriggerObject(child, triggerObject)

          # Add this objects children to it's namespace by their name
          if object:
            if child.name is not None:
              if child.name [:2] != '__':
                triggerObject.__dict__[child.name] = object

      #
      # populate the GObj's _localTriggerNamespace
      localNamespace = {'self':triggerObject}
      localNamespace.update(triggerObject.__dict__)
      gobjObject._localTriggerNamespace = localNamespace

    return triggerObject

#
# NamespaceElement
#
# Inherits GObj to gain it's parent/child system
#
class NamespaceElement(GObj):
  def __init__(self, parent):
    GObj.__init__(self,parent)
    self.__dict__['_parent'] = parent
    self._triggerProperties = NamespaceElementProperties()
    self._triggerSet = None
    self._triggerGet = None
    self._object = None

  #
  #
  #
  def __repr__(self):
    try:
      return "NamespaceElement(%s) at %s" % (self._object.__class__, id(self))
    except:
      return "NamespaceElement at %s" % (id(self))


  #
  # showTree
  #
  # Handy function to dump a rough look at the namespace
  # doesn't show things nested properly though
  def showTree(self, indent=0):
    print '  ' * indent + `self._type` + `self`
    for item in self.__dict__.keys():
      if item[:1] != '_':
        print '  ' * indent + ' :' + item
    for child in self._children:
      child.showTree(indent + 2)

  #
  # __setattr__
  #
  # This is called when trying to write something inside a trigger object
  # namespace.  It checks to see if the var name is already linked to a
  # NamespaceElement based object and calls that objects _triggerSet if it
  # exists.
  #
  # Example: form.block1.entry1 = "foo"
  #
  # The __setattr__ will execute at the block1 and call the functions that
  # are part of the entry1 object to set it's value
  #
  def __setattr__(self, name, value):
    if self.__dict__.has_key(name) and \
       isinstance(self.__dict__[name], NamespaceElement):
     if isinstance(self.__dict__[name], NamespaceElement):
      if self.__dict__[name]._triggerSet:
        self.__dict__[name]._triggerSet(value)
      else:
        GDebug.printMesg(1,'Trigger attempting to reset a form object')
    else:
      self.__dict__[name] = value

  #
  # __getattr__
  #
  # Only needed to return the NamespaceElementProperties
  # object
  #
  def __getattr__(self,name):
    if name == '__properties__':
        return self._triggerProperties
    else:
#      GDebug.printMesg(1,"AttributeError: %s" % name)
#      print self.__dict__
      raise AttributeError, '%s' % (name)

  #
  # __str__
  #
  # This executes at a different level than the __setattr__
  # While __setattr__ is executed in the parent object to protect
  # it's namespace object links, this routine is called by the
  # object referenced
  #
  # Example: foo = form.block1.entry1
  #
  # This __str__ would execute against the entry1 object
  #
  def __str__(self):
    if self._triggerGet:
      return str(self._triggerGet())
    else:
      return ""

  #
  # __str__
  #
  # This executes at a different level than the __setattr__
  # While __setattr__ is executed in the parent object to protect
  # it's namespace object links, this routine is called by the
  # object referenced
  #
  # Example: foo = int(form.block1.entry1)
  #
  # This __int__ would execute against the entry1 object
  #
  def __int__(self):
    if self._triggerGet:
      return int(self._triggerGet())
    else:
      return 0

  #
  # __cmp__
  #
  # Forces the system to compare the string values of
  # NamespaceElement objects and not their instances
  #
  def __cmp__(self,other):
    selfvalue = "%s" % self
    othervalue = "%s" % other

    if selfvalue == othervalue:
      return 0
    elif selfvalue < othervalue:
      return -1
    else:
      return 1


  #
  # __nonzero__
  #
  # Needed to make __len__ function below play nice
  #
  def __nonzero__(self):
    return 1

  #
  # __len__
  #
  # Implements len() support
  #
  def __len__(self):
    string =  "%s" % self
    return len(string)

  #
  # __getitem__
  #
  # implements support for sub selections of strings
  #
  # example: block1.fieldname[0:4]
  #
  def __getitem__(self, key) :
    string = self.__str__()
    return string[key.start:key.stop]


  #
  # __iter__
  #
  # Python iterator support
  #
  def __iter__(self):
     return iter(self._object)

#
# NamespaceFunction
#
# Accessor class for functions that are made available in the trigger
# namespace
#
class NamespaceFunction:
  def __init__(self, name, functionLink):
    self._name = name
    self._objectFunction = functionLink

  def __call__(self, *args):
    return self._objectFunction(*args)

#
# NamespaceElementProperties
#
# Accessor class for properties that are made available in this an
# object's __properties__ namespace
#
class NamespaceElementProperties:
  def __init__(self):
    self._properties = {}

  def addProperty(self,name, getFunc, setFunc):
    self._properties[name] = {'get':getFunc,
                              'set':setFunc,
                              }

  def __setattr__(self, name, value):
    # Hack to ensure that self._properties exists
    if not self.__dict__.has_key('_properties'):
      self.__dict__['_properties'] = {}

    if self._properties.has_key(name):
      # If none the it's readonly
      if self._properties[name]['set']:
        self._properties[name]['set'](value)
      else:
        GDebug.printMesg(1,'Attempt to set readonly property :%s' %(name))
    else:
      self.__dict__[name] = value

  def __getattr__(self,name):
    if self.__dict__['_properties'].has_key(name):
      return self._properties[name]['get']()
    else:
      raise AttributeError


