/*
**  ConsoleWindowController.m
**
**  Copyright (c) 2001, 2002, 2003
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "ConsoleWindowController.h"

#include "Constants.h"
#include "GNUMail+TaskManager.h"
#include "MailboxManagerController.h"
#include "Task.h"
#include "TaskManager.h"

#ifndef MACOSX
#include "ConsoleWindow.h"
#endif

#include <Pantomime/Message.h>
#include <Pantomime/URLName.h>

static ConsoleWindowController *singleInstance = nil;

//
//
//
@implementation ConsoleWindowController

- (id) initWithWindowNibName: (NSString *) windowNibName
{
#ifdef MACOSX
  
  self = [super initWithWindowNibName: windowNibName];
  
#else
  ConsoleWindow *aConsoleWindow;

  aConsoleWindow = [[ConsoleWindow alloc] initWithContentRect: NSMakeRect(100,100,500,280)
					  styleMask: NSTitledWindowMask|NSClosableWindowMask|NSResizableWindowMask
					  backing: NSBackingStoreBuffered
					  defer: NO];
  
  self = [super initWithWindow: aConsoleWindow];
  
  [aConsoleWindow layoutWindow];
  [aConsoleWindow setDelegate: self];

  // We link our outlets
  icon = aConsoleWindow->icon;
  tasksTableView = aConsoleWindow->tasksTableView;
  messagesTableView = aConsoleWindow->messagesTableView;
  currentTaskBox = aConsoleWindow->currentTaskBox;
  
  RELEASE(aConsoleWindow);
#endif

  [[self window] setTitle: _(@"GNUMail.app Console")];
  
  // We finally set our autosave window frame name and restore the one from the user's defaults.
  [[self window] setFrameAutosaveName: @"ConsoleWindow"];
  [[self window] setFrameUsingName: @"ConsoleWindow"];

  // We set our initial box's title.
  [currentTaskBox setTitle: _(@"No Tasks")];

  return self;
}


//
//
//
- (void) dealloc
{
  NSDebugLog(@"ConsoleWindowController: -dealloc");

#ifdef MACOSX
  [tasksTableView setDataSource: nil];
  [messagesTableView setDataSource: nil];
#endif

  RELEASE(allMessages);

  [super dealloc];
}


//
// action methods
//
- (IBAction) deleteClicked: (id) sender
{
  int aRow;
     
  NSDebugLog(@"Delete clicked.");

  aRow = [tasksTableView selectedRow];

  if ( aRow >= 0 )
    {
      Task *aTask;
      
      // No need to call reloadData here since it's called in -removeTask.
      aTask = [[[NSApp delegate] allTasks] objectAtIndex: aRow];
      
      if ( [aTask running] )
	{
	  NSRunInformationalAlertPanel(_(@"Delete error!"),
				       _(@"You can't delete a running task. Stop it first."),
				       _(@"OK"),
				       NULL, 
				       NULL,
				       NULL);
	  return;
	}

      [[NSApp delegate] removeTask: aTask];
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) runNowClicked: (id) sender
{
  int aRow;
     
  NSDebugLog(@"Run now clicked.");

  aRow = [tasksTableView selectedRow];

  if ( aRow >= 0 )
    {
      Task *aTask;
      
      aTask = [[[NSApp delegate] allTasks] objectAtIndex: aRow];
      
      if ( [aTask running] )
	{
	  NSRunInformationalAlertPanel(_(@"Run error!"),
				       _(@"The task is already running!"),
				       _(@"OK"),
				       NULL, 
				       NULL,
				       NULL);
	  return;
	}

      [aTask setDate: [NSDate date]];
      [tasksTableView setNeedsDisplay: YES];
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) saveClicked: (id) sender
{
  int aRow;
     
  NSDebugLog(@"Save in Drafts clicked.");

  aRow = [tasksTableView selectedRow];

  if ( aRow >= 0 )
    {
      URLName *theURLName;
      NSData *aData;
      Task *aTask;
      
      aTask = [[[NSApp delegate] allTasks] objectAtIndex: aRow];

      if ( [aTask running] )
	{
	  NSRunInformationalAlertPanel(_(@"Suspend error!"),
				       _(@"You can't save the message in Drafts if the task is running. Stop it first."),
				       _(@"OK"),
				       NULL, 
				       NULL,
				       NULL);
	  return;
	}

      // We finally get our URLName object.
      theURLName = [[URLName alloc] initWithString: [[[[[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"] objectForKey: [aTask key]] 
						       objectForKey: @"MAILBOXES"] objectForKey: @"DRAFTSFOLDERNAME"]
				    path: [[NSUserDefaults standardUserDefaults] objectForKey: @"LOCALMAILDIR"]];
      
      if ( [[aTask message] respondsToSelector: @selector(isEqualToData:)] )
	{
	  aData = [aTask message];
	}
      else
	{
	  aData = [[aTask message] dataValue];
	}

      [[MailboxManagerController singleInstance] addMessage: aData
						 toFolder: theURLName];
      RELEASE(theURLName);
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) stopClicked: (id) sender
{
  NSDebugLog(@"Stop clicked.");
  [[TaskManager singleInstance] stopCurrentTask];
}


//
//
//
- (IBAction) suspendClicked: (id) sender
{
  int aRow;
     
  NSDebugLog(@"Suspend clicked.");

  aRow = [tasksTableView selectedRow];

  if ( aRow >= 0 )
    {
      Task *aTask;
      
      aTask = [[[NSApp delegate] allTasks] objectAtIndex: aRow];

      if ( [aTask running] )
	{
	  NSRunInformationalAlertPanel(_(@"Suspend error!"),
				       _(@"You can't suspend a running task. Stop it first."),
				       _(@"OK"),
				       NULL, 
				       NULL,
				       NULL);
	  return;
	}

      [aTask setDate: [NSDate distantFuture]];
      [tasksTableView setNeedsDisplay: YES];
    }
  else
    {
      NSBeep();
    }
}


//
// delegate methods
//
- (void) windowWillClose: (NSNotification *) theNotification
{
  // Do nothing
}


//
//
//
- (void) windowDidLoad
{
  allMessages = [[NSMutableArray alloc] init];

#ifdef MACOSX
  [tasksTableView reloadData];
  [messagesTableView reloadData];
#endif
}


//
// Data Source methods
//
- (int) numberOfRowsInTableView: (NSTableView *)aTableView
{
  if ( aTableView == tasksTableView )
    {
      return [[[NSApp delegate] allTasks] count];
    }
  else
    {
      return [allMessages count];
    }
}


//
//
//
- (id)           tableView: (NSTableView *) aTableView
 objectValueForTableColumn: (NSTableColumn *) aTableColumn 
		       row: (int) rowIndex
{
  if ( aTableView == tasksTableView )
    {
      NSString *aString;
      Task *aTask;
      
      aTask = [[[NSApp delegate] allTasks] objectAtIndex: rowIndex];
      
      if ( [[[aTableColumn headerCell] stringValue] isEqual: _(@"Date")] )
	{
	  aString = [[aTask date] descriptionWithCalendarFormat: @"%H:%M:%S"
				  timeZone: nil
				  locale: nil];
	  return aString;
	}
      else
	{
	  if ( [aTask op] == SEND_SENDMAIL || [aTask op] == SEND_SMTP )
	    {
	      NSUserDefaults *aUserDefaults;
	      NSDictionary *allValues;
	      NSString *s1, *s2;
	      BOOL aBOOL;
	      
	      aUserDefaults = [NSUserDefaults standardUserDefaults];
	      allValues = [[[aUserDefaults objectForKey: @"ACCOUNTS"] objectForKey: [aTask key]]
			    objectForKey: @"SEND"];
	      
	      aBOOL = [[allValues objectForKey: @"TRANSPORT_METHOD"] intValue] == TRANSPORT_SMTP;
	      s1 = (aBOOL ? _(@"SMTP") : _(@"Mailer"));
	      s2 = (aBOOL ? [allValues objectForKey: @"SMTP_HOST"] : [allValues objectForKey: @"MAILER_PATH"]);
	      
	      if ( [aTask running] )
		{
		  aString = [NSString stringWithFormat: _(@"Sending - %@ - %@..."), s1, s2];
		}
	      else
		{
		  aString = [NSString stringWithFormat: _(@"In queue - Send - %@ - %@"), s1, s2];
		}
	    }
	  else
	    {
	      if ( [aTask running] )
		{
		  if ( [aTask op] == RECEIVE_POP3 )
		    {
		      aString = [NSString stringWithFormat: _(@"Receiving - POP3 - %@..."), [aTask key]];
		    }
		  else
		    {
		      // For now, we only have one subOp.
		      aString = [NSString stringWithFormat: _(@"Receiving - IMAP/Mailboxes status - %@..."), [aTask key]];
		    }
		}
	      else
		{
		  aString = [NSString stringWithFormat: _(@"In queue - Receive - %@"), [aTask key]];
		}
	    }
	  
	  return aString;
	}
    }
  else
    {
      ConsoleMessage *aMessage;
      
      aMessage = [allMessages objectAtIndex: rowIndex];
      
      if ( [[[aTableColumn headerCell] stringValue] isEqual: _(@"Date")] )
        {
          return [aMessage->date descriptionWithCalendarFormat: _(@"%H:%M:%S.%F")
			  timeZone: [aMessage->date timeZone]
			  locale: nil];
        }
      else
        {
          return aMessage->message;
        }
    }
  return nil;
}


//
// access / mutation method
//
- (NSTableView *) tasksTableView
{
  return tasksTableView;
}


//
//
//
- (NSTimer *) animation
{
  return animation;
}


//
// Other methods
//
- (void) addConsoleMessage: (NSString *) theString
{
  ConsoleMessage *aMessage;
  
  aMessage = [[ConsoleMessage alloc] initWithMessage: theString];
  
  [allMessages insertObject: aMessage
    	       atIndex: 0];
  RELEASE(aMessage);
  
  // We never keep more than 25 object in our array
  if ( [allMessages count] > 25 )
    {
      [allMessages removeLastObject];
    }
  
  [messagesTableView reloadData];
}


//
//
//
- (void) startAnimation
{
  // We don't start the animation if it's already started
  if ( !animation )
    {
      int i;

#ifdef MACOSX
      [progressIndicator startAnimation: self];
#endif

      animation_index = 1;
      animation = [NSTimer scheduledTimerWithTimeInterval: 0.1
			   target: self
			   selector: @selector(_updateAnimatedIcon:)
			   userInfo: nil
			   repeats: YES];
      RETAIN(animation);

      for (i = 0; i < [[GNUMail allMailWindows] count]; i++)
	{
	  [[[[GNUMail allMailWindows] objectAtIndex: i] windowController] startAnimation];
	}
    }
}


//
//
//
- (void) stopAnimation
{
  // We don't stop the animation if it has never been running!
  if ( animation )
    {
      int i;

      [animation invalidate];
      DESTROY(animation);

#ifdef MACOSX
      [progressIndicator stopAnimation: self];
#else      
      [icon setImage: nil];
#endif

      for (i = 0; i < [[GNUMail allMailWindows] count]; i++)
	{
	  [[[[GNUMail allMailWindows] objectAtIndex: i] windowController] stopAnimation];
	}
    }
}


//
// class methods
//
+ (id) singleInstance
{
  if ( singleInstance == nil )
    {
      singleInstance = [[ConsoleWindowController alloc] initWithWindowNibName: @"ConsoleWindow"];
    }

  return singleInstance;
}

@end


//
// ConsoleWindowController private implementation
//
@implementation ConsoleWindowController (Private)

- (BOOL) _noTaskInQueueIsRunning
{
  NSArray *allTasks;
  int i;

  allTasks = [[NSApp delegate] allTasks];
  
  for (i = 0; i < [allTasks count]; i++)
    {
      if ( [[allTasks objectAtIndex: i] running] || [[allTasks objectAtIndex: i] immediate] )
	{
	  return NO;
	}
    }

  return YES;
}


//
//
//
- (void) _updateAnimatedIcon: (id) sender
{
  if ( !animation || [self _noTaskInQueueIsRunning] )
    {
      [self stopAnimation];
      return;
    }

#ifndef MACOSX
  if (animation_index == 9)
    {
      animation_index = 1;
    }
  
  [icon setImage: [NSImage imageNamed: [NSString stringWithFormat: @"anim-logo-%d.tiff", animation_index]]];
  
  animation_index += 1;
#endif
}

@end


//
// 
//
@implementation ConsoleMessage

- (id) initWithMessage: (NSString *) theMessage
{
  self = [super init];

  message = RETAIN(theMessage);
  date = RETAIN([NSCalendarDate calendarDate]);

  return self;
}

- (void) dealloc
{
  RELEASE(message);
  RELEASE(date);

  [super dealloc];
}

@end
