/*
**  MailHeaderCell.m
**
**  Copyright (c) 2002, 2003
**
**  Author: Nicolas Roard <nicolas@roard.com>
**          Ludovic Marcotte <ludovic@Sophos.ca>
**          Francis Lachapelle <francis@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "MailHeaderCell.h"

#include "GNUMail.h"
#include "Constants.h"
#include "NSUserDefaults+Extensions.h"

//
// Simple category of NSBezierPath to easily draw a rectangle with
// custom rounded corners.
//
@implementation NSBezierPath (RoundRect)

- (void) appendBezierPathWithRoundedRectangle: (NSRect) aRect
                                   withRadius: (float) radius
{
  NSPoint topMid = NSMakePoint(NSMidX(aRect), NSMaxY(aRect));
  NSPoint topLeft = NSMakePoint(NSMinX(aRect), NSMaxY(aRect));
  NSPoint topRight = NSMakePoint(NSMaxX(aRect), NSMaxY(aRect));
  NSPoint bottomRight = NSMakePoint(NSMaxX(aRect), NSMinY(aRect));
  
  [self moveToPoint: topMid];
  [self appendBezierPathWithArcFromPoint: topLeft
	toPoint: aRect.origin
	radius: radius];
  [self appendBezierPathWithArcFromPoint: aRect.origin
	toPoint: bottomRight
	radius: radius];
  [self appendBezierPathWithArcFromPoint: bottomRight
	toPoint: topRight
	radius: radius];
  [self appendBezierPathWithArcFromPoint: topRight
	toPoint: topLeft
	radius: radius];
  [self closePath];
}

@end

@implementation MailHeaderCell

//
//
//
- (id) init 
{
  self = [super init];

  width = 0;

  color = [[NSUserDefaults standardUserDefaults] colorForKey: @"MAILHEADERCELL_COLOR"];

  if ( !color )
    {
      color = [NSColor colorWithCalibratedWhite: 0.9 alpha: 1.0];
    }
  
  RETAIN(color);

  allViews = [[NSMutableArray alloc] init];

  return self;
}


//
//
//
- (void) dealloc
{
  RELEASE(originalAttributedString);
  RELEASE(color);

  RELEASE(allViews);

  [super dealloc];
}



//
//
//
- (float) height
{
  return [[self attributedStringValue] size].height + 20;
}


//
//
//
- (float) width
{
  return width;
}


//
//
//
- (void) setWidth: (float) theWidth
{
  width = theWidth;
}


//
//
//
- (NSSize) cellSize
{
  return NSMakeSize (width, [self height]);
}


//
// other methods
//
- (void) addView: (id) theView
{
  if ( theView )
    {
      [allViews addObject: theView];
    }
}


//
//
//
- (BOOL) containsView: (id) theView
{
  return [allViews containsObject: theView];
}


//
// FIXME - Optimize.
//
- (void) resize: (id) sender
{
  NSRect aRect;
  
  aRect = [[[[GNUMail lastMailWindowOnTop] windowController] textView] frame];
  
  //if ( !NSEqualSizes(aRect.size, NSMakeSize(width, aRect.size.height)) )
  //  {
  [self setWidth: aRect.size.width];
  [super setAttributedStringValue: [self _fold]];
  //  }
}


//
//
//
- (void) setAttributedStringValue: (NSAttributedString *) theAttributedString
{
  originalAttributedString = RETAIN(theAttributedString);
  [super setAttributedStringValue: theAttributedString];
}


//
//
//
- (void) drawWithFrame: (NSRect) theFrame
		inView: (NSView *) theView
{  
  NSBezierPath *aRoundRect;
  NSView *aView;
  NSSize aSize;
  
  float current_x, current_y;
  int i;
  
  if ( ![theView window] )
    {
      return;
    }
  
  // We fill our header cell
#ifdef MACOSX
  theFrame.origin.y += 4;
#endif
  theFrame.size.width -= 10;
  [color set];
  aRoundRect = [NSBezierPath bezierPath];
  [aRoundRect appendBezierPathWithRoundedRectangle: theFrame
                                        withRadius: 8.0];
  [aRoundRect fill];
  
  current_x = theFrame.origin.x + theFrame.size.width;

  for (i = 0; i < [allViews count]; i++)
    {
      aView = [allViews objectAtIndex: i];

      // If our bundle doesn't provide an image, we draw it's view
      if ( ![aView respondsToSelector: @selector(image)] )
	{
	  if ( NSEqualRects([aView frame], NSZeroRect) )
	    {
	      continue;
	    }

	  aSize = [aView frame].size;
	  current_x = current_x - aSize.width - 10;
	  current_y = theFrame.origin.y + aSize.height + (theFrame.size.height - aSize.height)/2;
	  [aView drawRect: NSMakeRect(current_x, current_y, aSize.width, aSize.height)];
	}
      else
	{
	  NSImage *aImage;

	  aImage = [(id)aView image];
	  
	  if ( !aImage )
	    {
	      continue;
	    }
	  
	  aSize = [aImage size];
	  
	  current_x = current_x - aSize.width - 10;
	  current_y = theFrame.origin.y + aSize.height + (theFrame.size.height - aSize.height)/2;
	  
	  [aImage compositeToPoint: NSMakePoint(current_x, current_y)
		  operation: NSCompositeSourceAtop];
	}
    }

  // We adjust our frame and we draw our attributed string
  theFrame.origin.x += 8; 
  theFrame.size.width -= 16;
  theFrame.origin.y += 10;
  theFrame.size.height -= 20;

  [[self attributedStringValue] drawInRect: theFrame];
}


@end



//
// Private interface
//
@implementation MailHeaderCell (Private)

- (NSAttributedString *) _fold
{
  NSMutableAttributedString *aMutableAttributedString;
  NSScanner *aScanner;

  int previousLocation, currentLocation;
  float currentLineWidth;
  
  if ( !originalAttributedString )
    {
      return nil;
    }

  aMutableAttributedString = [[NSMutableAttributedString alloc] init];

  aScanner = [NSScanner scannerWithString: [originalAttributedString string]];
  
  currentLocation = previousLocation = currentLineWidth = 0;

  // We tokenize our headers...
  while ( [aScanner scanUpToCharactersFromSet: [NSCharacterSet whitespaceAndNewlineCharacterSet]
		    intoString: NULL] )
    {
      NSString *aString;
      
      NSRange aRange;
      float wordWidth;
 
      currentLocation = [aScanner scanLocation];
      wordWidth = 0;

      aRange =  NSMakeRange(previousLocation, currentLocation - previousLocation);
      aString = [[aScanner string] substringWithRange: aRange];
 
      
      if ( [[aScanner string] length] > NSMaxRange(aRange) &&
	   [[aScanner string] characterAtIndex: NSMaxRange(aRange)] == '\n' &&
	   (currentLineWidth + wordWidth) <= ([self width] - 100) )
	{
	  [aMutableAttributedString appendAttributedString:
				      [originalAttributedString attributedSubstringFromRange: aRange]];
	  currentLineWidth = 0;
	  previousLocation = currentLocation;
	  continue;
	}

      wordWidth = [[originalAttributedString attributedSubstringFromRange: aRange] size].width;

      if ( (currentLineWidth + wordWidth) > ([self width] - 100) )
	{
	  [aMutableAttributedString appendAttributedString: 
	  			      AUTORELEASE([[NSAttributedString alloc] initWithString: @"\n"])];
	  [aMutableAttributedString appendAttributedString:
				      [originalAttributedString attributedSubstringFromRange: aRange]];
	  
	  currentLineWidth = wordWidth;
	}
      else
	{
	  [aMutableAttributedString appendAttributedString:
				      [originalAttributedString attributedSubstringFromRange: aRange]];
	  currentLineWidth += wordWidth;
	}
      
      previousLocation = currentLocation;
    }


  return AUTORELEASE(aMutableAttributedString);
}

@end
